/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * $Log: RicohMP6200.cc,v $
 * Revision 1.2  1998/10/06 18:10:24  mueller
 * Disabled setting of Catalogue Number.
 *
 * Revision 1.1  1998/10/03 15:04:37  mueller
 * Initial revision
 *
 */

/* Driver for the Ricoh MP6200 drive. It's mainly SCSI-3/mmc compatible but
 * disk-at-once writing is done with the Philips CDD2x00 commands.
 */

static char rcsid[] = "$Id: RicohMP6200.cc,v 1.2 1998/10/06 18:10:24 mueller Exp $";

#include <config.h>

#include <string.h>
#include <assert.h>

#include "RicohMP6200.h"
#include "SubChannel.h"
#include "Toc.h"
#include "util.h"

RicohMP6200::RicohMP6200(ScsiIf *scsiIf, Toc *toc)
  : GenericMMC(scsiIf, toc), CDD2600Base(this)
{
  driverName_ = "Ricoh MP6200 - Version 0.1(alpha)";
  
  simulate_ = 1;
}

RicohMP6200::~RicohMP6200()
{
}

// static constructor
CdrDriver *RicohMP6200::instance(ScsiIf *scsiIf, Toc *toc)
{
  return new RicohMP6200(scsiIf, toc);
}


// Sets write parameters via mode page 0x05.
// return: 0: OK
//         1: scsi command failed
int RicohMP6200::setWriteParameters()
{
  unsigned char mp[0x38];

  if (getModePage(5/*write parameters mode page*/, mp, 0x38) != 0) {
    message(-2, "Cannot retrieve write parameters mode page.");
    return 1;
  }

  mp[0] &= 0x7f; // clear PS flag

  mp[2] &= 0xef;

  if (simulate_) {
    mp[2] |= 1 << 4; // test write
  }

  if (setModePage(mp) != 0) {
    message(-2, "Cannot set write parameters mode page.");
    return 1;
  }

  return 0;
}


int RicohMP6200::initDao()
{
  long n;
  blockLength_ = AUDIO_BLOCK_LEN;
  blocksPerWrite_ = scsiIf_->maxDataLen() / blockLength_;

  assert(blocksPerWrite_ > 0);

  if (modeSelectBlockSize(blockLength_) != 0 ||
      readSessionInfo(&leadInLen_, &leadOutLen_, 1) != 0) {
    return 1;
  }

  //modeSelectCatalog(toc_);
  setWriteParameters();

  // allocate buffer for write zeros
  n = blocksPerWrite_ * blockLength_;
  delete[] zeroBuffer_;
  zeroBuffer_ = new char[n];
  memset(zeroBuffer_, 0, n);

  return 0;
}

int RicohMP6200::startDao()
{
  long lba = 0;
  if (writeSession(toc_, multiSession_) != 0) {
    return 1;
  }

  message(0, "Writing lead-in and gap...");

  // write lead-in
  if (writeZeros(lba, leadInLen_ + 150/*2 second gap*/) != 0) {
    flushCache();
    return 1;
  }

  message(0, "");

  return 0;
}

int RicohMP6200::finishDao()
{
  long lba = 0;

  message(0, "Writing lead-out...");

  // write lead-out
  if (writeZeros(lba, leadOutLen_) != 0) {
    flushCache();
    return 1;
  }

  message(0, "\nFlushing cache...");
  
  if (flushCache() != 0) {
    return 1;
  }

  message(0, "");

  delete[] zeroBuffer_, zeroBuffer_ = NULL;

  return 0;
}

// Writes data to target, the block length depends on the actual writing mode
// and is stored internally. 'len' is number of blocks to write.
// 'lba' specifies the next logical block address for writing and is updated
// by this function but not used for writing
// return: 0: OK
//         1: scsi command failed
int RicohMP6200::writeData(long &lba, const char *buf, long len)
{
  assert(blocksPerWrite_ > 0);
  assert(blockLength_ > 0);
  int nwritten = 0;
  int writeLen = 0;
  unsigned char cmd[10];

  memset(cmd, 0, 10);
  cmd[0] = 0x2a; // WRITE1
  
  while (len > 0) {
    writeLen = (len > blocksPerWrite_ ? blocksPerWrite_ : len);

    cmd[7] = writeLen >> 8;
    cmd[8] = writeLen & 0xff;

    if (sendCmd(cmd, 10, (unsigned char *)(buf + (nwritten * blockLength_)),
		writeLen * blockLength_, NULL, 0) != 0) {
      message(-2, "Write data failed.");
      return 1;
    }

    lba += writeLen;

    len -= writeLen;
    nwritten += writeLen;
  }
      
  return 0;
}
