/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * $Log: dao.cc,v $
 * Revision 1.9  1998/10/03 14:33:46  mueller
 * Applied patch from Bjoern Fischer <bfischer@Techfak.Uni-Bielefeld.DE>:
 * Cosmetic changes and correction of real time scheduling selection.
 *
 * Revision 1.8  1998/09/22 19:14:29  mueller
 * Added locking of memory pages.
 *
 * Revision 1.7  1998/09/21 19:35:51  mueller
 * Added real time scheduling. Submitted by Bjoern Fischer <bfischer@Techfak.Uni-Bielefeld.DE>.
 *
 * Revision 1.6  1998/09/06 13:34:22  mueller
 * Use 'message()' for printing messages.
 *
 * Revision 1.5  1998/08/30 19:21:15  mueller
 * Rearranged the code.
 *
 * Revision 1.4  1998/08/15 20:47:16  mueller
 * Added support for GenericMMC driver.
 *
 */

static char rcsid[] = "$Id: dao.cc,v 1.9 1998/10/03 14:33:46 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <malloc.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/shm.h>

#ifdef HAVE_SYS_MMAN_H
#include <sys/mman.h>
#endif

/* Select POSIX scheduler interface for real time scheduling if possible */
#if (defined HAVE_SCHED_GETPARAM) && (defined HAVE_SCHED_GET_PRIORITY_MAX) && (defined HAVE_SCHED_SETSCHEDULER) && (!defined LINUX_QNX_SCHEDULING)
#define POSIX_SCHEDULING
#endif

#ifdef LINUX_QNX_SCHEDULING

#define SCHED_OTHER     0
#define SCHED_FIFO      1
#define SCHED_RR        2

struct sched_param {
  unsigned int priority;
  int fork_penalty_threshold;
  unsigned int starvation_threshold;
  unsigned int ts_max;
  unsigned int run_q, run_q_min, run_q_max;
};
extern "C" int sched_setparam __P((pid_t __pid, const struct sched_param *__param));
extern "C" int sched_getparam __P((pid_t __pid, struct sched_param *__param));
extern "C" int sched_setscheduler __P((pid_t __pid, int __policy, const struct sched_param *__param));
extern "C" int sched_getscheduler __P((pid_t __pid));

#elif defined POSIX_SCHEDULING

#include <sched.h>

#endif

#include "dao.h"
#include "util.h"

// buffer size in blocks
int BUFFER_SIZE = 250;

struct Message {
  long type;
  char *buffer;
  long bufLen;
};

static long MSGLEN = sizeof(Message) - sizeof(long);

static int TERMINATE = 0;

static RETSIGTYPE terminationRequest(int sig)
{
  if (sig == SIGQUIT) 
    TERMINATE = 1;

  signal(sig, terminationRequest);
}

static void writeSlave(CdrDriver *cdr, int testMode, long total, int msgId)
{
  long ret;
  long cnt = 0;
  long blkCount = 0;
  long lba = 0;
  long len = 0;
  char *buf = NULL;
  Message msg;
  long cntMb;
  long lastMb = 0;
  int daoStarted = 0;

  setsid(); // detach from controlling terminal

  while (1) {
    //message(3, "Slave: waiting for master.");
    do {
      if ((ret = msgrcv(msgId, (msgbuf *)&msg, MSGLEN, 1, 0)) < 0 &&
	  errno != EINTR) {
	message(-2, "Slave: msgrcv failed: %s", strerror(errno));
	exit(1);
      }
    } while (ret < 0 && errno == EINTR);

    len = msg.bufLen;
    buf = msg.buffer;

    if (len == 0) {
      // termination request
      message(0, "");
      if (testMode)
	message(0, "Read %ld blocks.", blkCount);
      else
	message(0, "Wrote %ld blocks.", blkCount);

      if (!testMode && daoStarted) {
	if (cdr->finishDao() != 0)
	  exit(1);
      }
      exit(0);
    }

    if (!testMode && !daoStarted) {
      daoStarted = 1;

      if (cdr->startDao() != 0)
	exit(1);
      
      message(0, "Writing audio data...");
    }

    cnt += len * AUDIO_BLOCK_LEN;
    blkCount += len;

    //message(3, "Slave: writing buffer %p (%ld).", buf, len);

    msg.type = 2;
    msg.buffer = NULL;

    if (!testMode) {
      if (cdr->writeData(lba, buf, len) != 0) {
	message(-2, "Slave: write of audio data failed.");

	cdr->flushCache();

	msg.bufLen = 1;	// notify master about error
      }
      else {
	cntMb = cnt >> 20;
	if (cntMb > lastMb) {
	  message(0, "Wrote %ld of %ld MB.\r", cnt >> 20, total >> 20);
	  fflush(stdout);
	  lastMb = cntMb;
	}

	msg.bufLen = 0; // indicates OK
      }
    }
    else {
      message(0, "Read %ld of %ld MB.\r", cnt >> 20, total >> 20);
      fflush(stdout);

      msg.bufLen = 0; // indicates OK
    }

    do {
      if ((ret = msgsnd(msgId, (msgbuf*)&msg, MSGLEN, 0)) < 0 && 
	  errno != EINTR) {
	message(-2, "Slave: msgsnd failed: %s", strerror(errno));
	exit(3);
      }
    } while (ret < 0 && errno == EINTR);
  }
}



static int reader(const Toc *toc, CdrDriver *cdr, int swap, int testMode,
		  int msgId, char *sbuf1, char *sbuf2)
{
  long length = toc->length().lba();
  char *buf = NULL;
  Message msg;
  long n;
  int status;

  if (cdr != NULL && cdr->bigEndianSamples() == 0) {
    // swap samples for little endian recorders
    swap = !swap;
  }
  message(2, "Swap: %d", swap);

  n = (length > BUFFER_SIZE ? BUFFER_SIZE : length);
  buf = sbuf1;

  //message(3, "Master: reading buffer %p.", buf);
  if (toc->readData(buf, n) != n) {
    message(-2, "Read of audio data failed.");
    return 1;
  }

  if (swap) {
    swapSamples((Sample *)buf, n * SAMPLES_PER_BLOCK);
  }

  if (TERMINATE) {
    return 3;
  }

  do {
    // notify slave that it can write buffer 'bufSel'
    msg.type = 1;
    msg.bufLen = n;
    msg.buffer = buf;

    if (msgsnd(msgId, (msgbuf*)&msg, MSGLEN, 0) < 0) {
      message(-2, "msgsnd failed: %s", strerror(errno));
      return 4;
    }

    length -= n;
    n = (length > BUFFER_SIZE ? BUFFER_SIZE : length);

    // switch to other buffer
    buf = (buf == sbuf1 ? sbuf2 : sbuf1);

    //message(3, "Master: reading buffer %p.", buf);
    if (toc->readData(buf, n) != n) {
      message(-2, "Read of audio data failed.");
      return 1;
    }

    if (swap) {
      swapSamples((Sample *)buf, n * SAMPLES_PER_BLOCK);
    }

    // Check if child has terminated
    if (wait3(&status, WNOHANG, NULL) > 0) {
      message(-2, "Child process terminated.");
      return 4;
    }
    
    // wait for slave to finish writing of previous buffer
    //message(3, "Master: waiting for slave.");
    do {
      if ((status = msgrcv(msgId, (msgbuf*)&msg, MSGLEN, 2, 0)) < 0) {
	if (errno == EINTR) {
	  if (wait3(&status, WNOHANG, NULL) > 0) {
	    message(-2, "Child process terminated.");
	    return 4;
	  }
	}
	else {
	  message(-2, "msgrcv failed: %s", strerror(errno));
	  return 4;
	}
      }
    } while (status < 0 && errno == EINTR);


    if (msg.bufLen != 0) {
      message(-2, "Writing failed - buffer under run?");
      return 5;
    }
  } while (length > 0 && !TERMINATE);

  if (TERMINATE) {
    message(-1, "Writing/simulation/read-test aborted on user request.");
    if (!testMode)
      cdr->flushCache();
    return 3;
  }

  // send finish message
  msg.type = 1;
  msg.bufLen = 0;
  msg.buffer = NULL;

  do {
    if ((status = msgsnd(msgId, (msgbuf*)&msg, MSGLEN, 0)) < 0) {
      if (errno != EINTR) {
	message(-2, "msgsnd failed: %s", strerror(errno));
	return 4;
      }
    }
  } while (status < 0 && errno == EINTR);

  return 0;
}


int writeDiskAtOnce(const Toc *toc, CdrDriver *cdr, int swap, int testMode)
{
  int bufSize = BUFFER_SIZE * AUDIO_BLOCK_LEN;
  long length = toc->length().lba();
  long total = length * AUDIO_BLOCK_LEN;
  int pid = 0;
  int status;
  int err = 0;
  int msgId = -1;
  int shmId1 = -1;
  int shmId2 = -1;
  char *sharedMem1 = NULL;
  char *sharedMem2 = NULL;

#if (defined POSIX_SCHEDULING) || (defined LINUX_QNX_SCHEDULING)
  struct sched_param schedp;
#endif

  TERMINATE = 0;
  //signal(SIGINT, terminationRequest);
  signal(SIGINT, SIG_IGN);
  signal(SIGCHLD, terminationRequest);
  signal(SIGQUIT, terminationRequest);
  
  if (toc->openData() != 0) {
    message(-2, "Cannot open audio data.");
    err = 1; goto fail;
  }

  if ((msgId = msgget(IPC_PRIVATE, 0600)) < 0) {
    message(-2, "Cannot create message queue: %s", strerror(errno));
    err = 1; goto fail;
  }

  if ((shmId1 = shmget(IPC_PRIVATE, bufSize, 0600)) < 0 ||
      (shmId2 = shmget(IPC_PRIVATE, bufSize, 0600)) < 0) {
    message(-2, "Cannot create shared memory segments: %s",
	    strerror(errno));
    err = 1; goto fail;
  }
  if ((sharedMem1 = (char *)shmat(shmId1, 0, 0)) <= 0 ||
      (sharedMem2 = (char *)shmat(shmId2, 0, 0)) <= 0) {
    message(-2, "Cannot get shared memory: %s", strerror(errno));
    err = 1; goto fail;
  }

  if (!testMode) {
    if (cdr->initDao() != 0) {
      err = 1; goto fail;
    }
  }

  // start writing slave process
  if ((pid = fork()) == 0) {
    // we are the new writing process
#ifdef LINUX_QNX_SCHEDULING

    if (geteuid() == 0) {
      sched_getparam (0, &schedp);
      schedp.run_q_min = schedp.run_q_max = 1;
      if (sched_setscheduler (0, SCHED_RR, &schedp) != 0) {
	message(-1, "Cannot setup real time scheduling: %s", strerror(errno));
      }
    }

#elif defined POSIX_SCHEDULING

    if (geteuid() == 0) {
      sched_getparam (0, &schedp);
      schedp.sched_priority = sched_get_priority_max (SCHED_RR) - 1;
      if (sched_setscheduler (0, SCHED_RR, &schedp) != 0) {
	message(-1, "Cannot setup real time scheduling: %s", strerror(errno));
      }
    }

#endif

#ifdef HAVE_MLOCKALL
    if (geteuid() == 0) {
      if (mlockall(MCL_CURRENT) != 0) {
	message(-1, "Cannot lock memory pages: %s", strerror(errno));
      }
    }
#endif

    writeSlave(cdr, testMode, total, msgId);
  }
  else if (pid < 0) {
    message(-2, "fork failed: %s", strerror(errno));
    err = 1; goto fail;
  }

#ifdef LINUX_QNX_SCHEDULING
  
  if (geteuid() == 0) {
    sched_getparam (0, &schedp);
    schedp.run_q_min = schedp.run_q_max = 2;
    if (sched_setscheduler (0, SCHED_RR, &schedp) != 0) {
      message(-1, "Cannot setup real time scheduling: %s", strerror(errno));
    }
    else {
      message(0, "Using Linux QNX real time scheduling.");
    }
  }
  else {
    message(-1, "No root permissions for real time scheduling.");
  }

#elif defined POSIX_SCHEDULING
  
  if (geteuid() == 0) {
    sched_getparam (0, &schedp);
    schedp.sched_priority = sched_get_priority_max (SCHED_RR) - 2;
    if (sched_setscheduler (0, SCHED_RR, &schedp) != 0) {
      message(-1, "Cannot setup real time scheduling: %s", strerror(errno));
    }
    else {
      message(0, "Using POSIX real time scheduling.");
    }
  }
  else {
    message(-1, "No root permissions for real time scheduling.");
  }
  
#endif

#ifdef HAVE_MLOCKALL
  if (geteuid() == 0) {
    if (mlockall(MCL_CURRENT) != 0) {
      message(-1, "Cannot lock memory pages: %s", strerror(errno));
    }
  }
#endif

  if (reader(toc, cdr, swap, testMode, msgId, sharedMem1, sharedMem2) == 0) {
    // collect writing process
    if (wait(&status) <= 0) {
      message(-2, "wait failed: %s", strerror(errno));
      err = 1;
    }
    else {
      pid = 0;
      if (WIFEXITED(status)) {
	if (WEXITSTATUS(status) != 0) {
	  err = 1;
	}
      }
      else {
	message(-2, "Child process terminated abnormally.");
	err = 1;
      }
    }
  }
  else {
    err = 1;
  }

fail:
#ifdef HAVE_MUNLOCKALL
  munlockall();
#endif

  toc->closeData();

  if (pid != 0) {
    kill(pid, SIGKILL);
  }

  if (shmId1 >= 0) {
    if (sharedMem1 != NULL) {
      if (shmdt(sharedMem1) != 0) {
	message(-2, "shmdt: %s", strerror(errno));
      }
    }
    if (shmctl(shmId1, IPC_RMID, NULL) != 0) {
      message(-2, "Cannot remove shared memory: %s", strerror(errno));
    }
  }

  if (shmId2 >= 0) {
    if (sharedMem2 != NULL) {
      if (shmdt(sharedMem2) != 0) {
	message(-2, "shmdt: %s", strerror(errno));
      }
    }
    if (shmctl(shmId2, IPC_RMID, NULL) != 0) {
      message(-2, "Cannot remove shared memory: %s", strerror(errno));
    }
  }

  if (msgId >= 0) {
    if (msgctl(msgId, IPC_RMID, NULL) != 0) {
      message(-2, "Cannot remove message queue: %s", strerror(errno));
    }
  }

  signal(SIGINT, SIG_DFL);
  signal(SIGCHLD, SIG_DFL);
  signal(SIGQUIT, SIG_DFL);

  return err;
}
