/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Log: main.cc,v $
 * Revision 1.14  1998/10/24 14:31:01  mueller
 * Added command 'read-cd'.
 * Added option '--fname' to specify the audio file name that is placed into
 * created toc-files.
 * Made 'rezeroUnit()' failure non fatal. Some ATAPI drives do not support
 * this SCSI command.
 * Added driver detection for Memorex drives and for HP CD-Writer+ 8100.
 * Added warning if toc exceeds capacity of CD-R.
 *
 * Revision 1.13  1998/10/03 14:34:29  mueller
 * Added inquiry string for YAMAHA CRW4260.
 * Added pausing before starting write or simulation.
 * Applied patch from Bjoern Fischer <bfischer@Techfak.Uni-Bielefeld.DE>:
 * Retries for test if drive is ready.
 *
 * Revision 1.12  1998/09/27 19:20:14  mueller
 * Added multi session mode.
 *
 * Revision 1.11  1998/09/21 19:29:08  mueller
 * Added inquiry string for YAMAHA CDR200.
 *
 * Revision 1.10  1998/09/19 19:15:43  mueller
 * Added inquiry string for PIONEER CD-ROM DR-U12X reader.
 *
 * Revision 1.9  1998/09/08 11:54:22  mueller
 * Extended disk info structure because CDD2000 does not support the
 * 'READ DISK INFO' command.
 *
 * Revision 1.8  1998/09/07 15:20:20  mueller
 * Reorganized read-toc related code.
 *
 * Revision 1.7  1998/09/06 13:34:22  mueller
 * Use 'message()' for printing messages.
 * Added inquiry string for Plextor writer.
 *
 * Revision 1.6  1998/09/02 18:49:13  mueller
 * Write the toc-file with real user and group id.
 *
 * Revision 1.5  1998/08/30 19:22:34  mueller
 * Added driver selection for Ricoh, Philips and Yamaha writers.
 * Now the disk state is checked before performing commands.
 *
 * Revision 1.4  1998/08/25 19:26:53  mueller
 * Added inquiry strings.
 * Added command disk-info.
 * Added drivers 'generic-mmc' and 'generic-mmc-raw'.
 *
 * Revision 1.3  1998/07/28 13:49:35  mueller
 * Command SHOW_TOC: Check consistency of toc-file after call to
 * 'showToc()'.
 *
 */

static char rcsid[] = "$Id: main.cc,v 1.14 1998/10/24 14:31:01 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <fstream.h>
#include <stdarg.h>

#include "util.h"
#include "Toc.h"
#include "ScsiIf.h"
#include "dao.h"
#include "CDD2600.h"
#include "PlextorReader.h"
#include "PlextorReaderScan.h"
#include "GenericMMC.h"
#include "GenericMMCraw.h"
#include "RicohMP6200.h"

enum Command { SHOW_TOC, SHOW_DATA, READ_TEST, SIMULATE, WRITE, READ_TOC,
               DISK_INFO, READ_CD };

static const char *PRGNAME = NULL;
static const char *SCSI_DEVICE = "/dev/cdrecorder";
static const char *TOC_FILE = NULL;
static const char *DRIVER_ID = NULL;
static const char *TOC_AUDIO_FILENAME = "data.wav";
static int WRITING_SPEED = -1;
static int EJECT = 0;
static int SWAP = 0;
static int MULTI_SESSION = 0;
static Command COMMAND;
static int VERBOSE = 0; // verbose level

typedef CdrDriver *(*CdrDriverConstructor)(ScsiIf *, Toc *);

struct DriverTable {
  char *driverId;
  char *vendor;
  char *model;
  CdrDriverConstructor constructor;
};

DriverTable DRIVER_TABLE[] = {
  { "cdd2600",         "PHILIPS", "CDD2600", &CDD2600::instance },
  { "cdd2600",         "PHILIPS", "CDD2000", &CDD2600::instance },
  { "cdd2600",         "IMS",     "CDD2000", &CDD2600::instance },
  { "cdd2600",         "HP",      "CD-Writer 6020", &CDD2600::instance },

  { "generic-mmc",     "HP",      "CD-Writer+ 8100", &GenericMMC::instance },
  { "generic-mmc",     "PLEXTOR", "CD-R   PX-R412", &GenericMMC::instance },
  { "generic-mmc",     "YAMAHA",  "CDR200", &GenericMMC::instance },
  { "generic-mmc",     "YAMAHA",  "CDR400", &GenericMMC::instance },
  { "generic-mmc",     "YAMAHA",  "CRW4260", &GenericMMC::instance },

  { "generic-mmc-raw", "MEMOREX", "CD-233E", &GenericMMCraw::instance },
  { "generic-mmc-raw", "MEMOREX", "CR-622", &GenericMMCraw::instance },
  { "generic-mmc-raw", "MEMOREX", "CRW-1662", &GenericMMCraw::instance },
  { "generic-mmc-raw", "PHILIPS", "CDD3600", &GenericMMCraw::instance },
  { "generic-mmc-raw", "PHILIPS", "CDD3610", &GenericMMCraw::instance },

  { "plextor",         "PIONEER", "CD-ROM DR-U16", &PlextorReader::instance },
  { "plextor",         "PIONEER", "CD-ROM DR-U12", &PlextorReader::instance },
  { "plextor",         "PLEXTOR", "CD-ROM", &PlextorReader::instance },

  { "plextor-scan",    "TEAC", "CD-ROM CD-532S", &PlextorReaderScan::instance},
  { "plextor-scan",    "PLEXTOR", "CD-ROM", &PlextorReaderScan::instance },

  { "ricoh-mp6200", "RICOH",   "MP6200S", &RicohMP6200::instance },
  { NULL, NULL, NULL, NULL }
};


void message(int level, const char *fmt, ...)
{
  long len = strlen(fmt);
  char last = len > 0 ? fmt[len - 1] : 0;

  va_list args;
  va_start(args, fmt);

  if (level < 0) {
    switch (level) {
    case -1:
      fprintf(stderr, "Warning: ");
      break;
    case -2:
      fprintf(stderr, "ERROR: ");
      break;
    case -3:
      fprintf(stderr, "INTERNAL ERROR: ");
      break;
    default:
      fprintf(stderr, "FATAL ERROR: ");
      break;
    }
    vfprintf(stderr, fmt, args);
    if (last != ' ' && last != '\r')
      fprintf(stderr, "\n");
    
    fflush(stderr);
    if (level <= -10)
      exit(-1);
  }
  else if (level <= VERBOSE) {
    vfprintf(stdout, fmt, args);
    if (last != ' ' && last != '\r')
      fprintf(stdout, "\n");

    fflush(stdout);
  }

  va_end(args);
}

void usage()
{
  message(0, "Cdrdao version %s\n", VERSION);
  message(0, "Writes audio CD-R in disk-at-once mode.\n");

  message(0, "Usage: %s command [options] toc-file", PRGNAME);

  message(0, 
"  command: show-toc  - prints out toc and exits\n\
           read-toc  - create toc file from audio CD\n\
           read-cd   - create toc and rip audio data from CD\n\
           show-data - prints out audio data and exits\n\
           read-test - reads all audio files and exits\n\
           disk-info - shows information about inserted medium\n\
           simulate  - performs a write simulation\n\
           write     - writes CD\n");

  message(0,
"  options: --device <device>       - sets generic SCSI device of CD-writer\n\
                                     (default: /dev/cdrecorder)\n\
           --driver <driver-id>    - force usage of specified driver\n\
           --speed <writing-speed> - selects writing speed\n\
           --multi                 - session is not closed\n\
           --eject                 - ejects cd after writing or simulation\n\
           --swap                  - swap byte order of audio files\n\
           --fname <audio-file>    - name of audio file placed in toc-file\n\
                                     (used by 'read-toc' and 'read-cd')");
}

int parseCmdline(int argc, char **argv)
{
  if (argc < 1) {
    return 1;
  }

  if (strcmp(*argv, "show-toc") == 0) {
    COMMAND = SHOW_TOC;
  }
  else if (strcmp(*argv, "read-toc") == 0) {
    COMMAND = READ_TOC;
  }
  else if (strcmp(*argv, "show-data") == 0) {
    COMMAND = SHOW_DATA;
  }
  else if (strcmp(*argv, "read-test") == 0) {
    COMMAND = READ_TEST;
  }
  else if (strcmp(*argv, "simulate") == 0) {
    COMMAND = SIMULATE;
  }
  else if (strcmp(*argv, "write") == 0) {
    COMMAND = WRITE;
  }
  else if (strcmp(*argv, "disk-info") == 0) {
    COMMAND = DISK_INFO;
  }
  else if (strcmp(*argv, "read-cd") == 0) {
    COMMAND = READ_CD;
  }
  else {
    message(-2, "Illegal command: %s", *argv);
    return 1;
  }

  argc--, argv++;

  while (argc > 0 && (*argv)[0] == '-') {
    if ((*argv)[1] != '-') {
      message(-2, "Illegal option: %s", *argv);
      return 1;
    }
    else {
      if (strcmp((*argv) + 2, "device") == 0) {
	if (argc < 2) {
	  message(-2, "Missing argument after: %s", *argv);
	  return 1;
	}
	else {
	  SCSI_DEVICE = argv[1];
	  argc--, argv++;
	}
      }
      else if (strcmp((*argv) + 2, "speed") == 0) {
	if (argc < 2) {
	  message(-2, "Missing argument after: %s", *argv);
	  return 1;
	}
	else {
	  WRITING_SPEED = atol(argv[1]);
	  if (WRITING_SPEED < 0) {
	    message(-2, "Illegal writing speed: %s", argv[1]);
	    return 1;
	  }
	  argc--, argv++;
	}
      }
      else if (strcmp((*argv) + 2, "eject") == 0) {
	EJECT = 1;
      }
      else if (strcmp((*argv) + 2, "swap") == 0) {
	SWAP = 1;
      }
      else if (strcmp((*argv) + 2, "multi") == 0) {
	MULTI_SESSION = 1;
      }
      else if (strcmp((*argv) + 2, "driver") == 0) {
	if (argc < 2) {
	  message(-2, "Missing argument after: %s", *argv);
	  return 1;
	}
	else {
	  DRIVER_ID = argv[1];
	  argc--, argv++;
	}
      }
      else if (strcmp((*argv) + 2, "fname") == 0) {
	if (argc < 2) {
	  message(-2, "Missing argument after: %s", *argv);
	  return 1;
	}
	else {
	  TOC_AUDIO_FILENAME = argv[1];
	  argc--, argv++;
	}
      }
      else {
	message(-2, "Illegal option: %s", *argv);
	return 1;
      }

      argc--, argv++;
    }
  }

  if (COMMAND != DISK_INFO) {
    if (argc < 1) {
      message(-2, "Missing toc-file.");
      return 1;
    }
    else if (argc > 1) {
      message(-2, "Expecting only one toc-file.");
      return 1;
    }
    TOC_FILE = *argv;
  }


  return 0;
}

void showToc(const Toc *toc)
{
  const Track *t;
  Msf start, end, index;
  int i;
  int n;
  int tcount = 1;
  char buf[14];

  if (toc->catalogValid()) {
    for (i = 0; i < 13; i++) 
      buf[i] = toc->catalog(i) + '0';
    buf[13] = 0;

    message(0, "CATALOG NUMBER: %s", buf);
  }

  for (t = toc->first(start, end); t != NULL; t = toc->next(start, end)) {
    message(0, "TRACK %2d:", tcount);
    if (t->isrcValid()) {
      message(0, "          ISRC %c%c %c%c%c %c%c %c%c%c%c%c",
	      t->isrcCountry(0), t->isrcCountry(1),
	      t->isrcOwner(0), t->isrcOwner(1), t->isrcOwner(2),
	      t->isrcYear(0) + '0', t->isrcYear(1) + '0',
	      t->isrcSerial(0) + '0', t->isrcSerial(1) + '0',
	      t->isrcSerial(2) + '0', t->isrcSerial(3) + '0',
	      t->isrcSerial(4) + '0');
    }
    message(0, "          COPY%sPERMITTED",
	    t->copyPermitted() ? " " : " NOT ");
    message(0, "          %sPRE-EMPHASIS",
	    t->preEmphasis() ? "" : "NO ");
    message(0, "          %s CHANNEL AUDIO",
	    t->audioType() == 0 ? "TWO" : "FOUR");
    if (t->start().lba() != 0) {
      message(0, "          PREGAP %s(%6ld)", 
	      t->start().str(), t->start().lba());
    }
    message(0, "          START  %s(%6ld)",
	    start.str(), start.lba());
    n = t->nofIndices();
    for (i = 0; i < n; i++) {
      index = start + t->getIndex(i);
      message(0, "          INDEX  %s(%6ld)",
	      index.str(), index.lba());
    }

    message(0, "          END%c   %s(%6ld)",
	    t->isPadded() ? '*' : ' ', end.str(), end.lba());

    tcount++;
  }
} 

void showData(const Toc *toc, int swap)
{
  long length = toc->length().lba();
  Sample buf[SAMPLES_PER_BLOCK];
  int i;
  unsigned long sampleNr = 0;

  if (toc->openData() != 0) {
    message(-2, "Cannot open audio data.");
    return;
  }

  while (length > 0) {
    if (toc->readData((char *)buf, 1) != 1) {
      message(-2, "Read of audio data failed.");
      return;
    }

    if (swap) {
      swapSamples(buf, SAMPLES_PER_BLOCK);
    }

    for (i = 0; i < SAMPLES_PER_BLOCK; i++) {
      message(0, "%7lu:%6d %6d", sampleNr, buf[i].left(), buf[i].right());
      sampleNr++;
    }
    length -= 1;
  }
}

void showDiskInfo(DiskInfo *di)
{
  const char *s1, *s2;

  message(0, "CD-R empty     : ");
  if (di->valid.empty)
    message(0, di->empty ? "yes" : "no");
  else 
    message(0, "n/a");

  message(0, "CD-RW          : ");
  if (di->valid.cdrw)
    message(0, di->cdrw ? "yes" : "no");
  else 
    message(0, "n/a");
    
  message(0, "Capacity       : ");
  if (di->valid.capacity)
    message(0, "%s (%ld blocks, %ld/%ld MB)", Msf(di->capacity).str(),
	   di->capacity,
	   (di->capacity * 2) >> 10,
	   (di->capacity * AUDIO_BLOCK_LEN) >> 20);
  else
    message(0, "n/a");

  message(0, "CD-R medium    : ");
  if (di->valid.manufacturerId && 
      CdrDriver::cdrVendor(di->manufacturerId, &s1, &s2)) {
    message(0, "%s", s1);
    message(0, "                 %s", s2);
  }
  else
    message(0, "n/a");

  message(0, "Recording Speed: ");
  if (di->valid.recSpeed)
    message(0, "%dX - %dX", di->recSpeedLow, di->recSpeedHigh);
  else
    message(0, "n/a");
}

CdrDriver *selectDriver(ScsiIf *scsiIf, Toc *toc)
{
  DriverTable *run = DRIVER_TABLE;
  const char *lastDriverId = NULL;
  const char *driverId;
  long i, n;
  long maxDriverIdLen = 0;

  if (DRIVER_ID != NULL) {
    while (run->driverId != NULL) {
      if (strcmp(DRIVER_ID, run->driverId) == 0) {
	return run->constructor(scsiIf, toc);
      }
      run++;
    }
    message(-2, "%s: Illegal driver ID, available drivers:",
	    DRIVER_ID);
  }
  else {
    while (run->driverId != NULL) {
      if (strcmp(run->vendor, scsiIf->vendor()) == 0 &&
	  strstr(scsiIf->product(), run->model) != NULL) {
	return run->constructor(scsiIf, toc);
      }
      run++;
    }
    message(-2, "No driver found for your model, available drivers:");
  }

  // determine maximal length of driver id strings
  for (run = DRIVER_TABLE; run->driverId != NULL; run++) {
    if ((n = strlen(run->driverId)) > maxDriverIdLen) {
      maxDriverIdLen = n;
    }
  }

  for (run = DRIVER_TABLE; run->driverId != NULL; run++) {
    if (lastDriverId == NULL || strcmp(lastDriverId, run->driverId) != 0) {
      driverId = run->driverId;
    }
    else {
      // don't repeat same driver ID
      driverId = "";
    }
    message(0, "%s ", driverId);

    n = maxDriverIdLen - strlen(driverId);
    for (i = 0; i < n; i++) {
      message(0, " ");
    }
    message(0, ": %s ", run->vendor);

    n = 9 - strlen(run->vendor);
    for (i = 0; i < n; i++) {
      message(0, " ");
    }
    message(0, "%s", run->model);

    lastDriverId = run->driverId;
  }

  return NULL;
}

int main(int argc, char **argv)
{
  int ret;
  DiskInfo *di = NULL;

  PRGNAME = *argv;

  if (parseCmdline(argc - 1, argv + 1) != 0) {
    usage();
    return -1;
  }

  Toc *toc = NULL;
  if (COMMAND != READ_TOC && COMMAND != DISK_INFO && COMMAND != READ_CD) {
    toc = Toc::read(TOC_FILE);

    if (toc == NULL) {
      return -1;
    }

    if (COMMAND != SHOW_TOC) {
      if (toc->check() != 0) {
	message(-10, "Toc file '%s' is inconsistent - exiting.",
		TOC_FILE);
	return -1;
      }
    }
  }

  ScsiIf *scsiIf = NULL;
  CdrDriver *cdr = NULL;
  int retry = 0;
#define MAX_RETRY 4

  if (COMMAND == SIMULATE || COMMAND == WRITE || COMMAND == READ_TOC ||
      COMMAND == DISK_INFO || COMMAND == READ_CD) {
    scsiIf = new ScsiIf(SCSI_DEVICE);

    if (scsiIf->init() != 0) {
      return -1;
    }
  
    message(0, "%s: %s %s\tRev: %s (%s)", SCSI_DEVICE, scsiIf->vendor(),
	   scsiIf->product(), scsiIf->revision(), scsiIf->revisionDate());
    
    if ((cdr = selectDriver(scsiIf, toc)) == NULL) {
      return -1;
    }

    message(0, "Using driver: %s\n", cdr->driverName());

    while (retry < MAX_RETRY) {
      if (retry++)
	sleep(3);
      if (!(ret = cdr->testUnitReady(1)))
	break;
      if (ret == 1) {
	return -1; // scsi command failed
      }
      message(-1, "Unit not ready, still trying...");
    }
    if (ret) {
      message(-10, "Unit not ready, giving up.");
      return -1;
    }

    ret = cdr->testUnitReady(1);
    if (ret == 1) {
      return -1; // scsi command failed
    }
    else if (ret >= 2) {
      message(-10, "Unit not ready.");
      return -1;
    }

    // clear unit attention
    cdr->rezeroUnit(0);

    if ((di = cdr->diskInfo()) == NULL) {
      message(-10, "Cannot get disk information.");
      return -1;
    }
  }

  switch (COMMAND) {
  case SHOW_TOC:
    showToc(toc);
    if (toc->check() != 0) {
      message(-1, "Toc file '%s' is inconsistent.", TOC_FILE);
    }
    break;

  case SHOW_DATA:
    showData(toc, SWAP);
    break;

  case READ_TEST:
    message(0, "Starting read test...");
    if (writeDiskAtOnce(toc, NULL, SWAP, 1) != 0) {
      message(-2, "Read test failed.");
      return -1;
    }
    break;

  case DISK_INFO:
    showDiskInfo(di);
    break;

  case READ_TOC:
    if (di->valid.empty && di->empty) {
      message(-2, "Inserted disk is empty.");
      return -1;
    }
    message(0, "Reading toc data...");

    if (access(TOC_FILE, R_OK) == 0) {
      message(-10, "File \"%s\" exists, will not overwrite.",
	      TOC_FILE);
      return -1;
    }

    if ((toc = cdr->readDiskToc(TOC_AUDIO_FILENAME)) == NULL) {
      return -1;
    }
    else {
      seteuid(getuid());
      setegid(getgid());
      ofstream out(TOC_FILE);
      if (!out) {
	message(-10, "Cannot open \"%s\" for writing: %s",
		TOC_FILE, strerror(errno));
	return -1;
      }
      toc->print(out);
    }
    break;
    
  case READ_CD:
    if (di->valid.empty && di->empty) {
      message(-2, "Inserted disk is empty.");
      return -1;
    }
    message(0, "Reading toc and audio data...");

    if (access(TOC_FILE, R_OK) == 0) {
      message(-10, "File \"%s\" exists, will not overwrite.",
	      TOC_FILE);
      return -1;
    }

    if ((toc = cdr->readDisk(TOC_AUDIO_FILENAME)) == NULL) {
      return -1;
    }
    else {
      seteuid(getuid());
      setegid(getgid());
      ofstream out(TOC_FILE);
      if (!out) {
	message(-10, "Cannot open \"%s\" for writing: %s",
		TOC_FILE, strerror(errno));
	return -1;
      }
      toc->print(out);
    }
    break;

  case WRITE:
    cdr->simulate(0);
    // fall through
    
  case SIMULATE:
    if (di->valid.empty && !di->empty) {
      message(-10, "Inserted disk is not empty.");
      return -1;
    }

    if (toc->length().lba() > di->capacity - 1) {
      message(-1, "Length of toc (%s, %ld blocks) exceeds capacity ",
	      toc->length().str(), toc->length().lba());
      message(0, "of CD-R (%s, %ld blocks).", Msf(di->capacity - 1).str(),
	      di->capacity - 1);
      // the '- 1' is for the CDD2600 which rejects a toc that takes
      // the full capacity
      message(-1, "Some drives may fail to record this toc.");
    }

    if (MULTI_SESSION != 0) {
      if (cdr->multiSession(1) != 0) {
	message(-10, "This driver does not support multi session discs.");
	return -1;
      }
    }

    if (WRITING_SPEED >= 0) {
      if (cdr->speed(WRITING_SPEED) != 0) {
	message(-10, "Writing speed %d not supported by device.",
		WRITING_SPEED);
	return -1;
      }
    }

    message(0, "Starting write ");
    if (cdr->simulate()) {
      message(0, "simulation ");
    }
    message(0, "at speed %d...", cdr->speed());
    if (cdr->multiSession() != 0) {
      message(0, "Using multi session mode.");
    }

    message(0, "Pausing 10 seconds - hit CTRL-C to abort.");
    sleep(10);

    if (cdr->startStopUnit(1) != 0 || 
	cdr->preventMediumRemoval(1) != 0) {
      return -1;
    }

    if (writeDiskAtOnce(toc, cdr, SWAP, 0) != 0) {
      if (cdr->simulate()) {
	message(-2, "Simulation failed.");
      }
      else {
	message(-2, "Writing failed.");
      }
      cdr->preventMediumRemoval(0);
      return -1;
    }

    if (cdr->simulate()) {
      message(0, "Simulation finished successfully.");
    }
    else {
      message(0, "Writing finished successfully.");
    }

    if (cdr->preventMediumRemoval(0) != 0) {
      return -1;
    }

    if (EJECT) {
      cdr->loadUnload(1);
    }
    break;
  }

  return 0;
}
