/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Log: TocParser.g,v $
 * Revision 1.4  1998/10/24 14:27:57  mueller
 * Improved consistency check for track length.
 *
 * Revision 1.3  1998/08/20 20:53:35  mueller
 * Added some compatibility code and notes.
 * Problem indicated by Joerg Schneider <joerg_schneider@gmx.net>
 *
 * Revision 1.2  1998/07/28 13:47:48  mueller
 * Automatic length determination of audio files is now done in 'AudioData'.
 *
 */

#header <<
#include <config.h>
#include "Toc.h"
#include "util.h"
>>

<<
#include "TocLexer.h"

/* Use 'ANTLRCommonToken' as 'ANTLRToken' to be compatible with some bug
 * fix releases of PCCTS-1.33. The drawback is that the token length is
 * limited to 100 characters. This might be a problem for long file names.
 * Define 'USE_ANTLRCommonToken' to 0 if you want to use the dynamic token
 * which handles arbitrary token lengths (there'll be still a limitation in
 * the lexer code but that's more than 100 characters). In this case it might
 * be necessary to adjust the types of the member functions to the types in
 * 'ANTLRAbstractToken' defined in PCCTSDIR/h/AToken.h'.
 */

#define USE_ANTLRCommonToken 1

#if USE_ANTLRCommonToken

typedef ANTLRCommonToken ANTLRToken;

#else

class ANTLRToken : public ANTLRRefCountToken {
private:
  ANTLRTokenType type_;
  int line_;
  ANTLRChar *text_;

public:
  ANTLRToken(ANTLRTokenType t, ANTLRChar *s) 
    : ANTLRRefCountToken(t, s)
  { 
    setType(t); 
    line_ = 0; 
    setText(s); 
  }
  ANTLRToken()
  { 
    setType((ANTLRTokenType)0); 
    line_ = 0; 
    setText("");
  }
  virtual ~ANTLRToken() { delete[] text_; }

#if 1
  // use this for basic PCCTS-1.33 release
  ANTLRTokenType getType()	 { return type_; }
  virtual int getLine()		 { return line_; }
  ANTLRChar *getText()		 { return text_; }
#else
  // use this for PCCTS-1.33 bug fix releases
  ANTLRTokenType getType() const { return type_; }
  virtual int getLine()    const { return line_; }
  ANTLRChar *getText()	   const { return text_; }
#endif

  void setType(ANTLRTokenType t) { type_ = t; }
  void setLine(int line)	 { line_ = line; }
  void setText(ANTLRChar *s)     { text_ = strdupCC(s); }

  virtual ANTLRAbstractToken *makeToken(ANTLRTokenType tt, ANTLRChar *txt,
					int line)
  {
    ANTLRAbstractToken *t = new ANTLRToken(tt,txt);
    t->setLine(line);
    return t;
  }
};
#endif
 >>

#lexclass START
#token Eof		"@"
#token                  "[\t\ ]+"    << skip(); >>
#token Comment          "//~[\n@]*"  << skip(); >>
#token                  "\n"         << newline(); skip(); >>
#token BeginString      "\""         << mode(STRING); >>
#token INTEGER          "[0-9]+"
#token TRACK            "TRACK"
#token AUDIO            "AUDIO"
#token INDEX            "INDEX"
#token CATALOG          "CATALOG"
#token ISRC             "ISRC"
#token NO               "NO"
#token COPY             "COPY"
#token PRE_EMPHASIS     "PRE_EMPHASIS"
#token TWOCHANNEL       "TWO_CHANNEL_AUDIO"
#token FOURCHANNEL      "FOUR_CHANNEL_AUDIO"

#lexclass STRING
#token EndString        "\""         << mode(START); >>
#token String           "~[\n\"\ \t]*"

#lexclass START


class TocParserGram {
<<
public:
  const char *filename_;
  int error_;

void syn(_ANTLRTokenPtr tok, ANTLRChar *egroup, SetWordType *eset,
	 ANTLRTokenType etok, int k);
>>

toc > [ Toc *t ]
  : << $t = new Toc; 
       Track *tr = NULL;
       int lineNr = 0;
       char *catalog = NULL;
    >>
  { CATALOG string > [ catalog ]
    << if (catalog != NULL) {
         if ($t->catalog(catalog) != 0) {
           fprintf(stderr, "%s:%d: Illegal catalog number: %s.\n",
                   filename_, $1->getLine(), catalog);
           error_ = 1;
         }
         delete[] catalog;
       }
    >>
  }
  ( track > [ tr, lineNr ]
    << if (tr != NULL) {
         if ($t->append(tr) != 0) {
           fprintf(stderr, "%s:%d: First track must not have a pregap.\n",
                   filename_, lineNr);
           error_ = 1;
         }
         delete tr, tr = NULL;
       }
    >>
  )+
  Eof
  ;
  // fail action
  << delete $t, $t = NULL;
     delete[] catalog;
  >>

track > [ Track *tr, int lineNr ]
  : << $tr = NULL;
       $lineNr = 0;
       SubTrack *st = NULL;
       char *isrcCode = NULL;
       Track::Type trackType;
       Msf length;
       Msf indexIncrement;
       Msf start;
       Msf startPos;
       int startPosLine = 0;
       int lineNr = 0;
       int flag = 1;
    >>
    TRACK << $lineNr = $1->getLine(); >>
    ( AUDIO << trackType = Track::AUDIO; >> )
    << $tr = new Track(trackType); >>

    (  ISRC string > [ isrcCode ]
       << if (isrcCode != NULL) {
            if ($tr->isrc(isrcCode) != 0) {
              fprintf(stderr, "%s:%d: Illegal ISRC code: %s.\n",
                      filename_, $1->getLine(), isrcCode);
              error_ = 1;
            }
            delete[] isrcCode;
          }
       >>
     | { NO << flag = 0; >> } COPY
       << $tr->copyPermitted(flag); flag = 1; >>
     | { NO << flag = 0; >> } PRE_EMPHASIS
       << $tr->preEmphasis(flag); flag = 1; >>
     | TWOCHANNEL 
       << $tr->audioType(0); >>
     | FOURCHANNEL
       << $tr->audioType(1); >>
    )*
    { "PREGAP" msf > [ length ] 
      << if (length.lba() == 0) {
	   fprintf(stderr, "%s:%d: Length of pregap is zero.\n",
	           filename_, $1->getLine());
	   error_ = 1;
	 }
         else {
           $tr->append(SubTrack(SubTrack::DATA, 
                                AudioData(AudioData::SILENCE,
                                          length.samples())));
           startPos = $tr->length();
	   startPosLine = $1->getLine();
         }
      >>
    }

    (  subTrack > [ st, lineNr ] 
       << $tr->append(*st);
          delete st, st = NULL;
       >>
     | "START" { msf > [ start ] }
       << if (startPosLine != 0) {
            fprintf(stderr, "%s:%d: Track start already defined.\n",
                    filename_, $1->getLine());
            error_ = 1;
          }
          else {
            if (start.lba() == 0) {
              start = $tr->length(); // retrieve current position
            }
            startPos = start;
	    startPosLine = $1->getLine();
          }
          start = Msf(0);
       >>
    )+

    ( INDEX msf > [ indexIncrement ]
      << if ($tr != NULL) {
           switch ($tr->appendIndex(indexIncrement)) {
           case 1:
             fprintf(stderr, "%s:%d: More than 98 index increments.\n",
                     filename_, lineNr);
             error_ = 1;
             break;

           case 2:
             fprintf(stderr, "%s:%d: Index beyond track end.\n",
                     filename_, lineNr);
             error_ = 1;
             break;

           case 3:
             fprintf(stderr, "%s:%d: Index at start of track.\n",
                     filename_, lineNr);
             error_ = 1;
             break;
           }
         }
      >>
    )*

    // set track start and check for minimal track length
    << if ($tr->start(startPos) != 0) {
         fprintf(stderr, "%s:%d: START %s behind track end.\n", filename_,
	         startPosLine, startPos.str());
         error_ = 1;
       }

       if ($tr->length().lba() - $tr->start().lba() < Msf(0, 4, 0).lba()) {
         fprintf(stderr, "%s:%d: Track length (excluding pre-gap) must be at least 4 seconds.\n",
                 filename_, $lineNr);
         error_ = 1;
       }
    >>

    ;
    // fail action
    << delete $tr, $tr = NULL;
       delete[] isrcCode;
    >>


subTrack > [ SubTrack *st, int lineNr ]
  : << $st = NULL;
       $lineNr = 0;
       char *filename = NULL;
       unsigned long start = 0;
       unsigned long len = 0;
    >>
    (  "FILE" string > [ filename ] samples > [start] { samples > [len] }
       << $st = new SubTrack(SubTrack::DATA,
	                     AudioData(filename, start, len));
          $lineNr = $1->getLine();
       >>
     | "SILENCE" samples > [len]
       << $st = new SubTrack(SubTrack::DATA,
 	                     AudioData(AudioData::SILENCE, len));
          $lineNr = $1->getLine();
          if (len == 0) {
	    fprintf(stderr, "%s:%d: Length of silence is zero.\n",
		    filename_, $lineNr);
	    error_ = 1;
	  }
       >>
    )
    ;
    // fail action
    << delete $st, $st = NULL;
       delete[] filename;
    >>

string > [ char *ret ]
 :  << $ret = NULL; >>
    BeginString String EndString
    << $ret = strdupCC($2->getText()); >>
    ;

uLong > [ unsigned long l ]
  : << $l = 0; >>
    INTEGER << $l = strtoul($1->getText(), NULL, 10); >>
    ;

integer > [ int i ]
  : << $i = 0; >>
    INTEGER << $i = atol($1->getText()); >>
    ;
    
msf > [ Msf m ]
  : << int min = 0;
       int sec = 0;
       int frac = 0;
       int err = 0;
    >>
    integer > [min] ":" integer > [sec] ":" integer > [frac]
    << if (min < 0) {
	 fprintf(stderr, "%s:%d: Illegal minute field: %d\n", filename_,
	         $2->getLine(), min);
         err = error_ = 1;
       }
       if (sec < 0 || sec > 59) {
	 fprintf(stderr, "%s:%d: Illegal second field: %d\n", filename_,
	         $4->getLine(), sec);
	 err = error_ = 1;
       }
       if (frac < 0 || frac > 74) {
	 fprintf(stderr, "%s:%d: Illegal fraction field: %d\n", filename_,
		 $4->getLine(), frac);
	 err = error_ = 1;
       }
	  
       if (err != 0) {
	 $m = Msf(0);
       }
       else {
	 $m = Msf(min, sec, frac);
       }
    >>
    ;


samples > [ unsigned long s ]
  : << Msf m; >>
    (  msf > [ m ] << $s = m.samples(); >>
     | uLong > [ $s ]
    )
    ;
    // fail action
    << $s = 0; >>

}

<<
void TocParserGram::syn(_ANTLRTokenPtr tok, ANTLRChar *egroup,
			SetWordType *eset, ANTLRTokenType etok, int k)
{
  int line;

  error_ = 1;
  line = LT(1)->getLine();

  fprintf(stderr, "%s:%d: syntax error at \"%s\"", filename_, line,
	  LT(1)->getText());
  if ( !etok && !eset ) {
    fprintf(stderr, "\n");
    return;
  }
  if ( k==1 ) {
    fprintf(stderr, " missing");
  }
  else {
    fprintf(stderr, "; \"%s\" not", LT(1)->getText());
    if ( set_deg(eset)>1 ) fprintf(stderr, " in");
  }
  if ( set_deg(eset)>0 )
    edecode(eset);
  else fprintf(stderr, " %s", token_tbl[etok]);

  if ( strlen(egroup) > 0 )
    fprintf(stderr, " in %s", egroup);
	
  fprintf(stderr, "\n");
}


Toc *parseToc(FILE *fp, const char *filename)
{
  DLGFileInput in(fp);
  TocLexer scan(&in);
  ANTLRTokenBuffer pipe(&scan);
  ANTLRToken aToken;
  scan.setToken(&aToken);
  TocParserGram parser(&pipe);

  parser.init();
  parser.filename_ = filename;
  parser.error_ = 0;

  Toc *t = parser.toc();

  if (parser.error_ != 0) {
    return NULL;
  }
  else {
    return t;
  }
}
>>
