/* Copyright (c) 1997 The Regents of the University of California.
* For information on usage and redistribution, and for a DISCLAIMER OF ALL
* WARRANTIES, see the file, "LICENSE.txt," in this distribution.  */

/* connective objects */

#include "m_pd.h"

#include <string.h>
extern t_pd *newest;

/* -------------------------- int ------------------------------ */
static t_class *pdint_class;

typedef struct _pdint
{
    t_object x_obj;
    t_float x_f;
} t_pdint;

static void *pdint_new(t_floatarg f)
{
    t_pdint *x = (t_pdint *)pd_new(pdint_class);
    x->x_f = f;
    outlet_new(&x->x_obj, &s_float);
    floatinlet_new(&x->x_obj, &x->x_f);
    return (x);
}

static void pdint_bang(t_pdint *x)
{
    outlet_float(x->x_obj.ob_outlet, (t_float)(int)(x->x_f));
}

static void pdint_float(t_pdint *x, t_float f)
{
    outlet_float(x->x_obj.ob_outlet, (t_float)(int)(x->x_f = f));
}

void pdint_setup(void)
{
    pdint_class = class_new(gensym("int"), (t_newmethod)pdint_new, 0,
    	sizeof(t_pdint), 0, A_DEFFLOAT, 0);
    class_addcreator((t_newmethod)pdint_new, gensym("i"), A_DEFFLOAT, 0);
    class_addbang(pdint_class, pdint_bang);
    class_addfloat(pdint_class, pdint_float);
}

/* -------------------------- float ------------------------------ */
static t_class *pdfloat_class;

typedef struct _pdfloat
{
    t_object x_obj;
    t_float x_f;
} t_pdfloat;

    /* "float," "symbol," and "pointer" are special because
    they're created by short-circuited messages to the "new"
    object which are handled specially in typedmess(). */

static void *pdfloat_new(t_pd *dummy, t_float f)
{
    t_pdfloat *x = (t_pdfloat *)pd_new(pdfloat_class);
    x->x_f = f;
    outlet_new(&x->x_obj, &s_float);
    floatinlet_new(&x->x_obj, &x->x_f);
    newest = &x->x_obj.ob_pd;
    return (x);
}

static void *pdfloat_new2(t_floatarg f)
{
    return (pdfloat_new(0, f));
}

static void pdfloat_bang(t_pdfloat *x)
{
    outlet_float(x->x_obj.ob_outlet, x->x_f);
}

static void pdfloat_float(t_pdfloat *x, t_float f)
{
    outlet_float(x->x_obj.ob_outlet, x->x_f = f);
}

void pdfloat_setup(void)
{
    pdfloat_class = class_new(gensym("float"), (t_newmethod)pdfloat_new, 0,
    	sizeof(t_pdfloat), 0, A_DEFFLOAT, 0);
    class_addcreator((t_newmethod)pdfloat_new2, gensym("f"), A_DEFFLOAT, 0);
    class_addbang(pdfloat_class, pdfloat_bang);
    class_addfloat(pdfloat_class, (t_method)pdfloat_float);
}

/* -------------------- send ------------------------------ */

static t_class *send_class;

typedef struct _send
{
    t_object x_obj;
    t_symbol *x_sym;
} t_send;

static void send_bang(t_send *x)
{
    if (x->x_sym->s_thing) pd_bang(x->x_sym->s_thing);
}

static void send_float(t_send *x, t_float f)
{
    if (x->x_sym->s_thing) pd_float(x->x_sym->s_thing, f);
}

static void send_symbol(t_send *x, t_symbol *s)
{
    if (x->x_sym->s_thing) pd_symbol(x->x_sym->s_thing, s);
}

static void send_pointer(t_send *x, t_gpointer *gp)
{
    if (x->x_sym->s_thing) pd_pointer(x->x_sym->s_thing, gp);
}

static void send_list(t_send *x, t_symbol *s, int argc, t_atom *argv)
{
    if (x->x_sym->s_thing) pd_list(x->x_sym->s_thing, s, argc, argv);
}

static void send_anything(t_send *x, t_symbol *s, int argc, t_atom *argv)
{
    if (x->x_sym->s_thing) typedmess(x->x_sym->s_thing, s, argc, argv);
}

static void *send_new(t_symbol *s)
{
    t_send *x = (t_send *)pd_new(send_class);
    x->x_sym = s;
    return (x);
}

static void send_setup(void)
{
    send_class = class_new(gensym("send"), send_new, 0,
    	sizeof(t_send), 0, A_DEFSYM, 0);
    class_addcreator(send_new, gensym("s"), A_DEFSYM, 0);
    class_addbang(send_class, send_bang);
    class_addfloat(send_class, send_float);
    class_addsymbol(send_class, send_symbol);
    class_addpointer(send_class, send_pointer);
    class_addlist(send_class, send_list);
    class_addanything(send_class, send_anything);
}
/* -------------------- receive ------------------------------ */

static t_class *receive_class;

typedef struct _receive
{
    t_object x_obj;
    t_symbol *x_sym;
} t_receive;

static void receive_bang(t_receive *x)
{
    outlet_bang(x->x_obj.ob_outlet);
}

static void receive_float(t_receive *x, t_float f)
{
    outlet_float(x->x_obj.ob_outlet, f);
}

static void receive_symbol(t_receive *x, t_symbol *s)
{
    outlet_symbol(x->x_obj.ob_outlet, s);
}

static void receive_pointer(t_receive *x, t_gpointer *gp)
{
    outlet_pointer(x->x_obj.ob_outlet, gp);
}

static void receive_list(t_receive *x, t_symbol *s, int argc, t_atom *argv)
{
    outlet_list(x->x_obj.ob_outlet, s, argc, argv);
}

static void receive_anything(t_receive *x, t_symbol *s, int argc, t_atom *argv)
{
    outlet_anything(x->x_obj.ob_outlet, s, argc, argv);
}

static void *receive_new(t_symbol *s)
{
    t_receive *x = (t_receive *)pd_new(receive_class);
    x->x_sym = s;
    pd_bind(&x->x_obj.ob_pd, s);
    outlet_new(&x->x_obj, 0);
    return (x);
}

static void receive_free(t_receive *x)
{
    pd_unbind(&x->x_obj.ob_pd, x->x_sym);
}

static void receive_setup(void)
{
    receive_class = class_new(gensym("receive"), receive_new, receive_free,
    	sizeof(t_receive), CLASS_NOINLET, A_SYMBOL, 0);
    class_addcreator(receive_new, gensym("r"), A_DEFSYM, 0);
    class_addbang(receive_class, receive_bang);
    class_addfloat(receive_class, (t_method)receive_float);
    class_addsymbol(receive_class, receive_symbol);
    class_addpointer(receive_class, receive_pointer);
    class_addlist(receive_class, receive_list);
    class_addanything(receive_class, receive_anything);
}

/* -------------------------- select ------------------------------ */

static t_class *sel1_class;

typedef struct _sel1
{
    t_object x_obj;
    t_float x_f;
    t_outlet *x_outlet1;
    t_outlet *x_outlet2;
} t_sel1;

static void sel1_float(t_sel1 *x, t_float f)
{
    if (f == x->x_f) outlet_bang(x->x_outlet1);
    else outlet_float(x->x_outlet2, f);
}

static t_class *sel2_class;

typedef struct _selectelement
{
    t_float e_f;
    t_outlet *e_outlet;
} t_selectelement;

typedef struct _sel2
{
    t_object x_obj;
    t_int x_nelement;
    t_selectelement *x_vec;
    t_outlet *x_rejectout;
} t_sel2;

static void sel2_float(t_sel2 *x, t_float f)
{
    t_selectelement *e;
    int nelement;
    for (nelement = x->x_nelement, e = x->x_vec; nelement--; e++)
    	if (e->e_f == f)
    {
    	outlet_bang(e->e_outlet);
    	return;
    }
    outlet_float(x->x_rejectout, f);
}

static void sel2_free(t_sel2 *x)
{
    freebytes(x->x_vec, x->x_nelement * sizeof(*x->x_vec));
}

static void *select_new(t_symbol *s, int argc, t_atom *argv)
{
    if (argc < 2)
    {
    	t_sel1 *x = (t_sel1 *)pd_new(sel1_class);
    	x->x_f = atom_getfloatarg(0, argc, argv);
    	x->x_outlet1 = outlet_new(&x->x_obj, &s_bang);
    	x->x_outlet2 = outlet_new(&x->x_obj, &s_float);
    	floatinlet_new(&x->x_obj, &x->x_f);
    	return (x);
    }
    else
    {
    	int n;
    	t_selectelement *e;
    	t_sel2 *x = (t_sel2 *)pd_new(sel2_class);
    	x->x_nelement = argc;
    	x->x_vec = (t_selectelement *)getbytes(argc * sizeof(*x->x_vec));
    	for (n = 0, e = x->x_vec; n < argc; n++, e++)
    	{
    	    e->e_outlet = outlet_new(&x->x_obj, &s_bang);
    	    e->e_f = atom_getfloatarg(n, argc, argv);
    	}
    	x->x_rejectout = outlet_new(&x->x_obj, &s_float);
    	return (x);
    }

}

void select_setup(void)
{
    sel1_class = class_new(gensym("select"), 0, 0,
    	sizeof(t_sel1), 0, 0);
    class_addfloat(sel1_class, sel1_float);

    sel2_class = class_new(gensym("select"), 0, sel2_free,
    	sizeof(t_sel2), 0, 0);
    class_addfloat(sel2_class, sel2_float);

    class_addcreator(select_new, gensym("select"),  A_GIMME, 0);
    class_addcreator(select_new, gensym("sel"),  A_GIMME, 0);
}

/* -------------------------- route ------------------------------ */

static t_class *route_class;

typedef struct _routeelement
{
    t_float e_f;
    t_outlet *e_outlet;
} t_routeelement;

typedef struct _route
{
    t_object x_obj;
    t_int x_nelement;
    t_routeelement *x_vec;
    t_outlet *x_rejectout;
} t_route;

static void route_list(t_route *x, t_symbol *sel, int argc, t_atom *argv)
{
    t_routeelement *e;
    int nelement;
    float f;
    if (!argc) return;
    else f = atom_getfloat(argv);
    for (nelement = x->x_nelement, e = x->x_vec; nelement--; e++)
    	if (e->e_f == f)
    {
    	outlet_list(e->e_outlet, 0, argc-1, argv+1);
    	return;
    }
    outlet_list(x->x_rejectout, 0, argc, argv);
}

static void route_free(t_route *x)
{
    freebytes(x->x_vec, x->x_nelement * sizeof(*x->x_vec));
}

static void *route_new(t_symbol *s, int argc, t_atom *argv)
{
    int n;
    t_routeelement *e;
    t_route *x = (t_route *)pd_new(route_class);
    x->x_nelement = argc;
    x->x_vec = (t_routeelement *)getbytes(argc * sizeof(*x->x_vec));
    for (n = 0, e = x->x_vec; n < argc; n++, e++)
    {
    	e->e_outlet = outlet_new(&x->x_obj, &s_list);
    	e->e_f = atom_getfloatarg(n, argc, argv);
    }
    x->x_rejectout = outlet_new(&x->x_obj, &s_list);
    return (x);
}

void route_setup(void)
{
    route_class = class_new(gensym("route"), route_new, route_free,
    	sizeof(t_route), 0, A_GIMME, 0);
    class_addlist(route_class, route_list);
}

/* -------------------------- pack ------------------------------ */

static t_class *pack_class;

typedef struct _pack
{
    t_object x_obj;
    t_int x_n;
    t_atom *x_vec;
    t_int x_nptr;
    t_gpointer *x_gpointer;
} t_pack;

static void *pack_new(t_symbol *s, int argc, t_atom *argv)
{
    t_pack *x = (t_pack *)pd_new(pack_class);
    t_atom defarg[2], *ap, *vec, *vp;
    t_gpointer *gp;
    int nptr = 0;
    int i;
    if (!argc)
    {
    	argv = defarg;
    	argc = 2;
    	SETFLOAT(&defarg[0], 0);
    	SETFLOAT(&defarg[1], 0);
    }
    x->x_n = argc;
    vec = x->x_vec = (t_atom *)getbytes(argc * sizeof(*x->x_vec));

    for (i = argc, ap = argv; i--; ap++)
    	if (ap->a_type == A_SYMBOL && ap->a_w.w_symbol == &s_pointer)
    	    nptr++;

    gp = x->x_gpointer = (t_gpointer *)t_getbytes(nptr * sizeof (*gp));
    x->x_nptr = nptr;

    for (i = 0, vp = x->x_vec, ap = argv; i < argc; i++, ap++, vp++)
    {
    	if (ap->a_type == A_FLOAT)
    	{
    	    *vp = *ap;
    	    if (i) floatinlet_new(&x->x_obj, &vp->a_w.w_float);
    	}
    	else if (ap->a_type == A_SYMBOL)
    	{
    	    char c = *ap->a_w.w_symbol->s_name;
    	    if (c == 's')
    	    {
    	    	SETSYMBOL(vp, &s_symbol);
    	    	if (i) symbolinlet_new(&x->x_obj, &vp->a_w.w_symbol);
    	    }
    	    else if (c == 'p')
    	    {
    	    	vp->a_type = A_POINTER;
    	    	vp->a_w.w_gpointer = gp;
    	    	gpointer_init(gp);
    	    	if (i) pointerinlet_new(&x->x_obj, gp);
    	    	gp++;
    	    }
    	    else
    	    {
    	    	if (c != 'f') error("pack: %s: bad type",
    	    	    ap->a_w.w_symbol->s_name);
    	    	SETFLOAT(vp, 0);
    	    	if (i) floatinlet_new(&x->x_obj, &vp->a_w.w_float);
    	    }
    	}
    }
    outlet_new(&x->x_obj, &s_list);
    return (x);
}

static void pack_bang(t_pack *x)
{
    int i;
    t_gpointer *gp;
    for (i = x->x_nptr, gp = x->x_gpointer; i--; gp++)
    	if (!gpointer_check(gp, 1))
    {
    	error("pack: stale pointer");
    	return;
    }
    outlet_list(x->x_obj.ob_outlet, &s_list, x->x_n, x->x_vec);
}

static void pack_pointer(t_pack *x, t_gpointer *gp)
{
    if (x->x_vec->a_type == A_POINTER)
    {
    	*x->x_gpointer = *gp;
    	pack_bang(x);
    }
    else error("pack_pointer: wrong type");
}

static void pack_float(t_pack *x, t_float f)
{
    if (x->x_vec->a_type == A_FLOAT)
    {
    	x->x_vec->a_w.w_float = f;
    	pack_bang(x);
    }
    else error("pack_float: wrong type");
}

static void pack_symbol(t_pack *x, t_symbol *s)
{
    if (x->x_vec->a_type == A_SYMBOL)
    {
    	x->x_vec->a_w.w_symbol = s;
    	pack_bang(x);
    }
    else error("pack_symbol: wrong type");
}

static void pack_free(t_pack *x)
{
    t_gpointer *gp;
    int i;
    for (gp = x->x_gpointer, i = x->x_nptr; i--; gp++)
    	gpointer_unset(gp);
    freebytes(x->x_vec, x->x_n * sizeof(*x->x_vec));
    freebytes(x->x_gpointer, x->x_nptr * sizeof(*x->x_gpointer));
}

static void pack_setup(void)
{
    pack_class = class_new(gensym("pack"), pack_new, pack_free,
    	sizeof(t_pack), 0, A_GIMME, 0);
    class_addbang(pack_class, pack_bang);
    class_addpointer(pack_class, pack_pointer);
    class_addfloat(pack_class, pack_float);
    class_addsymbol(pack_class, pack_symbol);
}

/* -------------------------- unpack ------------------------------ */

static t_class *unpack_class;

typedef struct unpackout
{
    t_atomtype u_type;
    t_outlet *u_outlet;
} t_unpackout;

typedef struct _unpack
{
    t_object x_obj;
    t_int x_n;
    t_unpackout *x_vec;
} t_unpack;

static void *unpack_new(t_symbol *s, int argc, t_atom *argv)
{
    t_unpack *x = (t_unpack *)pd_new(unpack_class);
    t_atom defarg[2], *ap;
    t_unpackout *u;
    int i;
    if (!argc)
    {
    	argv = defarg;
    	argc = 2;
    	SETFLOAT(&defarg[0], 0);
    	SETFLOAT(&defarg[1], 0);
    }
    x->x_n = argc;
    x->x_vec = (t_unpackout *)getbytes(argc * sizeof(*x->x_vec));
    for (i = 0, ap = argv, u = x->x_vec; i < argc; u++, ap++, i++)
    {
    	t_atomtype type = ap->a_type;
    	if (type == A_SYMBOL)
    	{
    	    char c = *ap->a_w.w_symbol->s_name;
    	    if (c == 's')
    	    {
    	    	u->u_type = A_SYMBOL;
    	    	u->u_outlet = outlet_new(&x->x_obj, &s_symbol);
    	    }
    	    else if (c == 'p')
    	    {
    	    	u->u_type =  A_POINTER;
    	    	u->u_outlet = outlet_new(&x->x_obj, &s_pointer);
    	    }
    	    else
    	    {
    	    	if (c != 'f') error("unpack: %s: bad type",
    	    	    ap->a_w.w_symbol->s_name);
    	    	u->u_type = A_FLOAT;
    	    	u->u_outlet = outlet_new(&x->x_obj, &s_float);
    	    }
    	}
    	else
    	{
    	    u->u_type =  A_FLOAT;
    	    u->u_outlet = outlet_new(&x->x_obj, &s_float);
    	}
    }
    return (x);
}

static void unpack_list(t_unpack *x, t_symbol *s, int argc, t_atom *argv)
{
    t_atom *ap;
    t_unpackout *u;
    int i;
    if (argc > x->x_n) argc = x->x_n;
    for (i = argc, u = x->x_vec + i, ap = argv + i; u--, ap--, i--;)
    {
    	t_atomtype type = u->u_type;
    	if (type != ap->a_type)
    	    error("unpack: type mismatch");
    	else if (type == A_FLOAT)
    	    outlet_float(u->u_outlet, ap->a_w.w_float);
    	else if (type == A_SYMBOL)
    	    outlet_symbol(u->u_outlet, ap->a_w.w_symbol);
    	else outlet_pointer(u->u_outlet, ap->a_w.w_gpointer);
    }
}

static void unpack_free(t_unpack *x)
{
    freebytes(x->x_vec, x->x_n * sizeof(*x->x_vec));
}

static void unpack_setup(void)
{
    unpack_class = class_new(gensym("unpack"), unpack_new, unpack_free,
    	sizeof(t_unpack), 0, A_GIMME, 0);
    class_addlist(unpack_class, unpack_list);
}

/* -------------------------- trigger ------------------------------ */

static t_class *trigger_class;

typedef struct triggerout
{
    t_atomtype u_type;	/* outlet type, with A_NULL standing for bang and
    	    	    	A_GIMME for list.  */
    t_outlet *u_outlet;
} t_triggerout;

typedef struct _trigger
{
    t_object x_obj;
    t_int x_n;
    t_triggerout *x_vec;
} t_trigger;

static void *trigger_new(t_symbol *s, int argc, t_atom *argv)
{
    t_trigger *x = (t_trigger *)pd_new(trigger_class);
    t_atom defarg[2], *ap;
    t_triggerout *u;
    int i;
    if (!argc)
    {
    	argv = defarg;
    	argc = 2;
    	SETSYMBOL(&defarg[0], &s_bang);
    	SETSYMBOL(&defarg[1], &s_bang);
    }
    x->x_n = argc;
    x->x_vec = (t_triggerout *)getbytes(argc * sizeof(*x->x_vec));
    for (i = 0, ap = argv, u = x->x_vec; i < argc; u++, ap++, i++)
    {
    	t_atomtype thistype = ap->a_type;
    	char c;
    	if (thistype == A_SYMBOL) c = ap->a_w.w_symbol->s_name[0];
    	else if (thistype == A_FLOAT) c = 'f';
    	if (c == 'p')
    	    u->u_type = A_POINTER,
    	    	u->u_outlet = outlet_new(&x->x_obj, &s_pointer);
    	else if (c == 'f')
    	    u->u_type = A_FLOAT, u->u_outlet = outlet_new(&x->x_obj, &s_float);
    	else if (c == 'b')
    	    u->u_type = A_NULL, u->u_outlet = outlet_new(&x->x_obj, &s_bang);
    	else if (c == 'l')
    	    u->u_type = A_GIMME, u->u_outlet = outlet_new(&x->x_obj, &s_list);
    	else if (c == 's')
    	    u->u_type = A_SYMBOL, u->u_outlet = outlet_new(&x->x_obj, &s_symbol);
    	else error("trigger: %s: bad type", ap->a_w.w_symbol->s_name);
    }
    return (x);
}

static void trigger_list(t_trigger *x, t_symbol *s, int argc, t_atom *argv)
{
    t_triggerout *u;
    int i;
    t_atom at;
    if (!argc)
    {
    	argc = 1;
    	SETFLOAT(&at, 0);
    	argv = &at;
    }
    for (i = x->x_n, u = x->x_vec + i; u--, i--;)
    {
    	if (u->u_type == A_FLOAT)
    	    outlet_float(u->u_outlet, atom_getfloat(argv));
    	else if (u->u_type == A_NULL) outlet_bang(u->u_outlet);
    	else if (u->u_type == A_SYMBOL)
    	    outlet_symbol(u->u_outlet, atom_getsymbol(argv));
    	else if (u->u_type == A_POINTER)
    	{
    	    if (argv->a_type != A_POINTER)
    	    	error("unpack: bad pointer");
    	    else outlet_pointer(u->u_outlet, argv->a_w.w_gpointer);
    	}
    	else outlet_list(u->u_outlet, &s_list, argc, argv);
    }
}

static void trigger_bang(t_trigger *x)
{
    trigger_list(x, 0, 0, 0);
}

static void trigger_pointer(t_trigger *x, t_gpointer *gp)
{
    t_atom at;
    SETPOINTER(&at, gp);
    trigger_list(x, 0, 1, &at);
}

static void trigger_float(t_trigger *x, t_float f)
{
    t_atom at;
    SETFLOAT(&at, f);
    trigger_list(x, 0, 1, &at);
}

static void trigger_symbol(t_trigger *x, t_symbol *s)
{
    t_atom at;
    SETSYMBOL(&at, s);
    trigger_list(x, 0, 1, &at);
}

static void trigger_free(t_trigger *x)
{
    freebytes(x->x_vec, x->x_n * sizeof(*x->x_vec));
}

static void trigger_setup(void)
{
    trigger_class = class_new(gensym("trigger"), trigger_new, trigger_free,
    	sizeof(t_trigger), 0, A_GIMME, 0);
    class_addcreator(trigger_new, gensym("t"), A_GIMME, 0);
    class_addlist(trigger_class, trigger_list);
    class_addbang(trigger_class, trigger_bang);
    class_addpointer(trigger_class, trigger_pointer);
    class_addfloat(trigger_class, (t_method)trigger_float);
    class_addsymbol(trigger_class, trigger_symbol);
}

/* -------------------------- spigot ------------------------------ */
static t_class *spigot_class;

typedef struct _spigot
{
    t_object x_obj;
    float x_state;
} t_spigot;

static void *spigot_new()
{
    t_spigot *x = (t_spigot *)pd_new(spigot_class);
    floatinlet_new(&x->x_obj, &x->x_state);
    outlet_new(&x->x_obj, 0);
    x->x_state = 0;
    return (x);
}

static void spigot_bang(t_spigot *x)
{
    if (x->x_state != 0) outlet_bang(x->x_obj.ob_outlet);
}

static void spigot_pointer(t_spigot *x, t_gpointer *gp)
{
    if (x->x_state != 0) outlet_pointer(x->x_obj.ob_outlet, gp);
}

static void spigot_float(t_spigot *x, t_float f)
{
    if (x->x_state != 0) outlet_float(x->x_obj.ob_outlet, f);
}

static void spigot_symbol(t_spigot *x, t_symbol *s)
{
    if (x->x_state != 0) outlet_symbol(x->x_obj.ob_outlet, s);
}

static void spigot_list(t_spigot *x, t_symbol *s, int argc, t_atom *argv)
{
    if (x->x_state != 0) outlet_list(x->x_obj.ob_outlet, s, argc, argv);
}

static void spigot_anything(t_spigot *x, t_symbol *s, int argc, t_atom *argv)
{
    if (x->x_state != 0) outlet_anything(x->x_obj.ob_outlet, s, argc, argv);
}

static void spigot_setup(void)
{
    spigot_class = class_new(gensym("spigot"), spigot_new, 0,
    	sizeof(t_spigot), 0, A_DEFSYM, 0);
    class_addbang(spigot_class, spigot_bang);
    class_addpointer(spigot_class, spigot_pointer);
    class_addfloat(spigot_class, spigot_float);
    class_addsymbol(spigot_class, spigot_symbol);
    class_addlist(spigot_class, spigot_list);
    class_addanything(spigot_class, spigot_anything);
}

static t_class *moses_class;   	/* ----------- moses --------------- */

typedef struct _moses
{
    t_object x_ob;
    t_outlet *x_out2;
    float x_y;
} t_moses;

static void *moses_new(t_floatarg f)
{
    t_moses *x = (t_moses *)pd_new(moses_class);
    floatinlet_new(&x->x_ob, &x->x_y);
    outlet_new(&x->x_ob, &s_float);
    x->x_out2 = outlet_new(&x->x_ob, &s_float);
    x->x_y = f;
    return (x);
}

static void moses_float(t_moses *x, t_float f)
{
    if (f < x->x_y) outlet_float(x->x_ob.ob_outlet, f);
    else outlet_float(x->x_out2, f);
}

static void moses_setup(void)
{
    moses_class = class_new(gensym("moses"), (t_newmethod)moses_new, 0,
    	sizeof(t_moses), 0, A_DEFFLOAT, 0);
    class_addfloat(moses_class, moses_float);
}

/* ----------------------- until --------------------- */

static t_class *until_class;

typedef struct _until
{
    t_object x_obj;
    int x_run;
} t_until;

static void *until_new(void)
{
    t_until *x = (t_until *)pd_new(until_class);
    inlet_new(&x->x_obj, &x->x_obj.ob_pd, gensym("bang"), gensym("bang2"));
    outlet_new(&x->x_obj, &s_bang);
    x->x_run = 0;
    return (x);
}

static void until_bang(t_until *x)
{
    x->x_run = 1;
    while (x->x_run) outlet_bang(x->x_obj.ob_outlet);
}

static void until_bang2(t_until *x)
{
    x->x_run = 0;
}

static void until_setup(void)
{
    until_class = class_new(gensym("until"), (t_newmethod)until_new, 0,
    	sizeof(t_until), 0, 0);
    class_addbang(until_class, until_bang);
    class_addmethod(until_class, until_bang2, gensym("bang2"), 0);
}

/* -------------- overall setup routine for this file ----------------- */

void x_connective_setup(void)
{
    pdint_setup();
    pdfloat_setup();
    send_setup();
    receive_setup();
    select_setup();
    route_setup();
    pack_setup();
    unpack_setup();
    trigger_setup();
    spigot_setup();
    moses_setup();
    until_setup();
}
