/* Copyright (c) 1997 The Regents of the University of California.
* For information on usage and redistribution, and for a DISCLAIMER OF ALL
* WARRANTIES, see the file, "LICENSE.txt," in this distribution.  */

/* clock objects */

#include "m_pd.h"
#include <stdio.h>
/* -------------------------- delay ------------------------------ */
static t_class *delay_class;

typedef struct _delay
{
    t_object x_obj;
    t_clock *x_clock;
    double x_deltime;
} t_delay;

static void delay_bang(t_delay *x)
{
    clock_delay(x->x_clock, x->x_deltime);
}

static void delay_stop(t_delay *x)
{
    clock_unset(x->x_clock);
}

static void delay_ft1(t_delay *x, t_floatarg g)
{
    if (g < 0) g = 0;
    x->x_deltime = g;
}

static void delay_float(t_delay *x, t_float f)
{
    delay_ft1(x, f);
    delay_bang(x);
}

static void delay_tick(t_delay *x)
{
    outlet_bang(x->x_obj.ob_outlet);
}

static void delay_free(t_delay *x)
{
    clock_free(x->x_clock);
}

static void *delay_new(t_floatarg f)
{
    t_delay *x = (t_delay *)pd_new(delay_class);
    delay_ft1(x, f);
    x->x_clock = clock_new(x, delay_tick);
    outlet_new(&x->x_obj, gensym("bang"));
    inlet_new(&x->x_obj, &x->x_obj.ob_pd, gensym("float"), gensym("ft1"));
    return (x);
}

static void delay_setup(void)
{
    delay_class = class_new(gensym("delay"), (t_newmethod)delay_new, delay_free,
    	sizeof(t_delay), 0, A_DEFFLOAT, 0);
    class_addcreator((t_newmethod)delay_new, gensym("del"), A_DEFFLOAT, 0);
    class_addbang(delay_class, delay_bang);
    class_addmethod(delay_class, delay_stop, gensym("stop"), 0);
    class_addmethod(delay_class, (t_method)delay_ft1,
    	gensym("ft1"), A_FLOAT, 0);
    class_addfloat(delay_class, (t_method)delay_float);
}

/* -------------------------- metro ------------------------------ */
static t_class *metro_class;

typedef struct _metro
{
    t_object x_obj;
    t_clock *x_clock;
    double x_deltime;
    int x_hit;
} t_metro;

static void metro_tick(t_metro *x)
{
    x->x_hit = 0;
    outlet_bang(x->x_obj.ob_outlet);
    if (!x->x_hit) clock_delay(x->x_clock, x->x_deltime);
}

static void metro_float(t_metro *x, t_float f)
{
    if (f != 0) metro_tick(x);
    else clock_unset(x->x_clock);
    x->x_hit = 1;
}

static void metro_bang(t_metro *x)
{
    metro_float(x, 1);
}

static void metro_stop(t_metro *x)
{
    metro_float(x, 0);
}

static void metro_ft1(t_metro *x, t_floatarg g)
{
    if (g < 1) g = 1;
    x->x_deltime = g;
}

static void metro_free(t_metro *x)
{
    clock_free(x->x_clock);
}

static void *metro_new(t_floatarg f)
{
    t_metro *x = (t_metro *)pd_new(metro_class);
    metro_ft1(x, f);
    x->x_hit = 0;
    x->x_clock = clock_new(x, metro_tick);
    outlet_new(&x->x_obj, gensym("bang"));
    inlet_new(&x->x_obj, &x->x_obj.ob_pd, gensym("float"), gensym("ft1"));
    return (x);
}

static void metro_setup(void)
{
    metro_class = class_new(gensym("metro"), (t_newmethod)metro_new,
    	metro_free, sizeof(t_metro), 0, A_DEFFLOAT, 0);
    class_addbang(metro_class, metro_bang);
    class_addmethod(metro_class, (t_method)metro_stop, gensym("stop"), 0);
    class_addmethod(metro_class, (t_method)metro_ft1, gensym("ft1"),
    	A_FLOAT, 0);
    class_addfloat(metro_class, (t_method)metro_float);
}

/* -------------------------- line ------------------------------ */
static t_class *line_class;

typedef struct _line
{
    t_object x_obj;
    t_clock *x_clock;
    double x_targettime;
    t_float x_targetval;
    double x_prevtime;
    t_float x_setval;
    int x_gotinlet;
    t_float x_grain;
    double x_1overtimediff;
    double x_in1val;
} t_line;

static void line_tick(t_line *x)
{
    double timenow = clock_getsystime();
    double msectogo = - clock_gettimesince(x->x_targettime);
    if (msectogo < 1E-9)
    {
    	outlet_float(x->x_obj.ob_outlet, x->x_targetval);
    }
    else
    {
    	outlet_float(x->x_obj.ob_outlet,
    	    x->x_setval + x->x_1overtimediff * (timenow - x->x_prevtime)
    	    	* (x->x_targetval - x->x_setval));
    	clock_delay(x->x_clock,
    	    (x->x_grain > msectogo ? msectogo : x->x_grain));
    }
}

static void line_float(t_line *x, t_float f)
{
    double timenow = clock_getsystime();
    if (x->x_gotinlet && x->x_in1val > 0)
    {
    	if (timenow > x->x_targettime) x->x_setval = x->x_targetval;
    	else x->x_setval = x->x_setval + x->x_1overtimediff *
    	    (timenow - x->x_prevtime)
    	    * (x->x_targetval - x->x_setval);
    	x->x_prevtime = timenow;
    	x->x_targettime = clock_getsystimeafter(x->x_in1val);
    	x->x_targetval = f;
    	line_tick(x);
    	x->x_gotinlet = 0;
    	x->x_1overtimediff = 1./ (x->x_targettime - timenow);
    	clock_delay(x->x_clock,
    	    (x->x_grain > x->x_in1val ? x->x_in1val : x->x_grain));
    
    }
    else
    {
    	clock_unset(x->x_clock);
    	x->x_targetval = x->x_setval = f;
    	outlet_float(x->x_obj.ob_outlet, f);
    }
    x->x_gotinlet = 0;
}

static void line_ft1(t_line *x, t_floatarg g)
{
    x->x_in1val = g;
    x->x_gotinlet = 1;
}

static void line_free(t_line *x)
{
    clock_free(x->x_clock);
}

static void *line_new(t_floatarg f, t_floatarg grain)
{
    t_line *x = (t_line *)pd_new(line_class);
    x->x_targetval = x->x_setval = f;
    x->x_gotinlet = 0;
    x->x_1overtimediff = 1;
    x->x_clock = clock_new(x, line_tick);
    x->x_targettime = x->x_prevtime = clock_getsystime();
    if (grain <= 0) grain = 20;
    x->x_grain = grain;
    outlet_new(&x->x_obj, gensym("float"));
    inlet_new(&x->x_obj, &x->x_obj.ob_pd, gensym("float"), gensym("ft1"));
    return (x);
}

static void line_setup(void)
{
    line_class = class_new(gensym("line"), (t_newmethod)line_new, line_free,
    	sizeof(t_line), 0, A_DEFFLOAT, A_DEFFLOAT, 0);
    class_addmethod(line_class, (t_method)line_ft1,
    	gensym("ft1"), A_FLOAT, 0);
    class_addfloat(line_class, (t_method)line_float);
}

/* -------------------------- timer ------------------------------ */
static t_class *timer_class;

typedef struct _timer
{
    t_object x_obj;
    double x_settime;
} t_timer;

static void timer_bang(t_timer *x)
{
    x->x_settime = clock_getsystime();
}

static void timer_bang2(t_timer *x)
{
    outlet_float(x->x_obj.ob_outlet, clock_gettimesince(x->x_settime));
}

static void *timer_new(t_floatarg f)
{
    t_timer *x = (t_timer *)pd_new(timer_class);
    timer_bang(x);
    outlet_new(&x->x_obj, gensym("float"));
    inlet_new(&x->x_obj, &x->x_obj.ob_pd, gensym("bang"), gensym("bang2"));
    return (x);
}

static void timer_setup(void)
{
    timer_class = class_new(gensym("timer"), (t_newmethod)timer_new, 0,
    	sizeof(t_timer), 0, A_DEFFLOAT, 0);
    class_addbang(timer_class, timer_bang);
    class_addmethod(timer_class, timer_bang2, gensym("bang2"), 0);
}

void x_time_setup(void)
{
    delay_setup();
    metro_setup();
    line_setup();
    timer_setup();
}
