// Sound recorder 0.05 Aug 01 1998, GPL 2 (see COPYRIGHTS)
// 1997, 1998  B. Warmerdam

#include "record.h"

int stop;				// Arggg... Shitty global.

/* Print the available commandline options */
void printOptions(bool useCDrom)
{
	cout << "Options:"
		"\n\t-c\tNumber of channels"
		"\n\t-s\tSamplerate of the recording"
		"\n\t-b\tBits per sample [8/16]"
		"\n\t-k\tKeep going if destination file already exists (overwrite)"
		"\n\t-P\tUse a higher priority for recording thread"
		"\n\t-f\tOutput format [wav/pcm]" << ends;
	if(useCDrom == true){
		cout << "\n\t-t\tTrack to start recording"
			"\n\t-p\tTrack to stop recording"
			"\n\t-l\tTracklist to record; comma separated list (no space)"
			"\n\t-e\tExecute this statement after recording (eg 'rm $file')"
			"\n\t-C\tFull cdrom recording (seperate files, ??_file)"
			"\n\t-S\tThe number of samples to record in samplerate"
			"\n\t-O\tStarttime of a sample (MM:SS)"
			"\n\t-q\tOutput no track information to screen"
			"\n\t-D\tCDROM device (default /dev/cdrom)"
			"\n\t-M\tMixer device (default /dev/mixer)"
			"\n\t-A\tAudio device (default /dev/dsp)"
			"\n\t-Q\tQuit on exec-error" << ends;
	}

	cout << "\n\t-h\tThis information"
		"\n" << endl;
		       
}

const bool fileCanBeCreated(struct recordSettings * recSettings)
{
	struct stat fstat;
	bool status = recSettings->getForceFileCreate();

	if(status == false && stat(recSettings->getIoFile(), & fstat) == -1){
		if(errno == ENOENT)
			status = true;
		else
			perror("Stat file");
	}
	return status;
}

/* Read from /dev/dsp the audio and record it in a file */
void recordIt(struct recordSettings * recSettings)
{
	char * audio_buffer;
	unsigned long len;
	unsigned long bufferSize;
	unsigned long recordedSamples = 0L;
	unsigned long numberOfSamples = recSettings->getNumberOfSamples();

	TFileFormat * oFile = 0;

	if(recSettings->getHighPrioOnAudioThr() == true){
		int policy;
		pthread_attr_t attr;
		struct sched_param sparam;
		
		pthread_getschedparam(pthread_self(), & policy, & sparam);
		sparam.sched_priority = sched_get_priority_max(policy);
		pthread_attr_init(&attr);
		pthread_attr_setschedparam(&attr, &sparam);
	}

	DspSetting dspSetting(recSettings->getSoundDevice(), recSettings->getSampleRate(), O_RDONLY, 
				recSettings->getRecordFormat(), recSettings->getChannels());
	Dsp dsp(dspSetting);

	switch(recSettings->getFileFormat()){
		case WAV:
			oFile = new TWave;
			break;
		case PCM:
			oFile = new TPcm;
	}

	if(dsp.init()){
		if(fileCanBeCreated(recSettings) == true){
			bufferSize = dsp.getOptimalBufferSize();
			cout << "Creating file: " << recSettings->getIoFile() << "\n" << endl;

			audio_buffer = new char[bufferSize];
			if(oFile->create(recSettings->getIoFile(), 1, recSettings->getChannels(),
					 recSettings->getBitsPerSample(), recSettings->getSampleRate()) == false){
				perror("Outputfile");
			} else {
				bool countSamples = numberOfSamples > 0;
				while((len = dsp.read(audio_buffer, bufferSize)) > 0){
					if(countSamples == true && len > numberOfSamples)
						len = numberOfSamples;

					oFile->write(audio_buffer, len);
					numberOfSamples -= len;
					recordedSamples += len;

					if(countSamples == true && numberOfSamples <= 0) stop = 1;
					if(stop) break;
				}
			}
			delete [] audio_buffer;

			cout << "\nEnd of recording: " << recSettings->getIoFile() <<
				" with " << recordedSamples << " samples" << endl;
		} else {
			cerr << "File can't be created (use -k to override "
				"overwrite protection)." << endl;
			stop = 1;
		}
	} else {
		stop = 2;
		cerr << "Error during initialisation of soundcard." << endl;
	}

	oFile->close();
	delete oFile;
}

/* Show info from cdrom-recording  */
void cdromShow(struct cdRecordData * cdRecData)
{
	CDplayer::CDtrack track;
	int time = 0;
	int savedCurrentTrack;
	int trk  = 0;
	int indx = 0;
	bool quiet = cdRecData->recSettings->getQuiet();

	if(cdRecData->recSettings->getHighPrioOnAudioThr() == true){
		int policy;
		pthread_attr_t attr;
		struct sched_param sparam;
		
		pthread_getschedparam(pthread_self(), & policy, & sparam);
		sparam.sched_priority = sched_get_priority_min(policy);
		pthread_attr_init(&attr);
		pthread_attr_setschedparam(&attr, &sparam);
	}

	sleep(1);
	do {
		savedCurrentTrack = time ? track.track() : 0;
		cdRecData->cdrom->getPlayingTrack(track);	
		if(time % 10 == 0){
			time = track.timeRemaining();
			if(quiet == false){
				trk = track.track();
				indx = track.index();
			}
		} else {
			time--;
		}
		if(quiet == false){
			cout << "Number " << trk << ":" << 
				setw(2) << setfill('0') << indx << 
				" " << time / 60 << ":" << 
				setw(2) << setfill('0') <<time % 60 <<
				"\r" << flush;
		}
		sleep(1);
	} while(!stop && (!savedCurrentTrack || (time > 0 && savedCurrentTrack == track.track())) && 
		cdRecData->cdrom->audio_status() == CDROM_AUDIO_PLAY);

	if(!stop)				// normal termination (no signals)
		stop = 1;
}

/* Show info from line-recording, stops on signal */
void lineShow(struct cdRecordData * cdRecData)
{
	int time = 1;
	bool quiet = cdRecData->recSettings->getQuiet();

	if(cdRecData->recSettings->getHighPrioOnAudioThr() == true){
		int policy;
		pthread_attr_t attr;
		struct sched_param sparam;
		
		pthread_getschedparam(pthread_self(), & policy, & sparam);
		sparam.sched_priority = sched_get_priority_min(policy);
		pthread_attr_init(&attr);
		pthread_attr_setschedparam(&attr, &sparam);
	}
	sleep(1);
	do {
		if(quiet == false){
			cout << "Time " << setw(2) << setfill('0') << 
				time / 60 << ":" << time % 60 << "\r" << ends;
		}
		sleep(1);
		time = time + 1;
	} while(!stop);
}

/* Break from recording loop and exit program */
void shutdown_signal(int signal)
{
	void connect_signal(int the_signal);

	connect_signal(signal);
	stop = 2;
}

/* Connect to a new signal handler */
void connect_signal(int the_signal)
{
	if(signal(the_signal, shutdown_signal) == SIG_ERR){
		perror("Signal error");
		exit(1);
	}
}

/* Get the next track from the track-record list */
int getNextTrack(unsigned & trackIdx, struct recordSettings & recSettings)
{
	if(trackIdx >= 1 && trackIdx <= 99){
		for(; trackIdx <= 99; trackIdx++){
			if(recSettings.track_list[trackIdx]){
				return recSettings.track_list[trackIdx++];
			}
		}
	}

	return -1;
}

/* Test if there are more than one song listed in te 'playlist' */
const bool isMultiTrack(struct recordSettings * recSettings)
{
	return recSettings->track_list[2] != 0;
}

void checkMixerSettingsForCd(struct recordSettings * recSettings)
{
	Mixer mixer;

	mixer.bindMixerToDevice(recSettings->getMixerDevice());

	Channel & channel = mixer.getChannel(SOUND_MIXER_CD);
	if(channel.canChannelRecord() == false){
		cerr << "CDROM device can't record!!" << endl;
		stop = 2;
	} else {
		if(channel.isChannelInRecordState() == false){
			channel.setChannelRecordState(true);
			cerr << "Warning: Settings of mixer were adjusted to record from CD!" << endl;
		}
	}
}

/* Control cdplayer to record a track from dsp */
void cdrom_record(struct recordSettings * recSettings)
{
	void * retval;
	unsigned current_track, trackIdx;
	pthread_t pa, pb;
	char file[256];
	const char * iofile = recSettings->getIoFile();
	struct cdRecordData cdRecData;

	if(iofile != 0){
		CDplayer cdrom;

		cdrom.setDevice(recSettings->getCdromDevice());
		if(cdrom.ready()){
			checkMixerSettingsForCd(recSettings);

			connect_signal(SIGINT);		// for now
			connect_signal(SIGTERM);
			connect_signal(SIGQUIT);

			trackIdx = 1;
			recSettings->setIoFile(file);	// Set pointer to file
			cdRecData.cdrom = & cdrom;
			cdRecData.recSettings = recSettings;
			while((current_track = getNextTrack(trackIdx, (*recSettings))) &&
					current_track <= cdrom.lastTrack() &&
					current_track > 0 && stop != 2){
				if(cdrom.isAudioTrack(current_track)){

					if(isMultiTrack(recSettings))
						sprintf(file, "%02d_%s", current_track, iofile);
					else
						sprintf(file, "%s", iofile);
					cdrom.play(current_track, current_track, recSettings->getStartTimeOfSamples());
					stop = 0;

					pthread_create(&pa,0,(void *(*)(void *)) recordIt, recSettings);
					pthread_create(&pb,0,(void *(*)(void *)) cdromShow, & cdRecData);
					pthread_join(pa,(void **)&retval);
					pthread_join(pb,(void **)&retval);

					if(stop == 1 && recSettings->getExecStmt() != 0){
						char * cmd = new char[50 + strlen(recSettings->getIoFile()) + 
								strlen(recSettings->getExecStmt())];
						sprintf(cmd, "declare -x file=%s;%s", recSettings->getIoFile(),
							recSettings->getExecStmt());

						int pid, status;
						if(!(pid = fork())){
							execl("/bin/sh", "sh", "-c", cmd, 0);
							exit(0);
						}

						waitpid(pid, &status, 0);

                                                if(status != 0 && recSettings->getQuitOnError() == true){
                                                        cerr << "Child exits with " << status << endl;
                                                        stop = 2;
                                                }
 
						delete [] cmd;
					}
					if(stop == 1) stop = 0;
				} else {
					cerr << "Cannot read CDROM data-tracks." << endl;
				}
			}
		}

		cdrom.stop();

		if(stop == 2){
			cerr << "Unlinking file due to abnormal termination." << endl;
			unlink(recSettings->getIoFile());
		}
	}
}

/* record the source on /dev/dsp and quit on signals */
void line_record(struct recordSettings * recSettings)
{
	void * retval;
	pthread_t pa, pb;
	struct cdRecordData cdRecData;

	stop = 0;
	if(recSettings->getIoFile()){
		connect_signal(SIGINT);		// for now
		connect_signal(SIGTERM);

		cout << "Record from dsp (no cdrom support).\nTo end the recording press CTRL-C" << endl;

		cdRecData.cdrom = 0;
		cdRecData.recSettings = recSettings;

		pthread_create(&pa,0,(void *(*)(void *)) recordIt, recSettings);
		pthread_create(&pb,0,(void *(*)(void *)) lineShow, &cdRecData);
		pthread_join(pa,(void **)&retval);
		pthread_join(pb,(void **)&retval);
	}
}

// Process info from a rc-file to recSettings object
void readSettings(RCfile & rc, recordSettings  & recSettings)
{
	if(rc.isOpen() == true){
		char buffer[LINEBUFFERSIZE];

		if(rc.getEntry("channels", buffer, sizeof(buffer)) == true){
			if(!strcmp(buffer, "mono")){
				recSettings.setChannels(1);
			} else {
				if(strcmp(buffer, "stereo")) {
					cerr << "Illegal channels setting: " <<
						buffer << " defaulting to stereo." << endl;
				}
				recSettings.setChannels(2);
			}
		}

		if(rc.getEntry("samplerate", buffer, sizeof(buffer)) == true){

			unsigned sampleRate;
			sscanf(buffer, "%ul", &sampleRate);

			recSettings.setSampleRate(sampleRate);
		}

		if(rc.getEntry("fileformat", buffer, sizeof(buffer)) == true){
			if(!strcmp(buffer, "wave")) recSettings.setFileFormat(WAV);
			if(!strcmp(buffer, "pcm")) recSettings.setFileFormat(PCM);
		}

		if(rc.getEntry("format", buffer, sizeof(buffer)) == true){
			struct sSoundType {char * name; int format;} r[] =
				{ {"MU LAW",    AFMT_MU_LAW},    {"A LAW", AFMT_A_LAW},
				  {"IMA ADPCM", AFMT_IMA_ADPCM}, {"U8",    AFMT_U8},
				  {"U16le",     AFMT_U16_LE},    {"U16be", AFMT_U16_BE},
				  {"S8",        AFMT_S8},        {"S16le", AFMT_S16_LE},
				  {"S16be",     AFMT_S16_BE}
				};

			for(int i = 0; i < 9; i++){
				if(!strcmp(buffer, r[i].name))
					recSettings.setRecordFormat(r[i].format);
			}
		}

		if(rc.getEntry("QuitOnExec", buffer, sizeof(buffer)) == true){
			if(!strcmp(buffer, "true")) recSettings.setQuitOnError(true);
			if(!strcmp(buffer, "false")) recSettings.setQuitOnError(false);
		}

		if(rc.getEntry("sounddevice", buffer, sizeof(buffer)) == true){
			int len = strlen(buffer);
			while(len > 0 && buffer[len-1] == ' ')
				buffer[--len] = '\0';
			recSettings.setSoundDevice(buffer);
		}

		if(rc.getEntry("mixerdevice", buffer, sizeof(buffer)) == true){
			int len = strlen(buffer);
			while(len > 0 && buffer[len-1] == ' ')
				buffer[--len] = '\0';
			recSettings.setMixerDevice(buffer);
		}

		if(rc.getEntry("cdromdevice", buffer, sizeof(buffer)) == true){
			int len = strlen(buffer);
			while(len > 0 && buffer[len-1] == ' ')
				buffer[--len] = '\0';
			recSettings.setCdromDevice(buffer);
		}
	}
}

/* Set the default settings for program */
void defaultSettings(recordSettings & recSettings)
{
	char * home, * home_env = getenv("HOME");
	home = new char[strlen(home_env) + 50];

	strcpy(home, "/etc/soundrecorder.conf");
	RCfile rc(home, false);

	strcpy(home, home_env);
	strcat(home, "/.soundrecorder");

	rc.open(home, false, true);
	readSettings(rc, recSettings);
	rc.close();

	delete [] home;
}

/* Create a list with the tracknumbers from -tlst argument */
void expandList(struct recordSettings & recSettings, char * track_list)
{
	unsigned i = 0, from, to = 1, dst = 1;
	while(i < strlen(track_list)){
		from = 0;
		if(track_list[i] < '0' || track_list[i] > '9'){
			if(track_list[i] == '-')
				from = to;
			to = 0;
			i++;
		}

		sscanf(track_list + i,"%d", & to);
		while(track_list[i] >= '0' && track_list[i] <= '9') i++;
		if(from){
			for(unsigned j = from + 1; j <= to; j++)
				recSettings.track_list[dst++] = j;
		} else {
			recSettings.track_list[dst++] = to;
		}
	}
}

const bool threadScheduleAvailable()
{
#ifdef _POSIX_THREAD_PRIORITY_SCHEDULING
	return true;
#else
	return false;
#endif
}

/* Extract extra parameters for recordsettings */
bool parseArgs(int argc, char ** argv, recordSettings & recSettings, bool useCDrom)
{
	bool status = true;
	long lng;
	int opt, i, min, sec, tstt = 0, tstp = 0;
	defaultSettings(recSettings);

	while((opt = getopt(argc, argv, useCDrom ? "c:s:f:t:p:l:e:S:O:D:M:A:b:ChkPqQ" : "c:s:f:b:hPq")) != -1){
		switch(opt){
			case '?':
			case 'h':
				printOptions(useCDrom);
				return false;
			case 'c':
				sscanf(optarg, "%lu", &lng);
				status &= recSettings.setChannels(lng);
				break;
			case 's':
				sscanf(optarg, "%lu", &lng);
				status &= recSettings.setSampleRate(lng);
				break;
			case 'f':
				if(!strcasecmp("wav", optarg)) status &= recSettings.setFileFormat(WAV);
				if(!strcasecmp("pcm", optarg)) status &= recSettings.setFileFormat(PCM);
				break;
			case 'b':
				sscanf(optarg, "%d", &i);
				recSettings.setBitsPerSample(i == 8 ? 8 : 16);
				recSettings.setRecordFormat(i == 8 ? AFMT_U8 : AFMT_S16_LE);
				break;
			case 't':
				sscanf(optarg, "%d", &tstt);
				recSettings.track_list[1] = tstt;
				break;
			case 'p':
				if(tstt > 0){
					sscanf(optarg, "%d", &tstp);
					for(i = 1; tstt <= tstp; i++){
						recSettings.track_list[i] = tstt++;
					}
				} else {
					cerr << "No start given with -t" << endl;
				}
				break;
			case 'l':
				expandList(recSettings, optarg);
				break;
			case 'e':
				status &= recSettings.setExecStmt(optarg);
				break;
			case 'C':
				for(i = 1; i < 99; i++)
					recSettings.track_list[i] = i;
				break;
			case 'S':
				sscanf(optarg,"%lu", &lng);
				status &= recSettings.setNumberOfSamples(lng);
				break;
			case 'O':
				sscanf(optarg, "%d:%d", &min, &sec);
				status &= recSettings.setStartTimeOfSamples(min * 60 + sec);
				break;
			case 'q':
				recSettings.setQuiet(true);
				break;
			case 'k':
				recSettings.setForceFileCreate(true);
				break;
			case 'P':
				recSettings.setHighPrioOnAudioThr(true);
				break;
			case 'Q':
				recSettings.setQuitOnError(true);
				break;
			case 'D':
				recSettings.setCdromDevice(optarg);
				break;
			case 'M':
				recSettings.setMixerDevice(optarg);
				break;
			case 'A':
				recSettings.setSoundDevice(optarg);
				break;
		}
	}

	status &= recSettings.setIoFile(argv[optind]);

	if(recSettings.getHighPrioOnAudioThr() == true && threadScheduleAvailable() == true){
		cerr << "Priority scheduling not supported on this system." << endl;
		recSettings.setHighPrioOnAudioThr(false);
	}
	
	if(status == false){
		printOptions(useCDrom);
	}

	return status;
}

/* Commandline coordinated use of binary */
int main(int argc, char **argv)
{
	cout << "Sound Recorder version " VERSION "\n"
		"Copyright (C) 1997, 1998 by B. Warmerdam under GPL.\n"
		"This program is free software and comes with ABSOLUTELY NO WARRANTY.\n" << endl;

	struct recordSettings recSettings;
	bool useCDrom = (strlen(argv[0]) >= 16 && ! strcmp(argv[0] + 
		strlen(argv[0]) - 16, "cdsound-recorder"));

	if(parseArgs(argc, argv, recSettings, useCDrom) == true){
		if(useCDrom == true){
			cdrom_record(& recSettings);
		} else {
			line_record(& recSettings);
		}
	}
	return 0;
}
