/*
  Converts an rdbtable record to HTML3 table rows.

  Author: Carlo Strozzi <carlos@linux.it>
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#define MAX_ARG_LENGTH    1024
#define LIST_ALL          "(all)"
#define EMPTY			  ""

void show_help( char *my_name)
{
    printf("
        NoSQL operator: %s

Usage:  %s [options]

Options:
    -h      Print this help info.
    -i 'L'  Blank-separated list of input fields. Only the table columns
            in list 'L' will be shown as input boxes in the output form.
            All the others will be read-only. 'L' must be one single token
            to the shell, so it must be quoted if it lists more than
            one column. The special word '(all)' means that all the
            columns are to be shown as input fields.
    -M nnn  Set maximum length for input fields (HTML 'maxlength' field
            attribute). If a table column is wider than 'nnn', then a
            textarea is displayed, otherwise an ordinary entry field is shown.
    -u      Prefix textarea field names with an underscore (_). This feature
            is meant for using with the 'uncgi' program. 
            (See http://hyperion.com/~koreth/uncgi.html for more details).

Converts an rdbtable row to HTML3 table rows. Each column in the rdbtable
is turned into a separated HTML table row. Only the last row of the input
rdbtable is used, regardless of whether the former contains more records. If
the input table has no records, then nothing is printed. Characters that are
special to the Unix shell must be escaped on the command line. %s
produces only the body of the HTML table; table heading and footing, as well
as any <FORM></FORM> tags, must be provided by the calling program.

Each HTML field is prefixed by a label (field name). The label is the first
blank-separated token of the corresponding column documentation text. If the
latter is blank then the rdbtable column name is used instead, followed by a
colon (:).
If more than one blank-separated words from the column documentation text
have to be used for an HTML field label, just escape blanks with the ASCII
sequence &#32; to form single tokens.  For instance, suppose that a column
documentation contains the phrase 'Job description of the person', and you
want 'Job description' to be the HTML field label, simply change the above
phrase into 'Job&#32;description of the person' in the column documentation,
to make the first two words be one single token.

This operator reads an rdbtable from STDIN and prints the HTML output to
STDOUT.

An example of usage of this operator from within a shell script is :

                 %s -i 'NAME JOB' < rdbtable

which means that only the columns NAME and JOB of 'rdbtable' will be displayed
as HTML entry fields, while all the others will be read-only text.


'$Id: nsq-t2html3.c,v 1.1 1998/05/29 20:43:01 carlos Exp $'

            ----------------------
NoSQL RDBMS, Copyright (C) 1998 Carlo Strozzi.
This program comes with ABSOLUTELY NO WARRANTY; for details
refer to the GNU General Public License.

You should have received a copy of the GNU General Public License
along with this program;  if not, write to the Free Software
Foundation, Inc., 59 Temple Place Suite 330, Boston, MA 02111-1307
USA.
            ----------------------\n",my_name,my_name,my_name,my_name,"\\");
}

int main( int  argc, char *argv[] ) {
  
  /* For getopt() */
  extern char* optarg;
  extern int optind;

  /* for the rest of the program. */
  register int
	a_loop;
  char
	*my_name=argv[0],
	cmd_buf[MAX_COMMAND_LENGTH],
	i_list[MAX_ARG_LENGTH]="",
	max_width[MAX_ARG_LENGTH]="64";

  int debug=0, uncgi=0, i_all=0;

  while ((a_loop = getopt(argc, argv, "xhui:M:")) != EOF) {
    switch (a_loop) {
      case 'h':
        show_help(my_name);
        exit(0);
        break;
      case 'x': 
        debug=1;
        break;
      case 'u': 
        uncgi=1;
        break;
      case 'i': 
        snprintf(i_list,MAX_ARG_LENGTH,"%s", optarg);
		if( !strcmp(i_list, LIST_ALL ))  i_all=1;
        break;
      case 'M': 
		snprintf(max_width,MAX_ARG_LENGTH,"%s", optarg);
        break;
      default:
        show_help(my_name);
        exit(1);
    }
  }

  snprintf(cmd_buf,MAX_COMMAND_LENGTH,"#
#
BEGIN { NULL=\"\"; FS=\"\\t\"; OFS=FS;
  # Get list of type=\"input\" fields.
  split(\"%s\", i_list, \" \")
  while( i_list[++c] != NULL ) f_type[i_list[c]] = \"i\"
}
# Table comments.
r==0 && $0 ~ /^ *#/ { next }
# Column names.
r==0 { split($0, c_names); r++; next; }
# Column definitions.
r == 1 {
  hdr_nr=NR; c=0;
  split($0,c_defs)
  while(c_defs[++c] != NULL) {
	a1 = substr(c_defs[c],1,index(c_defs[c],\" \"))
	gsub(/[^0-9]+/,NULL,a1)
	if( a1==NULL ) {
	  a1=c_defs[c]
	  c_width[c] = a1 + 0
	  c_label[c]=c_names[c] \":\"
	}
	else {
	  # Turn length into a number.
	  c_width[c] = a1 + 0
	  a2 = substr(c_defs[c],index(c_defs[c],\" \"))
	  sub(/^ */,NULL,a2)
	  c_label[c] = a2
	}
  }
  r++; next;
}
END {
  if( hdr_nr == NR ) { exit } # No records in table, just quit.
  c=0
  while( ++c <= NF ) {
	out_rec = \"<td>\" c_label[c] \"</td>\"
	f_name = c_names[c]
    if( %d || f_type[f_name] == \"i\" ) {
	  if( c_width[c] > %s ) {
		n_rows = int(c_width[c] / %s)
		add_row = c_width[c] %s %s
		if(add_row > 0) n_rows++
		# Uncgi handles a textarea specially if its name
		# starts with \"_\".
		if(%d) f_name = \"_\" f_name
		out_rec = out_rec \"<td><textarea name=\\042\"\
		  f_name \"\\042 rows=\\042\" n_rows \"\\042\
		  cols=\\042%s\\042>\" $c \"</textarea></td>\"
	  }
	  else {
		out_rec = out_rec \"<td><input type=text name=\\042\"\
		  f_name \"\\042 size=\\042\" c_width[c] \"\\042\
		  maxlength=\\042\" c_width[c] \"\\042 value=\\042\"\
		  $c \"\\042></td>\"
	  }
	}
	else out_rec = out_rec \"<td>\" $c \"</td>\"
	print \"<tr>\" out_rec \"</tr>\"
  }
}",i_list,i_all,max_width,max_width,"%",max_width,uncgi,max_width);

  if( debug )
	fprintf (stderr, "Generated AWK program:
	  ----------\n%s\n----------\n",cmd_buf);

  execlp(AWK,"awk",cmd_buf,NULL);
  exit(0);
}

