/*
  Converts an rdbtable to shell variable assignments.

  Author: Carlo Strozzi <carlos@linux.it>
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#define MAX_ARG_LENGTH  256
#define EMPTY			""

void show_help( char *my_name)
{
    printf("
        NoSQL operator: %s

Usage:  %s [options]

Options:
    -h    Print this help info.
    -p P  Prefix each output variable (column) name with string 'P'.

Converts an rdbtable to shell variable format (VARIABLE=value), handy
for grabbing a record into a shell program. If the rdbtable contains
more than one row of data, then the assignments will be those of the
last row. Any single-quotes in the 'value' part of the assignment are
escaped with the ASCII sequence '&#39;', not to cause troubles to the
invoking shell.

This operator reads an rdbtable from STDIN and prints the requested info
to STDOUT.

An example of usage of this operator from within a shell script is :

                 eval `%s < rdbtable`


'$Id: nsq-t2sh.c,v 1.1 1998/05/29 20:43:01 carlos Exp $'

            ----------------------
NoSQL RDBMS, Copyright (C) 1998 Carlo Strozzi.
This program comes with ABSOLUTELY NO WARRANTY; for details
refer to the GNU General Public License.

You should have received a copy of the GNU General Public License
along with this program;  if not, write to the Free Software
Foundation, Inc., 59 Temple Place Suite 330, Boston, MA 02111-1307
USA.
            ----------------------\n",my_name,my_name,my_name,"\\");
}

int main( int  argc, char *argv[] ) {
  
  /* For getopt() */
  extern char* optarg;
  extern int optind;

  /* for the rest of the program. */
  register int
	a_loop;
  char
	*my_name=argv[0],
	cmd_buf[MAX_COMMAND_LENGTH],
	prefix[MAX_ARG_LENGTH];

  int debug=0;

  while ((a_loop = getopt(argc, argv, "xhp:")) != EOF) {
    switch (a_loop) {
      case 'h':
        show_help(my_name);
        exit(0);
        break;
      case 'x': 
        debug=1;
        break;
      case 'p': 
        snprintf(prefix,MAX_ARG_LENGTH,"%s", optarg);
        break;
      default:
        show_help(my_name);
        exit(1);
    }
  }

  snprintf(cmd_buf,MAX_COMMAND_LENGTH,"#
#
BEGIN { NULL=\"\"; FS=\"\\t\"; OFS=FS;}
# Table comments.
r==0 && $0 ~ /^ *#/ { next }
# Column names.
r==0 { split($0, c_names); r++; next; }
# Column definitions.
r == 1 { hdr_nr=NR; r++; next; }
END {
  if( hdr_nr != NR )
    while( c_names[++c] != NULL ) {
	  # Any single quotes MUST be escaped in data.
	  gsub( \"'\", \"\\&#39;\", $c )
	  print \"%s\" c_names[c] \"='\" $c \"'\"
	}
  else
	while( c_names[++c] != NULL )
	  print \"%s\" c_names[c] \"='\" NULL \"'\"
}",prefix, prefix);

  if( debug )
	fprintf (stderr, "Generated AWK program:
	  ----------\n%s\n----------\n",cmd_buf);

  execlp(AWK,"awk",cmd_buf,NULL);
  exit(0);
}

