/*
 * html.c - display code for html
 *
 * This file contains the code for displaying HTML documents, scrolling them
 * and dealing with hypertext jumps by recognising which buttons are clicked.
 */
/*
 * Janne Saarela
 * janne.saarela@hut.fi
 * 28.7.1995
 *
 * The SEQTEXT element in the PaintStream has the additional
 * attribute ``emph'' of width of two bytes.
 * The handling of different emph attributes is not finalized.
 * I tried using < and > symbols from the adobe symbol font
 * but they do not seem to work properly. OVERHAT, OVERDOT,
 * OVERDDOT and OVERTILDE are to be implemented.
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "arena.h"
#include "defs.h"
#include "types.h"
#include "bridge.h"
#include "colour.h"
#ifdef BOOKMARKS
#  include "bookmark.h"
#endif
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
#  include "debug.h"
#endif
#include "display.h"
#include "dither.h"
#include "error.h"
#include "forms.h"
#include "html.h"
#include "image.h"
#include "main.h"
#include "spawn.h"
#include "parsehtml.h"
#include "status.h"
#include "style.h"
#include "util.h"
#include "x11.h"


Byte colour_text_ix, colour_background_ix;


/*
 * globals associated with detecting which object is under the mouse
 */
char *anchor_start, *anchor_end;


/*
 * The background frame structure specifies:
 *   a) where to start painting in this frame
 *   b) where this frame ends in paint buffer
 *   c) tree of nested frames which all intersect the top of the window
 */
Frame background;


/*
 * Copy line from str to LineBuf, skipping initial spaces and SGML <tags>.
 */
#define ARENA_CopyLine_BUFFER_CAPACITY 1024
char* CopyLine(char* str, unsigned int len) /* extremely lame implementation */
{
 int c, n, k;
 char* p;
 static char buf[ARENA_CopyLine_BUFFER_CAPACITY];


 if (len == 0) return "";
 if (len > (ARENA_CopyLine_BUFFER_CAPACITY - 1))
   len = (ARENA_CopyLine_BUFFER_CAPACITY - 1);

 for (p = buf; (len--) && (c = (unsigned char)*str); )
   {
    if ((n = (unsigned char)str[1]))
      {
      /*
       * Skip tag(s).
       */
       while (c == '<' &&
	      (n == '!' || n == '?' || n == '/' || isalpha(n)))
	 {
	  if ((str = ((n == '!') ?
		      FindEndComment(str) : ARENA_Index(str, '>'))))
	    {
	     if (*str)
	       {
		c = (unsigned char)*++str;
		n = (unsigned char)str[1];
		continue;
	       }
	    }

	  c = '\0';
	  break;
	 }

       if (c)
	 {
	  /* howcome 25/11/94: isalpha -> isalnum */
	  if (c == '&' && (isalnum(n) || n == '#'))
	    {
	     if ((n = entity(str + 1, &k)))
	       {
		str += k;
		*p++ = n;
		continue;
	       }
	    }

	  if (preformatted)
	    {
	     if (c == '\t')
	       {
		if ((k = 8 - (p - buf)%8) > 0)
		  for (len++; (k--) && (len--); ) *p++ = ' ';
	       }
	      else
	       {
		*p++ = c;
	       }

	     ++str;
	     continue;
	    }

	  if (isspace(c))
	    {
	     while (isspace((unsigned char)*++str));

	     c = ' ';
	    }
	   else
	    str++;

	  *p++ = c;
	 }
        else
	 break;
      }
     else
      {
       *p++ = c;
       break;
      }
   }

 *p = '\0';
 return buf;
}
#undef ARENA_CopyLine_BUFFER_CAPACITY


/*
 * Find title text as defined by ``<TITLE>title text</TITLE>''.
 */

#define MAXTITLE 128

char *TitleText(char *buf)
{
 return "Dummy Title";
}


/*
 * Move forwards thru frame, adjusting frame->top to last text line before top.
 * The routine iterates thru peer frames and recurses thru decendent frames.
 * The frame (and its descendents) are removed from the peer list if it
 * (and therefore they) finish before top.
 * This is implemented by returning the frame if it is needed
 * otherwise returning NULL
 */
/* Question: what happens when top is the bottom of the buffer?
 *           can this ever occur in practice, e.g. with a null file?
 */
Frame* FrameForward(Frame* frame, long top)
{
 long offset, height;
 unsigned int width, length, len;
 int tag, indent, style, border;
 unsigned char *p, *p_end, *obj;
 Frame *peer, *child, *last;
#ifdef STYLE_COLOUR_BORDER
 Byte cb_ix;
#endif
#ifdef ARENA_DEBUG
 char Iam[] = "FrameForward";
#endif


 if (frame == NULL) return NULL;

#ifdef ARENA_DEBUG
 if (DISPLAY_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam, " frame "POINTER_FORMAT", top %d.\n", frame, top);
#endif

 /* move down each of peer frames */
 peer = FrameForward(frame->next, top);

 /* Does this frame and its descendants end before top ? */
 if (frame->offset + frame->height <= top)
   {
    if (frame == &background) /* should never occur in practice! */
      {
       FreeFrames(background.child);
       background.child = NULL;
       background.top = (unsigned char*)PaintStreamCurrentPosition();
       return &background; /* never has peers ! */
      }

    /* remove self and any descendents BUT not our peers! */
    frame->next = NULL;  /* unlink from list before freeing */
    FreeFrames(frame);
    frame = NULL;
    return peer;
   }

 frame->next = peer;
 frame->child = FrameForward(frame->child, top);

 /* find last child in list to avoid inserting new children in reverse order */
 if ((last = frame->child)) while (last->next) last = last->next;

 /* Now move `frame->top' down until we reach top
  * and insert any new children at end of frame->child list.
  */
 p = frame->top;
 p_end = ((paint + frame->info) + FRAMESTLEN) + frame->length;

 while (p < p_end)
   {
    if (p >= (unsigned char*)PaintStreamCurrentPosition())
      {
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
       Arena_TracePrint(Iam,
			" Panic: ran off end of paint buffer! Exiting.\n");
# else
       Arena_PrintError(_("Panic: ran off end of paint buffer! Exiting.\n"));
#endif
       Exit(1);
      }

#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
    if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       " tag 0x%x read from "POINTER_FORMAT"."
		       " (p_end = "POINTER_FORMAT")\n",
		       *p, p, p_end);
#endif

    obj = p;
    tag = *p++;
    {
     Bool BreakBreak = False;

     switch (tag)
       {
       /*
	* If frame intersects top then create frame structure
	* and find paint position within it (and its children)
	* then insert it in front of frame->child list
	*/
        case BEGIN_FRAME:
	  offset = GetLong(p);

	  if (offset > top)
	    {
	     frame->top = obj;
	     BreakBreak = True;
	     break;
	    }

	  /* otherwise pickup frame header params */

	  indent = GetInt(p);
	   width = GetInt(p);
	  height = GetLong(p);
	  style = *p++;
	  border = *p++;
#ifdef STYLE_COLOUR_BORDER
	  cb_ix = *p++;
#endif
#ifdef STYLE_BACKGROUND
	  /* must be set to the structures defined by the style
	   * --Spif 18-Oct-95
	   */
	  SkipPointer(p);
#endif
	  length = GetInt(p);
	  if (offset + height > top)
	    {
	     child = (Frame *)Arena_MAlloc(sizeof(Frame), False);
	     child->next = NULL;
	     child->child = NULL;
	     child->offset = offset;
	     child->indent = indent;
	     child->width = width;
	     child->height = height;
	     child->info = obj - paint;
	     child->top = p;
	     child->length = length;
	     child->style = style;
	     child->border = border;
	     child->box_list = NULL;
#ifdef STYLE_COLOUR_BORDER
	     child->cb_ix = cb_ix;
#else
	     child->cb_ix = 0;
#endif
#ifdef ARENA_DEBUG
	  if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
	    Arena_TracePrint(Iam,
			     " frame = "POINTER_FORMAT","
			     " child = "POINTER_FORMAT".\n"
			     "\tinfo = 0x%x, top = 0x%x, length = %d.\n",
			     frame, child,
			     child->info, child->top, child->length);
#endif
	     FrameForward(child, top);

	     /* and insert new child in same order
	      * as it appears in paint buffer
	      */
	     if (last)
	       last->next = child;
	      else
	       frame->child = child;

	     last = child;
	    }

	  p += length; /* to skip over frame's contents */
	  break;

	     /* the END_FRAME is only used when scrolling backwards */
        case END_FRAME:
	  p += FRAMENDLEN - 1;
	  break;

	  /* stop if textline overlaps top or starts after it */
        case TEXTLINE:
	  offset  = GetLong(p);
	  if (offset >= top)
	    {
	     frame->top = obj;
	     BreakBreak = True;
	     break;
	    }

	  SkipInt(p); /* skip baseline */
	  SkipInt(p); /* skip   indent */
	  height = GetInt(p);

	  if (offset + height > top)
	    {
	     frame->top = obj;
	     BreakBreak = True;
	     break;
	    }

	  /* skip elements in text line to reach next object */
	  while ((tag = *p++) != '\0')
	    {
	     switch (tag & 0xF)
	       {
	        case RULE:
		  p += RULEFLEN - 1;
		  break;

	        case LINE:
		  p += LINEFLEN - 1;
		  break;

	        case BULLET:
		  p += BULLETFLEN - 1;
		  break;

	        case STRING:
		  p += STRINGFLEN - 1;
		  break;

	        case SEQTEXT:
		  SkipInt(p);	/* skip over emph */
		  p++;		/* skip over font */
#ifdef STYLE_COLOUR
		  p++;		/* skip over colour */
#endif
#ifdef STYLE_BACKGROUND
		 /* skip over background --Spif 18-Oct-95 */
		  SkipPointer(p);
#endif
		  SkipInt(p); /* skip over x position */
		  SkipInt(p); /* skip over y position */
		  len = *p++;
		  p += len;
		  break;

	        case IMAGE:
		  p += IMAGEFLEN - 1;
		  break;

	        case INPUT:
		  p += INPUTFLEN - 1;
		  break;

	        default:
		  /* safety net for garbled paint structure */
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
		  Arena_TracePrint(Iam,
				   " ERROR! Unexpected (1) internal tag 0x%x,"
				   " Exiting.\n",
				   tag);
# else
		  Arena_PrintError(_("Unexpected (%s) internal tag 0x%x.\n"),
				   "1", tag);
#endif
		  Exit(1);
	       }
	     /* End ``switch (tag & 0x0F)'' */

	     if (BreakBreak)
	       {
		BreakBreak = False;
	        break;
	       }
	    }
	  /* End ``while ((tag = *p++) != '\0')'' */
	  break;

	 /* safety net for garbled paint structure */
        default:
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
	  Arena_TracePrint(Iam,
			   " ERROR! Unexpected (2) internal tag 0x%x,"
			   " Exiting.\n", tag);
# else
	  Arena_PrintError(_("Unexpected (%s) internal tag 0x%x.\n"), "2",tag);
#endif
	  Exit(1);
	  break;
       }
     /* end ``switch (tag)'' */

     SkipInt(p);  /* skip over textline size param */
     if (BreakBreak) break;
    }
   }
 return frame;
}


/*
 * Move backwards through frame, adjusting frame->top to last text line
 * before top. The routine iterates through peer frames and recurses
 * through decendent frames. The frame (and its descendents) are removed
 * from the peer list if it (and therefore they) start at or after top.
 * This is implemented by returning the frame if it is needed otherwise
 * returning NULL.
 */
Frame* FrameBackward(Frame* frame, long top)
{
 Bool known;
 long offset, height;
 unsigned int width, length;
 int tag, indent, style, border, k;
 unsigned char *p, *p1, *p_begin, *obj;
 Frame *peer, *child;
#ifdef STYLE_COLOUR_BORDER
 Byte cb_ix;
#endif
#ifdef ARENA_DEBUG
 char Iam[] = "FrameBackward";
#endif


 if (frame == NULL) return NULL;

#ifdef ARENA_DEBUG
 if (DISPLAY_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam, " frame "POINTER_FORMAT", top %d.\n", frame, top);
#endif

 /* if this frame starts after top then remove from peer list */
 if (frame->offset >= top)
   {
    if (frame == &background)
      {
       FreeFrames(background.child);
       background.child = NULL;
       background.top = (unsigned char*)PaintStreamCurrentPosition();
       return &background; /* never has peers ! */
      }

    peer = frame->next;
    frame->next = NULL; /* unlink from list before freeing */
    FreeFrames(frame);
    frame = NULL;
    return FrameBackward(peer, top);
   }

 /* move backwards through peer frames */
 frame->next = FrameBackward(frame->next, top);

 /* move backwards through current children */
 frame->child = FrameBackward(frame->child, top);

 /* Now move `frame->top' back until we reach top
  * and insert any new children in front of frame->child list.
  */
 p = frame->top;
 p_begin = (paint + frame->info) + FRAMESTLEN;

 while (p_begin < p)
   {
   /* pop field size into k */
    p -= ARENA_INT_SIZE; k = GetInt(p); p -= ARENA_INT_SIZE;

#ifdef ARENA_DEBUG	/* QingLong.16-10-97: Sanity check. */
    if (k == 0)
      {
       Arena_TracePrint(Iam, " ERROR! `0' frame size causes deadloop.\n");
       Exit(-1);
      }
#endif

    {
     Bool BreakBreak = False;

#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
    if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       " tag 0x%x read from "POINTER_FORMAT"."
		       " (p_begin = "POINTER_FORMAT")\n",
		       *p, p, p_begin);
#endif

     obj = (p -= k); /* p now points to start of previous object */
     tag = *p++;

     switch (tag)
       {
        case BEGIN_FRAME:
	  p--;
	  break;

       /*
	* If frame intersects top then create frame structure
	* unless it already is one of my children and find
	* paint position within it (and its children)
	* then insert it in front of frame->child list
	*/
        case END_FRAME:
	  length = GetInt(p);
#ifdef ARENA_DEBUG	/* QingLong.16-10-97: Sanity check. */
	  if (length == 0)
	    {
	     Arena_TracePrint(Iam,
			      " ERROR! Zero frame length causes deadloop.\n"
			      "\tEndFrame at "POINTER_FORMAT".\n",
			      (p - ARENA_INT_SIZE - 1));
	     Exit(-1);
	    }
#endif
	  SkipInt(p);	/* Skip size field (constant) */
	  p = obj - (int)length;  /* kludge for bug in
				     HP92453-01 A.09.19 HP C Compiler */

	  /* p now points to BEGIN_FRAME tag */

	  if (*p != BEGIN_FRAME)
	    {
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
	      Arena_TracePrint(Iam,
			       " ERROR!\n"
			       "\tUnexpected (3) internal tag 0x%x,\n"
			       " when BEGIN_FRAME was expected, Exiting.\n",
			       tag);
# else
	      Arena_PrintError(_("Unexpected (%s) internal tag 0x%x.\n"),
			       "3", tag);
#endif
	      Exit(1);
	    }

	  p1 = p;

	  /* check if frame is one of my children, and hence already done */

	  for (known = False, child = frame->child; child; child = child->next)
	    {
	     if (child->info == p - paint)
	       {
		known = True;
		p = obj;
		break;
	       }
	    }

	  if (known) break;

	  p++; /* skip over frame tag */

	  offset = GetLong(p);
	  if (offset >= top)
	    {
	     p = obj;
	     break;
	    }

	  /* otherwise pickup frame header params */

	  indent = GetInt(p);
	  width = GetInt(p);
	  height = GetLong(p);
	  style = *p++;
	  border = *p++;
#ifdef STYLE_COLOUR_BORDER
	  cb_ix = *p++;
#endif
#ifdef STYLE_BACKGROUND
	  /* must be set to the structures defined by the style
	   * --Spif 18-Oct-95
	   */
	  SkipPointer(p);
#endif
	  length = GetInt(p);

#ifdef ARENA_DEBUG	 /* QingLong.16-10-97: Sanity check. */
	  if (length == 0)
	    Arena_TracePrint(Iam, " STRANGE! Zero frame length.\n");
#endif

	  child = (Frame*)Arena_MAlloc(sizeof(Frame), False);
	  child->next = NULL;
	  child->child = NULL;
	  child->offset = offset;
	  child->indent = indent;
	  child->width = width;
	  child->height = height;
	  child->info = p1 - paint;
	  child->top = p + length;
	  child->length = length;
	  child->style = style;
	  child->border = border;
	  child->box_list = NULL;
#ifdef STYLE_COLOUR_BORDER
	  child->cb_ix = cb_ix;
# else
	  child->cb_ix = 0;
#endif
#ifdef ARENA_DEBUG
	  if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
	    Arena_TracePrint(Iam,
			     " frame = "POINTER_FORMAT","
			     " child = "POINTER_FORMAT".\n"
			     "\tinfo = 0x%x, top = 0x%x, length = %d.\n",
			     frame, child,
			     child->info, child->top, child->length);
#endif
	  child = FrameBackward(child, top);

	  /* and insert new child in front of current children */
	  child->next = frame->child;
	  frame->child = child;

	  p = obj;
	  break;

        case TEXTLINE:
	  /* stop if textline overlaps top or starts before it */
	  offset = GetLong(p);
	  p = obj;
	  if (offset < top)
	    {
	     BreakBreak = True;
	     break;
	    }
	  break;

        default:
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
	  Arena_TracePrint(Iam,
			   " ERROR!\n"
			   "\tUnexpected (4) internal tag 0x%x,"
			   " when TEXTLINE was expected, Exiting.\n", tag);
# else
	  Arena_PrintError(_("Unexpected (%s) internal tag 0x%x.\n"),
			   "4", tag);
#endif
	  Exit(1);
	  break;
       }
     /* End ``switch (tag)'' */

     if (BreakBreak) break;
    }
   }
 /* End ``while (p > p_begin)'' */

 frame->top = p;
 return frame;
}


/*
 * FORCE h (pix from top) to be ON the document! Never beyond.
 */
long ValidateHTMLoffset(long theBufferHeight,
			ViewRectangle theViewWindowRect,
			long theOffset)
{
#ifdef ARENA_DEBUG
 char Iam[] = "ValidateHTMLoffset";
#endif


 if ((theOffset + theViewWindowRect.height) > theBufferHeight)
   {
    if ((theOffset = theBufferHeight - theViewWindowRect.height - 1) < 0)
      {
#ifdef ARENA_DEBUG
       if (PAINTSTREAM_TRACE)
	 Arena_TracePrint(Iam,
			  " offset = %ld. BEYOND END ... set to 0.\n",
			  theOffset);
#endif
       theOffset = 0;
      }
   }

 return theOffset;
}


/*
 * The frame->top points to the paint stream and is adjusted
 * to the first object that appears at the top of the window
 * for an offset of theOffset pixels from the start of the document.
 *
 * This involves a search through the paint stream,
 * and relies on the objects being ordered wrt increasing
 * pixel offset from the start of the document.
 *
 * The paint stream is organised as a sequence of nested frames,
 * intermingled with text lines.
 */
void UpdateHTMLvisible_top(ViewRectangle theViewWindowRect, long theOffset)
{
 if (theOffset)
   {
    if (theOffset > theViewWindowRect.y)
      FrameForward( &background, theOffset); /* search  forwards */
     else
      FrameBackward(&background, theOffset); /* search backwards */
   }
  else
   {
   /* shortcut to start */
    FreeFrames(background.child);
    background.child = NULL;
    background.top = paint + FRAMESTLEN;
   }
}


/*
 * The procedure validates the given desired position
 * and updates the point to the first visible object of the window.
 * Returns the pixel difference between the desired position
 * and the current position.
 */
long DeltaHTMLPosition(Doc* theDoc,
		       ViewRectangle* theViewWindowRect_p, long theOffset)
{
#ifdef ARENA_DEBUG
 char Iam[] = "DeltaHTMLPosition";
#endif

#ifdef ARENA_DEBUG
 if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam, " %d ---> %d.\n", theViewWindowRect_p->y, theOffset);
#endif


 theOffset = ValidateHTMLoffset(buf_height, (*theViewWindowRect_p), theOffset);
 UpdateHTMLvisible_top((*theViewWindowRect_p), theOffset);

 {
  long delta = theOffset - theViewWindowRect_p->y;

  theViewWindowRect_p->y = theOffset;
  return delta;
 }
}


void DrawHButton(GC theGC, Window theWindow,
		 int x, int y, int w, int h, Bool up)
{
 --x; ++y; ++w;  /* adjust drawing position */

 XSetForeground(display, theGC, (up ? windowTopShadow : windowBottomShadow));
 XFillRectangle(display, theWindow, theGC, x, y, w,     1);
 XFillRectangle(display, theWindow, theGC, x, y, 1, h - 1);

 XSetForeground(display, theGC, (up ? windowBottomShadow : windowTopShadow));
 XFillRectangle(display, theWindow, theGC,     x + 1, y + h - 1, w - 1,     1);
 XFillRectangle(display, theWindow, theGC, x + w - 1,     y + 1,     1, h - 2);
 XSetForeground(display, theGC, textColour);
}


/* Find which object if any is under the mouse at screen coords px, py
 * event is one of BUTTONUP, BUTTONDOWN, MOVEUP, MOVEDOWN
 */
Byte* WhichFrameObject(GC theGC, ViewWindow theViewWindow,
		       int event,
                       int indent,
                       int x, long y,
                       Byte* p1, Byte* p2,
                       int *type,
                       char **start, char **end,
                       int *dx, int *dy)
{
 char *s;
 Byte *p, *q;
 unsigned int tag, len, emph;
 int fnt, x1, y1, xi, yb, yi, y2, width, height;
 int style, border, length;
 long offset, str;
 BG_Style* bg_style;
#ifdef STYLE_COLOUR_BORDER
 Byte cb_ix;
#endif
#ifdef ARENA_DEBUG
 char Iam[] = "WhichFrameObject";
#endif


 for (p = p1; p < p2; )
   {
    tag = *p++;
    {
     Bool BreakBreak = False;

     switch (tag)
       {
        case BEGIN_FRAME:
	  offset = GetLong(p);

	  /* we are done if frame starts after bottom of window */
	  if ((y1 = offset) > theViewWindow.rect.y + theViewWindow.rect.height)
	    {
	     BreakBreak = True;
	     break;
	    }

	  /* otherwise pickup frame header params */

	  x1 = GetInt(p);
	  width  = GetInt(p);
	  height = GetLong(p);
	  style = *p++;
	  border = *p++;
#ifdef STYLE_COLOUR_BORDER
	  cb_ix = *p++;
#endif
#ifdef STYLE_BACKGROUND
	  /* must be set to the structures defined by the style
	   * --Spif 18-Oct-95
	   */
	  SkipPointer(p);
#endif
	  length = GetInt(p);

	  /* call self to find target object in this frame */
	  q = WhichFrameObject(theGC, theViewWindow,
			       event, x1, x, y,
			       p, p + length, /* howcome 25/1/95 */
			       type, start, end,
			       dx, dy);

	  if (q) return q;

	  p += length; /* to skip over frame's contents */
	  break;

        case END_FRAME:
	  p += FRAMENDLEN - 1;
	  break;

        case TEXTLINE:
	  offset = GetLong(p);

	  /* we are done if frame starts after bottom of window */
	  if ((y1 = offset) > theViewWindow.rect.y + theViewWindow.rect.height)
	    {
	     BreakBreak = True;
	     break;
	    }

	  yb = GetInt(p) + y1;
	  xi = GetInt(p) + indent;
	  height = GetInt(p);

#ifdef ARENA_DEBUG
	  if (yb <= 0)
	    if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
	      Arena_TracePrint(Iam,
			       " WEIRD? yb = %d, offset = %d, y1 = %d at "
			       POINTER_FORMAT".\n",
			       yb, offset, y1,
			       p - (ARENA_LONG_INT_SIZE +
				    ARENA_PAINTSTREAM_TAG_LEN +
				    ARENA_INT_SIZE +
				    ARENA_INT_SIZE +
				    ARENA_INT_SIZE));
#endif

	  while ((tag = *p++) != '\0')
	    {
	     switch (tag & 0xF)
	       {
	        case RULE:
		  p += RULEFLEN - 1;
		  break;

	        case LINE:
		  p += LINEFLEN - 1;
		  break;

	        case BULLET:
		  p += BULLETFLEN - 1;
		  break;

	        case STRING:
		  emph = GetInt(p);
		  fnt = *p++;
#ifdef STYLE_COLOUR
		  colour_text_ix = *p++;
#endif
#ifdef STYLE_BACKGROUND
		  /* must be set to the structures defined by the style
		   * --Spif 18-Oct-95
		   */
		  bg_style = (BG_Style*)GetPointer(p);
		  if (bg_style)
		    colour_background_ix = (bg_style->flag &
					   S_BACKGROUND_COLOUR) ?
		                                     rgb2ix(0,
							    bg_style->r,
							    bg_style->g,
							    bg_style->b,
			                                    0) : 255; /* GRR */
		   else
		    colour_background_ix = 255;	
#endif
		  yi = *p++ - 128;
		  x1 = GetInt(p) + xi;  /* here */
		  len = GetInt(p);
		  width = GetInt(p);

		  SkipPointer(p);
		  str = GetLong(p); /* only reference */

		  height = LineSpacing[fnt];
		  y2 = yb - yi - ASCENT(fnt) - 1;

		  if ((emph & EMPH_ANCHOR) &&
		      x1 <= x && x < x1 + width &&
		      y2 <= y && y < y2 + height)
		    {
		     s = (char *)str;

		     while (s > buffer)
		       {
			--s;

			if (s[0] == '<' && TOLOWER(s[1]) == 'a' && s[2] <= ' ')
			  break;
		       }

		     *start = s;    /* used in next stage */
		     s = (char*)str;

		     while (strncasecmp(s, "</a>", 4) != 0)
		       {
			if (*s == '\0') break;

			++s;
		       }

		     *end = s;    /* used in next stage */
		     *dx = *dy = -1;
		     *type = TAG_ANCHOR;

		     /* return pointer to start of anchor object */
		     return p - STRINGFLEN;
		    }
		  break;

	        case SEQTEXT:
		  SkipInt(p);	/* skip over emph */
		  p++;      /* skip over font */
#ifdef STYLE_COLOUR
		  p++;  /* skip over colour */ /* janet p += 2 */
#endif
#ifdef STYLE_BACKGROUND
		  /* skip over the background
		   * --Spif 18-Oct-95
		   */
		  SkipPointer(p);
#endif
		  SkipInt(p);  /* skip over x position */ 
		  SkipInt(p);  /* skip over y position */
		  len = *p++;
		  p += len;
		  break;

	        case IMAGE:
		  y2 = yb - GetInt(p); /* delta */
		  x1 = xi + GetInt(p); /* here */
		  width = GetInt(p);
		  height = GetInt(p);

		  if ((x1 <= x) && (x < x1 + width) &&
		      (y2 <= y) && (y < y2 + height))
		    {
		     SkipPixmap(p); /* past icon pixmap */
		     SkipPixmap(p); /* past mask pixmap */

		     str = GetLong(p);
#ifdef STYLE_BACKGROUND
		     /* must be set to the structures defined by the style
		      * --Spif 18-Oct-95
		      */
		     SkipPointer(p);
#endif
		     if (tag & EMPH_ANCHOR)
		       {
			*dx = *dy = -1;

			/* --Spif image input */
			if ((tag & EMPH_INPUT) == EMPH_INPUT)
			  {
			  /* can be removed, as INPUT is ANCHOR || ISMAP...*/
			   *dx = x - x1;
			   *dy = y - y2;
			  }
		         else
			  {
			   if (tag & ISMAP)
			     {
			      *dx = x - x1;
			      *dy = y - y2;
			     }
			  }

			s = (char*)str;

			/* --Spif 10-Oct-95 */
			if ((tag & EMPH_INPUT) == EMPH_INPUT)
			  {
			   while (buffer < s) /* find input...*/
			     {
			      --s;
			      if (strncasecmp(s, "<input", 6) != 0) break;
			     }

			   *start = s;
			   s = (char *)str;
			   while ( *s != '>')
			     {
			      if (*s == '\0') break;
			      s++;
			     }

			   *end = s;
			   *type = TAG_INPUT; /* ?? */

			   return p - IMAGEFLEN;
			  }
		         else
			  {
			  /* find anchor... must also find an input...*/
			   while (s > buffer)
			     {
			      --s;
			      if ((s[0] == '<') &&
				  (TOLOWER(s[1]) == 'a') &&
				  (s[2] <= ' '))
				break;
			     }

			   *start = s;    /* used in next stage */
			   s = (char *)str;

			   while (strncasecmp(s, "</a>", 4) != 0)
			     {
			      if (*s == '\0') break;
			      ++s;
			     }

			   *end = s;    /* used in next stage */
			   *type = TAG_ANCHOR;
			   return p - IMAGEFLEN;
			  }
		       }
		      else
		       {
			if (tag & ISMAP)
			  {
			   *dx = x - x1;
			   *dy = y - y2;
			   *start = (char*)str;
			   *end = (char*)str;
			   *type = TAG_IMG;
			   return p - IMAGEFLEN;
			  }
		       }
		    }
		   else
		    {
		    /* past pixmap, mask and buf pointer */
		     SkipPixmap(p);
		     SkipPixmap(p);
		     SkipPointer(p);
#ifdef STYLE_BACKGROUND
		     SkipPointer(p);
#endif
		    }
		  break;

	        case INPUT: /* --Spif only text... */
		  str = GetLong(p);
#ifdef STYLE_BACKGROUND
		  /* must be set to the structures defined by the style
		   * --Spif 18-Oct-95
		   */
		  SkipPointer(p);
#endif
		  if (ClickedInField(theGC, theViewWindow,
				     xi, yb, (Field*)str,
				     x, y, event))
		    {
		    /* updating *start --Spif 14-Oct-95 */ 
		      s = ((Field*)str)->name;
		      while (s > buffer)
			{
			 if (*s == '<')
			   {
			    if (strncasecmp(s + 1, "input",  5) == 0) break;
			    if (strncasecmp(s + 1, "select", 6) == 0) break;
			   }

			 s--;
			}

		      *start = s;
		      bufptr = s;
		      while(*s != '>') s++;
		      *end = s;
		      *type = TAG_INPUT;
		      return (p - INPUTFLEN); 
		    }
		  break;

	        default:
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
		  Arena_TracePrint(Iam,
				   " ERROR! Unexpected (5) internal tag 0x%x,"
				   " Exiting.\n",
				   tag);
# else
		  Arena_PrintError(_("Unexpected (%s) internal tag 0x%x.\n"),
				   "5", tag);
#endif
		  Exit(1);
	       }
	     /* End ``switch (tag & 0x0F)'' */

	     if (BreakBreak)
	       {
		BreakBreak = False;
	        break;
	       }
	    }
	  /* End ``while ((tag = *p++) != '\0')'' */
	  break;

        default:
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
	  Arena_TracePrint(Iam,
			   " ERROR! Unexpected (6) internal tag 0x%x,"
			   " Exiting.\n",
			   tag);
# else
	  Arena_PrintError(_("Unexpected (%s) internal tag 0x%x.\n"),
			   "6", tag);
#endif
	  Exit(1);
	  break;
       }
     /* End ``switch (tag)'' */

     SkipInt(p); /* skip final frame length field */
     if (BreakBreak) break;
    }
   }

 return NULL;
}


/*
 * Find which object contains given point.
 */
Byte* WhichObj(GC theGC, ViewWindow theViewWindow,
	       Frame* theFrame, int event, int x, int y,
               int *type, char **start, char **end, int *dx, int *dy)
{
 if (theFrame)
   {
    Byte* p;
#ifdef ARENA_DEBUG
    char Iam[] = "WhichObj";
#endif


#ifdef ARENA_DEBUG
    if (theFrame == theFrame->next)
      {
       Arena_TracePrint(Iam,
			" ERROR!"
			" Frames chain deadloop at "POINTER_FORMAT".\n",
			theFrame);
       return NULL;
      }
#endif

    if (theFrame->child)
      {
       if ((p = WhichObj(theGC, theViewWindow,
			 theFrame->child, event, x, y, type, start, end,
			 dx, dy)))
	 return p;
      }

    if (theFrame->next)
      {
       if ((p = WhichObj(theGC, theViewWindow,
			 theFrame->next,  event, x, y, type, start, end,
			 dx, dy)))
	 return p;
      }

    return WhichFrameObject(theGC, theViewWindow,
			    event,
			    theFrame->indent, x, y,
			    theFrame->top,
			    paint + theFrame->info +
			       FRAMESTLEN + theFrame->length,
			    type,
			    start, end,
			    dx, dy);
   }
  else
   return NULL;
}


/*
 * Find which hypertext button contains given point.
 */
Byte* WhichObject(GC theGC, ViewWindow theViewWindow,
		  int event,
		  int x, int y,
		  int *type, char **start, char **end,
		  int *dx, int *dy)
{
 x += theViewWindow.rect.x;
 y += theViewWindow.rect.y;

 if (focus &&
     focus->type == OPTIONLIST &&
     (focus->flags & CHECKED) &&
     ClickedInDropDown(theGC, theViewWindow, focus, -1, event, x, y))
   {
    *type = TAG_SELECT;
    *start = *end = focus->name;
    return paint + focus->object;
   }

 return WhichObj(theGC, theViewWindow,
		 &background, event, x, y, type, start, end, dx, dy);
}


/*
 * Drawn anchors in designated state.
 */
void DrawFrameAnchor(GC theGC, ViewWindow theViewWindow,
		     int up, int indent, Byte* start, Byte* end)
{
 char *s;
 Byte* p;
 unsigned int tag, len, emph;
 int fnt, x1, y1, xi, y2, yi, yb, width, height;
 int style, border, length;
 long offset, str;
 BG_Style* bg_style;
#ifdef STYLE_COLOUR_BORDER
 Byte cb_ix;
#endif
#ifdef ARENA_DEBUG
 char Iam[] = "DrawFrameAnchor";
#endif


 for (p = start; p < end;)
   {
    tag = *p++;
    {
     Bool BreakBreak = False;

     switch (tag)
       {
        case BEGIN_FRAME:
	  offset = GetLong(p);

	  /* we are done if frame starts after bottom of window */
	 if ((y1 = offset) >= theViewWindow.rect.y + theViewWindow.rect.height)
	    {
	     BreakBreak = True;
	     break;
	    }

	  /* otherwise pickup frame header params */

	  x1 = GetInt(p);
	  width = GetInt(p);
          height = GetLong(p);
	  style = *p++;
	  border = *p++;
#ifdef STYLE_COLOUR_BORDER
	  cb_ix = *p++;
#endif
#ifdef STYLE_BACKGROUND
	  /* must be set to the structures defined by the style
	   * --Spif 18-Oct-95
	   */
	  bg_style = (BG_Style *)GetPointer(p);
#endif
	  length = GetInt(p);

	  /* call self to draw anchors in this frame */

	  DrawFrameAnchor(theGC, theViewWindow, up, x1, p, p + length);
	  p += length; /* to skip over frame's contents */
	  break;

        case END_FRAME:
	  p += FRAMENDLEN - 1;
	  break;

        case TEXTLINE:
	  offset = GetLong(p);

	  /* y1 points to top of line */
	 if ((y1 = offset) >= theViewWindow.rect.y + theViewWindow.rect.height)
	    {
	     BreakBreak = True;
	     break;
	    }

	  yb = GetInt(p) - theViewWindow.rect.y + y1;
	  xi = GetInt(p) - theViewWindow.rect.x;
	  height = GetInt(p);
	  xi += indent;

	  while ((tag = *p++) != '\0')
	    {
	     switch (tag & 0xF)
	       {
	        case RULE:
		  p += RULEFLEN - 1;
		  break;

	        case LINE:
		  p += LINEFLEN - 1;
		  break;

	        case BULLET:
		  p += BULLETFLEN - 1;
		  break;

	        case STRING:
		  emph = GetInt(p);
		  fnt = *p++;
#ifdef STYLE_COLOUR
		  colour_text_ix = *p++;
#endif
#ifdef STYLE_BACKGROUND
		  /* must be set to the structures defined by the style
		   * --Spif 18-Oct-95
		   */
		  bg_style = (BG_Style*)GetPointer(p);
		  if (bg_style)
		    colour_background_ix = (bg_style->flag &
					   S_BACKGROUND_COLOUR) ?
                                                     rgb2ix(0,
							    bg_style->r,
							    bg_style->g,
							    bg_style->b,
							    0) : 255; /* GRR */
		   else
		    colour_background_ix = 255;	 
#endif
		  yi = *p++ - 128;
	          x1 = GetInt(p) + xi;  /* here */
	          len = GetInt(p);
		  width = GetInt(p);
		  str = GetLong(p);
		  SkipPointer(p); /* skip the reference pointer */
#ifdef STYLE_COLOUR
		  font = fnt;

		  SetColour(theGC, colour_text_ix, colour_background_ix);
		  SetFont(theGC, font);
# else
		  if (font != fnt)
		    {
		     font = fnt;
		     SetFont(theGC, font);
		    }
#endif

		  height = LineSpacing[font];
		  y2 = yb - yi - ASCENT(font) - 1;

		  if (emph & EMPH_ANCHOR) 
		    {
		     s = (char*)str;

		     if (anchor_start <= s && s <= anchor_end)
		       {
			DrawHButton(theGC, theViewWindow.win,
				    x1, y2, width, height, up);
		       }
		    }
		  break;

	        case SEQTEXT:
		  SkipInt(p); /* skip over emph */
		  p++;       /* skip over font */
#ifdef STYLE_COLOUR
		  p++;       /* skip over colour */
#endif
#ifdef STYLE_BACKGROUND
		  SkipPointer(p); /* skip over background */
#endif
		  SkipInt(p); /* skip over x position */ 
		  SkipInt(p); /* skip over y position */
		  len = *p++;
		  p += len;
		  break;

	        case IMAGE:
		  y2 = yb - GetInt(p);
		  x1 = xi + GetInt(p);  /* here */
		  width = GetInt(p);
		  height = GetInt(p);

		  /* pass pixmap and mask ids */
		  SkipPixmap(p);
		  SkipPixmap(p);

		  s = (char*)GetPointer(p);
#ifdef STYLE_BACKGROUND
		  SkipPointer(p); /* skip background */
#endif

		  if (tag & ISMAP && anchor_start == s)
		    {
		     x1 -= theViewWindow.rect.x;

		     DrawOutSet(theViewWindow.win, theGC,
				x1, y2, width, height);
		     XSetForeground(display, theGC, textColour);
		     DrawHButton(theGC, theViewWindow.win,
				 x1 + 4, y2 + 2, width - 7, height - 6, up);

		     width  -= 8;
		     height -= 8;
		     x1 += 4;
		     y2 += 4;
		    }
		  break;

	        case INPUT:
		  /* --Spif 13-Oct-95
		   * take care of this...it may be the cause of crashes
		   */
		  p += INPUTFLEN - 1;
		  break;

	        default:
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
		  Arena_TracePrint(Iam,
				   " ERROR! Unexpected (7) internal tag 0x%x,"
				   " Exiting.\n",
				   tag);
# else
		  Arena_PrintError(_("Unexpected (%s) internal tag 0x%x.\n"),
				   "7", tag);
#endif
		  Exit(1);
	       }
	     /* End ``switch (tag & 0x0F)'' */

	     if (BreakBreak)
	       {
		BreakBreak = False;
	        break;
	       }
	    }
	  /* End ``while ((tag = *p++) != '\0')'' */
	  break;

        default:
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
	  Arena_TracePrint(Iam,
			   " ERROR! Unexpected (8) internal tag 0x%x,"
			   " Exiting.\n",
			   tag);
# else
	  Arena_PrintError(_("Unexpected (%s) internal tag 0x%x.\n"),
			   "8", tag);
#endif
	  Exit(1);
	  break;
       }
     /* End ``switch (tag)'' */

     SkipInt(p); /* skip final frame length field */
     if (BreakBreak) break;
    }
   }

 XFlush(display);
}


void DrawAnchor(GC theGC, ViewWindow theViewWindow, Frame* frame, int state)
{
 if (frame->child) DrawAnchor(theGC, theViewWindow, frame->child, state);
 if (frame->next ) DrawAnchor(theGC, theViewWindow, frame->next,  state);
 DrawFrameAnchor(theGC, theViewWindow,
		 state, frame->indent,
		 frame->top, paint + frame->info + FRAMESTLEN + frame->length);
}


/*
 * Find topmost object's pointer to the html buffer.
 */
char *TopStrSelf(ViewWindow theViewWindow, Byte* start, Byte* end)
{
 Byte* p;
 char *s;
 unsigned int tag, len, emph, fnt;
 int style, border, length, x1, y1, xi, yb, yi, width, height;
 long offset, str;
#ifdef STYLE_COLOUR_BORDER
 Byte cb_ix;
#endif
#ifdef ARENA_DEBUG
 char Iam[] = "TopStrSelf";
#endif


#ifdef ARENA_DEBUG
 if (DISPLAY_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam,
		    " start "POINTER_FORMAT", end "POINTER_FORMAT".\n",
		    start, end);
#endif

 for (p = start; p < end; )
   {
    tag = *p++;
    {
     Bool BreakBreak = False;

     switch (tag)
       {
        case BEGIN_FRAME:
	  offset = GetLong(p);
	      x1 = GetInt(p);
	   width = GetInt(p);
	  height = GetLong(p);
	   style = *p++;
	  border = *p++;
#ifdef STYLE_COLOUR_BORDER
	  cb_ix = *p++;
#endif
#ifdef STYLE_BACKGROUND
	  /* must be set to the structures defined by the style
	   * --Spif 18-Oct-95
	   */
	  SkipPointer(p);
#endif
	  length  = GetInt(p);
	  if ((s = TopStrSelf(theViewWindow, p, p + length)))
	    return s;

	  p += length; /* to skip over frame's contents */
	  break;

        case END_FRAME:
	  p += FRAMENDLEN - 1;
	  break;

        case TEXTLINE:
	  offset = GetLong(p);

	  /* y1 points to top of line */
	 if ((y1 = offset) >= theViewWindow.rect.y + theViewWindow.rect.height)
	    {
	     BreakBreak = True;
	     break;
	    }

	  yb = GetInt(p) - theViewWindow.rect.y + y1;
	  xi = GetInt(p) - theViewWindow.rect.x;
	  height = GetInt(p);

	  while ((tag = *p++) != '\0')
	    {
	     switch (tag & 0xF)
	       {
	        case RULE:
		  p += RULEFLEN - 1;
		  break;

	        case LINE:
		  p += LINEFLEN - 1;
		  break;

	        case BULLET:
		  p += BULLETFLEN - 1;
		  break;

	        case STRING:
		  emph = GetInt(p);
		  fnt = *p++;
#ifdef STYLE_COLOUR
		  p++; /* skip over colour */
#endif
#ifdef STYLE_BACKGROUND
		  /* --Spif 20-Oct-95
		   * skip over style background
		   */
		  SkipPointer(p);
#endif
		  yi = *p++ - 128;
		  x1 = GetInt(p) + xi; /* here */
		  len = GetInt(p);
		  width = GetInt(p);
		  SkipPointer(p);
		  str = GetLong(p);
		  return (char*)str;

	        case SEQTEXT:
		  SkipInt(p); /* skip over emph */
		  p++;       /* skip over font */
#ifdef STYLE_COLOUR
		  p++; /* skip over colour */
#endif
#ifdef STYLE_BACKGROUND
		  /* skip over background
		   * --Spif 20-Oct-95
		   */
		  SkipPointer(p);
#endif
		  SkipInt(p); /* skip over x position */
		  SkipInt(p); /* skip over y position */
		  len = *p++;
		  p += len;
		  break;

	        case IMAGE:
		  p += IMAGEFLEN - 1;
		  break;

	        case INPUT:
		  p += INPUTFLEN - 1;
		  break;

	        default:
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
		  Arena_TracePrint(Iam,
				   " ERROR! Unexpected (9) internal tag 0x%x,"
				   " Exiting.\n",
				   tag);
# else
		  Arena_PrintError(_("Unexpected (%s) internal tag 0x%x.\n"),
				   "9", tag);
#endif
		  Exit(1);
	       }
	     /* End ``switch (tag & 0x0F)''*/

	     if (BreakBreak)
	       {
		BreakBreak = False;
	        break;
	       }
	    }
	  /* End ``while ((tag = *p++) != '\0')'' */
	  break;

        default:
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
	  Arena_TracePrint(Iam,
			   " ERROR! Unexpected (A) internal tag 0x%x,"
			   " Exiting.\n",
			   tag);
# else
	  Arena_PrintError(_("Unexpected (%s) internal tag 0x%x.\n"),
			   "A", tag);
#endif
	  Exit(1);
	  break;
       }
     /* End ``switch (tag)'' */

     SkipInt(p);  /* skip final frame length field */
     if (BreakBreak) break;
    }
   }

 return NULL;
}


char *TopStr(ViewWindow theViewWindow, Frame* frame)
{
 if (frame)
   {
    char *s;

    if (frame->child)
      if ((s = TopStr(theViewWindow, frame->child)))
	return s;

    if (frame->next)
      if ((s = TopStr(theViewWindow, frame->next)))
	return s;

    return TopStrSelf(theViewWindow,
		      frame->top,
		      ((paint + frame->info) + FRAMESTLEN) + frame->length);
   }
  else
   return NULL;
}


void DrawBorder(GC theGC, Window theWindow,
		int border,
		int x, int y, unsigned int w, unsigned int h,
		Byte cb_ix)
{
#ifdef STYLE_COLOUR_BORDER
 if (cb_ix < 128)
   {
    XSetForeground(display, theGC, stdcmap[cb_ix]);
   }
  else
   if (cb_ix < 144)
     {
      XSetForeground(display, theGC, greymap[cb_ix & 0xf]);
     }
#endif

 XFillRectangle(display, theWindow, theGC, x,         y, w, 1);
 XFillRectangle(display, theWindow, theGC, x + w,     y, 1, h);
 XFillRectangle(display, theWindow, theGC, x,     h + y, w, 1);
 XFillRectangle(display, theWindow, theGC, x,         y, 1, h);
}


void FreeBox(box_link* box_list)
{
 if (box_list)
   {
    box_link* current_link;
    box_link* next_link;

    current_link = box_list;
    next_link = box_list->next;

    while (next_link)
      {
       Free(current_link->box);
       Free(current_link);
       current_link = next_link;
       next_link = next_link->next;
      }

    Free(current_link->box);
    Free(current_link);
   }
}


/*
 * free frame structures except for background frame
 */
void FreeFrames(Frame* frame)
{
 if (frame)
   {
   /* Kai.Arstila@Helsinki.fi */

    if (frame->child) 
      {
       FreeFrames(frame->child);
       frame->child = NULL;
      }

    if (frame->next)
      {
       FreeFrames(frame->next);
       frame->next = NULL;
      }

    FreeBox(frame->box_list);
    Free(frame);
   }
}


void PaintSelf( GC theGC, ViewWindow theViewWindow, ViewRectangle theArea,
	       Frame* frame);
void PaintPeers(GC theGC, ViewWindow theViewWindow, ViewRectangle theArea,
		Frame* frame);
void PaintFrame(GC theGC, ViewWindow theViewWindow, ViewRectangle theArea,
		int textindent,
		unsigned char *p, unsigned char *p_end);


/*
 * Paint children then self - first called for background frame.
 */
void PaintSelf(GC theGC, ViewWindow theViewWindow, ViewRectangle theArea,
	       Frame* frame)
{
 ViewRectangle theFrameArea = { frame->indent, frame->offset,
				frame->width,  frame->height },
	       theFrameSubArea = {0,0,0,0};
 unsigned char *p1, *p2;
#ifdef ARENA_DEBUG
 char Iam[] = "PaintSelf";
#endif


#ifdef ARENA_DEBUG
 if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam,
		    " frame "POINTER_FORMAT" area %dx%d%+d%+d"
		    " in %dx%d%+d%+d.\n",
		    frame,
		    theArea.width, theArea.height, theArea.x, theArea.y,
		    theViewWindow.rect.width, theViewWindow.rect.height,
		    theViewWindow.rect.x, theViewWindow.rect.y);
#endif

 /* Test that this frame is at least partly visible */
 if (VisibleViewRectangle(theArea, theFrameArea, &theFrameSubArea))
   {
    if (frame->child)
      PaintPeers(theGC, theViewWindow, theArea, frame->child);

    if (frame->border)
      {
       DrawBorder(theGC, theViewWindow.win,
		  frame->border,
		  theFrameArea.x - theViewWindow.rect.x,
		  theFrameArea.y - theViewWindow.rect.y,
		  theFrameArea.width, theFrameArea.height,
		  frame->cb_ix);
      }

    p1 = frame->top;
    p2 = paint + frame->info + FRAMESTLEN + frame->length;
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
    if (TAG_TRACE)
      Arena_TracePrint(Iam,
		       " p1 = "POINTER_FORMAT", p2 = "POINTER_FORMAT","
		       "   frame "POINTER_FORMAT": info = %d, length = %d.\n",
		       p1, p2, frame, frame->info, frame->length);
#endif
    PaintFrame(theGC, theViewWindow, theFrameSubArea,
	       theFrameArea.x,
	       p1, p2);
   }
#ifdef ARENA_DEBUG
  else
   {
    if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       " frame "POINTER_FORMAT" appears NOT to be visible:\n"
		       "\tframe area %dx%d%+d%+d,\n"
		       "\tsubwin at %+d%+d, area %dx%d%+d%+d.\n",
		       frame,
		       theFrameArea.width, theFrameArea.height,
		       theFrameArea.x, theFrameArea.y,
		       theViewWindow.rect.x, theViewWindow.rect.y,
		       theArea.width, theArea.height, theArea.x, theArea.y);
   }
#endif
}


/* paint list of peer frames */
void PaintPeers(GC theGC, ViewWindow theViewWindow,
		ViewRectangle theArea, Frame* theFrame)
{
 while (theFrame)
   {
    PaintSelf(theGC, theViewWindow, theArea, theFrame);
    theFrame = theFrame->next;
   }
}


/*
 * p, p_end point to paint buffer.
 *
 * This routine recursively calls itself to paint nested frames
 * when it comes across the BEGIN_FRAME tag.
 * Note that the border for the current frame is already drawn ---
 * and saves having to pass the relevant params to this routine.
 */
void PaintFrame(GC theGC, ViewWindow theViewWindow, ViewRectangle theArea,
		int textindent,
		unsigned char *p, unsigned char *p_end)
{
 char *s;
 int indent, baseline, style, border, fnt, fnt_size, length,
     x1, y1, x2, y2, xb, yb, xbr, ybr, xr, yr, depth;
 unsigned int tag, len, width, emph;
 long offset, height, str;
#ifdef STYLE_BACKGROUND
 BG_Style* bg_style;
#endif
#ifdef STYLE_COLOUR_BORDER
 Byte cb_ix;
#endif
 XRectangle bgRect;
 ViewRectangle theFrameArea, theFrameSubArea = {0,0,0,0};
#ifdef ARENA_DEBUG
 void* tag_p = NULL;
 char Iam[] = "PaintFrame";
#endif


#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
 if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam,
		    " p = "POINTER_FORMAT", p_end = "POINTER_FORMAT","
		    " area is %dx%d%+d%+d.\n",
		    p, p_end,
		    theArea.width, theArea.height, theArea.x, theArea.y);
#endif


 while (p < p_end)
   {
    if (p >= (unsigned char*)PaintStreamCurrentPosition())
      {
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
       Arena_TracePrint(Iam,
			" Panic: ran off end of paint buffer! Exiting.\n");
# else
       Arena_PrintError(_("Panic: ran off end of paint buffer! Exiting.\n"));
#endif
       Exit(1);
      }

#ifdef ARENA_DEBUG
    tag_p = p;
#endif
    tag = *p++;
    {
     Bool BreakBreak = False;
     Bool PaintingFrameRecursionDone;
     Bool DoPrintTextLine;

     switch (tag)
       {
        case BEGIN_FRAME:
	  theFrameArea.y = offset = GetLong(p);

	  /* We are done if the frame starts after bottom of the window. */
	  if (theFrameArea.y > theArea.y + theArea.height)
	    {
	     BreakBreak = True;
	     break; 
	    }

	  /*
	   * Otherwise pickup frame header params.
	   */
	  theFrameArea.x      = indent =  (int)GetInt(p);
	  theFrameArea.width  =  width = GetInt(p);
	  theFrameArea.height = height = (long)GetLong(p);
	  style  = *p++;
	  border = *p++;
#ifdef STYLE_COLOUR_BORDER
	  cb_ix  = *p++;
#endif
#ifdef STYLE_BACKGROUND
	  bg_style = (BG_Style*)GetPointer(p);
#endif
	  length = GetInt(p);

#ifdef ARENA_DEBUG
	     if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
	       Arena_TracePrint(Iam,
				" frame at "POINTER_FORMAT":\n"
				"\tframe %dx%d%+d%+d,\n"
				"\tview at %+d%+d,  area %dx%d%+d%+d.\n",
				tag_p,
				theFrameArea.width, theFrameArea.height,
				theFrameArea.x,     theFrameArea.y,
				theViewWindow.rect.x, theViewWindow.rect.y,
				theArea.width, theArea.height,
				theArea.x,     theArea.y);
#endif

	  /* Check the frame visibility. */
	  if (VisibleViewRectangle(theArea, theFrameArea, &theFrameSubArea))
	    {
	     PaintingFrameRecursionDone = False;

#ifdef STYLE_BACKGROUND
	     if (bg_style)
	       {
		if ((bg_style->flag & S_BACKGROUND_COLOUR) ||
		    ((bg_style->flag & S_BACKGROUND_IMAGE) &&
		     ((bg_style->image) && (bg_style->image->pixmap != None))))
		  {
		   GC subGC = XCreateGC(display, theViewWindow.win, 0, NULL);

		   XCopyGC(display,
			   theGC,
			   (GCFunction | GCPlaneMask |
			    GCForeground | GCBackground |
			    GCLineWidth | GCLineStyle |
			    GCCapStyle | GCJoinStyle |
			    GCFillStyle | GCFillRule | GCTile | GCStipple |
			    GCTileStipXOrigin | GCTileStipYOrigin |
			    GCFont | GCSubwindowMode | GCGraphicsExposures |
			    GCClipXOrigin | GCClipYOrigin | GCClipMask |
			    GCDashOffset | GCDashList | GCArcMode),
			   subGC);

		   if (bg_style->flag & S_BACKGROUND_COLOUR)
		     {
		     /* background colour */
		      XSetForeground(display, subGC, rgb2colour(0,
							       bg_style->r,
							       bg_style->g,
							       bg_style->b,
							       0));

		      if (!(((bg_style->image) &&
			     (bg_style->image->pixmap != None)) &&
			    (bg_style->flag & S_BACKGROUND_IMAGE) &&
			    (bg_style->flag & S_BACKGROUND_X_REPEAT) &&
			    (bg_style->flag & S_BACKGROUND_Y_REPEAT)))
			XFillRectangle(display, theViewWindow.win, subGC,
				       (theFrameSubArea.x -
					theViewWindow.rect.x),
				       (theFrameSubArea.y -
					theViewWindow.rect.y),
				       theFrameSubArea.width,
				       theFrameSubArea.height);
		     }

		   if ((bg_style->flag & S_BACKGROUND_IMAGE) &&
		       ((bg_style->image) &&
			(bg_style->image->pixmap != None)))
		     {
		     /* background */
		      ViewWindow subViewWindow = { None,
						   { theFrameSubArea.x,
						     theFrameSubArea.y,
						     theFrameSubArea.width,
						     theFrameSubArea.height }};
		      XGCValues values;
		      unsigned int valuemask;
		      int x0, y0;

#ifdef ARENA_DEBUG
		      if (SCROLL_TRACE && VERBOSE_TRACE)
			Arena_TracePrint(Iam,
					 " "POINTER_FORMAT":"
					 " open %dx%d%+d%+d subWindow of frame"
					 " "POINTER_FORMAT".\n",
					 p,
					 theFrameSubArea.width,
					 theFrameSubArea.height,
					 theFrameSubArea.x,
					 theFrameSubArea.y,
					 tag_p);
#endif

		      subViewWindow.win = XCreatePixmap(display,
							RootWindow(display,
								   screen),
							theFrameSubArea.width,
							theFrameSubArea.height,
							DefaultDepth(display,
								     screen));

		      if (bg_style->flag & S_BACKGROUND_ORIGIN)
			{
			 x0 = bg_style->x_pos * (theViewWindow.rect.width  - bg_style->image->width)/100;
			 y0 = bg_style->y_pos * (theViewWindow.rect.height - bg_style->image->height)/100;
			}
		       else
			{
			 x0 = 0;
			 y0 = 0;
			}

		      if (!(bg_style->flag & S_BACKGROUND_FIXED))
			{
			 x0 -= theViewWindow.rect.x;
			 y0 -= theViewWindow.rect.y;
			}

		      values.fill_style = FillTiled;
		      values.tile = bg_style->image->pixmap;
		      values.ts_x_origin = x0;
		      values.ts_y_origin = y0;
		      values.graphics_exposures = False;
		      valuemask = (GCFillStyle | GCTile |
				   GCTileStipXOrigin | GCTileStipYOrigin |
				   GCGraphicsExposures);

		      XChangeGC(display, subGC, valuemask, &values);

		      if (bg_style->flag & S_BACKGROUND_X_REPEAT)
			{
			 bgRect.x = 0;
			 bgRect.width = theFrameSubArea.width;
			}
		       else
			{
			 bgRect.x = values.ts_x_origin;
			 bgRect.width = bg_style->image->width;
			}

		      if (bg_style->flag & S_BACKGROUND_Y_REPEAT)
			{
			 bgRect.y      = 0;
			 bgRect.height = theFrameSubArea.height;
			}
		       else
			{
			 bgRect.y      = values.ts_y_origin;
			 bgRect.height = bg_style->image->height; 
			}

		      XSetClipRectangles(display, subGC,
					 (theFrameSubArea.x -
					  theViewWindow.rect.x),
					 (theFrameSubArea.y -
					  theViewWindow.rect.y),
					 &bgRect, 1, Unsorted); 
		      XFillRectangle(display, theViewWindow.win, subGC,
				     theFrameSubArea.x - theViewWindow.rect.x,
				     theFrameSubArea.y - theViewWindow.rect.y,
				     theFrameSubArea.width,
				     theFrameSubArea.height);

		      XSetClipMask( display, subGC, None);
		      XSetFillStyle(display, subGC, FillSolid);

		      XCopyArea(display,
				theViewWindow.win, subViewWindow.win, subGC,
				theFrameSubArea.x - theViewWindow.rect.x,
				theFrameSubArea.y - theViewWindow.rect.y,
				theFrameSubArea.width, theFrameSubArea.height,
				0, 0);

		      PaintFrame(subGC, subViewWindow, theFrameSubArea,
				 indent,
				 p, p + length);

		      XCopyArea(display,
				subViewWindow.win, theViewWindow.win, theGC,
				0, 0,
				theFrameSubArea.width, theFrameSubArea.height,
				theFrameSubArea.x - theViewWindow.rect.x,
				theFrameSubArea.y - theViewWindow.rect.y);
		      XFreePixmap(display, subViewWindow.win);
#ifdef ARENA_DEBUG
		      if (SCROLL_TRACE && VERBOSE_TRACE)
			Arena_TracePrint(Iam,
					 " close %dx%d%+d%+d subWindow"
					 " been opened at "POINTER_FORMAT".\n",
					 theFrameSubArea.width,
					 theFrameSubArea.height,
					 theFrameSubArea.x,
					 theFrameSubArea.y,
					 p);
#endif
		     }
		    else
		     {
		      PaintFrame(subGC, theViewWindow, theFrameSubArea,
				 indent,
				 p, p + length);
		     }

		   XFreeGC(display, subGC);
		   PaintingFrameRecursionDone = True;
		  }
	       }

	     if (!PaintingFrameRecursionDone)
	       {
		PaintFrame(theGC, theViewWindow, theFrameSubArea,
			   indent,
			   p, p + length);
	       }
# else
	     {
	      long count = StyleGet(S_BACKGROUND);

	      if (count == 255) /* i.e. is the background transparent? */
		{
		 XFillRectangle(display, theViewWindow.win, gc_fill,
				x, y, w, h);
		}
	       else
		{
		 XSetForeground(display, theGC, ix2colour(count));
		 XSetFillStyle(display, theGC, FillSolid);
		 XFillRectangle(display, theViewWindow.win, theGC, x, y, w, h);
		}
	     }

	     /* recurse to paint this frame */
	     PaintFrame(theGC, theViewWindow, theFrameSubArea,
			indent,
			p, p + length);
#endif

	     if (border)
	       DrawBorder(theGC, theViewWindow.win,
			  border,
			  theFrameArea.x - theViewWindow.rect.x,
			  theFrameArea.y - theViewWindow.rect.y,
			  width, height, 0);
	    }
	   else
	    {
	    /* If totally invisible, painting children makes no sense. */
#ifdef ARENA_DEBUG
	     if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
	       Arena_TracePrint(Iam,
				" frame at "POINTER_FORMAT
				" appears NOT to be visible.\n",
				tag_p);
#endif
	    }

	  p += length; /* to skip over frame's contents */
	  break;

	     /* skip end of frame marker */
        case END_FRAME:
#ifdef ARENA_DEBUG
	     if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
	       {
		unsigned char* tmp_p = p;
		int the_frame_length = GetInt(tmp_p);

		tmp_p = tag_p - the_frame_length;
		Arena_TracePrint(Iam,
				 " frame end at "POINTER_FORMAT
				 " (of the frame from "POINTER_FORMAT").\n",
				 tag_p, tmp_p);
	       }
#endif
	  p += FRAMENDLEN - 1;  /* skip start/size params */
	  break;

        case TEXTLINE:
	  theFrameArea.y = offset = GetLong(p);

	  /* If the text starts after bottom of the window, we are done. */
	  if (theFrameArea.y > theArea.y + theArea.height)
	    {
	     BreakBreak = True;
	     break; 
	    }

	  /*
	   * Otherwise pickup frame header params.
	   */
	  baseline = (int)(GetInt(p));
	  theFrameArea.x      = indent =  (int)GetInt(p);
	  theFrameArea.width  = background.width; /* TEXTLINE width unknown? */
	  theFrameArea.height = height = (long)GetInt(p);

	  xbr = (xb = theFrameArea.x + textindent) - theViewWindow.rect.x;
	  ybr = (yb = theFrameArea.y +   baseline) - theViewWindow.rect.y;

	  /* If the text is totally invisible, X traffic makes no sense. */
	  DoPrintTextLine = VisibleViewRectangle(theArea, theFrameArea,
						 &theFrameSubArea);

#ifdef ARENA_DEBUG
	  if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
	    Arena_TracePrint(Iam,
			     " textline at "POINTER_FORMAT"%s:\n"
			     "\tframe %dx%d%+d%+d, (xbr = %+d, ybr = %+d),\n"
			     "\tsub window at %+d%+d,  area %dx%d%+d%+d.\n",
			     tag_p,
			     (DoPrintTextLine ? "" : " NOT visible"),
			     theFrameArea.width, theFrameArea.height,
			     theFrameArea.x, theFrameArea.y,
			     xbr, ybr,
			     theViewWindow.rect.x, theViewWindow.rect.y,
			     theArea.width, theArea.height,
			     theArea.x, theArea.y);
#endif

	  while ((tag = *p++) != '\0')
	    {
#ifdef ARENA_DEBUG
	     tag_p = p - 1;
#endif
	     switch (tag & 0xF)
	       {
	        case RULE:
#ifdef ARENA_DEBUG
		  if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
		    Arena_TracePrint(Iam,
				     " RULE (0x%x) at "POINTER_FORMAT".\n",
				     tag, tag_p);
#endif
		  x1 = GetInt(p) + xbr;
		  x2 = GetInt(p);
		  yr = GetInt(p);

		  if (tag & HLINE) x2 += x1;

#ifdef STYLE_COLOUR
		  colour_text_ix = *p++;
#endif

#ifdef STYLE_BACKGROUND
		 /* must be set to the structures defined by the style
		  * --Spif 18-Oct-95
		  */
		  SkipPointer(p);
#endif
		  if (DoPrintTextLine)
		    {
		     if (colour_text_ix == 255)
		       {
			XSetForeground(display, theGC, windowBottomShadow);
			XFillRectangle(display, theViewWindow.win, theGC,
				       x1, ybr - yr,     x2 - x1, 1);
			XSetForeground(display, theGC, windowTopShadow);
			XFillRectangle(display, theViewWindow.win, theGC,
				       x1, ybr - yr + 1, x2 - x1, 1);
			XSetForeground(display, theGC, textColour);
		       }
		      else
		       {
		       /* howcome 22/5/95 */
			XSetForeground(display, theGC, ix2colour(colour_text_ix));
			XFillRectangle(display, theViewWindow.win, theGC,
				       x1, ybr - yr, x2 - x1, 1);
		       }
		    }

		  break;

	        case LINE:
#ifdef ARENA_DEBUG
		  if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
		    Arena_TracePrint(Iam,
				     " LINE (0x%x) at "POINTER_FORMAT".\n",
				     tag, tag_p);
#endif
		  x1 = GetInt(p) + xbr;
		  x2 = GetInt(p) + xbr;
		  y1 = GetInt(p);
		  y2 = GetInt(p);
#ifdef STYLE_COLOUR
		  colour_text_ix = *p++;
#endif
#ifdef STYLE_BACKGROUND
		  SkipPointer(p);
#endif
		  if (DoPrintTextLine)
		    {
		     XSetForeground(display, theGC, ix2colour(colour_text_ix));
		     XDrawLine(display, theViewWindow.win, theGC,
			       x1, ybr - y1, x2, ybr - y2);
		    }
		  break;

	        case BULLET:
#ifdef ARENA_DEBUG
		  if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
		    Arena_TracePrint(Iam,
				     " BULLET (0x%x) at "POINTER_FORMAT".\n",
				     tag, tag_p);
#endif
		  x1 = GetInt(p) + xbr;
		  depth = GetInt(p);
		  fnt = *p++;	/* howcome 25/4/95 */
		  fnt_size = *p++;
#ifdef STYLE_COLOUR
		  colour_text_ix = *p++;
#endif
#ifdef STYLE_BACKGROUND
		  /* must be set to the structures defined by the style
		   * --Spif 18-Oct-95
		   */
		  bg_style = (BG_Style*)GetPointer(p);

		  if (bg_style)
		    colour_background_ix = (bg_style->flag &
					   S_BACKGROUND_COLOUR) ?
		                                     rgb2ix(0,
							    bg_style->r,
							    bg_style->g,
							    bg_style->b,
							    0) : 255; 
		   else
		    colour_background_ix = 255;	
#endif
		  if (DoPrintTextLine)
		    {
		     SetColour(theGC, colour_text_ix, colour_background_ix);
		     if (depth > 0)
		       XFillRectangle(display, theViewWindow.win, theGC,
				      x1, ybr - LineSpacing[fnt]*3/8,
				      LineSpacing[fnt]/4, 2);
		      else
		       {
			XFillRectangle(display, theViewWindow.win, theGC,
				       x1, ybr - LineSpacing[fnt]*3/8,
				       LineSpacing[fnt]/5, LineSpacing[fnt]/5);
		       }
		    }
		  break;

	        case STRING:
#ifdef ARENA_DEBUG
		  if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
		    Arena_TracePrint(Iam,
				     " STRING (0x%x) at "POINTER_FORMAT".\n",
				     tag, tag_p);
#endif
		  emph = GetInt(p);
		  fnt = *p++;
#ifdef STYLE_COLOUR
		  colour_text_ix = *p++;
#endif
#ifdef STYLE_BACKGROUND
		  /* --Spif 18-Oct-95:
		   * must be set to the structures defined by the style
		   */
		  bg_style = (BG_Style*)GetPointer(p);

		  if (bg_style)
		    {
		     if (bg_style->image)
		       colour_background_ix = 255; 
		      else
		       colour_background_ix = (bg_style->flag &
					      S_BACKGROUND_COLOUR) ?
			                                rgb2ix(0,
							       bg_style->r,
							       bg_style->g,
							       bg_style->b,
							       0) : 255;
		    }
		   else
		    colour_background_ix = 255;
#endif
		  yr = *p++ - 128; /* baseline offset */
		  x1 = GetInt(p) + xbr; /* here */
		  len = GetInt(p);
		  width = GetInt(p);

		  if (DoPrintTextLine)
		    {
		     char* theBuf = GetPointer(p);

		     preformatted = (tag & PRE_TEXT);
		     s = CopyLine(theBuf, len);

#ifdef STYLE_COLOUR
		     font = fnt;
		     SetColour(theGC, colour_text_ix, colour_background_ix);
		     SetFont(theGC, font);
# else
		     if (font != fnt) SetFont(theGC, (font = fnt));
#endif
		     if (emph & EMPH_HIGHLIGHT)
		       {
			XSetForeground(display, theGC, labelColour);
			XDrawString(display, theViewWindow.win, theGC,
				    x1, ybr - yr, s, len);
			XSetForeground(display, theGC, textColour);
		       }
		      else
		       {
#ifdef STYLE_COLOUR
			if (colour_background_ix == 255)
#endif
			  XDrawString(display, theViewWindow.win, theGC,
				      x1, ybr - yr, s, len);
#ifdef STYLE_COLOUR
			 else
			  XDrawImageString(display, theViewWindow.win, theGC,
					   x1, ybr - yr, s, len);
#endif
		       }

		     if (emph & EMPH_ANCHOR)
		       {
			y2 = ybr - yr - ASCENT(font) - 1;
			DrawHButton(theGC, theViewWindow.win,
				    x1, y2, width, LineSpacing[fnt], True);
		       }

		     if (emph & EMPH_UNDERLINE)
		       {
			y2 = ybr - yr + DESCENT(font);
			XFillRectangle(display, theViewWindow.win, theGC,
				       x1 + 1, y2 - LineThickness[font],
				       width - 2, LineThickness[font]);
		       }

		     if (emph & EMPH_STRIKE)
		       {
			y2 = ybr - yr - ASCENT(font)/3;
			XSetForeground(display,theGC, strikeColour);
			XFillRectangle(display, theViewWindow.win, theGC,
				       x1, y2 - LineThickness[font],
				       width, LineThickness[font]);
			XSetForeground(display,theGC, textColour);
		       }
		    }
		   else
		    {
		     SkipPointer(p); /* skip buf */
		    }

		  SkipPointer(p); /* skip ref */
		  break;

	        case SEQTEXT:
#ifdef ARENA_DEBUG
		  if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
		    Arena_TracePrint(Iam,
				     " SEQTEXT (0x%x) at "POINTER_FORMAT".\n",
				     tag, tag_p);
#endif
		  emph = GetInt(p);
		  fnt = *p++;
#ifdef STYLE_COLOUR
		  colour_text_ix = *p++;
#endif
#ifdef STYLE_BACKGROUND
		  /* must be set to the structures defined by the style
		   * --Spif 18-Oct-95
		   */
		  bg_style = (BG_Style*)GetPointer(p);

		  if (bg_style)
		    colour_background_ix = (bg_style->flag &
					   S_BACKGROUND_COLOUR) ?
		                                     rgb2ix(0,
							    bg_style->r,
							    bg_style->g,
							    bg_style->b,
							    0) : 255;
		   else
		    colour_background_ix = 255;	
#endif
		  x1 = xbr + GetInt(p);  /* here */
		  yr = GetInt(p);       /* y offset */
		  len = *p++;

		  if (DoPrintTextLine)
		    {
		     width = XTextWidth(Fonts[fnt], (char *)p, len);
#ifdef STYLE_COLOUR
		     font = fnt;
		     SetColour(theGC, colour_text_ix, colour_background_ix);
		     SetFont(theGC, font);

		     if (colour_background_ix == 255)
		       XDrawString(display, theViewWindow.win, theGC,
				   x1, ybr - yr, (char *)p, len);
		      else
		       XDrawImageString(display, theViewWindow.win, theGC,
					x1, ybr - yr, (char *)p, len);
# else
		     if (font != fnt)
		       {
			font = fnt;
			SetFont(theGC, font);
		       }

		     XDrawString(display, theViewWindow.win, theGC,
				 x1, ybr - yr, (char *)p, len);
#endif
		     /* different emphasis used in maths */

		     /* under effects */
		     if (emph & EMPH_UNDERLINE)
		       {
			y2 = ybr - yr + DESCENT(font);
			XFillRectangle(display, theViewWindow.win, theGC,
				       x1 + 1, y2 - LineThickness[font],
				       width - 2, LineThickness[font]);
		       }

		     if (emph & EMPH_UNDERLARR)
		       {
			int tmp = 60;

			y2 = ybr - yr + DESCENT(font);
			XFillRectangle(display, theViewWindow.win, theGC,
				       x1 + 1, y2 - LineThickness[font],
				       width, LineThickness[font]);
			SetFont(theGC, IDX_SUBSYMFONT);
			XDrawString(display, theViewWindow.win, theGC,
				    x1, y2 + 3, (char *)&tmp, 1);
			SetFont(theGC, font);
		       }

		     if (emph & EMPH_UNDERRARR)
		       {
			int tmp = 62;

			y2 = ybr - yr + DESCENT(font);
			XFillRectangle(display, theViewWindow.win, theGC,
				       x1 + 1, y2 - LineThickness[font],
				       width, LineThickness[font]);
			SetFont(theGC, IDX_SUBSYMFONT);
			XDrawString(display, theViewWindow.win, theGC,
				    x1 + width - 4, y2 + 3, (char *)&tmp, 1);
			SetFont(theGC, font);
		       }

		     if (emph & EMPH_UNDERHAT)
		       {
		       }

		     if (emph & EMPH_UNDERTILDE)
		       {
		       }

		     /* OVER effects */
		     if (emph & EMPH_OVERLINE)
		       {
			y2 = ybr - yr + DESCENT(font)/2 - ASCENT(font);
			XFillRectangle(display, theViewWindow.win, theGC,
				       x1 + 1, y2 - LineThickness[font],
				       width - 2, LineThickness[font]);
		       }

		     if (emph & EMPH_OVERLARR)
		       {
			int tmp = 60;

			y2 = ybr - yr + DESCENT(font)/2 - ASCENT(font);
			XFillRectangle(display, theViewWindow.win, theGC,
				       x1 + 1, y2 - LineThickness[font],
				       width - 2, LineThickness[font]);
			SetFont(theGC, IDX_SUBSYMFONT);
			XDrawString(display, theViewWindow.win, theGC,
				    x1, y2 + 3, (char *)&tmp, 1);
			SetFont(theGC, font);
		       }

		     if (emph & EMPH_OVERRARR)
		       {
			int tmp = 62;

			y2 = ybr - yr + DESCENT(font)/2 - ASCENT(font);
			XFillRectangle(display, theViewWindow.win, theGC,
				       x1, y2, width, 1);
			SetFont(theGC, IDX_SUBSYMFONT);
			XDrawString(display, theViewWindow.win, theGC,
				    x1 + width - 4, y2 + 3, (char *)&tmp, 1);
			SetFont(theGC, font);
		       }

		     if (emph & EMPH_OVERHAT)
		       {
		       }

		     if (emph & EMPH_OVERDOT)
		       {
		       }

		     if (emph & EMPH_OVERDDOT)
		       {
		       }

		     if (emph & EMPH_OVERTILDE)
		       {
		       }
		    }

		  p += len;
		  break;

	        case IMAGE:
		  {
		   Pixmap theIcon = None, theMask = None;

		   yr = -GetInt(p);
		   y2 = yb + yr;
		   yr += ybr;
		   xr = xbr + GetInt(p);  /* here */
		    width  = GetInt(p);
		    height = GetInt(p);
		   theIcon = GetPixmap(p);
		   theMask = GetPixmap(p);

		   SkipPointer(p);  /* skip past buffer pointer */
#ifdef STYLE_BACKGROUND
		   /* must be set to the structures defined by the style
		    * --Spif 18-Oct-95
		    */
		   SkipPointer(p);
#endif
#ifdef ARENA_DEBUG
		   if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
		     Arena_TracePrint(Iam,
				      " IMAGE (0x%x) %dx%d%+d%+d"
				      " at "POINTER_FORMAT".\n",
				      tag, width, height, xr, yr, tag_p);
#endif
		   if (DoPrintTextLine)
		     {
		      if (y2 > theArea.y + (int)theArea.height)	break;
		      if (y2 + height < theArea.y) continue;

		      if (tag & (ISMAP | EMPH_ANCHOR))
			{
			 DrawOutSet(theViewWindow.win, theGC,
				    xr, yr, width, height);
			 XSetForeground(display, theGC, textColour);
			 DrawHButton(theGC, theViewWindow.win,
				     xr + 4, yr + 2, width - 7, height - 6,
				     False);
			 width  -= 8;
			 height -= 8;
			 xr += 4;
			 yr += 4;
			}

		      if (theIcon != None)
			{
#ifdef ARENA_DEBUG
			 if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
			   Arena_TracePrint(Iam,
					    " drawing image "POINTER_FORMAT":"
					    " %dx%d%+d%+d.\n",
					    tag_p, width, height, xr, yr);
#endif
			 Arena_PutTransparentImage(display, theGC,
						   theIcon, theMask,
						   theViewWindow.win,
						   xr, yr, width, height);
			}
		       else
			{
			 if (!(tag & (ISMAP | EMPH_ANCHOR)))
			   {
			    XPoint points[3];

			    points[0].x = xr;
			    points[0].y = yr + IMG_INDICATOR;
			    points[1].x = xr;
			    points[1].y = yr;
			    points[2].x = xr + IMG_INDICATOR;
			    points[2].y = yr;
			    XDrawLines(display, theViewWindow.win, theGC,
				       points, 3, CoordModeOrigin);

			    points[0].x = xr + width - IMG_INDICATOR;
			    points[0].y = yr;
			    points[1].x = xr + width;
			    points[1].y = yr;
			    points[2].x = xr + width;
			    points[2].y = yr + IMG_INDICATOR;
			    XDrawLines(display, theViewWindow.win, theGC,
				       points, 3, CoordModeOrigin);

			    points[0].x = xr + width;
			    points[0].y = yr + height - IMG_INDICATOR;
			    points[1].x = xr + width;
			    points[1].y = yr + height;
			    points[2].x = xr + width - IMG_INDICATOR;
			    points[2].y = yr + height;
			    XDrawLines(display, theViewWindow.win,
				       theGC, points, 3, CoordModeOrigin);

			    points[0].x = xr + IMG_INDICATOR ;
			    points[0].y = yr + height;
			    points[1].x = xr;
			    points[1].y = yr + height;
			    points[2].x = xr;
			    points[2].y = yr + height - IMG_INDICATOR;
			    XDrawLines(display, theViewWindow.win, theGC,
				       points, 3, CoordModeOrigin);
			   }
			}
		     }
		  }
		  break;

	        case INPUT:
#ifdef ARENA_DEBUG
		  if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
		    Arena_TracePrint(Iam,
				     " INPUT (0x%x) at "POINTER_FORMAT".\n",
				     tag, tag_p);
#endif
		  str = GetLong(p);
#ifdef STYLE_BACKGROUND
		  /* must be set to the structures defined by the style
		   * --Spif 18-Oct-95
		   */
		  SkipPointer(p);
#endif
		  if (DoPrintTextLine)
		    {
#ifdef ARENA_DEBUG
		     if (PAINTSTREAM_TRACE)
		       {
			Field* f = (Field*)str;

			Arena_TracePrint(Iam,
					 " INPUT frame,   xb = %d,  yb = %d.\n"
					 "\tGot "POINTER_FORMAT
					 " Field struct from the stream:\n"
					 "\t\t name = \"%.*s\" (nlen = %d),\n"
					 "\t\tvalue = \"%s\",\n"
					 "\t\t    x = %d,   baseline = %ld\n"
					 "\t\twidth = %d,   height = %d,"
					 "   above = %d.\n",
					 xb, yb,
					 f,
					 f->nlen, f->name, f->nlen,
					 f->value,
					 f->x, f->baseline,
					 f->width, f->height, f->above);
		       }
#endif
		     PaintField(theGC, theViewWindow,
				xb + theViewWindow.rect.x,
				yb + theViewWindow.rect.y,
				(Field*)str);
		    }
		  break;

	        default:
		  /* safety net for garbled paint structure */
#ifdef ARENA_DEBUG
		  Arena_TracePrint(Iam,
				   " ERROR! Unexpected (B) internal tag 0x%x"
				   " at "POINTER_FORMAT", Exiting.\n",
				   tag, tag_p);
# else
		  Arena_PrintError(_("Unexpected (%s) internal tag 0x%x.\n"),
				   "B", tag);
#endif
		  Exit(1);
		  break;
	       }
	     /* End ``switch (tag & 0xF)'' */

	     if (BreakBreak)
	       {
		BreakBreak = False;
	        break;
	       }
	    }
	  /* End ``while ((tag = *p++) != '\0')'' */
	  break;

        default:
#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
	  Arena_TracePrint(Iam,
			   " ERROR! Unexpected (C) internal tag 0x%x"
			   " at "POINTER_FORMAT", Exiting.\n",
			   tag, tag_p);
# else
	  Arena_PrintError(_("Unexpected (%s) internal tag 0x%x.\n"),
			   "C", tag);
#endif
	  Exit(1);
	  break;
       }
     /* End ``switch (tag)'' */

     SkipInt(p);  /* skip size param to start of next object */
     if (BreakBreak) break;
    }
   }

#ifdef ARENA_DEBUG	/* QingLong.25-01-97 */
 if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam,
		    " after while p = "POINTER_FORMAT","
		    " p_end = "POINTER_FORMAT"\n",
		    p, p_end);
#endif
}


/*
 * search index for given keywords
 */
void SearchIndex(char* theKeywords)
{
#if 0
 static char* IndexSearchString = NULL;
 char *p, *q;
 char* keywords;


 if (Arena_StrLen(theKeywords))
   keywords = theKeywords;
  else
   if ((keywords = ArenaPromptUser(_("Search for?"), IndexSearchString, True)))
     {
      Free(IndexSearchString);
      IndexSearchString = keywords;
     }
    else
     return;

 p = SearchRef(keywords);

 if (*p)  /* attempt to get new document */
   {
   /* XDefineCursor(display, win, hourglass);*/
   /* XFlush(display);*/

    q = libGetDocument(p, NULL, REMOTE); /* howcome */

    if (q && *q)
      {
       CurrentDoc.offset = PixelOffset;
       PushDoc(CurrentDoc.offset);
       SetBanner(BANNER);

       SetCurrent();
       NewBuffer(q);

       if (DocIsIndex) ClearStatus();
      }

    DisplayDoc(CurrentDoc, False);
    XUndefineCursor(display, DocDispWin);
    XFlush(display);
   }
#endif
}

int WindowButtonDown(GC theGC, ViewWindow theViewWindow,
		     unsigned int button, int x, int y)
{
 int tag, dx, dy;
#if 0
#ifdef ARENA_DEBUG
 char Iam[] = "WindowButtonDown";
#endif
#endif

 /* the window should only be clickable for certain content_types and modes */

 if (CurrentDoc == NULL) return VOID;
 if (CurrentDoc->show_raw) return VOID;
 if (!DocHTML(CurrentDoc)) return VOID;

 anchor_start = anchor_end = 0;
 tag = 0;

 WhichObject(theGC, theViewWindow,
	     BUTTONDOWN, x, y,
	     &tag, &anchor_start, &anchor_end, &dx, &dy);

   /* toggle checked here ??? --SPif 13-oct-95 */
 if ((tag == TAG_INPUT) || (tag == TAG_SELECT)) return WINDOW;

 FinishCurrentFieldEditing(theGC, theViewWindow);

 if (SpawnIsWaitGui() == False && ((tag == TAG_ANCHOR) || (tag == TAG_IMG)))
   /* redraw anchor if type=image or src= */
   {
    DrawAnchor(theGC, theViewWindow, &background, 0);
    return WINDOW;
   }

 Beep();
 return VOID;
}


void WindowButtonUp(GC theGC, ViewWindow theViewWindow,
		    unsigned int button, int shifted, int px, int py)
{
#define WINDOW_BUTTON_UP__BUFFER_CAPACITY 16
 Byte* object;
 char *start, *end, *href, *name, *link, *value, *action;
 char buf[WINDOW_BUTTON_UP__BUFFER_CAPACITY];
 int tag, hreflen, namelen, align, ismap, dx, dy, width, height;
 int type, vlen, flags, maxlength, size, alen, nlen, link_capacity;
 FormSubmitMethod method;
 Image* image = NULL;
 Form* form = NULL;
#if 0
#ifdef ARENA_DEBUG
 char Iam[] = "WindowButtonUp";
#endif
#endif


/* the window should only be cliackable for certain content_types and modes */

 if (CurrentDoc == NULL) return;
 if (CurrentDoc->show_raw) return;
 if (!DocHTML(CurrentDoc)) return;

 if (anchor_start && anchor_end) DrawAnchor(theGC, theViewWindow,
					    &background, 1);

 tag = UNKNOWN; /* -- avoid any pb with this --Spif 16-Oct-95 */

 object = WhichObject(theGC, theViewWindow,
		      BUTTONUP, px, py, &tag, &start, &end, &dx, &dy);

 if (((tag == TAG_ANCHOR) || (tag == TAG_IMG)) &&
     (start == anchor_start) &&
     (  end == anchor_end))
   {
    if (tag == TAG_IMG)
      {
       bufptr = anchor_start + 5;
       ParseImageAttrs(&href, &hreflen, &align, &ismap, &width, &height);
#ifdef HAVE_SNPRINTF
       snprintf(buf, WINDOW_BUTTON_UP__BUFFER_CAPACITY-1, "?%d,%d", dx, dy);
# else
       sprintf(buf, "?%d,%d", dx, dy);
#endif
       link_capacity = hreflen + WINDOW_BUTTON_UP__BUFFER_CAPACITY + 1;
       link = (char *)Arena_MAlloc(link_capacity, False);
       memcpy(link, href, hreflen);
       link[hreflen] = '\0';
       strncat(link, buf, WINDOW_BUTTON_UP__BUFFER_CAPACITY - 1);
      }
     else /* tag == TAG_ANCHOR */
      {
       int class_len;
       char *class = NULL;

       bufptr = anchor_start + 3;
       ParseAnchorAttrs(&href, &hreflen, &name, &namelen, &class, &class_len);

       if (dx >= 0 && dy >= 0)
	 {
#ifdef HAVE_SNPRINTF
	  snprintf(buf, WINDOW_BUTTON_UP__BUFFER_CAPACITY-1, "?%d,%d", dx, dy);
# else
	  sprintf(buf, "?%d,%d", dx, dy);
#endif
	  link_capacity = hreflen + WINDOW_BUTTON_UP__BUFFER_CAPACITY + 1;
	  link = (char *)Arena_MAlloc(link_capacity, False);
	  memcpy(link, href, hreflen);
	  link[hreflen] = '\0';
	  strncat(link, buf, WINDOW_BUTTON_UP__BUFFER_CAPACITY - 1);
	 }
        else
	 {
	  link = (char *)Arena_MAlloc(hreflen+1, False);
	  memcpy(link, href, hreflen);
	  link[hreflen] = '\0';
	 }
      }
#if 0
    if (shifted)
      {
       if (CloneSelf()) OpenDoc(link);
      }
     else
#endif
    OpenDoc(link);
    Free(link);
   }
  else
   {
    if (tag == TAG_INPUT) /* --Spif FORM 10-Oct-95 */
      {
       bufptr = start + 7;
       ParseInputAttrs(&type, &name, &nlen, &value, &vlen,
		       &maxlength, &size, &flags, &image);
       /* backtrack to find the previous <FORM..> */
       for( ; (strncasecmp(bufptr, "<form", 5) != 0) && (*bufptr); bufptr--);
       bufptr += 5;
       ParseFormAttrs(&method, &action, &alen);  /* must test that
						    a form was found */
       form = FindFormByAction(action);
       if (form)
	 {
	  if (type == RESETBUTTON)
	    {
	    /* reset the whole form */
	     ResetForm(theGC, theViewWindow, form);
	    }
	   else
	    {
	     if (type == SUBMITBUTTON)
	       {
		if (CurrentDoc->edited_field)
		  FinishCurrentFieldEditing(theGC, theViewWindow);
		SubmitForm(form, form->method, action, alen, type,
			   name, nlen, value, vlen, image, dx, dy, bufptr);
	       }
	     else
	       {
	       /* not a submitbutton  but who cares ? ;)  -- Spif 18-Oct-95*/
	       }
	    }
	 }
      }
   }

 XFlush(display);
#undef WINDOW_BUTTON_UP__BUFFER_CAPACITY
}


Bool DisplayHTML(Doc* theDoc,
		 GC theGC, ViewWindow theViewWindow, ViewRectangle theArea)
{
 ViewRectangle theVisibleSubArea = {0,0,0,0};
#ifdef ARENA_DEBUG
 char Iam[] = "DisplayHTML";
#endif


 if (VisibleViewRectangle(theViewWindow.rect, theArea, &theVisibleSubArea))
   {
#ifdef ARENA_DEBUG
    if (SCROLL_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       " %dx%d%+d%+d ---> %dx%d%+d%+d.\n",
		       theArea.width, theArea.height, theArea.x, theArea.y,
		       theVisibleSubArea.width, theVisibleSubArea.height,
		       theVisibleSubArea.x, theVisibleSubArea.y);
#endif

    FormatElementStart(theDoc, TAG_HTML, NULL, 0);
    PaintFrame(theGC, theViewWindow, theVisibleSubArea, 0,
	       paint + background.info, PaintStreamCurrentPosition() - 1);
    FormatElementEnd(theDoc);

    if (focus)
      if ((focus->type == OPTIONLIST) && (focus->flags & CHECKED))
	PaintDropDown(theGC, theViewWindow, -1, focus);

    return True;
   }
  else
   return False;
}
