/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>

#include "commands.h"
#include "gzillabrowser.h"

/* todo:

   One menu per browser window.

   Get bw as callback data for commands.

   */

static void menus_init (void);
#if 0
static void menus_foreach (gpointer key,
						   gpointer value,
						   gpointer user_data);
#endif

static gint menus_install_accel (GtkWidget *widget,
								 gchar     *signal_name,
								 gchar      key,
								 gchar      modifiers,
								 gchar     *path);

static void menus_remove_accel (GtkWidget *widget,
								gchar     *signal_name,
								gchar     *path);



static GtkMenuEntry menu_items[] =
{
	  { "<Gzilla>/File/New Browser", "<alt>N", file_new_cmd_callback, NULL },
	  { "<Gzilla>/File/<separator>", NULL, NULL, NULL },
	  { "<Gzilla>/File/Open File...", "<alt>O", file_openfile_cmd_callback, NULL },
	  { "<Gzilla>/File/Open URL...", "<alt>L", file_openurl_cmd_callback, NULL },
	  { "<Gzilla>/File/<separator>", NULL, NULL, NULL },
	  { "<Gzilla>/File/Quit", "<alt>Q", file_quit_cmd_callback, NULL },
	  { "<Gzilla>/View/Reload", "<alt>R", view_reload_cmd_callback, NULL },
	  { "<Gzilla>/Go/Back", NULL, go_back_cmd_callback, NULL },
	  { "<Gzilla>/Go/Forward", NULL, go_forw_cmd_callback, NULL },
	  { "<Gzilla>/Go/Home", NULL, go_home_cmd_callback, NULL },
	  { "<Gzilla>/Go/Stop Loading", NULL, go_stop_cmd_callback, NULL },
	  { "<Gzilla>/Bookmarks/Add Bookmark", "<alt>A", bookmark_add_cmd_callback, NULL },
	  { "<Gzilla>/Bookmarks/<separator>", NULL, NULL, NULL }
};

static int nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);

static int initialize = TRUE;
static GtkMenuFactory *factory = NULL;
static GtkMenuFactory *subfactories[1];
static GHashTable *entry_ht = NULL;


void
menus_get_gzilla_menubar (GtkWidget           **menubar,
						  GtkAcceleratorTable **table)
{
	if (initialize)
			menus_init ();
	
	if (menubar)
			*menubar = subfactories[0]->widget;
	if (table)
			*table = subfactories[0]->table;
}


void
menus_create (GtkMenuEntry *entries,
			  int           nmenu_entries)
{
	char *accelerator;
	int i;
	
	if (initialize)
			menus_init ();
	
	if (entry_ht)
			for (i = 0; i < nmenu_entries; i++)
      {
		  accelerator = g_hash_table_lookup (entry_ht, entries[i].path);
		  if (accelerator)
			{
				if (accelerator[0] == '\0')
						entries[i].accelerator = NULL;
				else
						entries[i].accelerator = accelerator;
			}
      }
	
	gtk_menu_factory_add_entries (factory, entries, nmenu_entries);
	
	for (i = 0; i < nmenu_entries; i++)
			if (entries[i].widget)
      {
		  gtk_signal_connect (GTK_OBJECT (entries[i].widget), "install_accelerator",
							  (GtkSignalFunc) menus_install_accel,
							  entries[i].path);
		  gtk_signal_connect (GTK_OBJECT (entries[i].widget), "remove_accelerator",
							  (GtkSignalFunc) menus_remove_accel,
							  entries[i].path);
      }
}

void
menus_set_sensitive (char *path,
					 int   sensitive)
{
	GtkMenuPath *menu_path;
	
	if (initialize)
			menus_init ();
	
	menu_path = gtk_menu_factory_find (factory, path);
	if (menu_path)
			gtk_widget_set_sensitive (menu_path->widget, sensitive);
	else
			g_warning ("Unable to set sensitivity for menu which doesn't exist: %s", path);
}

void
menus_add_path (char *path,
				char *accelerator)
{
	if (!entry_ht)
			entry_ht = g_hash_table_new (g_string_hash, g_string_equal);
	
	g_hash_table_insert (entry_ht, path, accelerator);
}

#if 0
void
menus_quit ()
{
	FILE *fp;
	char filename[512];
	char *home_dir;
	
	if (!entry_ht)
			return;
	
	home_dir = getenv ("HOME");
	if (home_dir)
	  {
		  sprintf (filename, "%s/.gzilla/menurc", home_dir);
		  
		  fp = fopen (filename, "w");
		  if (!fp)
				  return;
		  
		  g_hash_table_foreach (entry_ht, menus_foreach, fp);
		  
		  fclose (fp);
	  }
}
#endif

static void
menus_init ()
{
	if (initialize)
	  {
		  initialize = FALSE;
		  
		  factory = gtk_menu_factory_new (GTK_MENU_FACTORY_MENU_BAR);
		  
		  subfactories[0] = gtk_menu_factory_new (GTK_MENU_FACTORY_MENU_BAR);
		  gtk_menu_factory_add_subfactory (factory, subfactories[0], "<Gzilla>");
		  
#if 0
		  parse_gimprc_file (".gimp/menurc");
#endif
		  
		  menus_create (menu_items, nmenu_items);
	  }
}

#if 0
static void
menus_foreach (gpointer key,
			   gpointer value,
			   gpointer user_data)
{
	fprintf ((FILE*) user_data, "(menu-path \"%s\" \"%s\")\n", (char*) key, (char*) value);
}
#endif

static gint
menus_install_accel (GtkWidget *widget,
					 gchar     *signal_name,
					 gchar      key,
					 gchar      modifiers,
					 gchar     *path)
{
	char accel[64];
	char *t1, t2[2];
	
	accel[0] = '\0';
	if (modifiers & GDK_CONTROL_MASK)
			strcat (accel, "<control>");
	if (modifiers & GDK_SHIFT_MASK)
			strcat (accel, "<shift>");
	if (modifiers & GDK_MOD1_MASK)
			strcat (accel, "<alt>");
	
	t2[0] = key;
	t2[1] = '\0';
	strcat (accel, t2);
	
	if (entry_ht)
	  {
		  t1 = g_hash_table_lookup (entry_ht, path);
		  g_free (t1);
	  }
	else
			entry_ht = g_hash_table_new (g_string_hash, g_string_equal);
	
	g_hash_table_insert (entry_ht, path, g_strdup (accel));
	
	return TRUE;
}

static void
menus_remove_accel (GtkWidget *widget,
					gchar     *signal_name,
					gchar     *path)
{
	char *t;
	
	if (entry_ht)
	  {
		  t = g_hash_table_lookup (entry_ht, path);
		  g_free (t);
		  
		  g_hash_table_insert (entry_ht, path, g_strdup (""));
	  }
}

/* New menus stuff here (based on GTK+ menus and not menufactory. */

/* Taken from gtkmenufactory.c.

   Todo: make it much more general, so that it can handle keys outside
   the 8-bit ASCII set. This will require changes to GTK :-( */
static void
gtk_menu_factory_parse_accelerator (char   *accelerator,
				    char   *accelerator_key,
				    guint8 *accelerator_mods)
{
  int done;

  g_return_if_fail (accelerator != NULL);
  g_return_if_fail (accelerator_key != NULL);
  g_return_if_fail (accelerator_mods != NULL);

  *accelerator_key = 0;
  *accelerator_mods = 0;

  done = FALSE;
  while (!done)
    {
      if (strncmp (accelerator, "<shift>", 7) == 0)
        {
          accelerator += 7;
          *accelerator_mods |= GDK_SHIFT_MASK;
        }
      else if (strncmp (accelerator, "<alt>", 5) == 0)
        {
          accelerator += 5;
          *accelerator_mods |= GDK_MOD1_MASK;
        }
      else if (strncmp (accelerator, "<control>", 9) == 0)
        {
          accelerator += 9;
          *accelerator_mods |= GDK_CONTROL_MASK;
        }
      else
        {
          done = TRUE;
          *accelerator_key = accelerator[0];
        }
    }
}

/* Make a new menu, insert it into the menu bar, and return it. */
GtkWidget *
gzilla_menu_new (GtkWidget *menubar,
		 const char *name,
		 gboolean right_justify,
		 BrowserWindow *bw)
{
  GtkWidget *menu;
  GtkWidget *menuitem;

  menu = gtk_menu_new ();
  menuitem = gtk_menu_item_new_with_label ((char *)name);
  if (right_justify)
    gtk_menu_item_right_justify (GTK_MENU_ITEM (menuitem));
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menubar), menuitem);
  gtk_widget_show (menuitem);

  gtk_menu_set_accelerator_table (GTK_MENU (menu), bw->accel_table);

  return menu;
}

/* Add an item to a menu, including the name, an accelerator (not yet
   implemented), and a callback function for activation. */
static GtkWidget *
gzilla_menu_add (GtkWidget *menu, const char *name, const char *accel,
		 void *data, BrowserWindow *bw,
		 void (*callback) (GtkWidget *widget, void *data))
{
  GtkWidget *menuitem;
  char accel_key;
  guint8 accel_mods;

  menuitem = gtk_menu_item_new_with_label ((char *)name);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);
  if (accel != NULL)
    {
      gtk_menu_factory_parse_accelerator ((char *)accel,
					  &accel_key, &accel_mods);
      gtk_widget_install_accelerator (menuitem,
				      bw->accel_table,
				      "activate",
				      accel_key,
				      accel_mods);
    }

  if (callback != NULL)
    gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			(GtkSignalFunc) callback,
			data);

  return menuitem;
}

/* Add a separator into the menu. */
static void
gzilla_menu_sep (GtkWidget *menu)
{
  GtkWidget *widget;

  widget = gtk_menu_item_new ();
  gtk_menu_append (GTK_MENU (menu), widget);
  gtk_widget_show (widget);
}

/* Make up a new menubar for a main browser window. The accelerator table
   is stored in bw->accel_table.

   Currently does not deal with dynamic menu items (bookmarks and history). */
GtkWidget *
gzilla_menu_mainbar_new (BrowserWindow *bw)
{
  GtkWidget *menubar;
  GtkWidget *file_menu;
  GtkWidget *view_menu;
  GtkWidget *go_menu;
  GtkWidget *bookmarks_menu;
  GtkWidget *help_menu;

  menubar = gtk_menu_bar_new ();

  bw->accel_table = gtk_accelerator_table_new ();
  gtk_accelerator_table_ref (bw->accel_table);

  file_menu = gzilla_menu_new (menubar, "File", FALSE, bw);

#if 0
#endif

  gzilla_menu_add (file_menu, "New browser", "<alt>N", bw, bw,
		   file_new_cmd_callback);
  gzilla_menu_sep (file_menu);
  gzilla_menu_add (file_menu, "Open File...", "<alt>O", bw, bw,
		   file_openfile_cmd_callback);
  gzilla_menu_add (file_menu, "Open URL...", "<alt>L", bw, bw,
		   file_openurl_cmd_callback);
  gzilla_menu_sep (file_menu);
  gzilla_menu_add (file_menu, "Test Gzw", "<alt>Z", bw, bw,
		   file_testgzw_cmd_callback);
  gzilla_menu_sep (file_menu);
  gzilla_menu_add (file_menu, "Quit", "<alt>Q", bw, bw,
		   file_quit_cmd_callback);

  view_menu = gzilla_menu_new (menubar, "View", FALSE, bw);

  gzilla_menu_add (view_menu, "Reload", "<alt>R", bw, bw,
		   view_reload_cmd_callback);

  go_menu = gzilla_menu_new (menubar, "Go", FALSE, bw);

  bw->back_menuitem = gzilla_menu_add (go_menu, "Back", NULL, bw, bw,
		   go_back_cmd_callback);
  bw->forw_menuitem = gzilla_menu_add (go_menu, "Forward", NULL, bw, bw,
		   go_forw_cmd_callback);
  gzilla_menu_add (go_menu, "Home", NULL, bw, bw,
		   go_home_cmd_callback);
  bw->stop_menuitem = gzilla_menu_add (go_menu, "Stop Loading", NULL, bw, bw,
				       go_stop_cmd_callback);

  bookmarks_menu = gzilla_menu_new (menubar, "Bookmarks", FALSE, bw);
  bw->bookmarks_menu = bookmarks_menu;

  gzilla_menu_add (bookmarks_menu, "Add Bookmark", "<alt>A", bw, bw,
		   bookmark_add_cmd_callback);
  gzilla_menu_sep (bookmarks_menu);

  help_menu = gzilla_menu_new (menubar, "Help", TRUE, bw);

  gzilla_menu_add (help_menu, "Gzilla homepage", NULL, bw, bw,
		   help_home_cmd_callback);

  return menubar;
}
