/* config.c - Read config file for redir. */

#ifndef lint
char RCS_config_c[] =
"$Id: config.c,v 2.0 1998/08/10 06:54:21 caleishm Exp caleishm $  Produced by Chris Leishman & Trevor Cohn.  Ormond College student IT department, 1998.\n";
#endif /* not lint */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include "config.h"
#include "main.h"
#include "utils.h"

#define DEFAULT_RANDOM 100
#define LINE_BUFF_SIZE 1000
#define COMMAND_SIZE   50
#define BASE_TYPE      0


/* Function prototypes */

static void add_redirects(item *, char *);
static str_list *get_keys(char *, int);
static str_list *file_get_keys(char *);
static void add_sub(int, item *, char *, int, FILE *);
static void do_random(item *, char *);
static void set_log(item *, char *);
static item *readconfig(FILE *, int, int, str_list *);
static item *createitem(void);
static str_list *createstr(char *);
static char *readline(FILE *, char *, int);
static char *getword(char *, char *, int);
static void die_config(void);


/* Keep track of config file line */
/*     Ok, I know, global variables suck, but I couldn't be bothered....*/

static int lineno = 1;


/* Open config file, then call recursive readconfig.  Return the data struct.*/

item *load_config(const char *file)
{
	FILE *cfp;
	item *data;

	if ((cfp = fopen(file, "r")) == NULL) {
		fprintf(stderr, "ERROR: Couldn't open file \"%s\"\n\n", file);
		exit(EXIT_FAILURE);
	}

	/* Call readconfig with BASE_TYPE, level 0, and no keys */

	if (!(data = readconfig(cfp, BASE_TYPE, 0, NULL))) 
		die_config();

	if (fclose(cfp) == EOF) {
		fprintf(stderr, "ERROR: Couldn't close file \"%s\"\n\n", file);
		exit(EXIT_FAILURE);
	}

	return data;
}


/* Recursive function for processing each event block. */
/* Recieves the event type and the search keys that apply. */

item *readconfig(FILE *cfp, int type, int level, str_list *keys)
{
	char line[LINE_BUFF_SIZE] = "";
	char command[COMMAND_SIZE] = "";
	item *thisitem;
	item *curr;
#ifdef DEBUG
	int c=2;
#endif


	/* Create the event data structure. */

	thisitem = createitem();
	thisitem->type = type;
	thisitem->keys = keys;


	/* Read a line, and determine the command */
	
	while ((readline(cfp, line, LINE_BUFF_SIZE)) 
	       && (line[0] != '\0') && (line[0] != '}'))
	{
		if (!getword(line, command, COMMAND_SIZE))
			die_config();

#ifdef DEBUG
		printf("\t\t++++%s:%s\n", command, line);
#endif


		/* Act on known commands. */

		if (!strncmp("redirect", command, COMMAND_SIZE))
			add_redirects(thisitem, line);
		else if (!strncmp("randomness", command, COMMAND_SIZE))
			do_random(thisitem, line);
		else if (!strncmp("logfile", command, COMMAND_SIZE))
			set_log(thisitem, line);
		else if (!strncmp("contains", command, COMMAND_SIZE))
			add_sub(CONTAINS, thisitem, line, level, cfp); 
		else if (!strncmp("hostcontains", command, COMMAND_SIZE))
			add_sub(HOSTCONTAINS, thisitem, line, level, cfp); 
		else if (!strncmp("pathcontains", command, COMMAND_SIZE)) 
			add_sub(PATHCONTAINS, thisitem, line, level, cfp); 
		else if (!strncmp("exempt", command, COMMAND_SIZE)) 
			add_sub(EXEMPT, thisitem, line, level, cfp); 
		else if (!strncmp("hostexempt", command, COMMAND_SIZE))
			add_sub(HOSTEXEMPT, thisitem, line, level, cfp); 
		else if (!strncmp("pathexempt", command, COMMAND_SIZE)) 
			add_sub(PATHEXEMPT, thisitem, line, level, cfp); 
		else {
			fprintf(stdout, "Unknown: %s\n", command);
			die_config();
		}
	}


	/* Die if we don't leave a block with } (except at base level). */
	/* Also detect if were unbalanced with }'s. */

	if ((level == 0 && line[0] == '}') || (level != 0 && line[0] != '}'))
		die_config();


	/* If we have sub structures, and redirects, then move the redirects to
	   a node at the very end of the sub structures as a default case. */

	if ((thisitem->redirect != NULL) && (thisitem->subs != NULL)) {
		curr = thisitem->subs;
		while(curr->next != NULL) {
#ifdef DEBUG
			c++;
#endif
			curr = curr->next;
		}
		curr->next = createitem();
		curr->next->redirect = thisitem->redirect;
		thisitem->redirect = NULL;
#ifdef DEBUG
		printf("Moving parent redirects to sublevel(%d), node %d.\n",
		        level+1, c);
#endif
	}

	return thisitem;
}


/* Routine for creating and adding a subevent to the current block */

void add_sub(int type, item *thisitem, char *line, int level, FILE *cfp)
{
	str_list *keys;
	item *nsub;
	item *curr;
#ifdef DEBUG
	int c = 1;
#endif

#ifdef DEBUG
	printf("Entering sublevel(%d) of type %d.\n", level+1, type);
#endif


	/* Get the keys to search for from after the command */

	if (!(keys = get_keys(line, type)))
		die_config();

	/* Depending on the type of event, act differently.  If a contains 
	   event then recurse back to readconfig.  If an exemption, then 
           just create a basic structure and add the keys to it. */

	if (type < EXEMPT) {
		if (!(nsub = readconfig(cfp, type, level+1, keys)))
			die_config();
	}
	else {
		nsub = createitem();
		nsub->keys = keys;
		nsub->type = type;
	}


	/* Add this subevent to the end of the subs list */

	if (thisitem->subs == NULL) {
		thisitem->subs = nsub;
	}
	else {
		curr = thisitem->subs;
#ifdef DEBUG
			c++;
#endif
		while(curr->next != NULL) {
#ifdef DEBUG
			c++;
#endif
			curr = curr->next;
		}
		curr->next = nsub;
	}
	curr = nsub;

#ifdef DEBUG
	printf("Exited sublevel(%d).  Joined to parent as node %d.\n", level+1,c);
#endif
}


/* Function to get search keys from off the string after a command */

str_list *get_keys(char *line, int type)
{
	str_list *head = NULL;
	str_list *curr = NULL;
	str_list *nstr = NULL;
	char buffer[LINE_BUFF_SIZE] = "";


	/* Get a key at a time */

	while(getword(line, buffer, LINE_BUFF_SIZE) 
	      && buffer[0] != '{' && buffer[0] != ';') {


		/* If key is "file" then get the next key and use this as a 
		   filename for getting keys from.  Otherwise, make a string 
		   node out of the key. */

		if (!strcmp("file", buffer)) {
			if (!getword(line, buffer, LINE_BUFF_SIZE))
				die_config();
#ifdef DEBUG
			printf("Reading from file %s\n", buffer);
#endif
			nstr = file_get_keys(buffer);
		}
		else {
#ifdef DEBUG
			printf("Adding key %s\n", buffer);
#endif
			nstr = createstr(buffer);

#ifndef NOSAVESKIP
			/* for Boyer-Moore seraching */
			nstr->skip = buildskip(buffer);
#endif
		}


		/* Add new key(s) into key list */

		if (curr == NULL) {
			head = curr = nstr;
		}
		else if (nstr != NULL) {
			while(curr->next != NULL)
				curr = curr->next;
			curr->next = nstr;
			curr = nstr;
		}
	}


	/* Different types should finish with a '{' or ';'.  Check this. */

	if (type < EXEMPT && buffer[0] == ';')
		die_config();
	if (type >= EXEMPT && buffer[0] == '{')
		die_config();

#ifdef DEBUG
	printf("Term keys on %s\n", buffer);
#endif
	return head;
}


/* Function to read keys from a file */

str_list *file_get_keys(char *file)
{
	str_list *head = NULL;
	str_list *curr = NULL;
	str_list *nstr = NULL;
	char buffer[LINE_BUFF_SIZE];
	FILE *fp;

	if ((fp = fopen(file, "r")) == NULL) {
		fprintf(stderr, "ERROR: Couldn't open file \"%s\"\n\n", file);
		exit(EXIT_FAILURE);
	}


	/* Read line at a time, remove \n, put in a node and add to list */

	while (fgets(buffer, LINE_BUFF_SIZE, fp)) {
		if (buffer[strlen(buffer)-1] == EOLN)
			buffer[strlen(buffer)-1] = EOS;
#ifdef DEBUG
			printf("Adding key %s from file\n", buffer);
#endif

		nstr = createstr(buffer);

#ifndef NOSAVESKIP
		/* for Boyer-Moore seraching */
		nstr->skip = buildskip(buffer);
#endif

		if (curr == NULL) {
			head = curr = nstr;
		}
		else if (nstr != NULL) {
			curr->next = nstr;
			curr = nstr;
		}
	}

	if (fclose(fp) == EOF) {
		fprintf(stderr, "ERROR: Couldn't close file \"%s\"\n\n", file);
		exit(EXIT_FAILURE);
	}

	return head;
}


/* Function to read and add redirect strings to redirect list */

void add_redirects(item *thisitem, char *line)
{
	str_list *redirs = NULL;
	str_list *nstr = NULL;
	char buffer[LINE_BUFF_SIZE] = "";


	/* Find end of list */

	if (thisitem->redirect != NULL) {
		redirs = thisitem->redirect;
		while(redirs->next != NULL) {
			redirs = redirs->next;
		}
	}


	/* Get a word at a time, put it in a node, and add to the list */

	while(getword(line, buffer, LINE_BUFF_SIZE) && buffer[0] != ';') {
#ifdef DEBUG
		printf("Adding redir %s\n", buffer);
#endif
		nstr = createstr(buffer);

		if (thisitem->redirect == NULL) {
			thisitem->redirect = nstr;
		}
		else {
			redirs->next = nstr;
		}
		redirs = nstr;
	}
#ifdef DEBUG
	printf("Term redir on %s\n", buffer);
#endif
}


/* Function to set the randomness of an event. Must be a percentage integer.*/

void do_random(item *thisitem, char *line) 
{
	int nrand;
	char buffer[LINE_BUFF_SIZE] = "";


	/* Get number, and make sure it valid. 
	   Also check that the line has ended.*/

	if (!getword(line, buffer, LINE_BUFF_SIZE) || line[0] != ';')
		die_config();

	if ((sscanf(buffer, "%d", &nrand) < 1) || (nrand < 0) || (nrand > 100))
		die_config();
#ifdef DEBUG
	printf("Setting randomness to %d\n", nrand);
#endif

	thisitem->random = nrand;
}


/* Function to set logfile name.  Requires access to global variable logfile */

void set_log(item *thisitem, char *line)
{
	char buffer[LINE_BUFF_SIZE] = "";


	/* If logfile already defined then free it */

	if (thisitem->logfile != NULL) {
#ifdef DEBUG
		printf("Freeing old logfile %s\n", thisitem->logfile);
#endif
		free(thisitem->logfile);
	}


	/* Get number, and make sure it valid. 
	   Also check that the line has ended.*/

	if (!getword(line, buffer, LINE_BUFF_SIZE) || line[0] != ';')
		die_config();

	thisitem->logfile = strdup(buffer);
#ifdef DEBUG
	printf("Setting logfile to %s\n", thisitem->logfile);
#endif
}


/* Creates an event item, setting default values. */

item *createitem(void)
{
	item *nitem;

	nitem = (item *) safemalloc(sizeof(item));
	nitem->keys = NULL;
	nitem->redirect = NULL;
	nitem->logfile = NULL;
	nitem->subs = NULL;
	nitem->next = NULL;
	nitem->type = 0;
	nitem->random = DEFAULT_RANDOM;

	return nitem;
}


/* Creates a string list node, and copy in string */

str_list *createstr(char *buffer)
{
	str_list *nstr;

	nstr = (str_list *) safemalloc(sizeof(str_list));
	nstr->data = strdup(buffer);
	nstr->next = NULL;

	return nstr;
}


/* Function to read a line from the file, ending on either a ';', '{' , '}'
   or EOF.  Any spaces are compressed to single space (except at start of 
   line).  Lines beginning with '#' are ignored.  Anything inside quotes 
   is left as is. */

char *readline(FILE *stream, char *s, int size)
{
	int i = 0;
	int c;
	int quoted = FALSE;

	while (i < (size - 1)) {
		do {
			c = tolower(fgetc(stream));
			if (c == '\n') lineno++;
		} while ((c == '\n') || (quoted == FALSE && 
		         (isspace(c) && (i < 1 || isspace(s[i-1])))));

		if (c == '\"' || c == '\'') {
			if (quoted) {
				quoted = FALSE;
			}
			else {
				quoted = TRUE;
			}
		}

		if (isspace(c)) {
			s[i++] = ' ';
		}
		else if (c == '#' && quoted == FALSE) {
			flush_to_eol(stream);
			lineno++;
		}
		else if ((c == ';' || c == '{' || c == '}') && quoted == FALSE)
		{
			s[i++] = c;
			s[i] = '\0';
			return s;
		}
		else if (c == EOF) {
			s[i] = '\0';
			return s;
		}
		else {
			s[i++] = c;
		}
	}

	flush_to_eol(stream);
	lineno++;
	s[i] = '\0';

	return s;
}


/* Function to extract a word from a string.  A word terminates at a space,
   ';' or '{'.  Anything inside quotes is a single word.  The word is
   removed from the original string. */

char *getword(char *source, char *dest, int d_size)
{
	int i = 0;
	int j = 0;
	int quoted = FALSE;

	if (source[0] == '\"' || source[0] == '\'') {
		quoted = TRUE;
		i++;
	}

	while(i < (d_size - 1) && source[i] != '\0'
	      && (quoted == TRUE || !isspace(source[i])))
	{
		if (quoted==TRUE && (source[i] == '\"' || source[i] == '\'')) 
		{
			i++;
			break;
		}
		if ((source[i] == ';' || source[i] == '{') && (i > 0))
			break;

		dest[j++] = source[i++];
	}
	dest[j] = '\0';
	j = 0;
	while(isspace(source[i]))
		i++;
	while(source[i] != '\0')
		source[j++] = source[i++];
	source[j] = '\0';

	return dest;
}


/* Function to print error and exit. */

void die_config(void)
{
	fprintf(stderr,"Invalid config file at line %d.\n", lineno);
	exit(EXIT_FAILURE);
}
