/* gtkfind.c */
/* gtk-based file finding and munging command */
/*
  gtkfind - a graphical "find" program
  Copyright (C) 1998  Matthew Grossman <mattg@oz.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/





#include"gtkfind.h"


/*********************/
/* globals           */

static char *version = "gtkfind v0.9";

static int run_only_once = 0; /* do we run only once? */

static int uid = -1;
static int gid = -1;

static char *window_output_text = NULL;


/*********************/
/* functions         */

int
main(int argc, char *argv[])
{
  
  gtk_init(&argc, &argv);

  if(get_bool_option(argc, argv, "-vanish"))
    run_only_once = 1;
  if(get_bool_option(argc, argv, "-version")) {
    printf("%s\n", version);
    gtk_exit(0);
  }
  if(get_bool_option(argc, argv, "-help")) {
    printf("%s - a graphical file finding program using the GTK+ toolkit\n",
	   version);
    printf("Type \"man gtkfind\" to view the manual page\n");
    printf("options:\n");
    printf("\t-version\tPrint the version and exit\n");
    printf("\t-help\t\tPrint this message and exit\n");
    printf("\t-vanish\t\tExit after the first search has completed\n");
    gtk_exit(0);
  }

  set_globals();
  
  make_widgets();

  g_set_message_handler(gtk_print1);
  
  gtk_main();
  
  return(0); /* to keep the compiler happy */
}




void
quit(gpointer *quit_button)
{
  gtk_exit(0);

  return;
}

void
find(gpointer *find_button, gpointer *pattern)
     /* traverse subdirectories, finding files whose names match pattern
	and print them to stdout */
{
  gchar *text = NULL;
  gchar *content_string = NULL;
  gchar *search_directory = NULL;
  GdkCursor *watch_cursor = NULL;
  GdkCursor *normal_cursor = NULL;
  char cwd[MAXPATHLEN];



  text = gtk_entry_get_text(GTK_ENTRY(pattern));
  if(strlen(text) == 0 && get_flag(WILDCARD_FILENAME_MATCH_P))
    text = "*";

  content_string = gtk_entry_get_text(GTK_ENTRY(content_pattern));

  search_directory = get_search_directory(directory);

  if(set_uid_and_gid() == 0)
    goto DONE;
  
  /* close already existing output window */
  if(output_toplevel) {
    gtk_widget_destroy(output_text);
    gtk_widget_destroy(output_toplevel);
    output_toplevel = NULL;
    output_text = NULL;
    reset_window_output_text();
  }

  if(set_global_dates() == 0)
    goto DONE;
  
  watch_cursor = gdk_cursor_new(GDK_WATCH);
  gdk_window_set_cursor(toplevel->window, watch_cursor);
  gdk_cursor_destroy(watch_cursor);

  normal_cursor = gdk_cursor_new(GDK_TOP_LEFT_ARROW);

  gtk_widget_hide(GTK_WIDGET(find_button));
  gdk_window_set_cursor(stop_button->window, normal_cursor);
  gtk_widget_show(stop_button);

  if(getcwd(cwd, MAXPATHLEN) == NULL) {
    print_error("getcwd: %s", strerror(errno));
    goto DONE;
  }
  
  if(get_flag(SEARCH_SUBDIRS_P))
    find_and_print(search_directory, text, content_string, 1);
  else
    find_and_print(search_directory, text, content_string, 0);

  chdir(cwd); /* we just have to end in the same directory we started in */

  gdk_window_set_cursor(toplevel->window, normal_cursor);
  gdk_cursor_destroy(normal_cursor);

  gtk_widget_hide(stop_button);
  gtk_widget_show(GTK_WIDGET(find_button));
  if(stop_flag == 1)
    stop_flag = 0;
  
  if(get_flag(PRINT_TO_WINDOW_P))
    make_output_window();
  
  if(run_only_once)
    gtk_exit(0);

 DONE:
  if(search_directory)
    free(search_directory);
  return;
}

void
find_and_print(char *dir_name, char *text, char *content_string, int recursep)
     /* search in dir_name and its subdirectories, if there is a shell command
	run it, default action is to print the file name */
{
  DIR *dp = NULL;
  struct dirent *dirent = NULL;
  struct stat sbuf;
  char full_file_name[MAXPATHLEN];
  char **registers = NULL;

#ifdef DEBUG
  fprintf(stderr, "find_and_print called with dir_name == %s\n", dir_name);
#endif

  dp = opendir(dir_name);
  if(!dp) {
    print_error("opendir: %s: Failed to open %s.\n", strerror(errno),
		dir_name);
    goto DONE;
  }

  while((dirent = readdir(dp)) != NULL) {
    if(registers) {
      free_glob_registers(registers);
      registers = NULL;
    }
    /* handle any pending events */
    while(gtk_events_pending())
      gtk_main_iteration();
    if(stop_flag) /* someone's clicked stop */
      goto DONE;
      
    if(strcmp(dirent->d_name, "..") == 0 ||
       strcmp(dirent->d_name, ".") == 0)
      continue;
    if(dirent->d_ino != 0) { /* file has not been deleted */

      /* avoid having those "//" in file names */
      if(dir_name[strlen(dir_name) - 1] == '/')
	sprintf(full_file_name, "%s%s", dir_name, dirent->d_name);
      else
	sprintf(full_file_name, "%s/%s", dir_name, dirent->d_name);

      if(lstat(full_file_name, &sbuf) < 0) {
	print_error("lstat: %s: Failed to read %s.\n", strerror(errno),
		  full_file_name);
	continue; /* we read the rest of the files in this directory */
      }

      if(match_mode(sbuf.st_mode) &&
	 match_type(sbuf.st_mode) &&
	 match_owner(sbuf) &&
	 match_atime(sbuf.st_atime) &&
	 match_ctime(sbuf.st_ctime) &&
	 match_mtime(sbuf.st_mtime) &&
	 (registers = match_filename(text, dirent->d_name)) != NULL &&
	 (strlen(content_string) == 0 ||
	  match_contents(full_file_name, sbuf, content_string))) {
	if(get_flag(SHELL_COMMAND_P)) {
	  if(chdir(dir_name) < 0) {
	    print_error("chdir: %s: Cannot chdir to %s.\n", strerror(errno),
		      dir_name);
	    goto DONE;
	  }
	  execute_shell_command(registers, full_file_name, sbuf);
	}
	else {
	  output_file_data(full_file_name, dirent->d_name, sbuf);
	}
      }
      if(recursep && (sbuf.st_mode & S_IFMT) == S_IFDIR) {
	find_and_print(full_file_name, text, content_string, 1);
      }
    }
  }

  
 DONE:
  if(dp) {
    closedir(dp);
  }
  if(registers)
    free_glob_registers(registers);
  return;
}


void
set_globals()
     /* set global variables at the start of the program */
{
  output_toplevel = NULL;
  output_text = NULL;

  return;
}


int
execute_shell_command(char **registers, char *full_file_name,
		      struct stat sbuf)
     /* execute a shell command on filename, return the exit status of the
	command, and print the output */
{
  char *command = NULL;
  int rv = -1;
  FILE *pipe = NULL;
  char *s = NULL;
  char buf[256];

  command = substitute_filename(gtk_entry_get_text(GTK_ENTRY(shell_command)),
				 registers);

  if(strlen(command) > 0) {
    pipe = popen(command, "r");

    while(fgets(buf, 256, pipe)) {
      if(!s)
	s = (char *)g_malloc0(256 * sizeof(char));
      else
	s = (char *)g_realloc(s, strlen(s) + 256 * sizeof(char));
      strcat(s, buf);
    }

    rv = pclose(pipe);

    if(get_flag(PRINT_FILENAME_ANYWAY_P) || rv == 0)
      output_file_data(full_file_name, registers[0], sbuf);

    if(s) {
      if(get_flag(PRINT_TO_WINDOW_P))
	print_to_window(s);
      if(get_flag(PRINT_TO_STDOUT_P))
	printf("%s", s);
    }

    g_free(s);
    g_free(command);
  }

  return(rv);
}

char *
substitute_filename(char *target, char **registers)
     /* substitute "\[0-9]" in target with filename */
{
  char *rv = NULL;
  char *copy = NULL;
  int i = 0;
  int reg = 0;
  int increment = 128;
  int size = 128;

  copy = (char *)g_malloc0(size * sizeof(char));

  while(target[i]) {
    if(target[i] == '\\' && isdigit(target[i + 1])) {
      reg = (int)target[i + 1] - 48; /* quick way to get a number from
					an ASCII digit */
      while(strlen(copy) + strlen(registers[reg]) + 1 >= size) {
	size += increment;
	copy = g_realloc(copy, size * sizeof(char));
      }
      strcat(copy, registers[reg]);
      i += 2;
    }
    else {
      while(strlen(copy) + 2 >= size) {
	size += increment;
	copy = (char *)g_realloc(copy, size * sizeof(char));
      }
      copy[strlen(copy)] = target[i++];
      copy[strlen(copy)] = '\0';
    }
  }

  rv = copy;

  return(rv);
}
      
char *get_search_directory(GtkWidget *directory)
     /* return the search directory, if it is null return "/", if it
	has a ~  or a . expand it */
{
  char *rv = NULL;
  char tmp[MAXPATHLEN + 1];

  strncpy(tmp, gtk_entry_get_text(GTK_ENTRY(directory)), MAXPATHLEN);
  tmp[MAXPATHLEN] = '\0';

  if(strlen(tmp) == 0) {
    rv = (char *)g_malloc(2);
    strcpy(rv, "/");
  }
  else
    rv = expand_path(tmp);

    return(rv);
}
      

int
match_type(unsigned short mode)
     /* type and mode are different things to people, even if to
	the system they are the same */
{
  int rv = 0;

  /* this is soooo stupid, I'm ashamed to write this */

  if((!get_flag(SETUID_P)) && 
     (!get_flag(SETGID_P)) && 
     (!get_flag(STICKY_P)) && 
     (!get_flag(DIRECTORY_P)) && 
     (!get_flag(REGULAR_P)) && 
     (!get_flag(RAW_DEVICE_P)) && 
     (!get_flag(BLOCK_DEVICE_P)) && 
     (!get_flag(SYMLINK_P)) && 
     (!get_flag(SOCKET_P)) && 
     (!get_flag(FIFO_P)))
    rv = 1;
  else if((get_flag(SETUID_P) && ((S_ISUID & mode))) ||
	   (get_flag(SETGID_P) && ((S_ISGID & mode))) ||
	   (get_flag(STICKY_P) && ((S_ISVTX & mode))) ||
	   (get_flag(DIRECTORY_P) && ((S_IFDIR == (mode & S_IFMT)))) ||
	   (get_flag(REGULAR_P) && ((S_IFREG == (mode & S_IFMT)))) ||
	   (get_flag(RAW_DEVICE_P) && ((S_IFCHR == (mode & S_IFMT)))) ||
	   (get_flag(BLOCK_DEVICE_P) && ((S_IFBLK == (mode & S_IFMT)))) ||
	   (get_flag(SYMLINK_P) && ((S_IFLNK == (mode & S_IFMT)))) ||
	   (get_flag(SOCKET_P) && ((S_IFSOCK == (mode & S_IFMT)))) ||
	   (get_flag(FIFO_P) && ((S_IFIFO == (mode & S_IFMT)))))
    rv = 1;
  
  return(rv);
}
     

int
match_mode(unsigned short mode)
     /* see if mode matches the selected modes */
{
  int rv = 0;

  /* ok, if we're not matching on any of the modes, we want to return 1 */

  if((!get_flag(OWNER_READ_P)) && (!get_flag(GROUP_READ_P)) &&
     (!get_flag(WORLD_READ_P)) && (!get_flag(OWNER_WRITE_P)) &&
     (!get_flag(GROUP_WRITE_P)) && (!get_flag(WORLD_WRITE_P)) &&
     (!get_flag(OWNER_EXEC_P)) && (!get_flag(GROUP_EXEC_P)) &&
     (!get_flag(WORLD_EXEC_P)))
    rv = 1;
  else { /* we are trying to match a mode, so check and return 1 if we
	    succeed */
    if((get_flag(OWNER_READ_P) && ((S_IREAD & mode))) ||
       (get_flag(GROUP_READ_P) && (((S_IREAD >> 3) & mode))) ||
       (get_flag(WORLD_READ_P) && (((S_IREAD >> 6) & mode))) ||
       (get_flag(OWNER_WRITE_P) && ((S_IWRITE & mode))) ||
       (get_flag(GROUP_WRITE_P) && (((S_IWRITE >> 3) & mode))) ||
       (get_flag(WORLD_WRITE_P) && (((S_IWRITE >> 6) & mode))) ||
       (get_flag(OWNER_EXEC_P) && ((S_IEXEC & mode))) ||
       (get_flag(GROUP_EXEC_P) && (((S_IEXEC >> 3) & mode))) ||
       (get_flag(WORLD_EXEC_P) && (((S_IEXEC >> 6) & mode))))
      rv = 1;
  }
  
  return(rv);
}


int
set_uid_and_gid()
     /* set the uid and gid variables */
{
  char *login = NULL;
  char *group = NULL;
  struct passwd *passwd = NULL;
  struct group *grp = NULL;
  int rv = 1;

  uid = -1;
  gid = -1;
  
  login = gtk_entry_get_text(GTK_ENTRY(login_entry));
  if(strlen(login) > 0) {
    if(get_flag(UID_NOT_LOGIN_P)) {
      uid = atoi(login);
    }
    else {
      if((passwd = getpwnam(login)) == NULL) {
	rv = 0;
	goto DONE;
      }
      else {
	uid = passwd->pw_uid;
      }
    }
  }

  group = gtk_entry_get_text(GTK_ENTRY(group_entry));
  if(strlen(group) > 0) {
    if(get_flag(GID_NOT_GROUP_P)) {
      gid = atoi(group);
    }
    else {
      if((grp = getgrnam(group)) == NULL) {
	rv = 0;
	goto DONE;
      }
      else
	gid = grp->gr_gid;
    }
  }
  
 DONE:
  return(rv);
}


int
match_owner(struct stat sbuf)
     /* if file matches uid and gid (or uid or gid is -1) return 1 */
{
  int rv = 1;
  
  if(uid >= 0 && sbuf.st_uid != uid)
    rv = 0;
  if(gid >= 0 && sbuf.st_gid != gid)
    rv = 0;

  return(rv);
}



void
reset_window_output_text()
     /* reset output_text */
{
  if(window_output_text) {
    free(window_output_text);
    window_output_text = NULL;
  }

  return;
}

void
output_toplevel_delete()
     /* we receive a delete event in the output window */
{
  gtk_widget_destroy(GTK_WIDGET(output_toplevel));
  output_toplevel = NULL;
  reset_window_output_text();
  
  return;
}

void
make_output_window()
     /* (re)make the output window.  See gtktest.c */
{
  GtkWidget *vbox = NULL;
  GtkWidget *table = NULL;
  GtkWidget *vscrollbar = NULL;
  GtkWidget *button = NULL;
  GtkWidget *hbox = NULL;
  GtkWidget *label = NULL;
  static GdkFont *output_font = NULL;
  static int c_width = 0;

  if(!output_font) {
    output_font = gdk_font_load("fixed");
    if(!output_font) {
      print_error("Unable to load font fixed\n");
      goto DONE;
    }
    c_width = gdk_char_width(output_font, ' '); /* fixed width font */
  }
  
  if(output_toplevel == NULL && window_output_text) {
    output_toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(output_toplevel), "gtkfind output");
    gtk_widget_set_usize(output_toplevel,
			 get_max_line_width(window_output_text, c_width) + 60,
			 200);
    gtk_window_set_policy(GTK_WINDOW(output_toplevel), TRUE, TRUE, FALSE);
    gtk_signal_connect(GTK_OBJECT(output_toplevel), "delete_event",
		       GTK_SIGNAL_FUNC(output_toplevel_delete), NULL);
    gtk_widget_show(output_toplevel);

    vbox = gtk_vbox_new(FALSE, 10);
    gtk_container_border_width(GTK_CONTAINER(vbox), 10);
    gtk_container_add(GTK_CONTAINER(output_toplevel), vbox);
    gtk_widget_show(vbox);

    table = gtk_table_new(1, 2, FALSE);
    gtk_table_set_col_spacing(GTK_TABLE(table), 0, 2);
    gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, TRUE, 0);
    gtk_widget_show(table);

    output_text = gtk_text_new (NULL, NULL);
    gtk_text_set_editable (GTK_TEXT (output_text), TRUE);
    gtk_table_attach (GTK_TABLE (table), output_text, 0, 1, 0, 1,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
    gtk_widget_show (output_text);
    
    vscrollbar = gtk_vscrollbar_new (GTK_TEXT (output_text)->vadj);
    gtk_table_attach (GTK_TABLE (table), vscrollbar, 1, 2, 0, 1,
		      GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
    gtk_widget_show (vscrollbar);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
    gtk_widget_show(hbox);

    button = gtk_button_new();
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(save_output_callback), NULL);
    gtk_container_add(GTK_CONTAINER(hbox), button);
    gtk_widget_show(button);

    label = gtk_label_new("Save");
    gtk_container_add(GTK_CONTAINER(button), label);
    gtk_widget_show(label);

    button = gtk_button_new();
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(output_toplevel_delete), NULL);
    gtk_container_add(GTK_CONTAINER(hbox), button);
    gtk_widget_show(button);

    label = gtk_label_new("Close");
    gtk_container_add(GTK_CONTAINER(button), label);
    gtk_widget_show(label);
    
    gtk_text_freeze(GTK_TEXT(output_text));
    gtk_text_insert(GTK_TEXT(output_text), output_font,
		    &(output_text)->style->black, NULL,
		    window_output_text, -1);
    gtk_text_thaw(GTK_TEXT(output_text));
  }

 DONE:
  
  return;
}

void
print_to_window(char *text)
     /* print text to window_output_text, appending a newline */
{
  if(!window_output_text)
    window_output_text = (char *)g_malloc0(strlen(text) + 2);
  else
    window_output_text =
      (char *)g_realloc(window_output_text,
			strlen(window_output_text) + strlen(text) + 2);

  strcat(window_output_text, text);
  strcat(window_output_text, "\n");

  return;
}

void
save_output_callback()
     /* save the output text to a file */
{
  GtkWidget *filesel = NULL;

  filesel = create_filesel("Save Output", save_output_ok, save_output_cancel);

  return;
}

void
save_output_cancel(GtkWidget *w, GtkFileSelection *filesel)
     /* close the filesel widget */
{
  gtk_widget_destroy(GTK_WIDGET(filesel));

  return;
}

void
save_output_ok(GtkWidget *w, GtkFileSelection *filesel)
     /* we selected ok in the Save Output widget */
{
  char *filename = NULL;
  FILE *fp = NULL;

  filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));

  if(strlen(filename) == 0)
    goto DONE;

  fp = fopen(filename, "w");
  if(!fp) {
    print_error("fopen: %s: Cannot open %s.\n", strerror(errno), filename);
    goto ERROR;
  }

  if(fputs(window_output_text, fp) == EOF) {
    print_error("Cannot write to file %s.\n", filename);
    goto ERROR;
  }

  gtk_widget_destroy(GTK_WIDGET(filesel));
  
 DONE:
  if(fp)
    fclose(fp);
  return;

 ERROR:
  goto DONE;
}


int
match_atime(time_t t)
{
  /*  time_t user_atime = 0; */
  int rv = 0;

  if(!get_flag(ATIME_ET_P) && !get_flag(ATIME_EQ_P) && !get_flag(ATIME_LT_P)) {
    rv = 1;
    goto DONE;
  }

  if(get_flag(ATIME_ET_P) && t < global_atime) {
    rv = 1;
    goto DONE;
  }
  else if(get_flag(ATIME_EQ_P) && t == global_atime) {
    rv = 1;
    goto DONE;
  }
  else if(get_flag(ATIME_LT_P) && t > global_atime) {
    rv = 1;
    goto DONE;
  }

 DONE:
  return(rv);
}

int
match_ctime(time_t t)
{
  int rv = 0;

  if(!get_flag(CTIME_ET_P) && !get_flag(CTIME_EQ_P) && !get_flag(CTIME_LT_P)) {
    rv = 1;
    goto DONE;
  }


  if(get_flag(CTIME_ET_P) && t < global_ctime) {
    rv = 1;
    goto DONE;
  }
  else if(get_flag(CTIME_EQ_P) && t == global_ctime) {
    rv = 1;
    goto DONE;
  }
  else if(get_flag(CTIME_LT_P) && t > global_ctime) {
    rv = 1;
    goto DONE;
  }

 DONE:
  return(rv);
}

int
match_mtime(time_t t)
{
  int rv = 0;

  if(!get_flag(MTIME_ET_P) && !get_flag(MTIME_EQ_P) && !get_flag(MTIME_LT_P)) {
    rv = 1;
    goto DONE;
  }


  if(get_flag(MTIME_ET_P) && t < global_mtime) {
    rv = 1;
    goto DONE;
  }
  else if(get_flag(MTIME_EQ_P) && t == global_mtime) {
    rv = 1;
    goto DONE;
  }
  else if(get_flag(MTIME_LT_P) && t > global_mtime) {
    rv = 1;
    goto DONE;
  }

 DONE:
  return(rv);
}

time_t
get_user_time(GtkWidget *month, GtkWidget *day, GtkWidget *year,
	      GtkWidget *hour, GtkWidget *minute, GtkWidget *second)
{
  time_t rv = 0;
  struct tm tm = { 0 };

  tm.tm_mon = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(month)) - 1;
  tm.tm_mday = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(day));
  tm.tm_year = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(year)) - 1900;
  tm.tm_hour = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(hour));
  tm.tm_min = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(minute));
  tm.tm_sec = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(second));

  rv = mktime(&tm);

  return(rv);
}
  
int
set_global_dates()
     /* set the global dates variable, return 0 if we have an insane time */
{
  int rv = 1;
  
  if(sane_date_p(atime_month_spin, atime_day_spin, atime_year_spin)) {
    global_atime = get_user_time(atime_month_spin, atime_day_spin,
				 atime_year_spin, atime_hour_spin,
				 atime_minute_spin, atime_second_spin);
  }
  else {
    print_error("Impossible atime, aborting find.\n");
    rv = 0;
  }
  
  if(sane_date_p(mtime_month_spin, mtime_day_spin, mtime_year_spin)) {
    global_mtime = get_user_time(mtime_month_spin, mtime_day_spin,
				 mtime_year_spin, mtime_hour_spin,
				 mtime_minute_spin, mtime_second_spin);
  }
  else {
    print_error("Impossible mtime, aborting find.\n");
    rv = 0;
  }
  
  if(sane_date_p(ctime_month_spin, ctime_day_spin, ctime_year_spin)) {
    global_ctime = get_user_time(ctime_month_spin, ctime_day_spin,
				 ctime_year_spin, ctime_hour_spin,
				 ctime_minute_spin, ctime_second_spin);
  }
  else {
    print_error("Impossible ctime, aborting find.\n");
    rv = 0;
  }

  return(rv);
}

int
sane_date_p(GtkWidget *month, GtkWidget *day, GtkWidget *year)
     /* if the date of the spin widgets is sane, return 1 */
{
  int rv = 1;

  int m = 0, d = 0, y = 0;

  m = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(month));
  d = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(day));
  y = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(year));

  if((m == 9 || m == 4 || m == 6 || m == 11) &&
     d == 31) {
    rv = 0;
  }
  else if(m == 2 && (d == 30 || d == 31)) {
    rv = 0;
  }
  else if(m == 2 && !leap_year_p(y) && d == 29)
    rv = 0;

  return(rv);
}

char **
match_filename(char *text, char *filename)
     /* if we match filename, return either a proper set of registers or
	just registers[0] */
{
  char **rv = NULL;

  if(get_flag(WILDCARD_FILENAME_MATCH_P))
    rv = glob_string(text, filename, 1);
  else {
    if(strlen(text) == 0 || strstr(filename, text)) {
      rv = allocate_glob_registers();
      rv[0] = (char *)g_malloc(sizeof(char) * (strlen(filename) + 1));
      if(!rv[0]) {
	print_error("match_filename: Can't malloc rv[0], failing.");
	goto ERROR;
      }
      strcpy(rv[0], filename);
    }
  }

 DONE:
  return(rv);
 ERROR:
  rv = NULL;
  goto DONE;
}

int
match_contents(char *full_file_name, struct stat sbuf, char *content_string)
     /* glob content_string into the contents of full_file_name */
{
  FILE *fp = NULL;
  char *s = NULL;
  int rv = 0;
  int i = 0;

  /* we will only match text in a regular file (or from a symbolic link) */
  if((sbuf.st_mode & S_IFMT) != S_IFREG && (sbuf.st_mode & S_IFMT) != S_IFLNK)
    goto DONE;

  fp = fopen(full_file_name, "r");
  if(!fp) {
    print_error("fopen: %s: Cannot open %s.\n", strerror(errno),
	      full_file_name);
    goto ERROR;
  }

  s = (char *)g_malloc0(sbuf.st_size + 1);
  if(!s) {
    print_error("g_malloc0: %s: Cannot malloc %ld bytes.\n", strerror(errno),
	      (long int)sbuf.st_size);
    /* assuming of course that we have not crashed */
    goto ERROR;
  }

  i = fread(s, 1, sbuf.st_size, fp);
  if(i != sbuf.st_size)
    print_error("Only read %ld out of %ld bytes from file %s.\n", (long int)i,
	      (long int)sbuf.st_size, full_file_name);
  s[sbuf.st_size] = '\0';

  if(get_flag(WILDCARD_CONTENTS_SEARCH_P)) {
    if(glob_string(content_string, s, 0))
      rv = 1;
  }
  else {
    if(strstr(s, content_string))
      rv = 1;
  }

 DONE:
  if(fp)
    fclose(fp);
  if(s)
    g_free(s);
  return(rv);

 ERROR:
  rv = 0;
  goto DONE;
}
	   
    
void
output_file_data(char *full_file_name, char *short_file_name,
		 struct stat sbuf)
     /* take care of all the details of outputting file data */
{
  char *s = NULL;

  if(get_flag(LONG_OUTPUT_P)) {
    s = make_long_format(full_file_name, short_file_name, sbuf);
    if(!s)
      goto DONE;
  }
  else
    s = full_file_name;

  if(get_flag(PRINT_TO_STDOUT_P))
    printf("%s\n", s);
  if(get_flag(PRINT_TO_WINDOW_P))
    print_to_window(s);

 DONE:
  
  if(s && get_flag(LONG_OUTPUT_P))
    g_free(s);
  return;
}
  
char *
make_long_format(char *full_file_name, char *short_file_name, 
		 struct stat sbuf)
     /* malloc and return a string containing the long format data for
	full_file_name */
{
  /* "+" are spaces */
  /* -rw-rw-r--+++1+mattg++++mattg+++++7985722+Aug+31+21:46+kernel-source-2.0.35-1.alpha.rpm */

  time_t t = 0;
  char *rv = NULL;
  char *s = NULL;
  int base_length = 55;
  int offset = 0;
  struct passwd *pwd;
  struct group *grp;
  struct tm *tm = NULL, *now = NULL;
  time_t diff = 0;
  unsigned short mode = 0;
  char c = 0;
  char tmp[MAXPATHLEN];
  int i = 0;

  s = (char *)g_malloc0(base_length * sizeof(char) +
			strlen(full_file_name) + 1);
  if(!s) {
    print_error("make_long_format: cannot malloc string.\n");
    goto ERROR;
  }

  /* file type */
  
  mode = sbuf.st_mode & S_IFMT;

  switch(mode) {
  case S_IFREG:
    c = '-';
    break;
  case S_IFDIR:
    c = 'd';
    break;
  case S_IFLNK:
    c = 'l';
    break;
  case S_IFSOCK:
    c = 's';
    break;
  case S_IFIFO:
    c = 'p';
    break;
  case S_IFCHR:
    c = 'c';
    break;
  case S_IFBLK:
    c = 'b';
    break;
  default:
    c = '?';
  }
  
  s[offset++] = c;

  /* owner permissions */
  
  sbuf.st_mode & S_IREAD ? (s[offset++] = 'r') : (s[offset++] = '-');
  sbuf.st_mode & S_IWRITE ? (s[offset++] = 'w') : (s[offset++] = '-');

  if(sbuf.st_mode & S_IEXEC) {
    if(sbuf.st_mode & S_ISUID)
      s[offset++] = 's';
    else
      s[offset++] = 'x';
  }
  else if(sbuf.st_mode & S_ISUID)
    s[offset++] = 'S';
  else
    s[offset++] = '-';

  /* group permissions */
  
  sbuf.st_mode & (S_IREAD >> 3) ? (s[offset++] = 'r') : (s[offset++] = '-');
  sbuf.st_mode & (S_IWRITE >> 3) ? (s[offset++] = 'w') : (s[offset++] = '-');

  if(sbuf.st_mode & (S_IEXEC >> 3)) {
    if(sbuf.st_mode & S_ISGID)
      s[offset++] = 's';
    else
      s[offset++] = 'x';
  }
  else if(sbuf.st_mode & S_ISGID)
    s[offset++] = 'S';
  else
    s[offset++] = '-';

  /* world permissions */
  
  sbuf.st_mode & (S_IREAD >> 6) ? (s[offset++] = 'r') : (s[offset++] = '-');
  sbuf.st_mode & (S_IWRITE >> 6) ? (s[offset++] = 'w') : (s[offset++] = '-');

  if(sbuf.st_mode & (S_IEXEC >> 6)) {
    if(sbuf.st_mode & S_ISVTX)
      s[offset++] = 't';
    else
      s[offset++] = 'x';
  }
  else if(sbuf.st_mode & S_ISVTX)
    s[offset++] = 'T';
  else
    s[offset++] = '-';

  /* number of hard links */
  
  sprintf(s + offset, "%4d", sbuf.st_nlink);

  offset += 4;

  s[offset++] = ' ';

  /* owner name */
  
  pwd = getpwuid(sbuf.st_uid);
  if(pwd)
    sprintf(s + offset, "%-8s", pwd->pw_name);
  else
    sprintf(s + offset, "%-8d", (int)sbuf.st_uid);

  offset += 8;
  s[offset++] = ' ';

  /* group name */
  
  grp = getgrgid(sbuf.st_gid);
  if(grp)
    sprintf(s + offset, "%-8s", grp->gr_name);
  else
    sprintf(s + offset, "%-8d", (int)sbuf.st_gid);

  offset += 8;
  s[offset++] = ' ';

  /* size || device numbers */

  if(mode == S_IFCHR || mode == S_IFBLK)
    sprintf(s + offset, "%3d,%4d", major(sbuf.st_rdev), minor(sbuf.st_rdev));
  else 
    sprintf(s + offset, "%8ld", (long)sbuf.st_size);
  offset += 8;

  s[offset++] = ' ';

  /* file modification time (mtime) */
  
  tm = (struct tm *)g_malloc0(sizeof(struct tm));
  memcpy(tm, localtime(&(sbuf.st_mtime)), sizeof(struct tm));

  now = (struct tm *)g_malloc0(sizeof(struct tm));
  t = time(NULL);
  memcpy(now, localtime(&t), sizeof(struct tm));

  diff = sbuf.st_mtime - t;
  
  /* if the file is more than 6 months old or more than an hour into the
     future, print the year, otherwise print the time of day */
  if((tm->tm_year == now->tm_year && now->tm_mon >= tm->tm_mon &&
      now->tm_mon - tm->tm_mon <= 6) ||
     (tm->tm_year == now->tm_year - 1 && now->tm_mon <= tm->tm_mon &&
      tm->tm_mon - now->tm_mon <= 6) ||
     (diff > 0 && diff < 3600)) {
    strftime(s + offset, 13, "%b %d %H:%M", tm);
    offset += 12;
    s[offset++] = ' ';
  }
  else {
    strftime(s + offset, 13, "%b %d  %Y", tm);
    offset += 12;
    s[offset++] = ' ';
  }

  /* filename */
  
  sprintf(s + offset, "%s", full_file_name);
  offset += strlen(full_file_name);

  /* if it's a symlink, add the -> and the name of the file it points too */
  if(mode == S_IFLNK) {
    if((i = readlink(full_file_name, tmp, MAXPATHLEN)) < 0) {
      print_error("readlink: %s", strerror(errno));
      print_error("readlink failed on %s", full_file_name);
    }
    tmp[i] = '\0';
    s = (char *)g_realloc(s, strlen(s) + strlen(" -> ") + i + 1);
    sprintf(s + offset, " -> %s", tmp);
  }
  
  rv = s;

 DONE:
  g_free(tm);
  g_free(now);
  return(rv);

 ERROR:
  rv = NULL;
  goto DONE;
}

void
print_error(const char *format, ...)
     /* use g_message to print the error, for util.h */
{
  va_list argp;
  char buffer[1024];
    /* this is arbitrary, unfortunately there is no
       portable way to ensure that our buffer is large
       enough... */

  va_start(argp, format);
  vsprintf(buffer, format, argp);
  g_message("%s", buffer);
  va_end(argp);
}
