/* save.c */
/* code for saving our search as a shell script */
/* Copyright (C) 1998 Matthew Grossman <mattg@oz.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include"save.h"

/* externs */

extern void print_error(char *format, ...);
extern char *get_search_directory(GtkWidget *directory);
extern GtkWidget *directory;
extern GtkWidget *pattern;
extern GtkWidget *login_entry;
extern GtkWidget *group_entry;
extern GtkWidget *atime_month_spin, *atime_day_spin, *atime_hour_spin,
  *atime_minute_spin, *atime_year_spin, *atime_second_spin;
extern GtkWidget *ctime_month_spin, *ctime_day_spin, *ctime_hour_spin,
  *ctime_minute_spin, *ctime_year_spin, *ctime_second_spin;
extern GtkWidget *mtime_month_spin, *mtime_day_spin, *mtime_hour_spin,
  *mtime_minute_spin, *mtime_year_spin, *mtime_second_spin;
extern time_t get_user_time(GtkWidget *month, GtkWidget *day, GtkWidget *year,
			    GtkWidget *hour, GtkWidget *minute,
			    GtkWidget *second);
extern GtkWidget *content_pattern;


/* ok:
   if we're matching times, we first need to create a file with touch that
   has the time we want. Then we can use -cnewer etc. and ! to match files
   earlier or later than our timing file.  We cannot seem to match files
   equal to our timing file without jumping through hoops (ie using 2 files)

   Then, we must delete any timing files we have created.
*/

static char *
dec_sec(time_t t)
     /* decrement time by 1 second */
{
  char *rv = NULL;
  static char s[32];
  struct tm tm;
  time_t dec = 0;

  memset(s, 0, 32);
  dec = t - 1;
  memcpy(&tm, localtime(&dec), sizeof(struct tm));
  strftime(s, 32, "%m%d%H%M%Y.%S", &tm);

  rv = s;
  return(rv);
}

static char *
inc_sec(time_t t)
     /* increment time by 1 second */
{
  static char s[32];
  struct tm tm;
  time_t inc = 0;
  char *rv = NULL;

  memset(s, 0, 32);
  inc = t + 1;
  memcpy(&tm, localtime(&inc), sizeof(struct tm));
  strftime(s, 32, "%m%d%H%M%Y.%S", &tm);

  rv = s;
  return(rv);
}

static char *
make_tmpfile()
     /* malloc and return a temporary filename */
{
  char *rv = NULL;

  rv = (char *)malloc(sizeof(char) * L_tmpnam + 1);
  if(!rv) {
    print_error("make_tmpfile: Can't malloc file");
    goto ERROR;
  }
  if(!tmpnam(rv)) {
    print_error("make_tmpfile: tmpnam: %s", strerror(errno));
    goto ERROR;
  }

 DONE:
  return(rv);
 ERROR:
  if(rv)
    free(rv);
  rv = NULL;
  goto DONE;
}
    

static char *
make_timing_string(char *afile, char *cfile, char *mfile, char *aeqfile1,
		   char *aeqfile2, char *ceqfile1, char *ceqfile2,
		   char *meqfile1, char *meqfile2)
{
  char *rv = NULL;
  static char s[2048]; /* big enough */
  char tmp[128];

  memset(s, 0, 2048);

  strcat(s, "\\( ");
  
  if(get_flag(ATIME_ET_P)) {
    sprintf(tmp, "\\( ! -anewer %s ! -path %s \\) -o ", afile, afile);
    strcat(s, tmp);
  }
  if(get_flag(ATIME_LT_P)) {
    sprintf(tmp, "\\( -anewer %s ! -path %s \\) -o ", afile, afile);
    strcat(s, tmp);
  }
  if(get_flag(ATIME_EQ_P)) {
    sprintf(tmp, "\\( ! -anewer %s -anewer %s ! \\( -path %s -o -path %s \\) \\) -o ",
	    aeqfile2, aeqfile1, aeqfile1, aeqfile2);
    strcat(s, tmp);
  }
  
  if(get_flag(CTIME_ET_P)) {
    sprintf(tmp, "\\( ! -cnewer %s ! -path %s \\) -o ", cfile, cfile);
    strcat(s, tmp);
  }
  if(get_flag(CTIME_LT_P)) {
    sprintf(tmp, "\\( -cnewer %s ! -path %s \\) -o ", cfile, cfile);
    strcat(s, tmp);
  }
  if(get_flag(CTIME_EQ_P)) {
    sprintf(tmp, "\\( ! -cnewer %s -cnewer %s ! \\( -path %s -o -path %s \\) \\) -o ",
	    ceqfile2, ceqfile1, ceqfile1, ceqfile2);
    strcat(s, tmp);
  }

  if(get_flag(MTIME_ET_P)) {
    sprintf(tmp, "\\( ! -newer %s ! -path %s \\) -o ", mfile, mfile);
    strcat(s, tmp);
  }
  if(get_flag(MTIME_LT_P)) {
    sprintf(tmp, "\\( -newer %s ~ -path %s \\) -o ", mfile, mfile);
    strcat(s, tmp);
  }
  if(get_flag(MTIME_EQ_P)) {
    sprintf(tmp, "\\( ! -newer %s -newer %s ! \\( -path %s -o -path %s \\) \\) -o ",
	    meqfile2, meqfile1, meqfile1, meqfile2);
    strcat(s, tmp);
  }
  strcat(s, "-false \\) ");

  rv = s;

  return(rv);
}

static int
make_filenames(char **afile, char **cfile, char **mfile, char **aeqfile1,
	       char **aeqfile2, char **ceqfile1, char **ceqfile2,
	       char **meqfile1, char **meqfile2)
     /* allocate and return the name of the files used for timing */
{
  int rv = 1;

  if(get_flag(ATIME_LT_P) || get_flag(ATIME_ET_P)) {
    if(!(*afile = make_tmpfile()))
      goto ERROR;
  }
  if(get_flag(ATIME_EQ_P)) {
    if(!(*aeqfile1 = make_tmpfile()))
      goto ERROR;
    if(!(*aeqfile2 = make_tmpfile()))
      goto ERROR;
  }

  if(get_flag(CTIME_LT_P) || get_flag(CTIME_ET_P)) {
    if(!(*cfile = make_tmpfile()))
      goto ERROR;
  }
  if(get_flag(CTIME_EQ_P)) {
    if(!(*ceqfile1 = make_tmpfile()))
      goto ERROR;
    if(!(*ceqfile2 = make_tmpfile()))
      goto ERROR;
  }

  if(get_flag(MTIME_LT_P) || get_flag(MTIME_ET_P)) {
    if(!(*mfile = make_tmpfile()))
      goto ERROR;
  }
  if(get_flag(MTIME_EQ_P)) {
    if(!(*meqfile1 = make_tmpfile()))
      goto ERROR;
    if(!(*meqfile2 = make_tmpfile()))
      goto ERROR;
  }

 DONE:
  return(rv);

 ERROR:
  if(*afile)
    free(*afile);
  if(*cfile)
    free(*cfile);
  if(*mfile)
    free(*mfile);
  if(*aeqfile1)
    free(*aeqfile1);
  if(*aeqfile2)
    free(*aeqfile2);
  if(*ceqfile1)
    free(*ceqfile1);
  if(*ceqfile2)
    free(*ceqfile2);
  if(*meqfile1)
    free(*meqfile1);
  if(*meqfile2)
    free(*meqfile2);
  rv = 0;
  goto DONE;
}

static char *
make_timing_files(char *afile, char *cfile, char *mfile, char *aeqfile1,
		  char *aeqfile2, char *ceqfile1, char *ceqfile2,
		  char *meqfile1, char *meqfile2)
     /* return the string to make the timing files or NULL */
{
  static char s[2048]; /* a long enough string */
  char *rv = NULL;
  char time[32];
  char tmp[128];
  time_t t = 0;

  memset(s, 0, 2048);


  if(get_flag(ATIME_ET_P) || get_flag(ATIME_LT_P)) {
    sprintf(time, "%02d%02d%02d%02d%04d.%02d",
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (atime_month_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (atime_day_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (atime_hour_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (atime_minute_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (atime_year_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (atime_second_spin)));
      
    sprintf(tmp, "touch -t %s %s\n", time, afile);
    strcat(s, tmp);
  }
  if(get_flag(ATIME_EQ_P)) {
    t = get_user_time(atime_month_spin, atime_day_spin, atime_year_spin,
		      atime_hour_spin, atime_minute_spin, atime_second_spin);
    sprintf(tmp, "touch -t %s %s\n", dec_sec(t), aeqfile1);
    strcat(s, tmp);
    sprintf(tmp, "touch -t %s %s\n", inc_sec(t), aeqfile2);
    strcat(s, tmp);
  }

  
  if(get_flag(CTIME_ET_P) || get_flag(CTIME_LT_P)) {
    sprintf(time, "%02d%02d%02d%02d%04d.%02d",
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (ctime_month_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (ctime_day_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (ctime_hour_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (ctime_minute_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (ctime_year_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (ctime_second_spin)));
    sprintf(tmp, "touch -t %s %s\n", time, cfile);
    strcat(s, tmp);
  }
  if(get_flag(CTIME_EQ_P)) {
    t = get_user_time(ctime_month_spin, ctime_day_spin, ctime_year_spin,
		      ctime_hour_spin, ctime_minute_spin, ctime_second_spin);
    sprintf(tmp, "touch -t %s %s\n", dec_sec(t), ceqfile1);
    strcat(s, tmp);
    sprintf(tmp, "touch -t %s %s\n", inc_sec(t), ceqfile2);
    strcat(s, tmp);
  }
  
  if(get_flag(MTIME_ET_P) || get_flag(MTIME_LT_P)) {
    sprintf(time, "%02d%02d%02d%02d%04d.%02d",
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (mtime_month_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(mtime_day_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(mtime_hour_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (mtime_minute_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (mtime_year_spin)),
	    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					     (mtime_second_spin)));
    sprintf(tmp, "touch -t %s %s\n", time, mfile);
    strcat(s, tmp);
  }
  if(get_flag(MTIME_EQ_P)) {
    t = get_user_time(mtime_month_spin, mtime_day_spin, mtime_year_spin,
		      mtime_hour_spin, mtime_minute_spin, mtime_second_spin);
    sprintf(tmp, "touch -t %s %s\n", dec_sec(t), meqfile1);
    strcat(s, tmp);
    sprintf(tmp, "touch -t %s %s\n", inc_sec(t), meqfile2);
    strcat(s, tmp);
  }

  rv = s;
  return(rv);
}

static char *
make_group_string()
     /* make the string to match the group or gid */
{
  char *rv = NULL;
  static char group[64];
  char *g = NULL;
  int gid = 0;
  struct group *grp = NULL;

  memset(group, 0, 64);

  g = gtk_entry_get_text(GTK_ENTRY(group_entry));
  if(strlen(g) > 0) {
    if(get_flag(GID_NOT_GROUP_P)) {
      gid = atoi(g);
    }
    else {
      if((grp = getgrnam(group)))
	gid = grp->gr_gid;
      else {
	print_error("make_group_string: invalid group");
	goto DONE;
      }
    }
    sprintf(group, "-gid %d", gid);
    rv = group;
  }

 DONE:
  return(rv);
}

static char *
make_user_string()
     /* make the string to match the user (or uid) */
{
  char *rv = NULL;
  static char user[64]; /* no way is our UID going to be more than 64 bytes */
  char *login = NULL;
  int uid = 0;
  struct passwd *passwd = NULL;

  memset(user, 0, 64);

  login = gtk_entry_get_text(GTK_ENTRY(login_entry));
  if(strlen(login) > 0) {
    if(get_flag(UID_NOT_LOGIN_P))
      uid = atoi(login);
    else {
      if((passwd = getpwnam(login)))
	uid = passwd->pw_uid;
      else {
	print_error("make_user_string: invalid login");
	goto DONE;
      }
    }
    sprintf(user, "-uid %d", uid);
    rv = user;
  }

 DONE:
  return(rv);
}

static char *
make_type_string()
     /* return the -type string that we can use, or NULL if no types */
{
  char *rv = NULL;
  static char type[100]; /* enough space for all possible types plus parens */

  memset(type, 0, 100);
  strcat(type, "\\( ");
  if(get_flag(DIRECTORY_P))
    strcat(type, "-type d -o ");
  if(get_flag(REGULAR_P))
    strcat(type, "-type f -o ");
  if(get_flag(RAW_DEVICE_P))
    strcat(type, "-type c -o ");
  if(get_flag(BLOCK_DEVICE_P))
    strcat(type, "-type b -o ");
  if(get_flag(SYMLINK_P))
    strcat(type, "-type l -o ");
  if(get_flag(SOCKET_P))
    strcat(type, "-type s -o ");
  if(get_flag(FIFO_P))
    strcat(type, "-type f -o ");

  if(type[3]) {
    strcat(type, "-false \\) ");
    rv = type;
  }
  
  return rv;
}

static char *
make_sticky_string()
     /* handle the sticky bit and such */
{
  char *rv = NULL;
  static char mode[6];
  int sticky = 0;

  if(get_flag(SETUID_P))
    sticky += 4;
  if(get_flag(SETGID_P))
    sticky += 2;
  if(get_flag(STICKY_P))
    sticky += 1;

  if(sticky) {
    sprintf(mode, "+%d000", sticky);
    rv = mode;
  }
  else
    rv = NULL;

  return(rv);
}

static char *
make_mode_string()
     /* return a pointer to a string of the mode, or NULL if we're not
	filtering by mode */
{
  char *rv = NULL;
  static char mode[6];
  int op = 0, gp = 0, wp = 0;

  memset(mode, 0, 6);

  if(get_flag(OWNER_READ_P))
    op += 4;
  if(get_flag(OWNER_WRITE_P))
    op += 2;
  if(get_flag(OWNER_EXEC_P))
    op += 1;

  if(get_flag(GROUP_READ_P))
    gp += 4;
  if(get_flag(GROUP_WRITE_P))
    gp += 2;
  if(get_flag(GROUP_EXEC_P))
    gp += 1;
  
  if(get_flag(WORLD_READ_P))
    wp += 4;
  if(get_flag(WORLD_WRITE_P))
    wp += 2;
  if(get_flag(WORLD_EXEC_P))
    wp += 1;

  
  if(op || gp || wp) {
    sprintf(mode, "+%d%d%d", op, gp, wp);
    rv = mode;
  }
  else
    rv = NULL;

  return(rv);
}

int
save_search_command(char *filename)
     /* open filename, print out the search command to it, close it */
{
  FILE *output = NULL;
  int rv = 0;
  char *afile = NULL, *cfile = NULL, *mfile = NULL;
  char *aeqfile1 = NULL, *aeqfile2 = NULL, *ceqfile1 = NULL, *ceqfile2 = NULL,
    *meqfile1 = NULL, *meqfile2 = NULL;

  output = fopen(filename, "w");
  if(!output) {
    print_error("save_search_command: Can't write %s", filename);
    goto ERROR;
  }

  fprintf(output, "#!/bin/sh\n");
  fprintf(output, "# program generated by gtkfind\n");
  fprintf(output, "# get the latest gktfind at http://www.oz.net/~mattg/download.html\n");
  fprintf(output, "# This script may not work with non-GNU find!\n\n");


  /* touch the timing files */
  {
    char *s = NULL;
    
    if(!make_filenames(&afile, &cfile, &mfile, &aeqfile1, &aeqfile2,
		       &ceqfile1, &ceqfile2, &meqfile1, &meqfile2))
      goto ERROR;
    s = make_timing_files(afile, cfile, mfile, aeqfile1, aeqfile2, ceqfile1,
			  ceqfile2, meqfile1, meqfile2);
    if(s)
      fprintf(output, "%s", s);
  }

  
  fprintf(output, "find ");

  {
    char *search_directory = NULL;
    
    search_directory = get_search_directory(directory);
    if(strlen(search_directory))
      fprintf(output, "%s ", search_directory);
    else
      fprintf(output, "/ ");
  }

  /* do we want to only search this directory? */
  if(!get_flag(SEARCH_SUBDIRS_P))
    fprintf(output, "-prune ");
  
  /* match on filename */
  
  {
    char *search_pattern = NULL;
    char *s = NULL;
	
    search_pattern = gtk_entry_get_text(GTK_ENTRY(pattern));

    if(strlen(search_pattern)) {
      if(get_flag(WILDCARD_FILENAME_MATCH_P))
	fprintf(output, "-name \'%s\' ", search_pattern);
      else {
	s = escape_wildcards(search_pattern);
	if(s) 
	  fprintf(output, "-name \'*%s*\' ", escape_wildcards(search_pattern));
      }
    }
    /* if we don't have a search pattern, then don't match on it... */
  }

  /* match on permissions */
  
  {
    char *s = make_mode_string();
    if(s)
      fprintf(output, "-perm %s ", s);
  }

  /* match on the sticky bit */

  {
    char *s = make_sticky_string();
    if(s)
      fprintf(output, "-perm %s ", s);
  }

  /* match on types */

  {
    char *s = make_type_string();
    if(s)
      fprintf(output, "%s ", s);
    
  }

  /* match on user */

  {
    char *s = make_user_string();
    if(s)
      fprintf(output, "%s ", s);
  }

  /* match on group */

  {
    char *s = make_group_string();
    if(s)
      fprintf(output, "%s ", s);
  }

  /* match on times */
  {
    char *s = NULL;

    if(afile || cfile || mfile || aeqfile1 || ceqfile1 || meqfile1) {
      s = make_timing_string(afile, cfile, mfile, aeqfile1, aeqfile2,
			     ceqfile1, ceqfile2, meqfile1, meqfile2);
      fprintf(output, "%s ", s);
    }
  }

  
  fprintf(output, "-print ");

  /* match on contents with | xargs grep */
  {
    char *pattern = NULL;
    char *s = NULL;

    pattern = gtk_entry_get_text(GTK_ENTRY(content_pattern));
    if(pattern) {
      fprintf(output, "| xargs egrep "); /* gotta use egrep */
      if(get_flag(WILDCARD_CONTENTS_SEARCH_P)) {
	s = glob2regex(pattern);
	if(s) {
	  fprintf(output, "'%s' ", s);
	  free(s);
	}
      }
      else
	fprintf(output, "'%s' ", pattern);
    }
  }
  
  fprintf(output, "\n");

  /* delete the timing files */
  if(afile)
    fprintf(output, "/bin/rm -f %s\n", afile);
  if(cfile)
    fprintf(output, "/bin/rm -f %s\n", cfile);
  if(mfile)
    fprintf(output, "/bin/rm -f %s\n", mfile);
  if(aeqfile1)
    fprintf(output, "/bin/rm -f %s %s\n", aeqfile1, aeqfile2);
  if(ceqfile1)
    fprintf(output, "/bin/rm -f %s %s\n", ceqfile1, ceqfile2);
  if(meqfile1)
    fprintf(output, "/bin/rm -f %s %s\n", meqfile1, meqfile2);
  
  rv = 1;
 DONE:
  if(output)
    fclose(output);
  if(afile)
    free(afile);
  if(cfile)
    free(cfile);
  if(mfile)
    free(mfile);
  if(aeqfile1)
    free(aeqfile1);
  if(aeqfile2)
    free(aeqfile2);
  if(ceqfile1)
    free(ceqfile1);
  if(ceqfile2)
    free(ceqfile2);
  if(meqfile1)
    free(meqfile1);
  if(meqfile2)
    free(meqfile2);
  return(rv);
 ERROR:
  rv = 0;

  goto DONE;
}

char *
escape_wildcards(char *s)
     /* copy s to rv, escaping any wildcards or quotes with \ */
{
  static char *rv = NULL;
  char *wilds = "*?\"'`[]{}";
  char *p = NULL, *q = NULL;

  if(!s || !strlen(s)) {
    print_error("escape_wildcards: NULL or empty string passed");
    goto ERROR;
  }

  if(rv)
    free(rv);
  rv = (char *)malloc(sizeof(char) * strlen(s) * 2);
  if(!rv) {
    print_error("escape_wildcards: Can't malloc rv");
    goto ERROR;
  }

  p = s;
  q = rv;
  while(*p) {
    if(strchr(wilds, *p)) {
      *q++ = '\\';
    }
    *q++ = *p++;
  }

  *q = '\0';

 DONE:
  return(rv);
 ERROR:
  if(rv)
    free(rv);
  rv = NULL;
  goto DONE;
}

      
    
  
  
