/*
 * Set the framebuffer parameters for bttv.
 *   tries first to ask the X-Server
 *   if this failes, it checks /dev/fb0
 *
 *  (c) 1998 Gerd Knorr <kraxel@cs.tu-berlin.de>
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <strings.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <sys/ioctl.h>
#include <linux/fb.h>

#include "config.h"

#include <asm/types.h>          /* XXX glibc */
#include "videodev.h"

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#ifdef HAVE_LIBXXF86DGA
# include <X11/extensions/xf86dga.h>
#endif

int    verbose = 1;
int    bpp     = 0;
char  *display = ":0.0";
char  *device  = "/dev/video";
char  *fbdev   = "/dev/fb0";

struct fb_fix_screeninfo   fix;
struct fb_var_screeninfo   var;

int
main(int argc, char *argv[])
{
    Display                 *dpy;
    Screen                  *scr;
    Window                   root;
    XWindowAttributes        wts;
    XPixmapFormatValues     *pf;
    struct video_capability  capability;
    struct video_buffer      fbuf;
    int                      fd,c,i,n,depth=0;
    char                     *h;
#ifdef HAVE_LIBXXF86DGA
    int                      width,bar,foo,flags=0;
    void                    *base = 0;
#endif

    if (NULL != (h = getenv("DISPLAY")))
	display = h;
    
    /* parse options */
    for (;;) {
	if (-1 == (c = getopt(argc, argv, "hqd:c:b:")))
	    break;
	switch (c) {
	case 'q':
	    verbose = 0;
	    break;
	case 'd':
	    display = optarg;
	    break;
	case 'c':
	    device = optarg;
	    break;
	case 'b':
	    bpp = atoi(optarg);
	    break;
	case 'h':
	default:
	    fprintf(stderr,
		    "usage: %s  [ options ] \n"
		    "\n"
		    "options:\n"
		    "    -q        quiet\n"
		    "    -d <dpy>  X11 Display     [%s]\n"
		    "    -c <dev>  video device    [%s]\n"
		    "    -b <n>    displays color depth is <n> bpp\n"
		    "              might be required for (and works\n"
		    "              only with) 24/32 bpp\n",
		    argv[0],
		    display,
		    device);
	    exit(1);
	}
    }

    if (display[0] != ':') {
	fprintf(stderr,
		"non-local display `%s' not allowed, using `:0.0' instead\n",
		display);
	display = ":0.0";
    }

    /* get screen params */
    if (NULL != (dpy = XOpenDisplay(display))) {
	scr  = DefaultScreenOfDisplay(dpy);
	root = DefaultRootWindow(dpy);
	XGetWindowAttributes(dpy, root, &wts);
	depth = wts.depth;

	pf = XListPixmapFormats(dpy,&n);
	for (i = 0; i < n; i++) {
	    if (pf[i].depth == depth) {
		depth = pf[i].bits_per_pixel;
		break;
	    }
	}

	if ((bpp == 32 || bpp == 24) && (depth == 32 || depth == 24))
	    depth = bpp;
	if (verbose)
	    fprintf(stderr,"x11: mode=%dx%d, %d bit color depth, probably %d bit/pixel\n",
			   wts.width,wts.height,wts.depth,depth);
	
#ifdef HAVE_LIBXXF86DGA
	if (XF86DGAQueryExtension(dpy,&foo,&bar)) {
	    XF86DGAQueryDirectVideo(dpy,XDefaultScreen(dpy),&flags);
	    if (flags & XF86DGADirectPresent) {
		XF86DGAGetVideoLL(dpy,XDefaultScreen(dpy),(int*)&base,&width,&foo,&bar);
		if (verbose)
		    fprintf(stderr,"dga: base=%p, width=%d\n", base, width);
	    }
	}
#endif
    } else {
	fprintf(stderr,"can't open x11 display %s\n",display);

	/* try framebuffer */
	if (-1 == (fd = open(fbdev,O_RDWR))) {
	    fprintf(stderr,"open %s: %s\n",fbdev,strerror(errno));
	    exit(1);
	}
	if (-1 == ioctl(fd,FBIOGET_FSCREENINFO,&fix)) {
	    perror("ioctl FBIOGET_FSCREENINFO");
	    exit(1);
	}
	if (-1 == ioctl(fd,FBIOGET_VSCREENINFO,&var)) {
	    perror("ioctl FBIOGET_VSCREENINFO");
	    exit(1);
	}
	if (fix.type != FB_TYPE_PACKED_PIXELS) {
	    fprintf(stderr,"can handle only packed pixel frame buffers\n");
	    exit(1);
	}
	close(fd);
	fprintf(stderr,"%s: %dx%dx%d @ %p\n",fbdev,
		var.xres_virtual,
		var.yres_virtual,
		var.bits_per_pixel,
		fix.smem_start);
    }

    /* open & check v4l device */
    if (-1 == (fd = open(device,O_RDWR))) {
	fprintf(stderr,"can't open %s: %s\n",device,strerror(errno));
	exit(1);
    }
    if (-1 == ioctl(fd,VIDIOCGCAP,&capability)) {
	fprintf(stderr,"%s: ioctl VIDIOCGCAP: %s\n",device,strerror(errno));
	exit(1);
    }
    if (!(capability.type & VID_TYPE_OVERLAY)) {
	fprintf(stderr,"%s: no overlay support\n",device);
	exit(1);
    }

    /* read-modify-write v4l screen parameters */
    if (-1 == ioctl(fd,VIDIOCGFBUF,&fbuf)) {
	fprintf(stderr,"%s: ioctl VIDIOCGFBUF: %s\n",device,strerror(errno));
	exit(1);
    }
    if (verbose)
	fprintf(stderr,"v4l: base=%p\n",fbuf.base);
#ifdef HAVE_LIBXXF86DGA
    if (base && fbuf.base != base) {
	fbuf.base = base;
	if (verbose)
	    fprintf(stderr,"setting v4l base to %p\n",fbuf.base);
    }
#endif
    if (dpy) {
	/* x11 */
	fbuf.depth        = (depth+7) & 0xf8;
#ifdef HAVE_LIBXXF86DGA
	fbuf.width        = (flags & XF86DGADirectPresent) ? width : wts.width;
#else
	fbuf.width        = wts.width;
#endif
	fbuf.height       = wts.height;
	fbuf.bytesperline = fbuf.width * fbuf.depth/8;
    } else {
	/* framebuffer */
	fbuf.depth        = (var.bits_per_pixel+7) & 0xf8;
	fbuf.width        = var.xres_virtual;
	fbuf.height       = var.yres_virtual;
	fbuf.bytesperline = fix.line_length;
    }

    /* XXX bttv confuses color depth and bits/pixel */
    if (wts.depth == 15)
	fbuf.depth = 15;

    if (-1 == ioctl(fd,VIDIOCSFBUF,&fbuf)) {
	fprintf(stderr,"%s: ioctl VIDIOCSFBUF: %s\n",device,strerror(errno));
	exit(1);
    }
    if (verbose)
	fprintf(stderr,"ok\n");

    return 0;
}
