/*
 * Copyright 1996, 1997, 1998 Computing Research Labs,
 * New Mexico State University
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE COMPUTING RESEARCH LAB OR NEW MEXICO STATE UNIVERSITY BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT
 * OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 * THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#ifndef lint
#ifdef __GNUC__
static char rcsid[] __attribute__ ((unused)) = "$Id: GEditTB.c,v 1.1 1998/07/24 16:12:08 mleisher Exp $";
#else
static char rcsid[] = "$Id: GEditTB.c,v 1.1 1998/07/24 16:12:08 mleisher Exp $";
#endif
#endif

#include <stdlib.h>
#include "GEditTBP.h"
#include "bitmaps.h"

static unsigned char bitswap[] = {
    0x00, 0x80, 0x40, 0xc0, 0x20, 0xa0, 0x60, 0xe0, 0x10, 0x90, 0x50, 0xd0,
    0x30, 0xb0, 0x70, 0xf0, 0x08, 0x88, 0x48, 0xc8, 0x28, 0xa8, 0x68, 0xe8,
    0x18, 0x98, 0x58, 0xd8, 0x38, 0xb8, 0x78, 0xf8, 0x04, 0x84, 0x44, 0xc4,
    0x24, 0xa4, 0x64, 0xe4, 0x14, 0x94, 0x54, 0xd4, 0x34, 0xb4, 0x74, 0xf4,
    0x0c, 0x8c, 0x4c, 0xcc, 0x2c, 0xac, 0x6c, 0xec, 0x1c, 0x9c, 0x5c, 0xdc,
    0x3c, 0xbc, 0x7c, 0xfc, 0x02, 0x82, 0x42, 0xc2, 0x22, 0xa2, 0x62, 0xe2,
    0x12, 0x92, 0x52, 0xd2, 0x32, 0xb2, 0x72, 0xf2, 0x0a, 0x8a, 0x4a, 0xca,
    0x2a, 0xaa, 0x6a, 0xea, 0x1a, 0x9a, 0x5a, 0xda, 0x3a, 0xba, 0x7a, 0xfa,
    0x06, 0x86, 0x46, 0xc6, 0x26, 0xa6, 0x66, 0xe6, 0x16, 0x96, 0x56, 0xd6,
    0x36, 0xb6, 0x76, 0xf6, 0x0e, 0x8e, 0x4e, 0xce, 0x2e, 0xae, 0x6e, 0xee,
    0x1e, 0x9e, 0x5e, 0xde, 0x3e, 0xbe, 0x7e, 0xfe, 0x01, 0x81, 0x41, 0xc1,
    0x21, 0xa1, 0x61, 0xe1, 0x11, 0x91, 0x51, 0xd1, 0x31, 0xb1, 0x71, 0xf1,
    0x09, 0x89, 0x49, 0xc9, 0x29, 0xa9, 0x69, 0xe9, 0x19, 0x99, 0x59, 0xd9,
    0x39, 0xb9, 0x79, 0xf9, 0x05, 0x85, 0x45, 0xc5, 0x25, 0xa5, 0x65, 0xe5,
    0x15, 0x95, 0x55, 0xd5, 0x35, 0xb5, 0x75, 0xf5, 0x0d, 0x8d, 0x4d, 0xcd,
    0x2d, 0xad, 0x6d, 0xed, 0x1d, 0x9d, 0x5d, 0xdd, 0x3d, 0xbd, 0x7d, 0xfd,
    0x03, 0x83, 0x43, 0xc3, 0x23, 0xa3, 0x63, 0xe3, 0x13, 0x93, 0x53, 0xd3,
    0x33, 0xb3, 0x73, 0xf3, 0x0b, 0x8b, 0x4b, 0xcb, 0x2b, 0xab, 0x6b, 0xeb,
    0x1b, 0x9b, 0x5b, 0xdb, 0x3b, 0xbb, 0x7b, 0xfb, 0x07, 0x87, 0x47, 0xc7,
    0x27, 0xa7, 0x67, 0xe7, 0x17, 0x97, 0x57, 0xd7, 0x37, 0xb7, 0x77, 0xf7,
    0x0f, 0x8f, 0x4f, 0xcf, 0x2f, 0xaf, 0x6f, 0xef, 0x1f, 0x9f, 0x5f, 0xdf,
    0x3f, 0xbf, 0x7f, 0xff,
};

/*
 * Sizes for the button shapes.
 */
#define _GETB_TOGGLE_SIZE 36
#define _GETB_BUTTON_SIZE 33

/**************************************************************************
 *
 * Button indexes.
 *
 **************************************************************************/

#define _GETB_DRAW_TOGGLE     0
#define _GETB_MOVE_TOGGLE     1
#define _GETB_COPY_TOGGLE     2
#define _GETB_FLIPH_BUTTON    3
#define _GETB_FLIPV_BUTTON    4
#define _GETB_SHEAR_BUTTON    5
#define _GETB_RLEFT_BUTTON    6
#define _GETB_RRIGHT_BUTTON   7
#define _GETB_ROTATE_BUTTON   8
#define _GETB_ULEFT_BUTTON    9
#define _GETB_UP_BUTTON       10
#define _GETB_URIGHT_BUTTON   11
#define _GETB_LEFT_BUTTON     12
#define _GETB_RIGHT_BUTTON    13
#define _GETB_DLEFT_BUTTON    14
#define _GETB_DOWN_BUTTON     15
#define _GETB_DRIGHT_BUTTON   16
#define _GETB_GLYPH_IMAGE     17

/**************************************************************************
 *
 * Callback values for each button.
 *
 **************************************************************************/

static int cback_values[] = {
    XmuttGLYPHEDIT_DRAW,
    XmuttGLYPHEDIT_MOVE,
    XmuttGLYPHEDIT_COPY,
    XmuttGLYPHEDIT_FLIP_HORIZ,
    XmuttGLYPHEDIT_FLIP_VERT,
    XmuttGLYPHEDIT_SHEAR,
    XmuttGLYPHEDIT_ROT_LEFT,
    XmuttGLYPHEDIT_ROT_RIGHT,
    XmuttGLYPHEDIT_ROTATE,
    XmuttGLYPHEDIT_SHIFT_ULEFT,
    XmuttGLYPHEDIT_SHIFT_UP,
    XmuttGLYPHEDIT_SHIFT_URIGHT,
    XmuttGLYPHEDIT_SHIFT_LEFT,
    XmuttGLYPHEDIT_SHIFT_RIGHT,
    XmuttGLYPHEDIT_SHIFT_DLEFT,
    XmuttGLYPHEDIT_SHIFT_DOWN,
    XmuttGLYPHEDIT_SHIFT_DRIGHT,
};

/**************************************************************************
 *
 * Help strings for context sensitive help.
 *
 **************************************************************************/

static String help_strings[18] = {
    "Draw",
    "Move",
    "Copy",
    "Flip Horizontally",
    "Flip Vertically",
    "Shear 45",
    "Rotate -90",
    "Rotate +90",
    "Rotate 359",
    "Shift Up+Left",
    "Shift Up",
    "Shift Up+Right",
    "Shift Left",
    "Shift Right",
    "Shift Down+Left",
    "Shift Down",
    "Shift Down+Right",
    "Glyph Image",
};

/**************************************************************************
 *
 * Utility functions.
 *
 **************************************************************************/

static void
#ifndef _NO_PROTO
_XmuttGEditTBoxCheckClass(Widget w)
#else
_XmuttGEditTBoxCheckClass(w)
Widget w;
#endif
{
    if (XtIsSubclass(w, xmuttGlyphEditToolboxWidgetClass) == False) {
        fprintf(stderr,
                "XmuttGlyphEditToolbox: %s not an XmuttGlyphEditToolboxWidget.\n",
                XtName(w));
        exit(1);
    }
}

static Dimension
#ifndef _NO_PROTO
_XmuttGEditTBoxPMargins(XmuttGlyphEditToolboxWidget gw)
#else
_XmuttGEditTBoxPMargins(gw)
XmuttGlyphEditToolboxWidget gw;
#endif
{
    return (gw->primitive.shadow_thickness << 2) +
        (gw->primitive.highlight_thickness << 1);
}

static void
#ifndef _NO_PROTO
_XmuttGEditTBoxPreferredGeometry(XmuttGlyphEditToolboxWidget gw,
                                 Dimension *width, Dimension *height)
#else
_XmuttGEditTBoxPreferredGeometry(gw, width, height)
XmuttGlyphEditToolboxWidget gw;
Dimension *width, *height;
#endif
{
    Dimension margin;
    bdf_bitmap_t *im;

    margin = _XmuttGEditTBoxPMargins(gw) + 4;

    *width = margin + (3 * (_GETB_TOGGLE_SIZE + 4)) + 4;
    *height = margin + (_GETB_TOGGLE_SIZE + 6) +
        ((5 * _GETB_BUTTON_SIZE) + 8);

    /*
     * If a glyph image was provided, adjust the height.  The width is highly
     * unlikely to be greater than the width of the toolbox buttons.
     */
    if ((im = gw->gedittb.gimage) != 0)
      *height += im->height + 6;
}

/*
 * Change window highlighting when focus is gained or lost.
 */
static void
#ifndef _NO_PROTO
_XmuttGEditTBoxHighlightWindow(Widget w)
#else
_XmuttGEditTBoxHighlightWindow(w)
Widget w;
#endif
{
    int i;
    XmuttGlyphEditToolboxWidget gw;
    XRectangle hi[4];

    if (!XtIsRealized(w))
      return;

    gw = (XmuttGlyphEditToolboxWidget) w;

    /*
     * Top of window.
     */
    hi[0].x = hi[0].y = 0;
    hi[0].width = gw->core.width;
    hi[0].height = gw->primitive.highlight_thickness;

    /*
     * Right side.
     */
    hi[1].x = gw->core.width - gw->primitive.highlight_thickness;
    hi[1].y = hi[0].y + gw->primitive.highlight_thickness;
    hi[1].width = gw->primitive.highlight_thickness;
    hi[1].height = gw->core.height - (gw->primitive.highlight_thickness << 1);

    /*
     * Bottom of window.
     */
    hi[2].x = hi[0].x;
    hi[2].y = gw->core.height - gw->primitive.highlight_thickness;
    hi[2].width = hi[0].width;
    hi[2].height = hi[0].height;

    /*
     * Left side.
     */
    hi[3].x = hi[0].x;
    hi[3].y = hi[1].y;
    hi[3].width = hi[1].width;
    hi[3].height = hi[1].height;

    if (gw->gedittb.hasfocus == True)
      XFillRectangles(XtDisplay(w), XtWindow(w), gw->primitive.highlight_GC,
                      hi, 4);
    else {
        for (i = 0; i < 4; i++)
          XClearArea(XtDisplay(w), XtWindow(w), hi[i].x, hi[i].y,
                     hi[i].width, hi[i].height, False);
    }
}

static void
#ifndef _NO_PROTO
_XmuttGEditTBoxShadowWindow(Widget w)
#else
_XmuttGEditTBoxShadowWindow(w)
Widget w;
#endif
{
    XmuttGlyphEditToolboxWidget gw;
    int i;
    XPoint br[3], tl[3];

    gw = (XmuttGlyphEditToolboxWidget) w;

    if (!XtIsRealized(w) || gw->primitive.shadow_thickness == 0)
      return;

    /*
     * Lower left corner outside line.
     */
    tl[0].x = gw->primitive.highlight_thickness;
    tl[0].y = gw->core.height - gw->primitive.highlight_thickness - 1;

    /*
     * Upper left corner outside line.
     */
    tl[1].x = tl[0].x;
    tl[1].y = gw->primitive.highlight_thickness;

    /*
     * Upper right corner outside line.
     */
    tl[2].x = gw->core.width - gw->primitive.highlight_thickness - 1;
    tl[2].y = tl[1].y;

    /*
     * Upper right corner outside line.
     */
    br[0].x = tl[2].x;
    br[0].y = tl[2].y + 1;

    /*
     * Lower right corner outside line.
     */
    br[1].x = br[0].x;
    br[1].y = gw->core.height - gw->primitive.highlight_thickness - 1;

    /*
     * Lower left corner outside line.
     */
    br[2].x = gw->primitive.highlight_thickness + 1;
    br[2].y = br[1].y;

    XDrawLines(XtDisplay(w), XtWindow(w), gw->primitive.top_shadow_GC,
               tl, 3, CoordModeOrigin);
    XDrawLines(XtDisplay(w), XtWindow(w), gw->primitive.bottom_shadow_GC,
               br, 3, CoordModeOrigin);

    /*
     * Draw the remaining shadows successively inward.
     */
    for (i = 1; i < gw->primitive.shadow_thickness; i++) {
        tl[0].x++;
        tl[0].y--;
        tl[1].x++;
        tl[1].y++;
        tl[2].x--;
        tl[2].y++;

        br[0].x--;
        br[0].y++;
        br[1].x--;
        br[1].y--;
        br[2].x++;
        br[2].y--;
        XDrawLines(XtDisplay(w), XtWindow(w), gw->primitive.top_shadow_GC,
                   tl, 3, CoordModeOrigin);
        XDrawLines(XtDisplay(w), XtWindow(w), gw->primitive.bottom_shadow_GC,
                   br, 3, CoordModeOrigin);
    }
}

static void
#ifndef _NO_PROTO
_XmuttGEditTBoxMakeBitmaps(Widget w)
#else
_XmuttGEditTBoxMakeBitmaps(w)
Widget w;
#endif
{
    XmuttGlyphEditToolboxWidget gw;
    XmuttGlyphEditToolboxWidgetClass gwc;
    _XmuttGEditTBoxButton *bp;

    gw = (XmuttGlyphEditToolboxWidget) w;
    gwc = (XmuttGlyphEditToolboxWidgetClass) XtClass(w);

    /*
     * Only create the bitmaps if this is the first instance of this
     * kind of widget.
     */
    if (gwc->gedittb_class.refcnt != 1)
      return;

    bp = gw->gedittb.buttons;

    gwc->gedittb_class.draw = bp[_GETB_DRAW_TOGGLE].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), draw_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.move = bp[_GETB_MOVE_TOGGLE].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), move_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.copy = bp[_GETB_COPY_TOGGLE].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), copy_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.fliph = bp[_GETB_FLIPH_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), fliph_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.flipv = bp[_GETB_FLIPV_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), flipv_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.shear = bp[_GETB_SHEAR_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), shear_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.rleft = bp[_GETB_RLEFT_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), rleft_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.rright = bp[_GETB_RRIGHT_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), rright_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.rotate = bp[_GETB_ROTATE_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), rotate_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.uleft = bp[_GETB_ULEFT_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), uleft_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.up = bp[_GETB_UP_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), up_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.uright = bp[_GETB_URIGHT_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), uright_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.left = bp[_GETB_LEFT_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), left_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.right = bp[_GETB_RIGHT_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), right_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.dleft = bp[_GETB_DLEFT_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), dleft_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.down = bp[_GETB_DOWN_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), down_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
    gwc->gedittb_class.dright = bp[_GETB_DRIGHT_BUTTON].pixmap =
        XCreateBitmapFromData(XtDisplay(w), XtWindow(w), dright_bits,
                              _GETB_WIDTH, _GETB_HEIGHT);
}

static void
#ifndef _NO_PROTO
_XmuttGEditTBoxAssignBitmaps(Widget w)
#else
_XmuttGEditTBoxAssignBitmaps(w)
Widget w;
#endif
{
    XmuttGlyphEditToolboxWidget gw;
    XmuttGlyphEditToolboxWidgetClass gwc;
    _XmuttGEditTBoxButton *bp;

    gw = (XmuttGlyphEditToolboxWidget) w;
    gwc = (XmuttGlyphEditToolboxWidgetClass) XtClass(w);

    bp = gw->gedittb.buttons;

    /*
     * Assign the pixmaps.
     */
    bp[_GETB_DRAW_TOGGLE].pixmap = gwc->gedittb_class.draw;
    bp[_GETB_MOVE_TOGGLE].pixmap = gwc->gedittb_class.move;
    bp[_GETB_COPY_TOGGLE].pixmap = gwc->gedittb_class.copy;
    bp[_GETB_FLIPH_BUTTON].pixmap = gwc->gedittb_class.fliph;
    bp[_GETB_FLIPV_BUTTON].pixmap = gwc->gedittb_class.flipv;
    bp[_GETB_SHEAR_BUTTON].pixmap = gwc->gedittb_class.shear;
    bp[_GETB_RLEFT_BUTTON].pixmap = gwc->gedittb_class.rleft;
    bp[_GETB_RRIGHT_BUTTON].pixmap = gwc->gedittb_class.rright;
    bp[_GETB_ROTATE_BUTTON].pixmap = gwc->gedittb_class.rotate;
    bp[_GETB_ULEFT_BUTTON].pixmap = gwc->gedittb_class.uleft;
    bp[_GETB_UP_BUTTON].pixmap = gwc->gedittb_class.up;
    bp[_GETB_URIGHT_BUTTON].pixmap = gwc->gedittb_class.uright;
    bp[_GETB_LEFT_BUTTON].pixmap = gwc->gedittb_class.left;
    bp[_GETB_RIGHT_BUTTON].pixmap = gwc->gedittb_class.right;
    bp[_GETB_DLEFT_BUTTON].pixmap = gwc->gedittb_class.dleft;
    bp[_GETB_DOWN_BUTTON].pixmap = gwc->gedittb_class.down;
    bp[_GETB_DRIGHT_BUTTON].pixmap = gwc->gedittb_class.dright;
}

static void
#ifndef _NO_PROTO
_XmuttGEditTBoxMakeGCs(Widget w, Boolean force)
#else
_XmuttGEditTBoxMakeGCs(w, force)
Widget w;
Boolean force;
#endif
{
    XmuttGlyphEditToolboxWidget gw;
    XtGCMask gcm;
    XGCValues gcv;

    gw = (XmuttGlyphEditToolboxWidget) w;

    gcm = GCForeground|GCBackground|GCFunction;

    if (gw->gedittb.gc == 0 || force == True) {
        if (gw->gedittb.gc != 0)
          XFreeGC(XtDisplay(w), gw->gedittb.gc);
        gcv.foreground = gw->primitive.foreground;
        gcv.background = gw->core.background_pixel;
        gcv.function = GXcopy;
        gw->gedittb.gc = XCreateGC(XtDisplay(w), XtWindow(w), gcm, &gcv);
    }

    if (gw->gedittb.fillgc == 0 || force == True) {
        if (gw->gedittb.fillgc != 0)
          XFreeGC(XtDisplay(w), gw->gedittb.fillgc);
        gcv.foreground = gw->core.background_pixel;
        gcv.background = gw->primitive.foreground;
        gcv.function = GXcopy;
        gw->gedittb.fillgc = XCreateGC(XtDisplay(w), XtWindow(w), gcm, &gcv);
    }

    if (gw->gedittb.armgc == 0 || force == True) {
        if (gw->gedittb.armgc != 0)
          XFreeGC(XtDisplay(w), gw->gedittb.armgc);
        gcv.foreground = gw->primitive.foreground;
        gcv.background = gw->gedittb.arm_color;
        gcv.function = GXcopy;
        gw->gedittb.armgc = XCreateGC(XtDisplay(w), XtWindow(w), gcm, &gcv);
    }

    if (gw->gedittb.armfillgc == 0 || force == True) {
        if (gw->gedittb.armfillgc != 0)
          XFreeGC(XtDisplay(w), gw->gedittb.armfillgc);
        gcv.foreground = gw->gedittb.arm_color;
        gcv.background = gw->primitive.foreground;
        gcv.function = GXcopy;
        gw->gedittb.armfillgc =
            XCreateGC(XtDisplay(w), XtWindow(w), gcm, &gcv);
    }

    if (gw->gedittb.tsgc == 0 || force == True) {
        if (gw->gedittb.tsgc != 0)
          XFreeGC(XtDisplay(w), gw->gedittb.tsgc);
        gw->gedittb.tsgc =
            XCreateGC(XtDisplay(w), XtWindow(w), 0, &gcv);
        XCopyGC(XtDisplay(w), gw->primitive.top_shadow_GC, gcm,
                gw->gedittb.tsgc);
    }
    if (gw->gedittb.bsgc == 0 || force == True) {
        if (gw->gedittb.bsgc != 0)
          XFreeGC(XtDisplay(w), gw->gedittb.bsgc);
        gw->gedittb.bsgc =
            XCreateGC(XtDisplay(w), XtWindow(w), 0, &gcv);
        XCopyGC(XtDisplay(w), gw->primitive.bottom_shadow_GC, gcm,
                gw->gedittb.bsgc);
    }
}

static void
#ifndef _NO_PROTO
_XmuttGEditTBoxDrawImage(Widget w)
#else
_XmuttGEditTBoxDrawImage(w)
Widget w;
#endif
{
    XmuttGlyphEditToolboxWidget gw;
    bdf_bitmap_t *im;

    gw = (XmuttGlyphEditToolboxWidget) w;

    if (!XtIsRealized(w) || (im = gw->gedittb.gimage) == 0)
      return;

    /*
     * Check to see if the pixmap exists for the image or not.  Create it if
     * necessary.
     */
    if (gw->gedittb.image == 0)
      gw->gedittb.image =
          XCreateBitmapFromData(XtDisplay(w), XtWindow(w), im->bitmap,
                                im->width, im->height);

    /*
     * Draw a rectangle and copy the image to the rectangle.
     */
    XDrawRectangle(XtDisplay(w), XtWindow(w), gw->gedittb.gc, im->x, im->y,
                   im->width + 3, im->height + 3);
    XCopyPlane(XtDisplay(w), gw->gedittb.image, XtWindow(w), gw->gedittb.gc,
               0, 0, im->width, im->height, im->x + 2, im->y + 2, 1);
}

static void
#ifndef _NO_PROTO
_XmuttGEditTBoxDrawToggle(Widget w, int which)
#else
_XmuttGEditTBoxDrawToggle(w, which)
Widget w;
int which;
#endif
{
    XmuttGlyphEditToolboxWidget gw;
    GC tgc, bgc, fgc, dgc;
    int i;
    _XmuttGEditTBoxButton *bp;
    XPoint toggle[6];

    if (!XtIsRealized(w))
      return;

    gw = (XmuttGlyphEditToolboxWidget) w;
    bp = gw->gedittb.buttons;

    /*
     * Choose the appropriate GCs.
     */
    if (bp[which].set == True) {
        tgc = gw->gedittb.bsgc;
        bgc = gw->gedittb.tsgc;
        fgc = gw->gedittb.armfillgc;
        dgc = gw->gedittb.armgc;
    } else {
        tgc = gw->gedittb.tsgc;
        bgc = gw->gedittb.bsgc;
        fgc = gw->gedittb.fillgc;
        dgc = gw->gedittb.gc;
    }

    /*
     * Set up the polygon for clearing the background first.
     */
    toggle[0].x = bp[which].x;
    toggle[0].y = bp[which].y;

    toggle[1].x = toggle[0].x - ((_GETB_TOGGLE_SIZE >> 1) + 2);
    toggle[1].y = toggle[0].y + ((_GETB_TOGGLE_SIZE >> 1) + 2);

    toggle[2].x = toggle[0].x - 1;
    toggle[2].y = toggle[1].y + (_GETB_TOGGLE_SIZE >> 1) + 1;

    toggle[3].x = toggle[0].x + (_GETB_TOGGLE_SIZE >> 1) + 2;
    toggle[3].y = toggle[1].y;

    toggle[4].x = toggle[0].x;
    toggle[4].y = toggle[0].y;

    /*
     * Clear the foreground.
     */
    XFillPolygon(XtDisplay(w), XtWindow(w), fgc, toggle, 5, Convex,
                 CoordModeOrigin);

    /*
     * Reset the extra points for drawing the polygon shadow.
     */
    toggle[3].x = toggle[0].x + 1;
    toggle[3].y = toggle[0].y + 1;

    toggle[4].x = toggle[0].x + (_GETB_TOGGLE_SIZE >> 1) + 2;
    toggle[4].y = toggle[1].y;

    toggle[5].x = toggle[2].x + 1;
    toggle[5].y = toggle[2].y + 1;

    XDrawLines(XtDisplay(w), XtWindow(w), tgc, toggle, 3, CoordModeOrigin);
    XDrawLines(XtDisplay(w), XtWindow(w), bgc, &toggle[3], 3, CoordModeOrigin);

    for (i = 1; i < 2; i++) {
        toggle[0].y++;
        toggle[1].x++;
        toggle[2].y--;
        toggle[3].y++;
        toggle[4].x--;
        toggle[5].y--;
        XDrawLines(XtDisplay(w), XtWindow(w), tgc, toggle, 3, CoordModeOrigin);
        XDrawLines(XtDisplay(w), XtWindow(w), bgc, &toggle[3], 3,
                   CoordModeOrigin);
    }

    /*
     * Copy the bitmap onto the polygon.
     */
    XCopyPlane(XtDisplay(w), bp[which].pixmap, XtWindow(w), dgc,
               0, 0, _GETB_WIDTH, _GETB_HEIGHT,
               bp[which].x - 7, bp[which].y + 12, 1);

}

static void
#ifndef _NO_PROTO
_XmuttGEditTBoxDrawButton(Widget w, int which)
#else
_XmuttGEditTBoxDrawButton(w, which)
Widget w;
int which;
#endif
{
    XmuttGlyphEditToolboxWidget gw;
    GC tgc, bgc, fgc, dgc;
    int i;
    _XmuttGEditTBoxButton *bp;
    XRectangle clear;
    XPoint br[3], tl[3];

    if (!XtIsRealized(w))
      return;

    gw = (XmuttGlyphEditToolboxWidget) w;
    bp = gw->gedittb.buttons;

    clear.x = bp[which].x + 2;
    clear.y = bp[which].y + 2;
    clear.width = clear.height = _GETB_BUTTON_SIZE - 2;

    /*
     * Set up the appropriate GC's.
     */
    if (bp[which].set == True) {
        tgc = gw->gedittb.bsgc;
        bgc = gw->gedittb.tsgc;
        fgc = gw->gedittb.armfillgc;
        dgc = gw->gedittb.armgc;
    } else {
        tgc = gw->gedittb.tsgc;
        bgc = gw->gedittb.bsgc;
        fgc = gw->gedittb.fillgc;
        dgc = gw->gedittb.gc;
    }

    XFillRectangle(XtDisplay(w), XtWindow(w), fgc,
                   clear.x, clear.y, clear.width, clear.height);
    XCopyPlane(XtDisplay(w), bp[which].pixmap, XtWindow(w), dgc,
               0, 0, _GETB_WIDTH, _GETB_HEIGHT,
               bp[which].x + ((_GETB_BUTTON_SIZE >> 1) - (_GETB_WIDTH >> 1)),
               bp[which].y + ((_GETB_BUTTON_SIZE >> 1) - (_GETB_HEIGHT >> 1)),
               1);

    /*
     * Lower left corner outside line.
     */
    tl[0].x = bp[which].x;
    tl[0].y = bp[which].y + (_GETB_BUTTON_SIZE - 1);

    /*
     * Upper left corner outside line.
     */
    tl[1].x = tl[0].x;
    tl[1].y = bp[which].y;

    /*
     * Upper right corner outside line.
     */
    tl[2].x = tl[0].x + _GETB_BUTTON_SIZE;
    tl[2].y = tl[1].y;

    /*
     * Upper right corner outside line.
     */
    br[0].x = tl[2].x;
    br[0].y = tl[2].y + 1;

    /*
     * Lower right corner outside line.
     */
    br[1].x = br[0].x;
    br[1].y = tl[0].y;

    /*
     * Lower left corner outside line.
     */
    br[2].x = tl[0].x + 1;
    br[2].y = br[1].y;

    XDrawLines(XtDisplay(w), XtWindow(w), tgc, tl, 3, CoordModeOrigin);
    XDrawLines(XtDisplay(w), XtWindow(w), bgc, br, 3, CoordModeOrigin);

    /*
     * Draw the remaining shadows successively inward.
     */
    for (i = 1; i < 2; i++) {
        tl[0].x++;
        tl[0].y--;
        tl[1].x++;
        tl[1].y++;
        tl[2].x--;
        tl[2].y++;

        br[0].x--;
        br[0].y++;
        br[1].x--;
        br[1].y--;
        br[2].x++;
        br[2].y--;
        XDrawLines(XtDisplay(w), XtWindow(w), tgc, tl, 3, CoordModeOrigin);
        XDrawLines(XtDisplay(w), XtWindow(w), bgc, br, 3, CoordModeOrigin);
    }
}

static void
#ifndef _NO_PROTO
_XmuttGEditTBoxPosition(XmuttGlyphEditToolboxWidget gw)
#else
_XmuttGEditTBoxPosition(gw)
XmuttGlyphEditToolboxWidget gw;
#endif
{
    XmuttGlyphEditToolboxWidgetClass gwc;
    int i, j, x, y, dx, dy;
    Dimension cp;
    _XmuttGEditTBoxButton *bp;
    bdf_bitmap_t *im;
    XRectangle button;
    XPoint toggle[5];

    gwc = (XmuttGlyphEditToolboxWidgetClass) XtClass((Widget) gw);

    /*
     * Set up the initial polygon for creating toggle regions.
     */
    toggle[0].x = ((_GETB_TOGGLE_SIZE >> 1) + 2);
    toggle[0].y = 0;

    toggle[1].x = toggle[0].x - ((_GETB_TOGGLE_SIZE >> 1) + 2);
    toggle[1].y = toggle[0].y + ((_GETB_TOGGLE_SIZE >> 1) + 2);

    toggle[2].x = toggle[0].x - 1;
    toggle[2].y = toggle[1].y + (_GETB_TOGGLE_SIZE >> 1) + 1;

    toggle[3].x = toggle[0].x + (_GETB_TOGGLE_SIZE >> 1) + 2;
    toggle[3].y = toggle[1].y;

    toggle[4].x = toggle[0].x;
    toggle[4].y = toggle[0].y;

    bp = gw->gedittb.buttons;

    y = (_XmuttGEditTBoxPMargins(gw) >> 1) + 2;

    cp = x = gw->core.width >> 1;

    /*
     * Position the top-left coordinates of the glyph image if it exists.
     */
    if ((im = gw->gedittb.gimage) != 0) {
        dx = im->x;
        dy = im->y;
        im->x = cp - ((im->width + 1) >> 1);
        im->y = y;

        if (gw->gedittb.image_region == 0) {
            gw->gedittb.image_region = XCreateRegion();
            button.x = im->x;
            button.y = im->y;
            button.width = im->width + 4;
            button.height = im->height + 4;
            XUnionRectWithRegion(&button, gw->gedittb.image_region,
                                 gw->gedittb.image_region);
        } else {
            dx = im->x - dx;
            dy = im->y - dy;
            XOffsetRegion(gw->gedittb.image_region, dx, dy);
        }

        y += im->height + 6;
    }

    bp[_GETB_DRAW_TOGGLE].x = cp - (_GETB_TOGGLE_SIZE + 6);
    bp[_GETB_DRAW_TOGGLE].y = y;
    dx = bp[_GETB_DRAW_TOGGLE].x - toggle[0].x;
    dy = y;

    /*
     * Position the toggle polygon at the beginning.
     */
    for (i = 0; i < 5; i++) {
        toggle[i].x += dx;
        toggle[i].y += dy;
    }

    for (i = _GETB_DRAW_TOGGLE; i <= _GETB_COPY_TOGGLE; i++) {
        if (bp[i].region != 0)
          XDestroyRegion(bp[i].region);
        bp[i].region = XPolygonRegion(toggle, 5, WindingRule);

        if (i > _GETB_DRAW_TOGGLE) {
            bp[i].x = bp[i - 1].x + (_GETB_TOGGLE_SIZE + 4) + 2;
            bp[i].y = y;
        }
        for (j = 0; j < 5; j++)
          toggle[j].x += _GETB_TOGGLE_SIZE + 6;
    }
    y += _GETB_TOGGLE_SIZE + 7;

    /*
     * Set up the initial rectangle for creating button regions.
     */
    button.x = button.y = 0;
    button.width = button.height = _GETB_BUTTON_SIZE;

    /*
     * Reset the starting X coordinate for layout of the three-button rows.
     */
    x = cp - ((_GETB_BUTTON_SIZE >> 1) + _GETB_BUTTON_SIZE + 2);

    for (i = _GETB_FLIPH_BUTTON; i <= _GETB_DRIGHT_BUTTON; i++) {
        if (bp[i].region == 0) {
            bp[i].region = XCreateRegion();
            XUnionRectWithRegion(&button, bp[i].region, bp[i].region);
        }
        switch (i) {
          case _GETB_FLIPH_BUTTON:
          case _GETB_FLIPV_BUTTON:
          case _GETB_RLEFT_BUTTON:
          case _GETB_RRIGHT_BUTTON:
          case _GETB_ULEFT_BUTTON:
          case _GETB_UP_BUTTON:
          case _GETB_LEFT_BUTTON:
          case _GETB_DLEFT_BUTTON:
          case _GETB_DOWN_BUTTON:
            dx = bp[i].x;
            dy = bp[i].y;
            bp[i].x = x;
            bp[i].y = y;
            dx = bp[i].x - dx;
            dy = bp[i].y - dy;
            XOffsetRegion(bp[i].region, dx, dy);
            x += _GETB_BUTTON_SIZE + 2;
            break;
          case _GETB_RIGHT_BUTTON:
            /*
             * Adjust the X offset by the width of one
             * button and drop through.
             */
            x += _GETB_BUTTON_SIZE + 2;
          case _GETB_SHEAR_BUTTON:
          case _GETB_ROTATE_BUTTON:
          case _GETB_URIGHT_BUTTON:
          case _GETB_DRIGHT_BUTTON:
            dx = bp[i].x;
            dy = bp[i].y;
            bp[i].x = x;
            bp[i].y = y;
            dx = bp[i].x - dx;
            dy = bp[i].y - dy;
            XOffsetRegion(bp[i].region, dx, dy);
            /*
             * Reset the X coordinate and move to the next row.
             */
            x = cp - ((_GETB_BUTTON_SIZE >> 1) + _GETB_BUTTON_SIZE + 2);
            y += _GETB_BUTTON_SIZE + 2;
            break;
        }
    }
}

static void
#ifndef _NO_PROTO
_XmuttGEditTBoxSetClip(Widget w)
#else
_XmuttGEditTBoxSetClip(w)
Widget w;
#endif
{
    XmuttGlyphEditToolboxWidget gw = (XmuttGlyphEditToolboxWidget) w;
    XRectangle clip;

    clip.x = clip.y = (_XmuttGEditTBoxPMargins(gw) >> 1);
    clip.width = gw->core.width - (clip.x << 1);
    clip.height = gw->core.height - (clip.y << 1);

    XSetClipRectangles(XtDisplay(w), gw->gedittb.gc, 0, 0, &clip, 1, YSorted);
    XSetClipRectangles(XtDisplay(w), gw->gedittb.fillgc,
                       0, 0, &clip, 1, YSorted);
    XSetClipRectangles(XtDisplay(w), gw->gedittb.armgc,
                       0, 0, &clip, 1, YSorted);
    XSetClipRectangles(XtDisplay(w), gw->gedittb.armfillgc,
                       0, 0, &clip, 1, YSorted);
    XSetClipRectangles(XtDisplay(w), gw->gedittb.tsgc,
                       0, 0, &clip, 1, YSorted);
    XSetClipRectangles(XtDisplay(w), gw->gedittb.bsgc,
                       0, 0, &clip, 1, YSorted);
}

static int
#ifndef _NO_PROTO
_XmuttGEditTBoxXYToButton(XmuttGlyphEditToolboxWidget gw, int x, int y)
#else
_XmuttGEditTBoxXYToButton(gw, x, y)
XmuttGlyphEditToolboxWidget gw;
int x, y;
#endif
{
    int i, which;
    _XmuttGEditTBoxButton *bp;
    
    bp = gw->gedittb.buttons;

    if (gw->gedittb.image_region != 0 &&
        XPointInRegion(gw->gedittb.image_region, x, y))
      return _GETB_GLYPH_IMAGE;

    for (i = _GETB_DRAW_TOGGLE, which = -1;
         which == -1 && i <= _GETB_DRIGHT_BUTTON; i++) {
        if (XPointInRegion(bp[i].region, x, y) == True)
          which = i;
    }

    return which;
}

/**************************************************************************
 *
 * Class methods.
 *
 **************************************************************************/

static void
#ifndef _NO_PROTO
ClassInitialize(void)
#else
ClassInitialize()
#endif
{
}

static void
#ifndef _NO_PROTO
ClassPartInitialize (WidgetClass wc)
#else
ClassPartInitialize (wc)
WidgetClass wc;
#endif
{
}

static void
#ifndef _NO_PROTO
Initialize(Widget req, Widget newone, ArgList args, Cardinal *num_args)
#else
Initialize(req, newone, args, num_args)
Widget req, newone;
ArgList args;
Cardinal *num_args;
#endif
{
    XmuttGlyphEditToolboxWidget gw;
    XmuttGlyphEditToolboxWidgetClass gwc;
    Screen *screen;
    Dimension wd, ht;
    Pixel unused;
    int i, x, y, bpr, byte;
    _XmuttGEditTBoxButton *bp;
    bdf_bitmap_t *gimage;

    gw = (XmuttGlyphEditToolboxWidget) newone;
    gwc = (XmuttGlyphEditToolboxWidgetClass) XtClass(newone);

    gwc->gedittb_class.refcnt++;

    /*
     * Get the color used to arm the buttons.
     */
    screen = XtScreen(newone);
    XmGetColors(screen, XDefaultColormapOfScreen(screen),
                gw->core.background_pixel, &unused, &unused, &unused,
                &gw->gedittb.arm_color);

    /*
     * Intialize the timer information.
     */
    gw->gedittb.timer = 0;
    gw->gedittb.timer_button = gw->gedittb.timer_count = 0;

    /*
     * Initialize the booleans.
     */
    gw->gedittb.hasfocus = False;

    /*
     * Initialize the GC's.
     */
    gw->gedittb.gc = gw->gedittb.fillgc =
        gw->gedittb.armgc = gw->gedittb.armfillgc =
        gw->gedittb.tsgc = gw->gedittb.bsgc = 0;

    /*
     * Initialize the fields that track the buttons pressed and moved over.
     */
    gw->gedittb.pressed = gw->gedittb.last = -1;

    /*
     * Initialize the button structures.
     */
    for (bp = gw->gedittb.buttons, i = _GETB_DRAW_TOGGLE;
         i <= _GETB_DRIGHT_BUTTON; i++, bp++) {
        /*
         * Assign the context sensitive help string.
         */
        bp->help = help_strings[i];

        /*
         * Initialize the region.
         */
        bp->region = 0;

        /*
         * Set the initial coordinates to 0, 0.
         */
        bp->x = bp->y = 0;

        /*
         * The Draw button will always be initialized as being set the first
         * time.
         */
        bp->set = (i == _GETB_DRAW_TOGGLE) ? True : False;

        switch (i) {
          case _GETB_DRAW_TOGGLE:
            bp->toggle = True;
            bp->other_toggles[0] = _GETB_MOVE_TOGGLE;
            bp->other_toggles[1] = _GETB_COPY_TOGGLE;
            break;
          case _GETB_MOVE_TOGGLE:
            bp->toggle = True;
            bp->other_toggles[0] = _GETB_DRAW_TOGGLE;
            bp->other_toggles[1] = _GETB_COPY_TOGGLE;
            break;
          case _GETB_COPY_TOGGLE:
            bp->toggle = True;
            bp->other_toggles[0] = _GETB_DRAW_TOGGLE;
            bp->other_toggles[1] = _GETB_MOVE_TOGGLE;
            break;
          default:
            bp->toggle = False;
            break;
        }
    }

    /*
     * Add the glyph image help message.
     */
    bp->help = help_strings[_GETB_GLYPH_IMAGE];

    /*
     * Initialize the glyph image if it exists.
     */
    gw->gedittb.image = 0;
    gw->gedittb.image_region = 0;
    if (gw->gedittb.gimage != 0) {
        /*
         * Copy the image.
         */
        gimage = (bdf_bitmap_t *) XtMalloc(sizeof(bdf_bitmap_t));
        gimage->x = gimage->y = 0;
        gimage->width = gw->gedittb.gimage->width;
        gimage->height = gw->gedittb.gimage->height;
        gimage->bytes = gw->gedittb.gimage->bytes;
        gimage->bitmap = (unsigned char *) XtMalloc(gimage->bytes);
        (void) memcpy((char *) gimage->bitmap, gw->gedittb.gimage->bitmap,
                      gimage->bytes);

        /*
         * Shift the bits around so a pixmap can be made from the data.
         */
        bpr = (gimage->width + 7) >> 3;
        for (y = 0; y < gimage->height; y++) {
            for (x = 0; x < bpr; x++) {
                byte = gimage->bitmap[(y * bpr) + x] & 0xff;
                gimage->bitmap[(y * bpr) + x] = bitswap[byte];
            }
        }

        gw->gedittb.gimage = gimage;
    }

    /*
     * Get the initial geometry.
     */
    _XmuttGEditTBoxPreferredGeometry(gw, &wd, &ht);
    gw->core.width = wd;
    gw->core.height = ht;

    /*
     * Position the buttons.
     */
    _XmuttGEditTBoxPosition(gw);
}

static void
#ifndef _NO_PROTO
Resize(Widget w)
#else
Resize(w)
Widget w;
#endif
{
    if (XtIsRealized(w))
      _XmuttGEditTBoxSetClip(w);

    _XmuttGEditTBoxPosition((XmuttGlyphEditToolboxWidget) w);
}

static void
#ifndef _NO_PROTO
Realize(Widget w, Mask *valueMask, XSetWindowAttributes *attributes)
#else
Realize(w, valueMask, attributes)
Widget w;
Mask *valueMask;
XSetWindowAttributes *attributes;
#endif
{
    XtCreateWindow(w, (unsigned int) InputOutput, (Visual *) CopyFromParent,
                   *valueMask, attributes);

    _XmuttGEditTBoxMakeBitmaps(w);
    _XmuttGEditTBoxAssignBitmaps(w);
    _XmuttGEditTBoxMakeGCs(w, False);
    _XmuttGEditTBoxSetClip(w);
}

static void
#ifndef _NO_PROTO
Destroy(Widget w)
#else
Destroy(w)
Widget w;
#endif
{
    int i;
    XmuttGlyphEditToolboxWidget gw;
    XmuttGlyphEditToolboxWidgetClass gwc;
    _XmuttGEditTBoxButton *bp;

    gw = (XmuttGlyphEditToolboxWidget) w;
    gwc = (XmuttGlyphEditToolboxWidgetClass) XtClass(w);

    /*
     * Destroy the GCs.
     */
    if (gw->gedittb.gc != 0)
      XFreeGC(XtDisplay(w), gw->gedittb.gc);
    if (gw->gedittb.fillgc != 0)
      XFreeGC(XtDisplay(w), gw->gedittb.fillgc);
    if (gw->gedittb.armgc != 0)
      XFreeGC(XtDisplay(w), gw->gedittb.armgc);
    if (gw->gedittb.armfillgc != 0)
      XFreeGC(XtDisplay(w), gw->gedittb.armfillgc);
    if (gw->gedittb.tsgc != 0)
      XFreeGC(XtDisplay(w), gw->gedittb.tsgc);
    if (gw->gedittb.bsgc != 0)
      XFreeGC(XtDisplay(w), gw->gedittb.bsgc);

    gw->gedittb.gc = gw->gedittb.fillgc =
        gw->gedittb.armgc = gw->gedittb.armfillgc =
        gw->gedittb.tsgc = gw->gedittb.bsgc = 0;

    /*
     * Destroy the regions.
     */
    if (gw->gedittb.image_region != 0)
      XDestroyRegion(gw->gedittb.image_region);
    gw->gedittb.image_region = 0;

    bp = gw->gedittb.buttons;
    for (i = _GETB_DRAW_TOGGLE; i <= _GETB_DRIGHT_BUTTON; i++) {
        if (bp[i].region != 0)
          XDestroyRegion(bp[i].region);
        bp[i].region = 0;
    }

    /*
     * Destroy the image pixmap if it exists.
     */
    if (gw->gedittb.image != 0)
      XFreePixmap(XtDisplay(w), gw->gedittb.image);
    gw->gedittb.image = 0;

    /*
     * Destroy the glyph image if it exists.
     */
    if (gw->gedittb.gimage != 0) {
        if (gw->gedittb.gimage->bytes > 0)
          XtFree((char *) gw->gedittb.gimage->bitmap);
        XtFree((char *) gw->gedittb.gimage);
    }
    gw->gedittb.gimage = 0;

    if (gwc->gedittb_class.refcnt > 0) {
        /*
         * Decrement the class reference count.
         */
        gwc->gedittb_class.refcnt--;

        /*
         * Destroy the pixmaps if this is the last instance of this kind of
         * widget.
         */
        if (gwc->gedittb_class.refcnt == 0) {
            if (gwc->gedittb_class.draw != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.draw);
            if (gwc->gedittb_class.move != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.move);
            if (gwc->gedittb_class.copy != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.copy);
            if (gwc->gedittb_class.fliph != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.fliph);
            if (gwc->gedittb_class.flipv != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.flipv);
            if (gwc->gedittb_class.shear != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.shear);
            if (gwc->gedittb_class.rleft != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.rleft);
            if (gwc->gedittb_class.rright != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.rright);
            if (gwc->gedittb_class.rotate != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.rotate);
            if (gwc->gedittb_class.uleft != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.uleft);
            if (gwc->gedittb_class.up != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.up);
            if (gwc->gedittb_class.uright != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.uright);
            if (gwc->gedittb_class.left != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.left);
            if (gwc->gedittb_class.right != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.right);
            if (gwc->gedittb_class.dleft != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.dleft);
            if (gwc->gedittb_class.down != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.down);
            if (gwc->gedittb_class.dright != 0)
              XFreePixmap(XtDisplay(w), gwc->gedittb_class.dright);
        }
    }
}

static void
#ifndef _NO_PROTO
Redisplay(Widget w, XEvent *event, Region region)
#else
Redisplay(w, event, region)
Widget w;
XEvent *event;
Region region;
#endif
{
    int i;
    XmuttGlyphEditToolboxWidget gw;
    _XmuttGEditTBoxButton *bp;

    _XmuttGEditTBoxHighlightWindow(w);
    _XmuttGEditTBoxShadowWindow(w);

    gw = (XmuttGlyphEditToolboxWidget) w;
    bp = gw->gedittb.buttons;

    /*
     * Draw the glyph image if it exists.
     */
    _XmuttGEditTBoxDrawImage(w);

    for (i = _GETB_DRAW_TOGGLE; i <= _GETB_DRIGHT_BUTTON; i++) {
        if (bp[i].toggle == False)
          _XmuttGEditTBoxDrawButton(w, i);
        else
          _XmuttGEditTBoxDrawToggle(w, i);
    }
}

static Boolean
#ifndef _NO_PROTO
SetValues(Widget curr, Widget req, Widget newone,
          ArgList args, Cardinal *num_args)
#else
SetValues(curr, req, newone, args, num_args)
Widget curr, req, newone;
ArgList args;
Cardinal *num_args;
#endif
{
    XmuttGlyphEditToolboxWidget cgw, ngw;
    Screen *screen;
    Pixel unused;
    Dimension width, height;
    int x, y, bpr, byte;
    short ox, oy, owidth, oheight;
    bdf_bitmap_t *gimage;
    Boolean redo, repos;

    cgw = (XmuttGlyphEditToolboxWidget) curr;
    ngw = (XmuttGlyphEditToolboxWidget) newone;

    redo = repos = False;

    /*
     * Check for changes in the width and height.
     */
    if (ngw->core.width != cgw->core.width ||
        ngw->core.height != cgw->core.height)
        redo = repos = True;

    /*
     * Check if the GC's need to be changed.
     */
    if (ngw->primitive.foreground != cgw->primitive.foreground ||
        ngw->core.background_pixel != cgw->core.background_pixel) {
        /*
         * Get the new arm color.
         */
        screen = XtScreen(newone);
        XmGetColors(screen, XDefaultColormapOfScreen(screen),
                    ngw->core.background_pixel, &unused, &unused, &unused,
                    &ngw->gedittb.arm_color);
        _XmuttGEditTBoxMakeGCs(newone, True);
        redo = True;
    }

    /*
     * Check to see if the glyph image changed.
     */
    if (ngw->gedittb.gimage != cgw->gedittb.gimage) {
        repos = False;

        /*
         * The glyph image has changed.  Delete the old one, free the
         * image pixmap (if it has been created), and copy the new
         * glyph image.
         */
        ox = oy = owidth = oheight = 0;
        if (cgw->gedittb.gimage != 0) {
            /*
             * Save the original width and height to determine
             * if a reposition needs to be done.
             */
            ox = cgw->gedittb.gimage->x;
            oy = cgw->gedittb.gimage->y;
            owidth = cgw->gedittb.gimage->width;
            oheight = cgw->gedittb.gimage->height;

            if (cgw->gedittb.gimage->bytes > 0)
              XtFree((char *) cgw->gedittb.gimage->bitmap);
            XtFree((char *) cgw->gedittb.gimage);
        }
        cgw->gedittb.gimage = 0;

        if (ngw->gedittb.gimage != 0) {
            /*
             * Copy the image.
             */
            gimage = (bdf_bitmap_t *) XtMalloc(sizeof(bdf_bitmap_t));
            gimage->width = ngw->gedittb.gimage->width;
            gimage->height = ngw->gedittb.gimage->height;
            gimage->bytes = ngw->gedittb.gimage->bytes;
            gimage->bitmap = (unsigned char *) XtMalloc(gimage->bytes);
            (void) memcpy((char *) gimage->bitmap, ngw->gedittb.gimage->bitmap,
                          gimage->bytes);

            /*
             * Shift the bits around so a pixmap can be made from the data.
             */
            bpr = (gimage->width + 7) >> 3;
            for (y = 0; y < gimage->height; y++) {
                for (x = 0; x < bpr; x++) {
                    byte = gimage->bitmap[(y * bpr) + x] & 0xff;
                    gimage->bitmap[(y * bpr) + x] = bitswap[byte];
                }
            }

            ngw->gedittb.gimage = gimage;

            if (gimage->width != owidth || gimage->height != oheight) {
                /*
                 * Since a size change is taking place, figure out
                 * the new geometry.
                 */
                _XmuttGEditTBoxPreferredGeometry(ngw, &width, &height);
                ngw->core.width = width;
                ngw->core.height = height;

                /*
                 * Destroy the image region if a reposition is going to occur.
                 */
                if (ngw->gedittb.image_region != 0) {
                    XDestroyRegion(ngw->gedittb.image_region);
                    ngw->gedittb.image_region = 0;
                }
                repos = redo = True;
            } else {
                /*
                 * Restore the original X and Y coordinates of the
                 * image because it didn't change.
                 */
                gimage->x = ox;
                gimage->y = oy;
            }
        } else if (owidth || oheight)
          repos = True;

        if (cgw->gedittb.image != 0)
          XFreePixmap(XtDisplay(curr), cgw->gedittb.image);
        cgw->gedittb.image = ngw->gedittb.image = 0;

        /*
         * If the widget will not be redrawn and the new glyph image didn't
         * change size, then simply redraw the glyph image.
         */
        if (redo == False && repos == False)
          _XmuttGEditTBoxDrawImage(newone);
    }

    /*
     * Reposition the glyph image and buttons if necessary.
     */
    if (repos == True)
      _XmuttGEditTBoxPosition(ngw);

    return redo;
}

/**************************************************************************
 *
 * API.
 *
 **************************************************************************/

void
#ifndef _NO_PROTO
XmuttGlyphEditToolboxSetToggle(Widget w, int toggle, Boolean activate)
#else
XmuttGlyphEditToolboxSetToggle(w, toggle, activate)
Widget w;
int toggle;
Boolean activate;
#endif
{
    int i, o;
    XmuttGlyphEditToolboxWidget gw;
    _XmuttGEditTBoxButton *bp;
    XmuttGlyphEditToolboxCallbackStruct cb;

    _XmuttGEditTBoxCheckClass(w);

    if (toggle < XmuttGLYPHEDIT_DRAW || toggle > XmuttGLYPHEDIT_COPY)
      return;

    /*
     * Decrement the toggle index to get the usable value.
     */
    toggle--;

    gw = (XmuttGlyphEditToolboxWidget) w;

    bp = gw->gedittb.buttons;

    if (bp[toggle].set == False) {
        bp[toggle].set = True;
        if (XtIsRealized(w)) {
            /*
             * Turn off the other toggles.
             */
            for (i = 0; i < 2; i++) {
                o = bp[toggle].other_toggles[i];
                if (bp[o].set == True) {
                    bp[o].set = False;
                    _XmuttGEditTBoxDrawToggle(w, o);
                }
            }
            /*
             * Turn on the one that was set.
             */
            _XmuttGEditTBoxDrawToggle(w, toggle);
        }
    }

    /*
     * Set up and call the callback if requested.
     */
    if (activate == True) {
        cb.reason = XmuttGLYPHEDIT_OPERATION;
        cb.event = 0;
        cb.op_action = cback_values[toggle];
        XtCallCallbackList(w, gw->gedittb.activate, (XtPointer) &cb);
    }
}

int
#ifndef _NO_PROTO
XmuttGlyphEditToolboxQueryToggle(Widget w)
#else
XmuttGlyphEditToolboxQueryToggle(w)
Widget w;
#endif
{
    int i, which;
    XmuttGlyphEditToolboxWidget gw;
    _XmuttGEditTBoxButton *bp;

    _XmuttGEditTBoxCheckClass(w);

    gw = (XmuttGlyphEditToolboxWidget) w;

    bp = gw->gedittb.buttons;

    for (which = -1, i = _GETB_DRAW_TOGGLE;
         which == -1 && i <= _GETB_COPY_TOGGLE; i++) {
        if (bp[i].set == True)
          which = i;
    }
    return (which >= 0) ? cback_values[which] : which;
}

/**************************************************************************
 *
 * Action functions.
 *
 **************************************************************************/

static void
#ifndef _NO_PROTO
GainFocus(Widget w, XEvent *event, String *params, Cardinal *num_params)
#else
GainFocus(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
#endif
{
    XmuttGlyphEditToolboxWidget gw;

    /*
     * There is only one way to really gain the focus and that is
     * from a send_event.
     */
    if (!event->xfocus.send_event)
      return;

    gw = (XmuttGlyphEditToolboxWidget) w;
    gw->gedittb.hasfocus = True;

    _XmuttGEditTBoxHighlightWindow(w);
}

static void
#ifndef _NO_PROTO
LoseFocus(Widget w, XEvent *event, String *params, Cardinal *num_params)
#else
LoseFocus(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
#endif
{
    XmuttGlyphEditToolboxWidget gw;

    /*
     * There is only one way to really lose the focus and that is
     * from a send_event.
     */
    if (!event->xfocus.send_event)
      return;

    gw = (XmuttGlyphEditToolboxWidget) w;
    gw->gedittb.hasfocus = False;

    _XmuttGEditTBoxHighlightWindow(w);
}

static void
#ifndef _NO_PROTO
_XmuttGEditActivateCallback(XtPointer client_data, XtIntervalId *id)
#else
_XmuttGEditActivateCallback(client_data, id)
XtPointer client_data;
XtIntervalId *id;
#endif
{
    Widget w;
    XmuttGlyphEditToolboxWidget gw;
    XmuttGlyphEditToolboxCallbackStruct cb;

    w = (Widget) client_data;
    gw = (XmuttGlyphEditToolboxWidget) w;

    /*
     * Call the activate callback for the button in question.
     */
    cb.reason = XmuttGLYPHEDIT_OPERATION;
    cb.event = 0;
    cb.op_action = cback_values[gw->gedittb.timer_button];
    XtCallCallbackList(w, gw->gedittb.activate, (XtPointer) &cb);

    /*
     * Increment the number of times the timer caused the callback to be
     * called.
     */
    gw->gedittb.timer_count++;

    /*
     * Add the timer again, because the toolkit removed it this call.
     */
    gw->gedittb.timer = XtAppAddTimeOut(XtWidgetToApplicationContext(w),
                                        100, _XmuttGEditActivateCallback,
                                        (XtPointer) w);
}

static void
#ifndef _NO_PROTO
ArmButton(Widget w, XEvent *event, String *params, Cardinal *num_params)
#else
ArmButton(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
#endif
{
    XmuttGlyphEditToolboxWidget gw;
    _XmuttGEditTBoxButton *bp;
    int i, b, o;
    XmuttGlyphEditToolboxCallbackStruct cb;

    gw = (XmuttGlyphEditToolboxWidget) w;
    bp = gw->gedittb.buttons;

    b = _XmuttGEditTBoxXYToButton(gw, event->xbutton.x, event->xbutton.y);

    if (b < 0 || b > _GETB_DRIGHT_BUTTON)
      return;

    if (bp[b].toggle == True && bp[b].set == False) {
        for (i = 0; i < 2; i++) {
            o = bp[b].other_toggles[i];
            if (bp[o].set == True) {
                bp[o].set = False;
                _XmuttGEditTBoxDrawToggle(w, o);
            }
        }
        /*
         * Activate the toggle.
         */
        cb.reason = XmuttGLYPHEDIT_OPERATION;
        cb.event = event;
        cb.op_action = cback_values[b];
        XtCallCallbackList(w, gw->gedittb.activate, (XtPointer) &cb);
    }
    bp[b].set = True;
    if (bp[b].toggle == False) {
        /*
         * Draw the button and start the timer for continual pressing of
         * the button.
         */
        _XmuttGEditTBoxDrawButton(w, b);

        /*
         * Only add the timer to the shift buttons.
         */
        if (b >= _GETB_ULEFT_BUTTON && b <= _GETB_DRIGHT_BUTTON) {
            gw->gedittb.timer_count = 0;
            gw->gedittb.timer_button = b;
            gw->gedittb.timer =
                XtAppAddTimeOut(XtWidgetToApplicationContext(w),
                                200, _XmuttGEditActivateCallback,
                                (XtPointer) w);
        }
    } else
      _XmuttGEditTBoxDrawToggle(w, b);

    gw->gedittb.pressed = b;
}

static void
#ifndef _NO_PROTO
DisarmButton(Widget w, XEvent *event, String *params, Cardinal *num_params)
#else
DisarmButton(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
#endif
{
    int b, which;
    Boolean call_callback;
    _XmuttGEditTBoxButton *bp;
    XmuttGlyphEditToolboxWidget gw;
    XmuttGlyphEditToolboxCallbackStruct cb;

    gw = (XmuttGlyphEditToolboxWidget) w;

    /*
     * Set the flag that tells this routine whether it should call the
     * callback or if the callback was handled by the timer.
     */
    call_callback = (gw->gedittb.timer_count == 0) ? True : False;

    /*
     * If a timer exists, remove it.  Only call the activate callback
     * if the timer was on and was actually called.
     */
    if (gw->gedittb.timer != 0) {
        XtRemoveTimeOut(gw->gedittb.timer);
        gw->gedittb.timer = 0;
        gw->gedittb.timer_button = gw->gedittb.timer_count = 0;
    }

    bp = gw->gedittb.buttons;

    b = _XmuttGEditTBoxXYToButton(gw, event->xbutton.x, event->xbutton.y);

    if (b < 0 || b > _GETB_DRIGHT_BUTTON)
      return;

    which = (b != gw->gedittb.pressed) ? gw->gedittb.pressed : b;
    if (bp[which].toggle == False) {
        bp[which].set = False;
        if (which == b && call_callback == True) {
            cb.reason = XmuttGLYPHEDIT_ACTION;
            cb.event = event;
            cb.op_action = cback_values[b];
            XtCallCallbackList(w, gw->gedittb.activate, (XtPointer) &cb);
        }
        _XmuttGEditTBoxDrawButton(w, which);
    }
    gw->gedittb.pressed = -1;
}

static void
#ifndef _NO_PROTO
ContextHelp(Widget w, XEvent *event, String *params, Cardinal *num_params)
#else
ContextHelp(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
#endif
{
    XmuttGlyphEditToolboxWidget gw;
    _XmuttGEditTBoxButton *bp;
    int b;
    XmuttGlyphEditToolboxCallbackStruct cb;

    gw = (XmuttGlyphEditToolboxWidget) w;
    bp = gw->gedittb.buttons;

    b = _XmuttGEditTBoxXYToButton(gw, event->xbutton.x, event->xbutton.y);

    if (b < 0) {
        if (gw->gedittb.last != b) {
            /*
             * Cause the last message to be erased.
             */
            cb.reason = XmuttGLYPHEDIT_CONTEXT_HELP;
            cb.event = event;
            cb.help = " ";
            XtCallCallbackList(w, gw->gedittb.help, (XtPointer) &cb);
        }
        gw->gedittb.last = b;
        return;
    }

    if (b != gw->gedittb.last) {
        cb.reason = XmuttGLYPHEDIT_CONTEXT_HELP;
        cb.event = event;
        cb.help = bp[b].help;
        XtCallCallbackList(w, gw->gedittb.help, (XtPointer) &cb);
    }
    gw->gedittb.last = b;
}

static void
#ifndef _NO_PROTO
Traverse(Widget w, XEvent *event, String *params, Cardinal *num_params)
#else
Traverse(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
#endif
{
    switch (params[0][0]) {
      case 'n':
        (void) XmProcessTraversal(w, XmTRAVERSE_NEXT_TAB_GROUP);
        break;
      case 'p':
        (void) XmProcessTraversal(w, XmTRAVERSE_PREV_TAB_GROUP);
        break;
      case 'r':
        (void) XmProcessTraversal(w, XmTRAVERSE_RIGHT);
        break;
      case 'l':
        (void) XmProcessTraversal(w, XmTRAVERSE_LEFT);
        break;
      case 'u':
        (void) XmProcessTraversal(w, XmTRAVERSE_UP);
        break;
      case 'd':
        (void) XmProcessTraversal(w, XmTRAVERSE_DOWN);
        break;
    }
}

static char default_trans[] = "\
<FocusIn>: gain-focus()\n\
<FocusOut>: lose-focus()\n\
~Shift ~Ctrl ~Meta ~Alt<Key>Tab: traverse(next)\n\
Shift ~Ctrl ~Meta ~Alt<Key>Tab: traverse(prev)\n\
<Btn1Down>: arm-button()\n\
<Btn1Up>: disarm-button()\n\
<Motion>: context-help()\n\
";

static XtActionsRec actions[] = {
    {"traverse", Traverse},
    {"gain-focus", GainFocus},
    {"lose-focus", LoseFocus},
    {"arm-button", ArmButton},
    {"disarm-button", DisarmButton},
    {"context-help", ContextHelp},
};

static XtResource resources[] = {
    {
        XmNtraversalOn,
        XmCTraversalOn,
        XmRBoolean,
        sizeof(Boolean),
        XtOffset(XmPrimitiveWidget, primitive.traversal_on),
        XmRImmediate,
        (XtPointer) False,
    },
    {
        XmNnavigationType,
        XmCNavigationType,
        XmRNavigationType,
        sizeof(unsigned char),
        XtOffset(XmPrimitiveWidget, primitive.navigation_type),
        XmRImmediate,
        (XtPointer) XmTAB_GROUP,
    },
    {
        XmNcontextHelpCallback,
        XmCContextHelpCallback,
        XmRCallback,
        sizeof(XtCallbackList),
        XtOffset(XmuttGlyphEditToolboxWidget, gedittb.help),
        XmRImmediate,
        (XtPointer) 0,
    },
    {
        XmNactivateCallback,
        XmCCallback,
        XmRCallback,
        sizeof(XtCallbackList),
        XtOffset(XmuttGlyphEditToolboxWidget, gedittb.activate),
        XmRImmediate,
        (XtPointer) 0,
    },
    {
        XmNglyphImage,
        XmCGlyphImage,
        XmRPointer,
        sizeof(bdf_bitmap_t *),
        XtOffset(XmuttGlyphEditToolboxWidget, gedittb.gimage),
        XmRImmediate,
        (XtPointer) 0,
    },
};

externaldef(xmuttglyphedittboxclassrec)
XmuttGlyphEditTBoxClassRec xmuttGlyphEditTBoxClassRec = {
    {
	(WidgetClass) &xmPrimitiveClassRec,	/* superclass		*/
	"XmuttGlyphEditToolbox",		/* class_name		*/
	sizeof(XmuttGlyphEditTBoxRec),		/* widget_size		*/
        ClassInitialize,			/* class_initialize	*/
	ClassPartInitialize,			/* chained class init	*/
	FALSE,					/* class_inited		*/
	Initialize,				/* initialize		*/
        NULL,					/* initialize hook	*/
	Realize,				/* realize		*/
	actions,				/* actions		*/
	XtNumber(actions),			/* num_actions		*/
	resources,				/* resources		*/
	XtNumber(resources),			/* num_resources	*/
	NULLQUARK,				/* xrm_class		*/
	TRUE,					/* compress_motion	*/
	XtExposeCompressMaximal,		/* compress_exposure	*/
        TRUE,					/* compress enter/exit	*/
	FALSE,					/* visible_interest	*/
	Destroy,				/* destroy		*/
	Resize,					/* resize		*/
	Redisplay,				/* expose		*/
	(XtSetValuesFunc) SetValues,		/* set_values		*/
        NULL,					/* set values hook	*/
        XtInheritSetValuesAlmost,		/* set values almost	*/
        NULL,					/* get values hook	*/
	NULL,					/* accept_focus		*/
        XtVersion,				/* version		*/
        NULL,					/* callback offsetlst	*/
        default_trans,				/* default trans	*/
	XtInheritQueryGeometry,			/* query geo proc	*/
	XtInheritDisplayAccelerator,		/* display accelerator	*/
        NULL,					/* extension record reserved*/
    },

    {						/* XmPrimitive		*/
        (XtWidgetProc) _XtInherit,		/* border_highlight	*/
        (XtWidgetProc) _XtInherit,		/* border_unhighlight	*/
        NULL,					/* translations		*/
        NULL,					/* arm_and_activate	*/
        NULL,					/* syn resources	*/
        0,					/* num syn_resources	*/
        0,					/* extension		*/
    },

    {						/* XmuttGlyphEditToolbox*/
        0,					/* Draw pixmap.		*/
        0,					/* Move pixmap.		*/
        0,					/* Copy pixmap.		*/
        0,					/* Flip horiz pixmap.	*/
        0,					/* Flip vert pixmap.	*/
        0,					/* Shear pixmap.	*/
        0,					/* Rotate right pixmap.	*/
        0,					/* Rotate left pixmap.	*/
        0,					/* Rotate pixmap.	*/
        0,					/* Up-left pixmap.	*/
        0,					/* Up pixmap.		*/
        0,					/* Up-right pixmap.	*/
        0,					/* Left pixmap.		*/
        0,					/* Right pixmap.	*/
        0,					/* Down-left pixmap.	*/
        0,					/* Down pixmap.		*/
        0,					/* Down-right pixmap.	*/
        0,					/* Class reference count*/
        0,					/* Extension record pointer.*/
    }
};

externaldef(xmuttglyphedittoolboxwidgetclass)
WidgetClass xmuttGlyphEditToolboxWidgetClass =
(WidgetClass) &xmuttGlyphEditTBoxClassRec;
