// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: aquire.cc,v 1.35 1998/10/31 01:36:39 jgg Exp $
/* ######################################################################

   This is the Archive Aquire engine.

   This class is responsible for putting files into and taking files out
   of a directory. The directory to use is specified by the caller and
   must contain a partial/ sub directory. It can Aquire files from a 
   variety of sources and provides a method for external URI handlers
   to be invoked (see files.sgml for information). Communication between
   the various processes is done with pipes and a simple single line
   format.
   
   A few internal methods are provided here because they are trivial, they
   follow exactly the same rules as the external methods, but have the
   added advantage of being able to read (but not write!) to the internal
   structures.
   
   Each download directy has a lock file associated with it. When 
   OutputDir is called the directory is locked so only one process may
   write to it.
   
   ##################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#include <pkglib/aquire.h>
#include <pkglib/error.h>
#include <fileutl.h>
#include <options.h>

#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <dirent.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#include <utime.h>
#include <time.h>
#include <algo.h>
									/*}}}*/

// Aquire::Worker::Worker - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* We want the FD to be non blocking. The select loop notifies us when
   data is ready and we internally buffer the fd. */
pkgAquire::Worker::Worker(pkgAquire &Owner,pid_t Pid,int Fd) : Pid(Pid), 
                       Fd(Fd), Owner(Owner), Cur(0), From(0), To(0), Buf("")
{
   fcntl(Fd,F_SETFL,O_NONBLOCK);
};
									/*}}}*/
// Aquire::Worker::~Worker - Destructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
pkgAquire::Worker::~Worker() 
{
   close(Fd);
};
									/*}}}*/
// Aquire::Worker::Read - Read from the status fd			/*{{{*/
// ---------------------------------------------------------------------
/* This pulls single lines out of the status FD, anything that does not
   start with a letter and a space is dumped to the log. When Sz = 0
   it means the pipe is closed */
bool pkgAquire::Worker::Read()
{
   char Buffer[2048];
   strcpy(Buffer,Buf.c_str());
   int Sz = read(Fd,Buffer + Buf.length(),sizeof(Buffer) - Buf.length());
   
   // 0 size means the FD is closed
   if (Sz == 0)
      return Error();
   
   // -1 means an error
   if (Sz < 0)
   {
      // EAGAIN simply means there is no data on the FD
      if (errno != EAGAIN)
	 return Error();
      else
	 return true;
   }
   
   Sz += Buf.length();
   
   // Grok the buffer for newlines
   char *I = Buffer;
   char *Start = Buffer;
   for (; I < Buffer + Sz; I++)
   {
      if (*I == '\n')
      {
	 if (I - Start <= 2 || Start[1] != ' ' || Start[0] == ' ')
	    clog << string(Start,I-Start) << endl;
	 else
	    Line(Start[0],string(Start+2,I-Start-2));
	    
	 Start = I + 1;
	 continue;
      }
   }

   // Store the last bit
   Buf = string(Start,I - Start,Buffer + Sz - I);

   return true;
}
									/*}}}*/
// Aquire::Worker::Error - Handle an error on the Fd			/*{{{*/
// ---------------------------------------------------------------------
/* This almost always means that the process is gone. There are some
   odd problems with read returning 0 so we don't use wnohang. */
bool pkgAquire::Worker::Error()
{
   int Stat = 0;
   if (waitpid(Pid,&Stat,0) <= 0)
   {
      cout << "Waited but nothing was there" << endl;
      return true;
   }

   // Check the error code
   if (WIFEXITED(Stat) == 0 || WEXITSTATUS(Stat) != 0)
   {
      if (Cur != 0)
      {
	 Cur->Status = Item::Error;
	 Cur->ErrorText = "Bad return code from subprocess";
      }
   }
   
   // Run the done handler
   if (Cur != 0)
   {
      Owner.Log(StatDoneFile,Cur,"");
      ItemDone();
   }
   
   return false;
}
									/*}}}*/
// Aquire::Worker::Line - Handle an incomming status line		/*{{{*/
// ---------------------------------------------------------------------
/* This handles all of the lines from the method. Each line is defined
   in files.sgml and indicates what is happening with the method. */
void pkgAquire::Worker::Line(char Mode,string Line)
{
   switch (Mode)
   {
      // Changing to a new file
      case 'F':
      {
	 if (Cur != 0)
	 {
	    Owner.Log(StatDoneFile,Cur,"");
	    ItemDone();
	 }
	 
	 // Strip out the URI
	 vector<Item *>::iterator I = Owner.List.begin();
	 for (;I != Owner.List.end() && !((*I)->URI == Line); I++);
	 if (I == Owner.List.end())
	 {
	    Cur = 0;
	    clog << "Method gave incorrect URI in F tag " << Line << endl;
	    break;
	 }
	 
	 Cur = *I;
	 Cur->Status = Item::Downloading;
	 
	 /* Generate a full file name, this is so clients can easially
	    stat the file for progress */
	 OutputFile = Owner.iOutputDir + "partial/" + Cur->OutputName;
	 Owner.Log(StatNewFile,Cur,"");
	 break;
      }
      
      // Files expected size
      case 'S':
      {
	 if (Cur == 0)
	    break;
	 if (Cur->ExpectedSize == 0)
	    Cur->ExpectedSize = atoi(Line.c_str());
	 else
	 {
	    if ((signed)Cur->ExpectedSize != atoi(Line.c_str()))
	    {
	       Cur->Status = Item::Error;
	       Cur->ErrorText = "The size of the file is not what I expected";
	    }
	 }
	 break;
      }

      // MD5 result
      case 'M':
      {
	 if (Cur == 0 || Cur->MD5Sum.empty() == true)
	    break;
	 if (!(Line == Cur->MD5Sum))
	 {
	    Cur->Status = Item::Error;
	    Cur->ErrorText = "Incorrect MD5Sum";
	 }
	 break;
      }
      
      // Error text
      case 'E':
      {
	 // Put the error text on every item in the group
	 if (Cur == 0)
	 {
	    Owner.Log(StatError,0,Line);
	    if (From == 0 || To == 0)
	       break;
	    for (vector<Item *>::iterator I = From; I != To; I++)
	    {
	       (*I)->ErrorText = Line;
	       (*I)->Status = Item::Error;
	    }
	    break;
	 }
	 
	 if (Cur->Status == Item::Error)
	    break;
	 
	 Cur->ErrorText = Line;
	 Cur->Status = Item::Error;
	 Owner.Log(StatError,Cur,Line);
	 break;
      }

      // Reference a Fsys file
      case 'R':
      {
	 if (Cur == 0)
	    break;

	 if (FileExists(Line) == false)
	 {
	    Cur->ErrorText = string("Can't open ") + Line;
	    Cur->Status = Item::Error;
	    break;
	 }
	 
	 // Method is giving a path but we dont want that..
	 if (Cur->FinalPath == 0)
	 {
	    // Less than ideal
	    CopyFile(Line,Cur->OutputName);
	    break;
	 }

	 *Cur->FinalPath = Line;
	 Owner.Log(StatReference,Cur,Line);
	 break;
      }
      
      // Information
      case 'I':
      Info = Line;
      
      Owner.Log(StatInfo,Cur,Info);
      break;

      // Log line
      case 'L':
      break;
      
      default:
      clog << "Method gave unknown mode '" << Mode << "'" << endl;
      clog << Line << endl;
   };
}
									/*}}}*/
// Aquire::Worker::ItemDone - Called on an item transition		/*{{{*/
// ---------------------------------------------------------------------
/* This is called whenever the worker task finishes a URI. It implements
   the core of the state machine. Things can go forwars in Status
   skipping over some irrelivant steps depending on the setup. */
void pkgAquire::Worker::ItemDone()
{
   if (Cur->ErrorText.empty() == false)
   {
      Cur->Status = Item::Error;
      return;
   }
      
   if (Owner.CdPartial(Cur->ErrorText) == false)
   {
      Cur->Status = Item::Error;
      return;
   }

   string To = string("../") + Cur->OutputName;
   switch (Cur->Status)
   {
      case Item::Error:
      case Item::Waiting:
      case Item::Complete:
      break;
	 
      // The file just finished downloading
      case Item::Downloading:
      {
	 /* We check if the output file exists, and .. exists
	    then it hasnt been changed */
	 struct stat Buf;
	 if ((Cur->FinalPath == 0 || Cur->FinalPath->empty() == true) &&
	     stat(Cur->OutputName.c_str(),&Buf) != 0)
	 {
	    // Check if an uncompressed version was transfered
	    if (Cur->CompressSuffix.empty() == false &&
		stat((Cur->OutputName + ".decomp").c_str(),&Buf) == 0)
	    {
	       Cur->Status = Item::Decompressing;
	       ItemDone();
	       break;
	    }
	    
	    if (stat(To.c_str(),&Buf) == 0)
	    {
	       Cur->Status = Item::Complete;

	       if (Cur->ExpectedSize == 0)
		  Cur->ExpectedSize = Buf.st_size;
	       break;
	    }
	    else
	    {
	       // No file was downloaded and no file already exists.
	       Cur->Status = Item::Error;
	       Cur->ErrorText = "Download failed without error";
	       break;
	    }
	 }
	 else
	    if (Cur->FinalPath != 0)
	       stat(Cur->FinalPath->c_str(),&Buf);
	 
	 // Check for decompression
	 if (Cur->CompressSuffix.empty() == true)
	 {
	    if (Cur->FinalPath != 0 && Cur->FinalPath->empty() == false)
	    {
	       Cur->Status = Item::Complete;
	       break;
	    }
	    
	    Cur->Status = Item::Partial;
	    ItemDone();
	    break;
	 }
	 
	 Cur->Status = Item::Decompressing;
	 Cur->Time.actime = Buf.st_atime;
	 Cur->Time.modtime = Buf.st_mtime;	 
	 
	 /* Rename to the compressed name or directly use the final path
	    file */
	 string CompName = Cur->OutputName;;
	 if (Cur->FinalPath != 0 && Cur->FinalPath->empty() == false)
	    CompName = *Cur->FinalPath;
	 	    
	 if (Cur->CompressSuffix == ".gz")
	 {
	    Cur->Status = Item::Decompressing;
	    Owner.RunDecompressor("gzip","-dc",CompName,Cur);
	 }
	 else
	    _error->Error("Unkown compression type %s",Cur->CompressSuffix.c_str());
	 break;
      }

      // The file finished decompressing, output is in .decomp
      case Item::Decompressing:
      {
	 string Out = Cur->OutputName + ".decomp";
	 utime(Out.c_str(),&Cur->Time);
	 
	 // Move out of the partial dir and into the completed section
	 if (rename(Out.c_str(),To.c_str()) != 0)
	 {
	    Cur->Status = Item::Error;
	    Cur->ErrorText = "Unable to move out of the partial directory";
	 }
	 else
	    Cur->Status = Item::Complete;
	 
	 unlink(Cur->OutputName.c_str());
	 
	 // Store the final path.
	 if (Cur->FinalPath != 0)
	    *Cur->FinalPath = Cur->OutputName;
	 break;
      }
	 
      case Item::Partial:
      {
	 // Move out of the partial dir and into the completed section
	 Cur->Status = Item::Partial;
	 if (rename(Cur->OutputName.c_str(),To.c_str()) != 0)
	 {
	    Cur->Status = Item::Error;
	    Cur->ErrorText = "Unable to move out of the partial directory";
	 }
	 else
	    Cur->Status = Item::Complete;
	 
	 // Store the final path.
	 if (Cur->FinalPath != 0)
	    *Cur->FinalPath = Cur->OutputName;
	 
	 break;
      }      
   };

   chdir(Owner.StartDir.c_str());
   return;
}
									/*}}}*/

// Aquire::Get - Enqueue an item					/*{{{*/
// ---------------------------------------------------------------------
/* If the URI ends in .gz but Output doesn't then it will be decompressed.
   If size is not 0 the final file size must exactly match the given size
   If MD5Sum is not empty then a MD5 check will be performed 
   If FinalPath is 0 then all files are copied into the current dir 
   otherwise FinalPath is set relative to the OutputDirectory.
 
   Decompression and a non-zero Final path are exclusive. */
bool pkgAquire::Get(string URI,string Output,string GetInfo,
		    unsigned long Size,string MD5Sum,string *FinalPath)
{
   // We can not mix update and archive requests
   if (URI.empty() == true)
      return false;

   for (vector<Item *>::iterator I = List.begin(); I != List.end(); I++)
      if ((*I)->URI == URI)
	 return _error->Error("Attempt to fetch the same file twice %s",URI.c_str());

   /* It is a file URI, we can directly grab the file in some cases. If it
      can't be found then we just fall through to the full aquire. This
      saves an extra stat and prevents things from being logged */
   if (FinalPath != 0 && strncmp("file:",URI.c_str(),5) == 0)
   {
      string Path(URI,5);
      
      struct stat Buf;
      if (stat(Path.c_str(),&Buf) == 0 && (unsigned)Buf.st_size == Size)
      {
	 *FinalPath = Path;
	 return true;
      }      
   }

   /* Check if the file already exists, if a md5 and a size is given
      that means that the file is unique (by filename!) and it's
      existance means it is the newest/best/etc version */
   if (MD5Sum.empty() == false || Size != 0)
   {
      // File exists and the size is ok.
      struct stat Buf;
      if (stat((iOutputDir+Output).c_str(),&Buf) == 0 && 
	  (unsigned)Buf.st_size == Size)
      {
	 if (FinalPath != 0)
	    *FinalPath = Output;
	 return true;
      }
      
      /* Hmm, a size difference should not be possible, it would mean
         non unique file names or someone futzed with the archives dir! */
      unlink((iOutputDir+Output).c_str());
   }
   
   if (FinalPath != 0)
      *FinalPath = "";

   Item *Itm = new Item;
   Itm->URI = URI;
   Itm->OutputName = Output;
   Itm->ExpectedSize = Size;
   Itm->MD5Sum = MD5Sum;
   Itm->FinalPath = FinalPath;
   Itm->GetInfo = GetInfo;
   iFetchBytes += Size;
   
   // Strip off the suffixes of both strings
   string::size_type UriEnd = URI.rfind('.');
   string::size_type OutEnd = Output.rfind('.');
   string Suffix;
   if (UriEnd != string::npos)
      Suffix = string(URI,UriEnd);
   string OSuffix;
   if (OutEnd != string::npos)
      OSuffix = string(Output,OutEnd);
   
   // Check if we should be decompressing
   if (!(Suffix == OSuffix))
   {
      // Decompression can directly use final path to advoid a copy.
      Itm->FinalPath = &Itm->CompFinalPath;
      Itm->CompressSuffix = Suffix;
   }
   
   // Estimate the size based on the existing file.
   if (Itm->ExpectedSize == 0)
   {
      struct stat Buf;
      if (stat((iOutputDir+Output).c_str(),&Buf) == 0)
	 Itm->EstimatedSize = Buf.st_size;
   }
   
   List.push_back(Itm);
   return true;
}
									/*}}}*/
// Aquire::Run - Run the queue						/*{{{*/
// ---------------------------------------------------------------------
/* This initiates the whole process, Worker::ItemDone picks up where this
   leaves off. It forks off all of the method handlers in parrallel and
   then returns. */
static int Sorter(const void *a,const void *b)
{
   pkgAquire::Item &A = **(pkgAquire::Item **)a;
   pkgAquire::Item &B = **(pkgAquire::Item **)b;

   if (B < A)
      return 1;
   if (A < B)
      return -1;
   return 0;
}
bool pkgAquire::Run()
{
   // Stash the current dir.
   StartDir = SafeGetCWD();

   // Check for enough free space
   struct statfs Buf;
   if (statfs(iOutputDir.c_str(),&Buf) != 0)
      return _error->Errno("statfs","Couldn't determine free space in %s",
			   iOutputDir.c_str());
   if (unsigned(Buf.f_bfree) < iFetchBytes/Buf.f_bsize)
      return _error->Error("Sorry, you don't have enough free space in %s",
			   iOutputDir.c_str());

   qsort(List.begin(),List.size(),sizeof(Item *),Sorter);
   for (vector<Item *>::iterator I = List.begin(); I != List.end();)
   {
      // Strip the scheme
      string Scheme((*I)->URI,0,(*I)->URI.find(':'));
      string Unique((*I)->URI,0,(*I)->URI.find('/',Scheme.length()+3));
      
      vector<Item *>::iterator J = I + 1;
      for (;J != List.end(); J++)
      {
	 if (Unique.length() > (*J)->URI.length())
	    break;
	 
	 if (strncmp(Unique.begin(),(*J)->URI.begin(),Unique.length()) != 0)
	    break;
      }
      
      // IPC channel to our subprocess..
      int Files[2];
      if (pipe(Files) < 0)
	 return _error->Errno("pipe","Couldn't setup IPC");
      SetCloseExec(Files[0],true);
      SetCloseExec(Files[1],false);
      
      // Create the subprocess..
      pid_t P = fork();
      if (P < 0)
      {
	 close(Files[0]);
	 close(Files[1]);
	 return _error->Errno("fork","Couldn't fork URI handler");
      }
      
      // Main process records the child
      if (P > 0)
      {
	 close(Files[1]);
	 Worker *Work = new Worker(*this,P,Files[0]);
	 Work->From = I;
	 Work->To = J;
	 RegisterWorker(Work);
	 I = J;
	 continue;
      }

      string Error;
      if (CdPartial(Error) == false)
      {
	 cout << "E " << Error << endl;
	 exit(100);
      }
      
      // Internal file scheme
      if (Scheme == "file")
      {
	 cout.rdbuf(new filebuf(Files[1]));
	 for (;I != J; I++)
	    if ((*I)->Status == Item::Waiting)
	       SchemeFile(string((*I)->URI,(*I)->URI.find(':')+1),**I);
	 
	 exit(0);
      }
      
      // External scheme
      string Method = string(PKG_DEB_LB_METHODS) + Scheme;
      char const **Args = new const char *[(J - I)*2 + 3];
      char const **Cur = Args;
      
      // Setup the args
      *Cur++ = Method.c_str();
      for (vector<Item *>::iterator K = I;K != J; K++)
      {
	 if ((*K)->Status != Item::Waiting)
	    continue;
	 
	 *Cur++ = (*K)->URI.c_str();
	 *Cur++ = (*K)->OutputName.c_str();
      }
      *Cur = 0;
      
      // Setup the FDs
      dup2(Files[1],STDOUT_FILENO);
      dup2(((filebuf *)clog.rdbuf())->fd(),STDERR_FILENO);
      close(Files[1]);
      close(Files[0]);
      SetCloseExec(STDOUT_FILENO);
      SetCloseExec(STDERR_FILENO);
      execv(Args[0],(char **)Args);

      // Exec error
      cout << "E Unable to exec scheme " << Method << endl;      
      exit(0);
   }

   iFetchBytes = 0;
   return true;
}
									/*}}}*/
// Aquire::SchemeFile - The file scheme					/*{{{*/
// ---------------------------------------------------------------------
/* In Update mode this IMS checks the source file and if changed/new copies
   into partial. In Archive mode this size checks the file and sets FinalPath
   to point to the full location in the fs. The cwd should be the partial
   dir. 
 
   This should return false in the case of an extruciatingly important
   error, for which there is none in this case*/
bool pkgAquire::SchemeFile(string Loc,Item &Itm)
{
   cout << "F " << Itm.URI << endl;

   if (Loc[0] != '/')
   {
      cout << "E File schemes must have absolute paths" << endl;
      return false;
   }

   // We look for uncompressed versions if decompressing is being used
   if (Itm.CompressSuffix.empty() == false)
   {
      string Uncomp(Loc,0,Loc.length() - Itm.CompressSuffix.length());
      struct stat Src;
      if (stat(Uncomp.c_str(),&Src) == 0)
      {
	 string Out = Itm.OutputName + ".decomp";
	 
	 // We have to copy the file into partial
	 if (CopyFile(Uncomp,Out) == false)
	 {
	    cout << "E Unable to copy " << Uncomp << " to " << Out << endl;
	    return true;
	 }
	 
	 // Copy over the modification time
	 struct utimbuf Buf;
	 Buf.actime = Src.st_atime;
	 Buf.modtime = Src.st_mtime;
	 utime(Out.c_str(),&Buf);
	 return true;
      }	 
   }
   
   // Soft fail, if it is important then the caller will bitch.
   struct stat Src;
   if (stat(Loc.c_str(),&Src) != 0)
   {      
      cout << "E Unable to stat " << Loc << ' ' << strerror(errno) << endl;
      return false;
   }

   // Caller will be happy with a generated path..
   if (Itm.FinalPath != 0)
   {
      cout << "S " << Src.st_size << endl;      
      cout << "R " << Loc << endl;
      return true;
   }
   
   // Attempt to stat the old file (../)
   struct stat Old;
   string OldName = "../" + Itm.OutputName;
   if (stat(OldName.c_str(),&Old) == 0)
   {
      // We dont compare sizes when decompressing.
      if (Old.st_mtime == Src.st_mtime &&
	  (Old.st_size == Src.st_size || Itm.CompressSuffix.empty() == false))
	 return true;
   }
   
   cout << "S " << Src.st_size << endl;
   
   // We have to copy the file into partial
   if (CopyFile(Loc,Itm.OutputName) == false)
   {
      cout << "E Unable to copy " << Loc << " to " << Itm.OutputName << endl;
      return true;
   }
   
   // Copy over the modification time
   struct utimbuf Buf;
   Buf.actime = Src.st_atime;
   Buf.modtime = Src.st_mtime;
   utime(Itm.OutputName.c_str(),&Buf);
   
   return true;
}
									/*}}}*/
// Aquire::RegisterWorker - Register a worker class			/*{{{*/
// ---------------------------------------------------------------------
/* This is a trivial limited version that just blocks the caller till the
   worker is done. */
bool pkgAquire::RegisterWorker(Worker *Work)
{
   while (Work->Read() == true)
      sleep(1);
   delete Work;
   return true;
}
									/*}}}*/
// Aquire::RunDecompressor - Run a decompression program		/*{{{*/
// ---------------------------------------------------------------------
/* The decompressor is forked off and has its stdin set to the source
   file and its stdout set to source+.decomp. This allows good handling
   of error conditions and resume support in the methods. We attach
   a communication pipe that the method will close when it exits thus
   providing a quick way to reap it. */
bool pkgAquire::RunDecompressor(string Program,string Args,string File,
				Item *For)
{
   /* IPC channel to our subprocess, the compressors do not use this but
      it allows us to know when the process died */
   int Files[2];
   if (pipe(Files) < 0)
      return _error->Errno("pipe","Couldn't setup IPC");
   SetCloseExec(Files[0],true);
   SetCloseExec(Files[1]);

   string OutN = For->OutputName + ".decomp";
   int Out = open(OutN.c_str(),O_WRONLY | O_TRUNC | O_CREAT,0666);
   if (Out < 0 )
   {
      close(Files[0]);
      close(Files[1]);
      return _error->Errno("open","Couldn't create/open %s",OutN.c_str());
   }
   
   int In = open(File.c_str(),O_RDONLY);
   if (In < 0 )
   {
      close(Files[0]);
      close(Files[1]);
      close(Out);
      return _error->Errno("open","Couldn't open %s",For->OutputName.c_str());
   }
   
   Log(StatNewFile,For,"");
      
   // Create the subprocess..
   pid_t P = fork();
   if (P < 0)
   {
      close(Files[0]);
      close(Files[1]);
      close(Out);
      close(In);
      return _error->Errno("fork","Couldn't fork URI handler");
   }
   
   // Main process records the child
   if (P > 0)
   {
      close(Files[1]);
      close(In);
      close(Out);
      Worker *Work = new Worker(*this,P,Files[0]);
      Work->Cur = For;
      RegisterWorker(Work);
      return true;
   }

   dup2(Out,STDOUT_FILENO);
   dup2(In,STDIN_FILENO);
   dup2(((filebuf *)clog.rdbuf())->fd(),STDERR_FILENO);
   close(Out);
   close(In);
   SetCloseExec(STDIN_FILENO);
   SetCloseExec(STDOUT_FILENO);
   SetCloseExec(STDERR_FILENO);
   SetCloseExec(Files[1]);
   execlp(Program.c_str(),Program.c_str(),Args.c_str(),0);
   
   cout.rdbuf(new filebuf(Files[1]));
   cout << "E Unable to exec " << Program << endl;
   exit(1);
}
									/*}}}*/
// Aquire::DoClean - Remove old files					/*{{{*/
// ---------------------------------------------------------------------
/* This purges all files that are not mentioned in the current get list.
   It does this in the partial dir and the download dir. */
bool pkgAquire::DoClean(string Dir)
{
   DIR *D = opendir(Dir.c_str());   
   if (D == 0)
      return _error->Errno("opendir","Unable to read %s",Dir.c_str());
   
   if (chdir(Dir.c_str()) != 0)
   {
      closedir(D);
      return _error->Errno("chdir","Unable to change to ",Dir.c_str());
   }
   
   for (struct dirent *Dir = readdir(D); Dir != 0; Dir = readdir(D))
   {
      // Skip some files..
      if (strcmp(Dir->d_name,"lock") == 0 ||
	  strcmp(Dir->d_name,"partial") == 0 ||
	  strcmp(Dir->d_name,".") == 0 ||
	  strcmp(Dir->d_name,"..") == 0)
	 continue;
      
      // Look in the get list
      vector<Item *>::iterator I = List.begin();
      for (; I != List.end(); I++)
	 if ((*I)->OutputName == Dir->d_name)
	    break;
      
      // Nothing found, nuke it
      if (I == List.end())
	 unlink(Dir->d_name);
   };
   
   chdir(StartDir.c_str());
   closedir(D);
   return true;
}
									/*}}}*/
// Aquire::Clean - Clean the directories				/*{{{*/
// ---------------------------------------------------------------------
/* Calls DoClean on both the final directory and the partial dir to 
   wipe all the old gunk out. */
bool pkgAquire::Clean()
{
   return DoClean(iOutputDir) && DoClean(iOutputDir + "partial/");
}
									/*}}}*/
// Aquire::CdParital - Change to the partial directory			/*{{{*/
// ---------------------------------------------------------------------
/* Error handling cover for chdir */
bool pkgAquire::CdPartial(string &Error)
{
   if (chdir((iOutputDir + "partial/").c_str()) != 0)
   {
      Error = string("Unable to chdir to ") + iOutputDir  + "partial/";
      return false;
   }
   
   return true;
}
									/*}}}*/
// Aquire::OutputDir - Change the output directory			/*{{{*/
// ---------------------------------------------------------------------
/* */
bool pkgAquire::OutputDir(string Dir)
{
   if (Dir.length() == 0 || Dir[Dir.length() - 1] != '/')
      return false;
   
   // Store the starting directory
   StartDir = SafeGetCWD();
   
   // Make sure the partial dir and parent dir exist
   string Error;
   iOutputDir = Dir;
   if (CdPartial(Error) == false)
   {
      iOutputDir = "";
      return _error->Errno("chdir","%s",Error.c_str());
   }
   
   // Create the lock file
   close(Lock);
   Lock = -1;
   Lock = GetLock("../lock");
   chdir(StartDir.c_str());
   
   if (Lock == -1)
      return _error->Error("Couldn't lock the Output dir, %s another process "
			   "is using it",Dir.c_str());
   
   return true;
}
									/*}}}*/
// Aquire::SetCloseExec - Set the close on exec flag			/*{{{*/
// ---------------------------------------------------------------------
/* For some reason the flag occasionaly does not get set right.. odd. */
void pkgAquire::SetCloseExec(int Fd,bool Close)
{
   if (fcntl(Fd,F_SETFD,(Close == false)?0:FD_CLOEXEC) != 0)
   {
      cerr << "FATAL -> Could not set close on exec " << strerror(errno) << endl;
      exit(100);
   }   
}
									/*}}}*/
// Aquire::Item::Item - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
pkgAquire::Item::Item() : Status(Waiting), ExpectedSize(0), 
                          EstimatedSize(0), FinalPath(0)
{
}
									/*}}}*/
// Aquire::Item << - Write the item to a stream				/*{{{*/
// ---------------------------------------------------------------------
/* */
ostream &operator <<(ostream &O,pkgAquire::Item &Itm)
{
   O << Itm.URI << " -> " << Itm.OutputName;
   return O;
}
									/*}}}*/
