#include "ckcsym.h"
#ifdef NOLOCAL
char *connv = "";
#else
char *connv = "CONNECT Command for UNIX:select(), 6.1.093, 23 April 1998";

/*  C K U C N S  --  Terminal connection to remote system, for UNIX  */
/*
  Author: Frank da Cruz <fdc@columbia.edu>,
  Columbia University Academic Information Systems, New York City.

  Copyright (C) 1998, Trustees of Columbia University in the City of New
  York.  The C-Kermit software may not be, in whole or in part, licensed or
  sold for profit as a software product itself, nor may it be included in or
  distributed with commercial products or otherwise distributed by commercial
  concerns to their clients or customers without written permission of the
  Office of Kermit Development and Distribution, Columbia University.  This
  copyright notice must not be removed, altered, or obscured.
*/

/*
  This version of the UNIX CONNECT module uses select(), which is required for
  Kerberos encryption.  Thus it can be used only on UNIX systems that support
  select() on both TCP/IP and serial connections.  A separate module that uses
  a completely portable fork() structure can be used on systems where select()
  is not available or does not work as required.
*/

#include "ckcdeb.h"			/* Common things first */
#include <errno.h>			/* Error numbers */
#ifdef __linux__
#include <sys/time.h>			/* For FD_blah */
#endif /* __linux__ */

/* Kermit-specific includes */

#include "ckcasc.h"			/* ASCII characters */
#include "ckcker.h"			/* Kermit things */
#include "ckucmd.h"			/* For xxesc() prototype */
#include "ckcnet.h"			/* Network symbols */
#ifndef NOCSETS
#include "ckcxla.h"			/* Character set translation */
#endif /* NOCSETS */

#ifdef BEBOX 
#include <kernel/OS.h>
#include <socket.h>
#include <signal.h>
#include <stdio.h>
#endif /* BEBOX */

/* Internal function prototypes */

_PROTOTYP( VOID ttflux, (void) );
_PROTOTYP( VOID doesc, (char) );
_PROTOTYP( static VOID logchar, (char) );
_PROTOTYP( int hconne, (void) );
#ifndef NOSHOW
_PROTOTYP( VOID shomdm, (void) );
#endif /* NOSHOW */
_PROTOTYP( static int kbget, (void) );
_PROTOTYP( static int ckcputf, (void) );
_PROTOTYP( static VOID ck_sndmsg, (void) );

/* External variables */

extern struct ck_p ptab[];

extern int local, escape, duplex, parity, flow, seslog, sessft, debses,
 mdmtyp, ttnproto, cmask, cmdmsk, network, nettype, deblog, sosi, tnlm,
 xitsta, what, ttyfd, ttpipe, quiet, backgrd, pflag, tt_crd, tn_nlm, ttfdflg,
 tt_escape, justone, carrier;

extern long speed;
extern char ttname[], sesfil[], myhost[], *ccntab[];
#ifdef TNCODE
extern int tn_b_nlm, me_binary, u_binary;
#endif /* TNCODE */

#ifdef CK_TRIGGER
extern char * tt_trigger[], * triggerval;
#endif /* CK_TRIGGER */

extern int nopush;

#ifdef CK_APC
extern int apcactive;			/* Application Program Command (APC) */
extern int apcstatus;			/* items ... */
static int apclength = 0;          
#ifdef DCMDBUF
extern char *apcbuf;
#else
extern char apcbuf[];
#endif /* DCMDBUF */
static int apcbuflen = APCBUFLEN - 2;
extern int autodl, protocol;		/* Auto download */
#endif /* CK_APC */

#ifdef CK_AUTODL
extern CHAR ksbuf[];
#endif /* CK_AUTODL */

#ifdef CK_ENCRYPTION
extern int me_auth;
#endif /* CK_ENCRYPTION */

#ifdef CK_XYZ
#ifdef XYZ_INTERNAL
static int zmdlok = 1;			/* Zmodem autodownloads available */
#else
static int zmdlok = 0;			/* Depends on external protocol def */
#endif /* XYZ_INTERNAL */
#else
static int zmdlok = 0;			/* Not available at all */
#endif /* CK_XYZ */

#ifndef NOSETKEY			/* Keyboard mapping */
extern KEY *keymap;			/* Single-character key map */
extern MACRO *macrotab;			/* Key macro pointer table */
static MACRO kmptr = NULL;		/* Pointer to current key macro */
#endif /* NOSETKEY */

/* Global variables local to this module */

static int
  active = 0,
  quitnow = 0,				/* <esc-char>Q was typed */
  dohangup = 0,				/* <esc-char>H was typed */
  inshift = 0,				/* SO/SI shift states */
  outshift = 0;

static char ecbuf[10], *ecbp;		/* Escape char buffer & pointer */

#ifdef CK_SMALL
#define IBUFL 1536			/* Input buffer length */
#else
#define IBUFL 4096
#endif /* CK_SMALL */

static int obc = 0;			/* Output buffer count */

#ifndef OXOS
#define OBUFL 1024			/* Output buffer length */
#else
#define OBUFL IBUFL
#endif /* OXOS */

#define TMPLEN 200			/* Temporary message buffer length */
#ifdef DYNAMIC
static char *ibuf = NULL, *obuf = NULL, *temp = NULL; /* Buffers */
#else
static char ibuf[IBUFL], obuf[OBUFL], temp[TMPLEN];
#endif /* DYNAMIC */

#ifdef DYNAMIC
static char *ibp;			/* Input buffer pointer */
#else
static char *ibp = ibuf;		/* Input buffer pointer */
#endif /*DYNAMIC */
static int ibc = 0;			/* Input buffer count */

#ifdef DYNAMIC
static char *obp;			/* Output buffer pointer */
#else
static char *obp = obuf;		/* Output buffer pointer */
#endif /* DYNAMIC */

/* Character-set items */

#ifndef NOCSETS
#ifdef CK_ANSIC /* ANSI C prototypes... */
extern CHAR (*xls[MAXTCSETS+1][MAXFCSETS+1])(CHAR); /* Character set */
extern CHAR (*xlr[MAXTCSETS+1][MAXFCSETS+1])(CHAR); /* translation functions */
static CHAR (*sxo)(CHAR);	/* Local translation functions */
static CHAR (*rxo)(CHAR);	/* for output (sending) terminal chars */
static CHAR (*sxi)(CHAR);	/* and for input (receiving) terminal chars. */
static CHAR (*rxi)(CHAR);
#else /* Not ANSI C... */
extern CHAR (*xls[MAXTCSETS+1][MAXFCSETS+1])();	/* Character set */
extern CHAR (*xlr[MAXTCSETS+1][MAXFCSETS+1])();	/* translation functions. */
static CHAR (*sxo)();		/* Local translation functions */
static CHAR (*rxo)();		/* for output (sending) terminal chars */
static CHAR (*sxi)();		/* and for input (receiving) terminal chars. */
static CHAR (*rxi)();
#endif /* CK_ANSIC */
extern int language;		/* Current language. */
static int langsv;		/* For remembering language setting. */
extern struct csinfo fcsinfo[]; /* File character set info. */
extern int tcsr, tcsl;		/* Terminal character sets, remote & local. */
static int tcs;			/* Intermediate ("transfer") character set. */
#endif /* NOCSETS */

/*
  We do not need to parse and recognize escape sequences if we are being built
  without character-set support AND without APC support.
*/
#ifdef NOCSETS				/* No character sets */
#ifndef CK_APC				/* No APC */
#ifndef NOESCSEQ
#define NOESCSEQ			/* So no escape sequence recognizer */
#endif /* NOESCSEQ */
#endif /* CK_APC */
#endif /* NOCSETS */

/* Child process events and messages */

#define CEV_NO  0			/* No event */
#define CEV_HUP 1			/* Communications hangup */
#define CEV_PAD 2			/* X.25 - change PAD parameters */
#define CEV_DUP 3			/* Toggle duplex */
#define CEV_APC 4			/* Execute APC */
#ifdef TNCODE
#define CEV_MEBIN 5			/* Change of me_binary */
#define CEV_UBIN 6			/* Change of u_binary */
#endif /* TNCODE */
#define CEV_ADL 7			/* Autodownload */
#define CEV_AUL 8			/* Autoupload */
#define CEV_TRI 9			/* Trigger string */

#ifdef NOESCSEQ
#define chkaes(x) 0
#else
/*
  As of edit 178, the CONNECT command skips past ANSI escape sequences to
  avoid translating the characters within them.  This allows the CONNECT
  command to work correctly with a host that uses a 7-bit ISO 646 national
  character set, in which characters like '[' would normally be translated
  into accented characters, ruining the terminal's interpretation (and
  generation) of escape sequences.

  As of edit 190, the CONNECT command responds to APC escape sequences
  (ESC _ text ESC \) if the user SETs TERMINAL APC ON or UNCHECKED, and the
  program was built with CK_APC defined.

  Non-ANSI/ISO-compliant escape sequences are not handled.
*/

/* States for the escape-sequence recognizer. */
  
#define ES_NORMAL 0			/* Normal, not in an escape sequence */
#define ES_GOTESC 1			/* Current character is ESC */
#define ES_ESCSEQ 2			/* Inside an escape sequence */
#define ES_GOTCSI 3			/* Inside a control sequence */
#define ES_STRING 4			/* Inside DCS,OSC,PM, or APC string */
#define ES_TERMIN 5			/* 1st char of string terminator */

static int
  escseq = 0,				/* 1 = Recognizer is active */
  inesc = ES_NORMAL,			/* State of sequence recognizer */
  oldesc = -1;				/* Previous state of recognizer */
/*
  ANSI escape sequence handling.  Only the 7-bit form is treated, because
  translation is not a problem in the 8-bit environment, in which all GL
  characters are ASCII and no translation takes place.  So we don't check
  for the 8-bit single-character versions of CSI, DCS, OSC, APC, or ST.
  Here is the ANSI sequence recognizer state table, followed by the code
  that implements it.

  Definitions:
    CAN = Cancel                       01/08         Ctrl-X
    SUB = Substitute                   01/10         Ctrl-Z
    DCS = Device Control Sequence      01/11 05/00   ESC P
    CSI = Control Sequence Introducer  01/11 05/11   ESC [
    ST  = String Terminator            01/11 05/12   ESC \
    OSC = Operating System Command     01/11 05/13   ESC ]
    PM  = Privacy Message              01/11 05/14   ESC ^
    APC = Application Program Command  01/11 05/15   ESC _

  ANSI escape sequence recognizer:

    State    Input  New State  ; Commentary

    NORMAL   (start)           ; Start in NORMAL state

    (any)    CAN    NORMAL     ; ^X cancels
    (any)    SUB    NORMAL     ; ^Z cancels

    NORMAL   ESC    GOTESC     ; Begin escape sequence
    NORMAL   other             ; NORMAL control or graphic character

    GOTESC   ESC               ; Start again
    GOTESC   [      GOTCSI     ; CSI
    GOTESC   P      STRING     ; DCS introducer, consume through ST
    GOTESC   ]      STRING     ; OSC introducer, consume through ST
    GOTESC   ^      STRING     ; PM  introducer, consume through ST
    GOTESC   _      STRING     ; APC introducer, consume through ST
    GOTESC   0..~   NORMAL     ; 03/00 through 17/14 = Final character
    GOTESC   other  ESCSEQ     ; Intermediate or ignored control character

    ESCSEQ   ESC    GOTESC     ; Start again
    ESCSEQ   0..~   NORMAL     ; 03/00 through 17/14 = Final character
    ESCSEQ   other             ; Intermediate or ignored control character

    GOTCSI   ESC    GOTESC     ; Start again
    GOTCSI   @..~   NORMAL     ; 04/00 through 17/14 = Final character
    GOTCSI   other             ; Intermediate char or ignored control char

    STRING   ESC    TERMIN     ; Maybe have ST
    STRING   other             ; Consume all else

    TERMIN   \      NORMAL     ; End of string
    TERMIN   other  STRING     ; Still in string
*/
/*
  chkaes() -- Check ANSI Escape Sequence.
  Call with EACH character in input stream.
  Sets global inesc variable according to escape sequence state.
  Returns 0 normally, 1 if an APC sequence is to be executed.
*/
int
#ifdef CK_ANSIC
chkaes(char c)
#else
chkaes(c) char c;
#endif /* CK_ANSIC */
/* chkaes */ {

    debug(F101,"chkaes c","",c);

    oldesc = inesc;			/* Remember previous state */
    if (c == CAN || c == SUB)		/* CAN and SUB cancel any sequence */
      inesc = ES_NORMAL;
    else				/* Otherwise */
      switch (inesc) {			/* enter state switcher */

	case ES_NORMAL:			/* NORMAL state */
	  if (c == ESC)			/* Got an ESC */
	    inesc = ES_GOTESC;		/* Change state to GOTESC */
	  break;			/* Otherwise stay in NORMAL state */

	case ES_GOTESC:			/* GOTESC state */
	  if (c == '[')			/* Left bracket after ESC is CSI */
	    inesc = ES_GOTCSI;		/* Change to GOTCSI state */
	  else if (c == 'P' || (c > 0134 && c < 0140)) { /* P, [, ^, or _ */
	      inesc = ES_STRING;	/* Switch to STRING-absorption state */
#ifdef CK_APC
	      if (c == '_' && apcstatus != APC_OFF) { /* If APC not disabled */
		  debug(F100,"CONNECT APC begin","",0);
		  apcactive = APC_REMOTE; /* Set APC-Active flag */
		  apclength = 0;	/* and reset APC buffer pointer */
	      }
#endif /* CK_APC */
	  } else if (c > 057 && c < 0177) /* Final character '0' thru '~' */
	    inesc = ES_NORMAL;		/* Back to normal */
	  else if (c != ESC)		/* ESC in an escape sequence... */
	    inesc = ES_ESCSEQ;		/* starts a new escape sequence */
	  break;			/* Intermediate or ignored ctrl char */

	case ES_ESCSEQ:			/* ESCSEQ -- in an escape sequence */
	  if (c > 057 && c < 0177)	/* Final character '0' thru '~' */
	    inesc = ES_NORMAL;		/* Return to NORMAL state. */
	  else if (c == ESC)		/* ESC ... */
	    inesc = ES_GOTESC;		/* starts a new escape sequence */
	  break;			/* Intermediate or ignored ctrl char */

	case ES_GOTCSI:			/* GOTCSI -- In a control sequence */
	  if (c > 077 && c < 0177)	/* Final character '@' thru '~' */
	    inesc = ES_NORMAL;		/* Return to NORMAL. */
	  else if (c == ESC)		/* ESC ... */
	    inesc = ES_GOTESC;		/* starts over. */
	  break;			/* Intermediate or ignored ctrl char */

	case ES_STRING:			/* Inside a string */
	  if (c == ESC)			/* ESC may be 1st char of terminator */
	    inesc = ES_TERMIN;		/* Go see. */
#ifdef CK_APC
	  else if (apcactive && (apclength < apcbuflen)) /* If in APC, */
	    apcbuf[apclength++] = c;	/* deposit this character. */
	  else {			/* Buffer overrun */
	      apcactive = 0;		/* Discard what we got */
	      apclength = 0;		/* and go back to normal */
	      apcbuf[0] = 0;		/* Not pretty, but what else */
	      inesc = ES_NORMAL;	/* can we do?  (ST might not come) */
	  }
#endif /* CK_APC */
	  break;			/* Absorb all other characters. */

	case ES_TERMIN:			/* May have a string terminator */
	  if (c == '\\') {		/* which must be backslash */
	      inesc = ES_NORMAL;	/* If so, back to NORMAL */
#ifdef CK_APC
	      if (apcactive) {		/* If it was an APC string, */
		  debug(F101,"CONNECT APC terminated","",c);
		  apcbuf[apclength] = NUL; /* terminate it and then ... */
		  return(1);
	      }
#endif /* CK_APC */
	  } else {			/* Otherwise */
	      inesc = ES_STRING;	/* Back to string absorption. */
#ifdef CK_APC
	      if (apcactive && (apclength+1 < apcbuflen)) { /* In APC string */
		  apcbuf[apclength++] = ESC; /* deposit the Esc character */
		  apcbuf[apclength++] = c;   /* and this character too */
	      }
#endif /* CK_APC */
	  }
      }
    return(0);
}
#endif /* NOESCSEQ */

/*  C K C P U T C  --  C-Kermit CONNECT Put Character to Screen  */
/*
  Output is buffered to avoid slow screen writes on fast connections.
  NOTE: These could (easily?) become macros ...
*/
static int
ckcputf() {				/* Dump the output buffer */
    int x = 0;
    if (obc > 0)			/* If we have any characters, */
      x = conxo(obc,obuf);		/* dump them, */
    obp = obuf;				/* reset the pointer */
    obc = 0;				/* and the counter. */
    return(x);				/* Return conxo's return code */
}

int
ckcputc(c) int c; {
    int x;

    *obp++ = c & 0xff;			/* Deposit the character */
    obc++;				/* Count it */
    if (ibc == 0 ||			/* If input buffer about empty */
	obc == OBUFL) {			/* or output buffer full */
	debug(F101,"CONNECT CKCPUTC obc","",obc);
	x = conxo(obc,obuf);		/* dump the buffer, */
	obp = obuf;			/* reset the pointer */
	obc = 0;			/* and the counter. */
	return(x);			/* Return conxo's return code */
    } else return(0);
}

/*  C K C G E T C  --  C-Kermit CONNECT Get Character  */
/*
  Buffered read from communication device.
  Returns the next character, refilling the buffer if necessary.
  On error, returns ttinc's return code (see ttinc() description).
  Dummy argument for compatible calling conventions with ttinc()
  so a pointer to this function can be passed to tn_doop().
*/
int
ckcgetc(dummy) int dummy; {
    int c, n;
  
#ifdef CK_ENCRYPTION
    /* No buffering for possibly encrypted connections */
    if (network && ttnproto == NP_TELNET && me_auth)
      return(ttinc(0));
#endif /* CK_ENCRYPTION */

    if (ibc < 1) {			/* Need to refill buffer? */
	ibc = 0;			/* Yes, reset count */
	ibp = ibuf;			/* and buffer pointer */
	c = ttinc(0);			/* Read one character, blocking */
	if (c < 0) {			/* If error, return error code */
	    return(c);
	} else {			/* Otherwise, got one character */
	    *ibp++ = c;			/* Advance buffer pointer */
	    ibc++;			/* and count. */
	}
	if ((n = ttchk()) > 0) {	/* Any more waiting? */
	    if (n > (IBUFL - ibc))	/* Get them all at once. */
	      n = IBUFL - ibc;		/* Don't overflow buffer */
	      if ((n = ttxin(n,(CHAR *)ibp)) > 0)
		ibc += n;		/* Advance counter */
	} else if (n < 0) {		/* Error? */
	    return(n);			/* Return the error code */
	}
	ibp = ibuf;			/* Point to beginning of buffer */
    }
    c = *ibp++ & 0xff;			/* Get next character from buffer */
    ibc--;				/* Reduce buffer count */
    /* debug(F000,"CKCGETC","",c); */
    return(c);				/* Return the character */
}

/*
   Keyboard handling, buffered for speed, which is needed when C-Kermit is
   in CONNECT mode between two other computers that are transferring data.
*/
static char *kbp;			/* Keyboard input buffer pointer */
static int kbc;				/* Keyboard input buffer count */

#ifdef CK_SMALL				/* Keyboard input buffer length */
#define KBUFL 32			/* Small for PDP-11 UNIX */
#else
#define KBUFL 257			/* Regular kernel size for others */
#endif /* CK_SMALL */

#ifdef DYNAMIC
static char *kbuf = NULL;
#else
static char kbuf[KBUFL];
#endif /* DYNAMIC */

/* Macro for reading keystrokes. */

#define CONGKS() (((--kbc)>=0) ? ((int)(*kbp++) & 0377) : kbget())

/*
  Note that we call read() directly here, normally a no-no, but in this case
  we know it's UNIX and we're only doing what coninc(0) would have done,
  except we're reading a block of characters rather than just one.  There is,
  at present, no conxin() analog to ttxin() for chunk reads, and instituting
  one would only add function-call overhead as it would only be a wrapper for
  a read() call anyway.
*/
/*
  Another note: We stick in this read() till the user types something.
  But the other (lower) fork is running too, and on TELNET connections,
  it will signal us to indicate echo-change negotiations, and this can
  interrupt the read().  Some UNIXes automatically restart the interrupted
  system call, others return from it with errno == EINTR.
*/
static int				/* Keyboard buffer filler */
kbget() {
#ifdef EINTR
    int tries = 10;			/* If read() is interrupted, */
    int ok = 0;
    while (tries-- > 0) {		/* try a few times... */
#endif /* EINTR */
	if ((kbc = conchk()) < 1)	/* How many chars waiting? */
	  kbc = 1;			/* If none or dunno, wait for one. */
	else if (kbc > KBUFL)		/* If too many, */
	  kbc = KBUFL;			/* only read this many. */
	if ((kbc = read(0, kbuf, kbc)) < 1) { /* Now read it/them. */
	    debug(F101,"CONNECT kbget errno","",errno);	/* Got an error. */
#ifdef EINTR
	    if (errno == EINTR)		/* Interrupted system call. */
	      continue;			/* Try again, up to limit. */
	    else			/* Something else. */
#endif /* EINTR */
	      return(-1);		/* Pass along read() error. */
	}
#ifdef EINTR
	else { ok = 1; break; }
    }
    if (!ok) return(-1);
#endif /* EINTR */
    kbp = kbuf;				/* Adjust buffer pointer, */
    kbc--;				/* count, */
    return((int)(*kbp++) & 0377);	/* and return first character. */
}

#ifdef BEBOX
/*
 *--------------------------------------------------------------
 *
 * CreateSocketPair --
 *
 *	This procedure creates a connected socket pair
 *
 * Results:
 *	0 if OK, the error if not OK.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int
socketpair(int *pair) {
    int servsock;
    int val;
    struct sockaddr_in serv_addr, cli_addr;
    extern char myipaddr[];

    debug(F110,"socketpair",myipaddr,0);

    if (myipaddr[0] == 0)
      getlocalipaddr();

    servsock = socket(AF_INET, SOCK_STREAM, 0);
    if (servsock == 0) {
	return h_errno;
    }
    debug(F111,"socketpair","socket",servsock);

    memset(&serv_addr, 0, sizeof(serv_addr));
    serv_addr.sin_family = AF_INET;
    serv_addr.sin_addr.s_addr = htonl(INADDR_ANY);
    serv_addr.sin_port = htons(0);

    val = sizeof(serv_addr);
    if (bind(servsock, (struct sockaddr *) &serv_addr, val) < 0) {
	closesocket(servsock);
	return h_errno;
    }
    debug(F111,"socketpair","bind",0);

    listen(servsock, 1);
    debug(F111,"socketpair","listen",0);

    if (getsockname(servsock, (struct sockaddr *) &serv_addr, &val) < 0) {
	closesocket(servsock);
	return h_errno;
    }
    debug(F111,"socketpair","getsockname",0);

    pair[0] = socket(AF_INET, SOCK_STREAM, 0);
    if (pair[0] == 0) {
	closesocket(servsock);
	return h_errno;
    }
    debug(F111,"socketpair","socket",pair[0]);

    memset(&cli_addr, 0, sizeof(cli_addr));
    cli_addr.sin_family = AF_INET;
    cli_addr.sin_addr.s_addr = inet_addr(myipaddr[0]?myipaddr:"127.0.0.1");
    cli_addr.sin_port = serv_addr.sin_port;

    if (connect(pair[0],(struct sockaddr *) &cli_addr, sizeof(cli_addr)) < 0) {
	closesocket(pair[0]);
	closesocket(servsock);
	return h_errno;
    }
    debug(F111,"socketpair","connect",0);

    pair[1] = accept(servsock, (struct sockaddr *) &serv_addr, &val);
    if (pair[1] == 0) {
	closesocket(pair[0]);
	closesocket(servsock);
	return h_errno;
    }
    debug(F111,"socketpair","accept",pair[1]);

    closesocket(servsock);
    debug(F111,"socketpair","closesocket",0);
    return 0;
}

long
kbdread(void * param) {
    int sock = (int) param;
    char ch;
    int rc = 0;

    debug(F111,"kbdread","sock",sock);
    
    while (rc >= 0) {
	rc = read(fileno(stdin), &ch, 1); /* Read a character. */
	if (rc > 0) {
	    rc = send(sock,&ch,1,0);
	    /* debug(F000,"kbdread","send()",ch); */
	    printf("\r\ngot: %c rc = %d\r\n",ch,rc);
	} else
	  msleep(100);
    }
    debug(F110,"kbdread","terminating",0);
    return(rc);
}
#endif /* BEBOX */

int
conect() {
    int rc = 0;				/* Return code: 0 = fail, 1 = OK */
    int x = 0;
    register int c, c2, csave;		/* Characters */
#ifdef TNCODE
    int tx;				/* For Telnet negotiations */
#endif /* TNCODE */
#ifdef CK_APC
    int apcrc;				/* For APC */
#endif /* CK_APC */
    int n, kbin, scrnout;		/* select() items... */
    fd_set in, out;			/* File descriptor sets */
    int gotnet = 0;			/* Flag for net ready to read */
    int gotkbd = 0;			/* Flag for keyboard ready to read */
#ifdef BEBOX
    int tid = 0;			/* Thread ID */
    int pair[2];			/* Socket Pair */
    CHAR ch;
    CHAR buf[64];
#endif /* BEBOX */

#ifdef BEBOX
    {
	/* Create a socket pair to be used for the keyboard input */
	if (socketpair(pair)) {
	    debug(F110,"conect","unable to create socket pair",0);
	    return(-1);
	}
	debug(F111,"connect","socket pair[0]",pair[0]);
	debug(F111,"connect","socket pair[1]",pair[1]);

	/* Assign one end of the socket to kbin */
	kbin = pair[0];
        tid = spawn_thread(kbdread,
			   "Kbd to Socket Pair", 
			    B_NORMAL_PRIORITY,
			   (void *)pair[1]
			   );
        resume_thread(tid);
	debug(F110,"connect","tid",tid);
    }
#else /* BEBOX */
    kbin = fileno(stdin);		/* stdin file descriptor */
#endif /* BEBOX */

    scrnout = fileno(stdout);		/* stdout file descriptor */

    ttimoff();				/* Turn off any timer interrupts */
    if (!local) {			/* Be sure we're not in remote mode */
#ifdef NETCONN
	printf("Sorry, you must SET LINE or SET HOST first\n");
#else
	printf("Sorry, you must SET LINE first\n");
#endif /* NETCONN */
	return(0);
    }
    if (speed < 0L && network == 0 && ttfdflg == 0) {
	printf("Sorry, you must SET SPEED first\n");
	return(0);
    }
#ifdef TCPSOCKET
    if (network && !ttpipe && (nettype != NET_TCPB)) {
	printf("Sorry, network type not supported\n");
	return(0);
    }
#endif /* TCPSOCKET */

#ifdef DYNAMIC
    if (!ibuf) {
	if (!(ibuf = malloc(IBUFL+1))) { /* Allocate input line buffer */
	    printf("Sorry, CONNECT input buffer can't be allocated\n");
	    return(0);
	} else {
	    ibp = ibuf;
	    ibc = 0;
	}
    }
    if (!obuf) {
	if (!(obuf = malloc(OBUFL+1))) {    /* Allocate output line buffer */
	    printf("Sorry, CONNECT output buffer can't be allocated\n");
	    return(0);
	} else {
	    obp = obuf;
	    obc = 0;
	}
    }
    if (!kbuf) {
	if (!(kbuf = malloc(KBUFL+1))) { /* Allocate keyboard input buffer */
	    printf("Sorry, CONNECT keyboard buffer can't be allocated\n");
	    return(0);
	}
    }
    if (!temp) {
	if (!(temp = malloc(TMPLEN+1))) { /* Allocate temporary buffer */
	    printf("Sorry, CONNECT temporary buffer can't be allocated\n");
	    return(0);
	}
    }
#else
    obp = obuf;
    obc = 0;
#endif /* DYNAMIC */

    kbp = kbuf;				/* Always clear these. */
    *kbp = NUL;				/* No need to preserve them between */
    kbc = 0;				/* CONNECT sessions. */

#ifdef DEBUG
    if (deblog) {
	debug(F101,"CONNECT conect entry ttyfd","",ttyfd);
	debug(F101,"CONNECT conect entry ibc","",ibc);
	debug(F101,"CONNECT conect entry obc","",obc);
	debug(F101,"CONNECT conect entry kbc","",kbc);
#ifdef CK_TRIGGER
	debug(F110,"CONNECT conect trigger",tt_trigger[0],0);
#endif /* CK_TRIGGER */
	if (ttyfd > -1) {
	    n = ttchk();
	    debug(F101,"CONNECT conect entry ttchk","",n);
	}
    }    
#endif /* DEBUG */    

    if (ttyfd < 0) {			/* If communication device not open */
	debug(F101,"CONNECT ttnproto","",ttnproto);
	debug(F111,"CONNECT opening",ttname,0); /* Open it now */
	if (ttopen(ttname,
		   &local,
		   network ? -nettype : mdmtyp,
		   0
		   ) < 0) {
	    sprintf(temp,"Sorry, can't open %s",ttname);
	    perror(temp);
	    debug(F110,"CONNECT open failure",ttname,0);
	    return(0);
	}
    }
    dohangup = 0;			/* Hangup not requested yet */

    if (!quiet
#ifdef CK_APC
	&& !apcactive
#endif /* CK_APC */
	) {
#ifdef NETCONN
	if (network) {
#ifdef CK_ENCRYPTION
	    extern int me_encrypt, u_encrypt;
	    if (ck_krb_encrypting() && ck_krb_decrypting())
	      printf("SECURE connection to host %s",ttname);
	    else
#endif /* CK_ENCRYPTION */
	      if (ttpipe)
		printf("Connecting via command \"%s\"",ttname);
	      else
		printf("Connecting to host %s",ttname);
	} else {
#endif /* NETCONN */
	    printf("Connecting to %s",ttname);
	    if (speed > -1L) printf(", speed %ld",speed);
#ifdef NETCONN
	}
#endif /* NETCONN */
	if (tt_escape) {
	    printf(".\r\nThe escape character is Ctrl-%c (ASCII %d, %s)\r\n",
		   ctl(escape), escape,
		   (escape == 127 ? "DEL" : ccntab[escape]));
	    printf("Type the escape character followed by C to get back,\r\n");
	    printf("or followed by ? to see other options.\r\n");
	} else {
	    printf(".\r\n\nESCAPE CHARACTER IS DISABLED\r\n\n");
	}
	if (seslog) {
	    printf("(Session logged to %s, ",sesfil);
	    printf("%s)\r\n", sessft ? "binary" : "text");
	}
	if (debses) printf("Debugging Display...)\r\n");
	fflush(stdout);
    }

/* Condition console terminal and communication line */	    

    if (conbin((char)escape) < 0) {
	printf("Sorry, can't condition console terminal\n");
	return(0);
    }
    debug(F101,"CONNECT cmask","",cmask);
    debug(F101,"CONNECT cmdmsk","",cmdmsk);
    debug(F101,"CONNECT speed before ttvt","",speed);
    if ((n = ttvt(speed,flow)) < 0) {	/* Enter "virtual terminal" mode */
	if (!network) {
	    debug(F101,"CONNECT ttvt","",n);
	    tthang();			/* Hang up and close the device. */
	    ttclos(0);
	    if (ttopen(ttname,		/* Open it again... */
		       &local,
		       network ? -nettype : mdmtyp,
		       0
		       ) < 0) {
		sprintf(temp,"Sorry, can't reopen %s",ttname);
		perror(temp);
		return(0);
	    }
	    if (ttvt(speed,flow) < 0) {	/* Try virtual terminal mode again. */
		conres();		/* Failure this time is fatal. */
		printf("Sorry, Can't condition communication line\n");
		return(0);
	    }
	}
    }
    debug(F101,"CONNECT ttvt ok, escape","",escape);

    debug(F101,"CONNECT carrier-watch","",carrier);
    if (!network && (carrier != CAR_OFF)) {
	int x;
	x = ttgmdm();
	debug(F100,"CONNECT ttgmdm","",x);
	if ((x > -1) && !(x & BM_DCD)) {
	    debug(F100,"CONNECT ttgmdm CD test fails","",x);
	    conres();
	    printf("?Carrier required but not detected.\n");
	    printf(" To CONNECT to a serial device that is not presenting\n");
	    printf(" the Carrier Detect signal, first tell C-Kermit to\n");
	    printf(" SET CARRIER-WATCH OFF.\n\n");
	    return(0);
	}
	debug(F100,"CONNECT ttgmdm ok","",0);
    }
#ifndef NOCSETS
/* Set up character set translations */

    tcs = gettcs(tcsr,tcsl);		/* Get intermediate set. */

    if (tcsr == tcsl) {			/* Remote and local sets the same? */
	sxo = rxo = NULL;		/* If so, no translation. */
	sxi = rxi = NULL;
    } else {				/* Otherwise, set up */
	sxo = xls[tcs][tcsl];		/* translation function */
	rxo = xlr[tcs][tcsr];		/* pointers for output functions */
	sxi = xls[tcs][tcsr];		/* and for input functions. */
	rxi = xlr[tcs][tcsl];
    }
/*
  This is to prevent use of zmstuff() and zdstuff() by translation functions.
  They only work with disk i/o, not with communication i/o.  Luckily Russian
  translation functions don't do any stuffing...
*/
    langsv = language;
#ifndef NOCYRIL
    if (language != L_RUSSIAN)
#endif /* NOCYRIL */
      language = L_USASCII;

#ifdef COMMENT
#ifdef DEBUG
    if (deblog) {
	debug(F101,"CONNECT tcs","",tcs);
	debug(F101,"CONNECT tcsl","",tcsl);
	debug(F101,"CONNECT tcsr","",tcsr);
	debug(F101,"CONNECT fcsinfo[tcsl].size","",fcsinfo[tcsl].size);
	debug(F101,"CONNECT fcsinfo[tcsr].size","",fcsinfo[tcsr].size);
    }
#endif /* DEBUG */
#endif /* COMMENT */

#ifdef CK_XYZ
#ifndef XYZ_INTERNAL
    {
	extern int binary;		/* See about ZMODEM autodownloads */
	char * s;
	s = binary ? ptab[PROTO_Z].p_b_rcmd : ptab[PROTO_Z].p_t_rcmd;
	if (!s) s = "";
	zmdlok = (*s != NUL);		/* OK if we have external commands */
    }
#endif /* XYZ_INTERNAL */
#endif /* CK_XYZ */

#ifndef NOESCSEQ
/*
  We need to activate the escape-sequence recognition feature when:
   (a) translation is elected, AND
   (b) the local and/or remote set is a 7-bit set other than US ASCII.
  Or:
   SET TERMINAL APC is not OFF (handled in the next statement).
*/
    escseq = (tcs != TC_TRANSP) &&	/* Not transparent */
      (fcsinfo[tcsl].size == 128 || fcsinfo[tcsr].size == 128) && /* 7 bits */
	(fcsinfo[tcsl].code != FC_USASCII); /* But not ASCII */
#endif /* NOESCSEQ */
#endif /* NOCSETS */

#ifndef NOESCSEQ
#ifdef CK_APC
    escseq = escseq || (apcstatus != APC_OFF);
    apcactive = 0;			/* An APC command is not active */
    apclength = 0;			/* ... */
#endif /* CK_APC */
    inesc = ES_NORMAL;			/* Initial state of recognizer */
    debug(F101,"CONNECT escseq","",escseq);
#endif /* NOESCSEQ */

    FD_ZERO(&in);			/* Clear select() structs */
    FD_ZERO(&out);
    active = 1;

    while (active) {			/* Big loop... */

	gotkbd = 0;
	gotnet = ttpeek();		/* Something sitting in ckutio buf */

	if (
#ifndef NOSETKEY
	    !kmptr			/* Check for key macro active */
#else
	    1
#endif /* NOSETKEY */
	    ) {
	    if (obc) {			/* No key macro - set up for select */
		FD_SET(ttyfd, &out);	/* Have stuff to send to net */
	    } else {
		FD_SET(kbin, &in);	/* Need to read stuff from keyboard */
	    }
#ifdef BEBOX 
	    if (!(ibc || gotnet))
		FD_SET(ttyfd, &in);	/* Need to read stuff from net */
#else /* BEBOX */
	    if (ibc || gotnet) {
		FD_SET(scrnout, &out);	/* Have stuff to put on screen */
	    } else {
		FD_SET(ttyfd, &in);	/* Need to read stuff from net */
	    }
#endif /* BEBOX */

	    /* Wait till the first one of the above is ready for i/o */

	    c = select(16, &in, &out, 0, 0);
	    debug(F101,"CONNECT select()","",c);
	    if (c < 1) {
#ifdef EINTR
		if (c == -1) {
		    if (errno == EINTR) {
			continue;
		    }
		}
#endif /* EINTR */
		sleep(1);
		continue;
	    }
#ifndef BEBOX 
	    if (FD_ISSET(scrnout, &out)) {
		debug(F100,"CONNECT SELECT scrnout","",0);
	    }
#endif /* BEBOX */
	    if (FD_ISSET(ttyfd, &in)) {	/* Read from net? */
		debug(F100,"CONNECT SELECT ttyfd","",0);
		FD_CLR(ttyfd, &in);
		gotnet = 1;		/* Net is ready */
	    }
	    if (FD_ISSET(kbin, &in)) {	/* Read from keyboard? */
		debug(F100,"CONNECT SELECT kbin","",0);
		FD_CLR(kbin, &in);
		gotkbd = 1;		/* Keyboard is ready */
	    }
	}
	debug(F101,"CONNECT gotkbd","",gotkbd);
	debug(F101,"CONNECT kbc","",kbc);
	debug(F101,"CONNECT kmptr","",kmptr);
	while (gotkbd || kbc > 0 /* If we have keyboard chars */
#ifndef NOSETKEY
	       || kmptr
#endif /* NOSETKEY */
	       ) {
#ifndef NOSETKEY
	    if (kmptr) {		/* Have current macro? */
		debug(F100,"CONNECT kmptr non NULL","",0);
		if ((c = (CHAR) *kmptr++) == NUL) { /* Get char from it */
		    kmptr = NULL;	/* If no more chars,  */
		    debug(F100,"CONNECT macro empty, continuing","",0);
		    continue;	/* reset pointer and continue */
		}
		debug(F000,"CONNECT char from macro","",c);
	    } else {			/* No macro... */
#endif /* NOSETKEY */
#ifdef BEBOX
		{
		    int rc = 0;
		    if ((rc = recv(kbin,buf,1,0)) > 0)
		      c = buf[0];
		    else
		      c = -1;
		    debug(F111,"recv","rc",rc);
		    printf("\r\nrecv: %c rc=%d\r\n",buf[0],rc);
		}
#else /* BEBOX */
		c = CONGKS();		/* Yes, read from keyboard */
#endif /* BEBOX */
		gotkbd = 0;		/* Turn off select() result flag */
#ifndef NOSETKEY
	    }
#endif /* NOSETKEY */
	    if (c == -1) {
#ifdef EINTR
		if (errno == EINTR)
		  continue;
#endif /* EINTR */
		conoc(BEL);
		goto conret0;
	    }
	    c &= cmdmsk;		/* Do any requested masking */
#ifndef NOSETKEY
/*
  Note: kmptr is NULL if we got character c from the keyboard, and it is
  not NULL if it came from a macro.  In the latter case, we must avoid
  expanding it again.
*/
	    if (!kmptr && macrotab[c]) { /* Macro definition for c? */
		debug(F000,"CONNECT macro key",macrotab[c],c);
		kmptr = macrotab[c];	/* Yes, set up macro pointer */
		continue;		/* and restart the loop, */
	    } else c = keymap[c];	/* else use single-char keymap */
#endif /* NOSETKEY */
	    if (
#ifndef NOSETKEY
		!kmptr &&
#endif /* NOSETKEY */
		(tt_escape && (c & 0x7f) == escape)) { /* Escape char? */
		debug(F000,"CONNECT got escape","",c);
#ifdef BEBOX
		if (recv(kbin,buf,1,0)>=0)
		  c = buf[0];
		else
		  c = -1;
#else /* BEBOX */
		c = CONGKS() & 0177;	/* Read argument */
#endif /* BEBOX */
		doesc((char) c);	/* Handle it */
		continue;		/* Back to loop */
	    }
	    csave = c;			/* Save it before translation */
					/* for local echoing. */
#ifndef NOCSETS
#ifdef NOESCSEQ
	    /* Translate character sets */
	    if (sxo) c = (*sxo)(c); /* From local to intermediate. */
	    if (rxo) c = (*rxo)(c); /* From intermediate to remote. */
#else
	    if (inesc == ES_NORMAL) { /* If not inside escape seq.. */
		/* Translate character sets */
		if (sxo) c = (*sxo)((char)c); /* Local-intermediate */
		if (rxo) c = (*rxo)((char)c); /* Intermediate-remote */
	    }
	    if (escseq)
	      apcrc = chkaes((char)c);

#endif /* NOESCSEQ */
#endif /* NOCSETS */
/*
 If Shift-In/Shift-Out is selected and we have a 7-bit connection,
 handle shifting here.
*/
	    if (sosi) {			/* Shift-In/Out selected? */
		if (cmask == 0177) {	/* In 7-bit environment? */
		    if (c & 0200) {	/* 8-bit character? */
			if (outshift == 0) { /* If not shifted, */
			    ttoc(dopar(SO)); /* shift. */
			    outshift = 1;
			}
		    } else {
			if (outshift == 1) { /* 7-bit character */
			    ttoc(dopar(SI)); /* If shifted, */
			    outshift = 0;    /* unshift. */
			}
		    }
		}
		if (c == SO) outshift = 1;   /* User typed SO */
		if (c == SI) outshift = 0;   /* User typed SI */
	    }
	    c &= cmask;			/* Apply Kermit-to-host mask now. */
	    if (c == '\015') {		/* Carriage Return */
		int stuff = -1;
		if (tnlm) {		/* TERMINAL NEWLINE ON */
		    stuff = LF; 	/* Stuff LF */
#ifdef TNCODE
		} else if (network &&	/* TELNET NEWLINE ON/OFF/RAW */
			   (ttnproto == NP_TELNET)) {
		    switch (!me_binary ? tn_nlm : tn_b_nlm) {
		      case TNL_CRLF:
			stuff = LF;
			break;
		      case TNL_CRNUL:
			stuff = NUL;
			break;
		    }
#endif /* TNCODE */
		}
		if (stuff > -1) {
		    ttoc(dopar('\015'));	/* Send CR */
		    if (duplex) conoc('\015');	/* Maybe echo CR */
		    c = stuff;			/* Char to stuff */
		    csave = c;
		}
	    }
#ifdef TNCODE
/* If user types the 0xff character (TELNET IAC), it must be doubled. */
	    else		/* Not CR */
	      if ((dopar((CHAR) c) == IAC) && /* IAC (0xff) */
		  network && (ttnproto == NP_TELNET)) {	/* Send one copy now */
		  ttoc((char)IAC); /* and the other one just below. */
	      }
#endif /* TNCODE */
	    /* Send the character */

	    if (ttoc((char)dopar((CHAR) c)) > -1) {
		if (duplex) {		/* If half duplex, must echo */
		    if (debses)
		      conol(dbchr(csave)); /* the original char */
		    else		/* not the translated one */
		      conoc((char)csave);
		    if (seslog) {	/* And maybe log it too */
			c2 = csave;
			if (sessft == 0 && csave == '\r')
			  c2 = '\n';
			logchar((char)c2);
		    }
		}
	    } else {
		perror("\r\nCan't send character");
		active = 0;
		break;
	    }
	}
	if (FD_ISSET(ttyfd, &out)) {
	    FD_CLR(ttyfd, &out);
	}
	while (gotnet || ibc > 0) {
	    gotnet = 0;
	    c = ckcgetc(0);		/* Get next character */
	    /* debug(F101,"CONNECT c","",c); */
	    if (c < 0) {		/* Failed... */
		ckcputf();		/* Flush CONNECT output buffer */
		if (!quiet) {
		    printf("\r\nCommunications disconnect ");
		    if ( c == -3
#ifdef ultrix
/* This happens on Ultrix if there's no carrier */
			&& errno != EIO
#endif /* ultrix */
#ifdef UTEK
/* This happens on UTEK if there's no carrier */
			&& errno != EWOULDBLOCK
#endif /* UTEK */
			)
		      perror("\r\nCan't read character");
		}
#ifdef NOSETBUF
		fflush(stdout);
#endif /* NOSETBUF */
		tthang();		/* Hang up the connection */
		debug(F111,"CONNECT i/o error 1",ck_errstr(),errno);
		goto conret0;
	    }
#ifdef TNCODE
	    tx = 0;
	    if ((c == IAC) && network && (ttnproto == NP_TELNET)) {
#ifdef CK_ENCRYPTION
		int x_auth = me_auth;
#else
		int x_auth = 0;
#endif /* CK_ENCRYPTION */
		int me_bin = me_binary;
		int u_bin = u_binary;
		debug(F100,"CONNECT got IAC","",0);
		ckcputf();		/* Dump screen-output buffer */
		if ((tx = tn_doop((CHAR)(c & 0xff),duplex,ckcgetc)) == 0) {
		    if (me_bin != me_binary) {
			me_bin = me_binary;
		    } else if (u_bin != u_binary) {
			u_bin = u_binary;
#ifdef CK_ENCRYPTION
/* 
  Here we have to push back any bytes we have read using block reads, so we
  can read them again using single-character reads, so they can be decrypted
  in case there was a switch to encryption in the block.  Note that we can't
  handle switches in the encryption state itself this way -- which would be
  nice, since it would eliminate the need for single-character reads.  Why?
  Because if a series of characters has already been decrypted that shouldn't
  have been, then (a) it's ruined, and (b) so is the state of the decryption
  machine.  Too bad.
*/
		    } else if (me_auth != 0 && me_auth != x_auth) {
			if (ttpushback((CHAR *)ibp,ibc) > -1) {
			    ibc = 0;
			    ibp = ibuf;
			}
#endif /* CK_ENCRYPTION */
		    }
		    continue;
		} else if (tx == -1) {	/* I/O error */
		    if (!quiet)
		      printf("\r\nCommunications disconnect ");
#ifdef NOSETBUF
		    fflush(stdout);
#endif /* NOSETBUF */
		    debug(F111,"CONNECT i/o error 2",ck_errstr(),errno);
		    goto conret0;
		} else if ((tx == 1) && (!duplex)) { /* ECHO change */
		    duplex = 1;		/* Turn on local echo */
		    continue;
		} else if ((tx == 2) && (duplex)) { /* ECHO change */
		    duplex = 0;
		    continue;
		} else if (tx == 3) {	/* Quoted IAC */
		    c = parity ? 127 : 255;
		} else continue;	/* Negotiation OK, get next char. */
	    } else if (parity)
	      c &= 0x7f;

#endif /* TNCODE */
	    if (debses) {		/* Output character to screen */
		char *s;		/* Debugging display... */
		s = dbchr(c);
		while (*s)
		  ckcputc(*s++);
	    } else {			/* Regular display ... */
		c &= cmask;		/* Apply Kermit-to-remote mask */
#ifdef CK_AUTODL
/*
  Autodownload.  Check for Kermit S packet prior to translation, since that
  can change the packet and make it unrecognizable (as when the terminal
  character set is an ISO 646 one)...  Ditto for Zmodem start packet.
*/
		if (autodl) {		/* Autodownload enabled? */
		    int k;
		    k = kstart((CHAR)c); /* Kermit S or I packet? */
#ifdef CK_XYZ			
		    if (!k && zmdlok)	/* Or an "sz" start? */
		      k = zstart((CHAR)c);
#endif /* CK_XYZ */
		    if (k) {
			int ksign = 0;
			debug(F101,"CONNECT autodownload k","",k);
			if (k < 0) { /* Minus-Protocol? */
#ifdef NOSERVER
			    goto noserver; /* Need server mode for this */
#else
			    ksign = 1; /* Remember */
			    k = 0 - k; /* Convert to actual protocol */
			    justone = 1; /* Flag for protocol module */
#endif /* NOSERVER */
			} else
			  justone = 0;
			k--;		/* Adjust [kz]start's return value */
			if (k == PROTO_K
#ifdef CK_XYZ
			    || k == PROTO_Z
#endif /* CK_XYZ */
			    ) {

			    /* Notify parent */
			    sprintf(apcbuf,
				    "set proto %s, %s, set proto %s",
				    ptab[k].p_name,
				    ksign ? "server" : "receive",
				    ptab[protocol].p_name
				    );
			    apclength = strlen(apcbuf);
			    debug(F111,"CONNECT ksbuf",ksbuf,k);
			    debug(F110,"CONNECT autodownload",apcbuf,0);
			    apcactive = APC_LOCAL;
			    ckcputf();	/* Force screen update */
			    goto conret1;
			}
		    }
		}
#ifdef NOSERVER
	      noserver:
#endif /* NOSERVER */
#endif /* CK_AUTODL */
		if (sosi) {		/* Handle SI/SO */
		    if (c == SO) {	/* Shift Out */
			inshift = 1;
			continue;
		    } else if (c == SI) { /* Shift In */
			inshift = 0;
			continue;
		    }
		    if (inshift) c |= 0200; 
		}
#ifndef NOCSETS
		if (
#ifndef NOESCSEQ
		    inesc == ES_NORMAL /* If not in an escape sequence */
#else
		    1
#endif /* NOESCSEQ */
		    ) {			/* Translate character sets */
		    if (sxi) c = (*sxi)((CHAR)c);
		    if (rxi) c = (*rxi)((CHAR)c);
		}
#endif /* NOCSETS */

#ifndef NOESCSEQ
		if (escseq)		/* If handling escape sequences */
		  apcrc = chkaes((char)c); /* update our state */

#ifdef CK_APC
/*
  If we are handling APCs, we have several possibilities at this point:
   1. Ordinary character to be written to the screen.
   2. An Esc; we can't write it because it might be the beginning of an APC.
   3. The character following an Esc, in which case we write Esc, then char,
      but only if we have not just entered an APC sequence.
*/
		if (escseq && apcstatus != APC_OFF) {
		    if (inesc == ES_GOTESC)	/* Don't write ESC yet */
		      continue;
		    else if (oldesc == ES_GOTESC && !apcactive) {
			ckcputc(ESC);	/* Write saved ESC */
			if (seslog) logchar((char)ESC);
		    } else if (apcrc) {	/* We have an APC */
			debug(F111,"CONNECT APC complete",apcbuf,apclength);
			ckcputf();		/* Force screen update */
			goto conret1;
		    }
		}
#endif /* CK_APC */
#endif /* NOESCSEQ */

		if (
#ifdef CK_APC
		    !apcactive		/* Ignore APC sequences */
#else
		    1
#endif /* CK_APC */
		    ) {
		    c &= cmdmsk;	/* Apply command mask. */
		    if (c == CR && tt_crd) { /* SET TERM CR-DISPLA CRLF? */
			ckcputc(c);	     /* Yes, output CR */
			if (seslog) logchar((char)c);
			c = LF;		/* and insert a linefeed */
		    }
		    ckcputc(c);		/* Write character to screen */
		}
		if (seslog) logchar((char)c); /* Handle session log */
#ifdef CK_TRIGGER
		/* Check for trigger string */	
		if (tt_trigger[0]) if (autoexitchk((CHAR)c) > -1) {
		    ckcputf();		/* Force screen update */
#ifdef NOSETBUF
		    fflush(stdout);	/* I mean really force it */
#endif /* NOSETBUF */
		    goto conret1;
		}
#endif /* CK_TRIGGER */
	    }
	}
#ifndef BEBOX 
	if (FD_ISSET(scrnout, &out)) {
	    FD_CLR(scrnout, &out);
	}
#endif /* BEBOX */
    } /* End of big loop */
  conret1:				/* Come here to succeed */
    rc = 1;
  conret0:				/* Common exit point */
#ifdef BEBOX
    {
	long ret_val;
	closesocket(pair[0]);
	closesocket(pair[1]);
	x = kill(tid,SIGKILLTHR);	/* Kill thread */
	wait_for_thread (tid, &ret_val);
    }
#endif /* BEBOX */
    conres();
    if (dohangup > 0) {
#ifdef NETCONN
	if (network)
	  ttclos(0);
#endif /* NETCONN */
#ifndef NODIAL
	if (dohangup > 1)		/* User asked for it */
	  if (mdmhup() < 1)		/* Maybe hang up via modem */
#endif /* NODIAL */
	    tthang();			/* And make sure we don't hang up */
	dohangup = 0;			/* again unless requested again. */
    }
    if (quitnow)			/* Exit now if requested. */
      doexit(GOOD_EXIT,xitsta);
    if (!quiet
#ifdef CK_APC
	&& !apcactive
#endif /* CK_APC */
	)
      printf("(Back at %s)", *myhost ? myhost : "local UNIX system");
#ifdef CK_APC
    if (!apcactive)
#endif /* CK_APC */
      printf("\n");
    what = W_NOTHING;			/* So console modes set right. */
#ifndef NOCSETS
    language = langsv;			/* Restore language */
#endif /* NOCSETS */
#ifdef CK_APC
    debug(F101,"CONNECT exit apcactive","",apcactive);
    debug(F101,"CONNECT exit justone","",justone);
#endif /* CK_APC */
    return(1);
}

/*  H C O N N E  --  Give help message for connect.  */

int
hconne() {
    int c;
    static char *hlpmsg[] = {
"\r\n  ? for this message",
"\r\n  0 (zero) to send a null",
"\r\n  B to send a BREAK",
#ifdef CK_LBRK
"\r\n  L to send a Long BREAK",
#endif /* CK_LBRK */
#ifdef NETCONN
"\r\n  I to send a network interrupt packet",
#ifdef TCPSOCKET
"\r\n  A to send Are You There?",
#endif /* TCPSOCKET */
#endif /* NETCONN */
"\r\n  U to hangup and close the connection",
"\r\n  Q to hangup and quit Kermit",
"\r\n  S for status",
#ifdef NOPUSH
"\r\n  ! to push to local shell (disabled)",
"\r\n  Z to suspend (disabled)",
#else
"\r\n  ! to push to local shell",
#ifdef NOJC
"\r\n  Z to suspend (disabled)",
#else
"\r\n  Z to suspend",
#endif /* NOJC */
#endif /* NOPUSH */
"\r\n  \\ backslash code:",
"\r\n    \\nnn  decimal character code",
"\r\n    \\Onnn octal character code",
"\r\n    \\Xhh  hexadecimal character code",
"\r\n    terminate with carriage return.",
"\r\n Type the escape character again to send the escape character, or",
"\r\n press the space-bar to resume the CONNECT command.\r\n\r\n",
"" };
    conol("\r\nPress C to return to ");
    conol(*myhost ? myhost : "the C-Kermit prompt");
    conol(", or:");
    conola(hlpmsg);			/* Print the help message. */
    conol("Command>");			/* Prompt for command. */
    c = CONGKS() & 0177;		/* Get character, strip any parity. */
    /* No key mapping or translation here */
    if (c != CMDQ)
      conoll(""); 
   return(c);				/* Return it. */
}


/*  D O E S C  --  Process an escape character argument  */

VOID
#ifdef CK_ANSIC
doesc(char c)
#else
doesc(c) char c;
#endif /* CK_ANSIC */
/* doesc */ {
    CHAR d;
  
    debug(F101,"CONNECT doesc","",c);
    while (1) {
	if (c == escape) {		/* Send escape character */
	    d = dopar((CHAR) c); ttoc((char) d); return;
    	} else				/* Or else look it up below. */
	    if (isupper(c)) c = tolower(c);

	switch(c) {

	case 'c':			/* Escape back to prompt */
	case '\03':
	    active = 0; conol("\r\n"); return;

	case 'b':			/* Send a BREAK signal */
	case '\02':
	    ttsndb(); return;

#ifdef NETCONN
	case 'i':			/* Send Interrupt */
	case '\011':
#ifdef TCPSOCKET
#ifndef IP
#define IP 244
#endif /* IP */
	    if (network && ttnproto == NP_TELNET) { /* TELNET */
		temp[0] = (CHAR) IAC;	/* I Am a Command */
		temp[1] = (CHAR) IP;	/* Interrupt Process */
		temp[2] = NUL;
		ttol((CHAR *)temp,2);
	    } else 
#endif /* TCPSOCKET */
	      conoc(BEL);
	    return;

#ifdef TCPSOCKET
	case 'a':			/* "Are You There?" */
	case '\01':
#ifndef AYT
#define AYT 246
#endif /* AYT */
	    if (network && ttnproto == NP_TELNET) {
		temp[0] = (CHAR) IAC;	/* I Am a Command */
		temp[1] = (CHAR) AYT;	/* Are You There? */
		temp[2] = NUL;
		ttol((CHAR *)temp,2);
	    } else conoc(BEL);
	    return;
#endif /* TCPSOCKET */
#endif /* NETCONN */

#ifdef CK_LBRK
	case 'l':			/* Send a Long BREAK signal */
	    ttsndlb(); return;
#endif /* CK_LBRK */

	case 'u':			/* Hangup */
     /*	case '\010': */			/* No, too dangerous */
	    dohangup = 2; active = 0; conol("\r\nHanging up "); return;

	case 'q':			/* Quit */
	    dohangup = 2; quitnow = 1; active = 0; conol("\r\n"); return;

	case 's':			/* Status */
#ifdef NETCMD
	    if (ttpipe)
	      sprintf(temp, "\r\nConnected via \"%s\"", ttname);
	    else
#endif /* NETCMD */
	      sprintf(temp,
		    "\r\nConnected %s %s", network ? "to" : "through", ttname);
	    conol(temp);
	    if (speed >= 0L) {
		sprintf(temp,", speed %ld", speed);
		conoll(temp);
	    } else conoll("");
	    sprintf(temp,
		    "Terminal bytesize: %d, Command bytesize: %d, Parity: ",
		    (cmask  == 0177) ? 7 : 8,
		    (cmdmsk == 0177) ? 7 : 8 );
	    conol(temp);

	    switch (parity) {
	      case  0:  conoll("none");  break;
	      case 'e': conoll("even");  break;
	      case 'o': conoll("odd");   break;
	      case 's': conoll("space"); break;
	      case 'm': conoll("mark");  break;
	    }
	    sprintf(temp,"Terminal echo: %s", duplex ? "local" : "remote");
	    conoll(temp);
	    if (seslog) {
		conol("Logging to: "); conoll(sesfil);
            }
#ifndef NOSHOW
	    if (!network) shomdm();
#endif /* NOSHOW */
	    return;

	case 'h':			/* Help */
	case '?':			/* Help */
	    c = hconne(); continue;

	case '0':			/* Send a null */
	    c = '\0'; d = dopar((CHAR) c); ttoc((char) d); return;

	case 'z': case '\032':		/* Suspend */
#ifndef NOPUSH
	    if (!nopush)
	      stptrap(0);
	    else
	      conoc(BEL);
#else
	    conoc(BEL);
#endif /* NOPUSH */
	    return;

	case '@':			/* Start inferior command processor */
	case '!':
#ifndef NOPUSH
	    if (!nopush) {
		conres();		      /* Put console back to normal */
		zshcmd("");		      /* Fork a shell. */
		if (conbin((char)escape) < 0) {
		    printf("Error resuming CONNECT session\n");
		    active = 0;
		}
	    } else conoc(BEL);
#else
	    conoc(BEL);
#endif /* NOPUSH */
	    return;

	case SP:			/* Space, ignore */
	    return;

	default:			/* Other */
	    if (c == CMDQ) {		/* Backslash escape */
		int x;
		ecbp = ecbuf;
		*ecbp++ = c;
		while (((c = (CONGKS() & cmdmsk)) != '\r') && (c != '\n'))
		  *ecbp++ = c;
		*ecbp = NUL; ecbp = ecbuf;
		x = xxesc(&ecbp);	/* Interpret it */
		if (x >= 0) {		/* No key mapping here */
		    c = dopar((CHAR) x);
		    ttoc((char) c);
		    return;
		} else {		/* Invalid backslash code. */
		    conoc(BEL);
		    return;
		}
	    }
	    conoc(BEL); return; 	/* Invalid esc arg, beep */
    	}
    }
}

static
VOID
#ifdef CK_ANSIC
logchar(char c)
#else
logchar(c) char c;
#endif /* CK_ANSIC */
/* logchar */ {			/* Log character c to session log */
    if (seslog) 
      if ((sessft != 0) ||
	  (c != '\r' &&
	   c != '\0' &&
	   c != XON &&
	   c != XOFF))
	if (zchout(ZSFILE,c) < 0) {
	    conoll("");
	    conoll("ERROR WRITING SESSION LOG, LOG CLOSED!");
	    seslog = 0;
	}
}
#endif /* NOLOCAL */
