;; Stepper Package for Scheme
;; --------------------------
;; Author: Peter Baumgartner, Universitaet Koblenz-Landau, Koblenz, Germany
;; Mail: peter@uni-koblenz.de Web: http://www.uni-koblenz.de/~peter
;;
;; Single-step evaluation of scheme code
;; tracing, break points.
;;
;; System requirements:
;; Bigloo scheme system by Manual serrano, tested with version 1.9b
;; You are invited to port this to other dialects, maybe slib would
;; be a good idea; I plan this to do in the future, but when???
;;
;; Documentation: see README
;;
;; Bugs:
;; stepping a lambda term results in a proc which steps its body.
;; is this desired? or should use explicit lambda-step instead??
;;

(define *stepper-version* "0.1c - 10/3/98, Peter Baumgartner")

;; --------------- user maintainable variables ---------------
;; in interaction, the most recent *stepper-offered-dynamic-history-length*
;; entries in the stepper dynamic history shall be bound to variables
;; !0, !1, ...
;; these are the relevant variables:
(define *stepper-offered-dynamic-history-length* 8)
(define *stepper-dynamic-history-names* '(!0 !1 !2 !3 !4 !5 !6 !7))
;; this list should be in accordance with 
;; *stepper-offered-dynamic-history-length*

(define *return-to-continue* "[RETURN to continue] ")
;; the string to be display during interaction after some value
;; has been computed, as initiated by the creep or skip command
;; legal values: 
;;    if *return-to-continue* is #f the nothing is diplayed
;;    otherwise the string bound to *return-to-continue* is displayed 
;;    and thee user is queried for some trivial input
;; 

(define *stepper-macros-as-functions* '(and or)) 
;; these macros are treated as functions, i.e. they are not 
;; subject to expand before decoration
;; 'and' and 'or' are treated as way, because otherwise
;; the user will see unexpected things


(define *stepper-interaction-command-style* 'prolog-like)
;; affects in interaction how to name commands. Legal values are
;;
;; prolog-like:
;; c  - creep          means single step through next form
;; s  - skip           means skip stepping the evaluation of next form
;; st - skip trace     like skip, but show trace
;; 
;; gdb-like:
;; s  - step           means single step through next form
;; c  - continue       means skip stepping the evaluation of next form
;; ct - skip trace     like continue, but show trace


;; --------------- end user maintainable variables ---------------

;; --------------- user interface --------------------------------

;; Call one of these from repl, e.g. (step (fib (* 2 3))) will
;; evaluate form in single-step mode. Notice, however, that fib has
;; to be defined via define-step, or loaded from a file
;; via loads

(define-macro (step form)
  (stepper-reset 'creep)
  (stepper-make-safe-eval-form (stepper-decorate-form form ''())))

;; trace - like step, but don't stop, show only entry and exit
(define-macro (trace form)
  (stepper-reset 'trace)
  (set! *stepper-skip-level* -1)
  (stepper-make-safe-eval-form (stepper-decorate-form form ''()))
  )

;; leap - eval silently like step, but don't stop, show only entry and exit
(define-macro (leap form)
  (stepper-reset 'leap)
  (set! *stepper-skip-level* -1)
  (stepper-make-safe-eval-form (stepper-decorate-form form ''()))
  )

;; add (remove) breakpoint to (from) defined function
(define-macro (breakf name)
  `(begin (putprop! ',name 'stepper-break #t) ',name))

(define-macro (unbreakf name)
  `(begin (remprop! ',name 'stepper-break) ',name))

;; the following two should only be used for top-level definitions:
;; define-step: like define, but body of defined proc is to be stepped
(define-macro (define-step expr . vals)
  (stepper-decorate-define `(define ,expr  ,@vals) ''()))

;; lambda-step: like lambda, but body is decorated to step through
(define-macro (lambda-step  formals . body)
  (stepper-decorate-lambda `(lambda ,formals ,@body)  ''())
  )

;; --------- this here is for test purposes ------------
;;
;; shows the actual expansion of 'step':
(define-macro (step-test form)
;    (pp (stepper-decorate-form form 'stepper-bindings))
  (pp 
   (stepper-make-safe-eval-form (stepper-decorate-form form ''()))
   ) 
  #t)

;; this is useful to see the result of the stepping-decoration of define-step
(define-macro (define-step-test expr . vals)
  (pp (stepper-decorate-define `(define ,expr  ,@vals) ''())))


;; --------- end user interface - private part -------------
;; useful macros:
(define-macro (pop! li) `(let ((r (car ,li)))
			   (set! ,li (cdr ,li))
			   r))
(define-macro (push! el li) `(set! ,li (cons ,el ,li)))

(define-macro (when test . forms)
  `(if ,test (begin ,@forms)))

(define-macro (incval! var)
  `(set! ,var (+ ,var 1)))

(define-macro (decval! var)
  `(set! ,var (- ,var 1)))



;; some stepper-internal variables:
(define *stepper-level* -1)
(define *stepper-mode* 'creep)
(define *stepper-skip-level* #unspecified)

(define *stepper-dynamic-history* '()) ; stack  of forms called
(define *stepper-bindings-history* '()) ; stack  of (local) bindings in effect

(define stepper-list list) ;; to avoid conflict with user named variable
                           ;; or parameter "list"

(define (stepper-reset #!optional (mode #f))
;  (set! stepper-bindings '())
  (set! *stepper-dynamic-history* '())
  (set! *stepper-bindings-history* '())
  (set! *stepper-level* -1)
  (cond ((and mode (not (eq? *stepper-mode* mode)))
	 (set! *stepper-mode* mode)
	 (display* "[STEPPER set to " mode " mode]")
	 (newline)))
  (set! *stepper-skip-level* #unspecified)
  (set! stepper-abort-handler stepper-undefined-abort-handler)
  )

(define *stepper-abbrev-max-length* 4)
(define *stepper-abbrev-max-depth* 3)
(define (stepper-printer-abbrev expr)
  (define (abbrev exp len dep)
    (cond ((not (pair? exp)) exp)
	  ((null? exp) '())
	  ((eqv? dep 0) (list '...))
	  ((eqv? len 0) (list '...))
	  (else (cons (abbrev (car exp) 
			      *stepper-abbrev-max-length*
			      (- dep 1))
		      (abbrev (cdr exp) (- len 1) dep)))))
  (display (abbrev expr *stepper-abbrev-max-length* *stepper-abbrev-max-depth*)))

;; the procedure to print values for stepper interaction
(define stepper-printer stepper-printer-abbrev)
;;(define stepper-printer pp) 

;; bindings stuff
;; a binding is a pair of a variable and a value
;; needed to mirror internal (non top-level) variables and the values
;; bound to them
;; 
(define (stepper-extend-bindings vars vals old)
  (append (map cons vars vals) old))

(define (stepper-extend-binding var val old)
  (cons (cons var val) old))


(define (stepper-set!-binding var val bindings)
  (let ((pair (assq var bindings)))
    (if pair (set-cdr! pair val)
	;; else it will be set globally
	)))

(define (stepper-set!-bindings vars vals bindings)
  (for-each (lambda (var val)
	      (stepper-set!-binding var val bindings))
	    vars vals))

(define (stepper-new-vars-bindings vars bindings)
  (stepper-extend-bindings vars (map (lambda (var) 'unbound) vars) bindings))

  
(define (stepper-bindings-to-let bindings form)
  ;; convert the BINDINGS to a let-form which binds the
  ;; vars to their vals, the body of the let is FORM
  `(let ,(map (lambda (binding) (list (car binding) (list 'quote 
							  (cdr binding))))
	      bindings)
     ,form))



;; main entry point for decoration of an expression
(define (stepper-decorate-form form bindings)
  (cond ((pair? form)
	 ;; some special cases
	 (case (car form)
	   ((lambda lambda-step) 
	    (stepper-wrap-i/c 
	     form
	     (stepper-decorate-lambda form bindings)
	     bindings))
	   ((define define-step)
	    ;; this won't work, because if we come here we usually 
	    ;; decorate a form coming from a FILE-stepper.scm file
	    ;; and the user would get wrong position information
	    ;; wrt. FILE.scm:
;	    (stepper-remember-source form)
	    (stepper-decorate-define form bindings)
	    )
	   ((set!)
	    (stepper-wrap-i/c
	     form
	     (stepper-decorate-set! form bindings)
	     bindings))
	   ((bind-exit)
	    (stepper-wrap-i/c
	     form
	     (stepper-decorate-bind-exit! form bindings)
	     bindings))
	   ((define-macro define-expander)
	    form)
	   ((do)
	    (stepper-wrap-i/c
	     form
	     (stepper-decorate-do form bindings)
	     bindings))
	   ((let)
	    (stepper-wrap-i/c
	     form
	     (stepper-decorate-let form bindings)
	     bindings))
	   ((letrec)
	    (stepper-wrap-i/c
	     form
	     (stepper-decorate-letrec form bindings)
	     bindings))
	   ((case)
	    (stepper-wrap-i/c
	     form
	     (stepper-decorate-case form bindings)
	     bindings))
	   ((assert)
	    (let ((vars (cadr form))
		  (test (caddr form)))
	      `(if (not ,test)
		   (begin 
		     (notify-error "assert" "assertion failed" ',form)
		     (stepper-interaction 
		      ',form
		      (stepper-extend-bindings ',vars (stepper-list ,@vars)
					     ,bindings))
		     (stepper-cleanup-interaction #unspecified)))))
	   ((break)
	    `(begin 
	       (stepper-interaction 
		(cons 'break (stepper-list ,@(cdr form)))
		,bindings)
	       (stepper-cleanup-interaction #unspecified)))
	   ((let*)
	    (stepper-wrap-i/c
	     form
	     (stepper-decorate-let* form bindings)
	     bindings))
	   ((if)
	    (stepper-wrap-i/c
	     form
	     (stepper-decorate-if form bindings)
	     bindings))
	   ((quasiquote)
	    (stepper-wrap-i/c
	     form
	     (stepper-decorate-quasiquote form bindings)
	     bindings))
	   ((cond)
	    (stepper-wrap-i/c
	     form
	     (stepper-decorate-cond form bindings)
	     bindings))
	   ((begin)
	    (stepper-wrap-i/c
	     form
	     `(begin ,@(stepper-decorate-form-sequence (cdr form) bindings))
	     bindings))
	   ((quote) 
	    (stepper-wrap-i/c form `',(cadr form) bindings))
	   (else
	    (stepper-decorate-application form bindings))))
	(else
	 (stepper-wrap-i/c form form bindings)
	 )))


(define (stepper-wrap-i/c form-text form-expr bindings)
  `(let ((form-text ',form-text))
     (stepper-interaction form-text ,bindings)
     (stepper-continue form-text ,form-expr)))


(define (stepper-decorate-quasiquote form bindings)
  (define (stepper-decorate-inside-quasiquote expr bindings)
    (cond ((not (pair? expr))
	   expr)
	  ((eq? (car expr) 'unquote)
	   (list 'unquote (stepper-decorate-form (cadr expr) bindings)))
	  ((eq? (car expr) 'unquote-splicing)
	   (list `unquote-splicing (stepper-decorate-form (cadr expr) bindings)))
	  (else (cons (stepper-decorate-inside-quasiquote (car expr) bindings)
		      (stepper-decorate-inside-quasiquote (cdr expr) bindings)))))
  (list 'quasiquote (stepper-decorate-inside-quasiquote (cadr form) bindings))
; this won't work: !!!!!!!
;  `(quasiquote ,(stepper-decorate-inside-quasiquote (cadr form) bindings))
)


(define (stepper-decorate-application form bindings)
  (define (stepper-decorate-fun fun bindings)
    (cond ((pair? fun) 
	   ;; (lambda ...) expressions, for instance
	   (stepper-decorate-form fun bindings))
	  (else fun))
    )
  (let ((fun (car form))
	(args (cdr form)))
;;      form
;;      `(,(stepper-decorate-fun fun bindings)
;;        ,@(stepper-decorate-form-sequence args bindings))
;;      bindings)
;; don't need this, because macroexpand everything before decoration
     (cond ((and (symbol? fun) 
		 (not (member fun *stepper-macros-as-functions*)))
	    (let ((exp-form (expand-once form)))
	      (if (and (pair? exp-form) (eq? (car exp-form) fun))
		  ;; nothing changed - so this is not a macro application
		  (stepper-wrap-i/c 
		   form
		   `(,(stepper-decorate-fun fun bindings)
		     ,@(stepper-decorate-form-sequence args bindings))
		   bindings)
		  ;; else have a macro application - try again
		  (stepper-decorate-form exp-form bindings))))
	   (else
 	   ;; assume proper function application
	    (stepper-wrap-i/c 
	     form
	     `(,(stepper-decorate-fun fun bindings)
	       ,@(stepper-decorate-form-sequence args bindings))
	     bindings)
	    ))
     ))

(define (stepper-decorate-set! form bindings)
  (let ((var (cadr form))
	(val (caddr form)))
    `(begin
       (set! ,var
	    ,(stepper-decorate-form val bindings))
       (stepper-set!-binding ',var ,var ,bindings)
       #unspecified)))

(define (stepper-decorate-bind-exit! form bindings)
  (let ((escape (cadr form))
	(body (caddr form)))
    `(bind-exit ,escape
	    ,(stepper-decorate-form body bindings))))

(define (stepper-decorate-if form bindings)
  (let ((test (cadr form))
	(then-part (caddr form))
	(else-part (cdddr form)))
    `(if ,(stepper-decorate-form test bindings)
	 ,(stepper-decorate-form then-part bindings)
	 ,@(stepper-decorate-form-sequence else-part bindings))))

(define (stepper-decorate-case form bindings)
  (let ((case-expr (cadr form))
	(case-body (cddr form)))
    `(case ,(stepper-decorate-form case-expr bindings)
       ,@(map (lambda (case-expr)
		`(,(car case-expr)
		  ,@(stepper-decorate-form-sequence (cdr case-expr)
					    bindings)))
	      case-body))))

(define (stepper-decorate-cond form bindings)
  (let ((cond-body (cdr form)))
    `(cond ,@(map (lambda (cond-expr)
		   `(,(if (eq? (car cond-expr) 'else)
			  'else
			  (stepper-decorate-form (car cond-expr) bindings))
		     ,@(if (and (not (null? (cdr cond-expr)))
				(eq? (cadr cond-expr) '=>))
			   (cons '=> 
				 (stepper-decorate-form-sequence (cddr cond-expr)
							 bindings))
			   (stepper-decorate-form-sequence (cdr cond-expr)
						   bindings))))
		  cond-body))))
    
(define (stepper-decorate-define form bindings)
  (let ((expr (cadr form))
	(vals (cddr form))
	define-var            ;; the variable for the final define
	define-decorated-expr ;; the decorated expression for the final define
	define-expr           ;; the undecorated expression
	(top-level-define? (equal? bindings ''())) ;; see below for why that
	)
    (cond ((pair? expr)
	   ;; have a procedure def.
	   (let* ((name (car expr))
		  (formals (cdr expr))
		  (lambda-expr `(lambda ,formals ,@vals))
		  )
	     (set! define-var name)
	     (set! define-decorated-expr 
		   (stepper-decorate-lambda lambda-expr bindings name))
	     (set! define-expr lambda-expr)))

	  ((and (not (null? vals))
		(pair? (car vals))
		(eq? (caar vals) 'lambda))
	   ;; special case recognized: (define foo (lambda ....))
	     (set! define-var expr)
	     (set! define-decorated-expr 
		   (stepper-decorate-lambda (car vals) bindings expr))
	     (set! define-expr (car vals)))

	  (else 	   
	   ;; have (define VAR FORM) don't step into evaluation of FORM
	   ;; could not load file then without interaction
	   (set! define-var expr)
	   (set! define-decorated-expr (car vals))
	   (set! define-expr (car vals))))

    ;; this is the decorated define:
    `(begin
       (define ,define-var ,define-decorated-expr)
       ,@(if top-level-define?
	     '() ;; this indicates top-level definition - not mirrored
	     ;; otherwise bindings holds the name of the
	     ;; variable holding the bindings for the body surrounding
	     ;; the current define; in this case we have an internal define
	     ;; and the define acts as a letrec
	     ;; this suffices to mirror this:
	     (list 
	      `(set! ,bindings (stepper-extend-binding ',define-var
						     ',define-expr
						     ,bindings))))
       ',define-var)))


;;; (let* ((v1 x1) ... ) body)
;;; ->
;;; (let ((stepper-bindings bindings))
;;;    (let* ((v1 (let ((X (stepper-decorate-form x1)))
;;;                  (set! stepper-bindings (stepper-extend-bindings '(v1) (list X) stepper-bindings))
;;;                  X)
;;;               ... ) body)

(define (stepper-decorate-let* form bindings)
  (let ((var-bindings (cadr form))
	(body (cddr form)))
    `(let ((stepper-bindings ,bindings))
       (let* ,(map (lambda (vb)
		     (if (symbol? vb) 
			 '(,vb 
			   (set! stepper-bindings (stepper-extend-bindings '(,vb)
							     #unspecified
							     stepper-bindings))
			   #unspecified)
			 `(,(car vb) 
			   (let ((X ,@(stepper-decorate-form-sequence (cdr vb)
							      'stepper-bindings)))
			     (set! stepper-bindings (stepper-extend-bindings '(,(car vb))
							       (stepper-list X)
							       stepper-bindings))
			     X))))
		   var-bindings)
	 ,@(stepper-decorate-form-sequence body 'stepper-bindings)
;; 	 ,(stepper-wrap-i/c 
;; 	   `(begin ,@body)
;; 	   `(begin ,@(stepper-decorate-form-sequence body 'stepper-bindings))
;; 	   'stepper-bindings)
	 ))))


;;; (let name ((v1 x1) ... ) body)
;;; ->
;;; (let name ((v1 (stepper-decorate-form x1)) ... )
;;;    (let ((stepper-bindings (stepper-extend-bindings '(x1 ...) (list x1 ...))))
;;;       body))


(define (stepper-decorate-let form bindings)
  (let* ((name (if (symbol? (cadr form)) (cadr form) #f))
	 (var-bindings (if name (caddr form) (cadr form)))
	 (body (if name (cdddr form) (cddr form))))
    `(let ,@(if name (list name) '())
       ,(map (lambda (vb)
	       (if (symbol? vb) 
		   vb 
		   `(,(car vb) ,@(stepper-decorate-form-sequence (cdr vb)
							 bindings))))
	     var-bindings)
       (let ((stepper-bindings
	      (stepper-extend-bindings 
	       ',(map (lambda (vb)
			(if (symbol? vb) vb (car vb)))
		      var-bindings)
	       (stepper-list ,@(map (lambda (vb)
			      (if (symbol? vb)
				  #unspecified 
				  ;; else get the value - was bound above
				  (car vb)))
			    var-bindings))
	       ,bindings)))
	 ,@(stepper-decorate-form-sequence body 'stepper-bindings)
;; 	 ,(stepper-wrap-i/c 
;; 	   `(begin ,@body)
;; 	   `(begin ,@(stepper-decorate-form-sequence body 'stepper-bindings))
;; 	   'stepper-bindings)
	 ))))


(define (stepper-decorate-do form bindings)
  (let* ((var-bindings (cadr form))
	 (vars (map (lambda (vb)
		      (if (symbol? vb) vb (car vb)))
		    var-bindings))
	 (terminate-expr (caddr form))
	 (body (cdddr form)))
    `(let ((stepper-bindings (stepper-new-vars-bindings ',vars ,bindings)))
       (do ,(map (lambda (vb)
		   (if (symbol? vb) 
		       vb 
		       `(,(car vb) ,@(stepper-decorate-form-sequence (cdr vb)
							     'stepper-bindings))))
		 var-bindings)
	   ((begin (stepper-set!-bindings ',vars (stepper-list ,@vars) stepper-bindings)
		   ,(stepper-decorate-form (car terminate-expr) 'stepper-bindings))
	    ;; this was the test expr
	    ,@(stepper-decorate-form-sequence (cdr terminate-expr) 'stepper-bindings))
	 ,@(stepper-decorate-form-sequence body 'stepper-bindings)
;; 	 ,(stepper-wrap-i/c 
;; 	   `(begin ,@body)
;; 	   `(begin ,@(stepper-decorate-form-sequence body 'stepper-bindings))
;; 	   'stepper-bindings)
	 )
       )))


(define (stepper-decorate-letrec form bindings)
  (let* ((var-bindings (cadr form))
	 (vars (map (lambda (vb)
		      (if (symbol? vb) vb (car vb)))
		    var-bindings))
	 (body (cddr form)))
    `(let ((stepper-bindings (stepper-new-vars-bindings ',vars ,bindings)))
       (letrec ,(map (lambda (vb)
		       (if (symbol? vb) 
			   vb 
			   `(,(car vb) ,@(stepper-decorate-form-sequence 
					  (cdr vb)
					  'stepper-bindings))))
		     var-bindings)
	 (stepper-set!-bindings ',vars (stepper-list ,@vars) stepper-bindings)
	 ,@(stepper-decorate-form-sequence body 'stepper-bindings)
;; 	 ,(stepper-wrap-i/c 
;; 	   `(begin ,@body)
;; 	   `(begin ,@(stepper-decorate-form-sequence body 'stepper-bindings))
;; 	   'stepper-bindings)
	 )
       )))

;;; (do  ((v1 i1 s1) ... )
;;;      (test exp1 ... )
;;;     body)
;;; ->
;;; (do ((v1 (stepper-decorate-form i1) (stepper-decorate-form s1)) ... )
;;;    ((let ((stepper-bindings (stepper-extend-bindings '(x1 ...) (list x1 ...))))
;;;         (stepper-decorate-form test))
;;;     (let ((stepper-bindings (stepper-extend-bindings '(x1 ...) (list x1 ...))))
;;;         (stepper-decorate-form-sequence exp1 ...)) )
;;;   (let ((stepper-bindings (stepper-extend-bindings '(x1 ...) (list x1 ...))))
;;;       (stepper-decorate-form-sequence body)))



;; (lambda (Arg1 ... ArgN) Body) ->
;; (lambda (Arg1 ... ArgN)
;;   (let ((bindings 
;; 	 (stepper-extend-bindings '(Arg1 ... ArgN) 
;; 			(list Arg1 ... ArgN))))
;;     Body))

(define (stepper-decorate-lambda form bindings #!optional (defname #f))
  (let* ((args (cadr form))
	 (proper-args-list
	  (letrec ((f (lambda (arg)
			(cond ((null? arg) '())
			      ((symbol? (cdr arg)) 
			       ;; optinal arguments!!
			       (list (car arg) 
				     (cdr arg)))
			      (else (cons (car arg)
					  (f (cdr arg))))))))
	    (if (symbol? args) 
		(list args)
		(f args))))
	 (body (cddr form))
	 )
;    (if defname (putprop! defname 'stepper-source form))
    `(lambda ,args 
       (let ((stepper-bindings
	      (stepper-extend-bindings ',proper-args-list
			     (stepper-list ,@proper-args-list)
			     ,bindings))
	     (form-text (stepper-list 'body-of ',(or defname form)
			      'formal-parameters ',proper-args-list
			      'evaled-arguments (stepper-list ,@proper-args-list)))
	     )
 	 (stepper-interaction form-text stepper-bindings)
	 ;; [ this is a bit ugly, but cannot do much more elegantly ...
 	 ,@(stepper-decorate-form-sequence (but-last-pair body)
 					   'stepper-bindings)
	 ;; the last form in body is modified so that at the end
	 ;; the cleanup is done and the result is the value of the
	 ;; last form
	 (let ((result ,(stepper-decorate-form (car (last-pair body))
					       'stepper-bindings)))

	   ;; (stepper-cleanup-interaction result)
	   (stepper-continue form-text result)
	   )
	 ;; ... in particular, THIS won't work, 
	 ;; because local defines in body would get lost to 
	 ;; scheme interpreter (because 'defines' would be in an
	 ;; argument position (of stepper-continue) ]
;; 	 (stepper-interaction form-text stepper-bindings)
;; 	 (stepper-continue form-text 
;; 		      (begin ,@(stepper-decorate-form-sequence 
;; 				body 
;; 				'stepper-bindings)))
	 ))))


(define (stepper-decorate-form-sequence form-sequence bindings)
  (map (lambda (form)
	 (stepper-decorate-form form bindings))
       form-sequence))


;; --- end of decoration procedures

(define (stepper-indent n) 
  (do ((i 0 (+ i 1)))
      ((>= i n) #unspecified)
    (display " ")))

(define (stepper-notify text form)
  (stepper-indent *stepper-level*) (display text)
  (stepper-printer form)
  form)

(define (stepper-cleanup-interaction val)
  (if (>= *stepper-level* 0) (decval! *stepper-level*))
  (pop! *stepper-dynamic-history*)
  (pop! *stepper-bindings-history*)
  val)

(define (stepper-continue form val)
  (cond ((and (eq? *stepper-mode* 'skip)
	      (> *stepper-level* *stepper-skip-level*)))
	((eq? *stepper-mode* 'leap))
	((and (eq? *stepper-mode* 'trace)
	      (> *stepper-level* *stepper-skip-level*))
	 (stepper-notify "EXIT: " form)
	 (display " => ")
	 (stepper-printer val)
	 (newline))
	(else
	 (stepper-notify "EXIT: " form)
	 (display " => ")
	 (stepper-printer val)
	 (newline)
	 (stepper-indent *stepper-level*)
	 (when *return-to-continue*
	       (display *return-to-continue*)
	       (read-string (current-input-port) 20)) ; some value
	 ))
  (stepper-cleanup-interaction val))

(define (stepper-error-handler escape proc mes obj)
  (notify-error proc mes obj)
  (newline)
  (let ((bindings (pop! *stepper-bindings-history*)))
    (stepper-pp-dynamic-history *stepper-dynamic-history*)
    (newline)
    (stepper-pp-bindings bindings)
    (newline)
    ;; something more clever should be done here...
    (stepper-reset)
    (escape #unspecified)))

(define (stepper-undefined-abort-handler val)
  (print "[stepper-abort-handler not initialized.]")
  )

(define (stepper-make-abort-handler cont)
  (lambda (val)
    (print "[Aborting]")
    (stepper-reset)
    (cont val)))

(define stepper-abort-handler stepper-undefined-abort-handler)



;; For some reason I don't understand, this does not work.
;; aborting is without any effect (Manuel?)
;; (define (stepper-make-safe-eval-form form)
;;   ;; install the stepper and abort handlers and eval form
;;   `(bind-exit (stepper-abort-handler)
;; ;	      ,form
;; 	      (try ,form stepper-error-handler)
;; 	      ))

;; this works:
(define (stepper-make-safe-eval-form form)
  ;; install the error and abort-command handlers and eval form
  `(call/cc (lambda (cont)
	      (set! stepper-abort-handler 
		    (stepper-make-abort-handler cont))
	      (try ,form stepper-error-handler)
	      )))

(define (stepper-pp-dynamic-history h)
  (print "Evaluation stack:")
  (print "-----------------")
;; this was the previous form:
;;   (for-each (lambda (form)
;; 	      (stepper-printer form)
;; 	      (newline))
;; 	    h)
  (do ((rest-names *stepper-dynamic-history-names*
		   (cdr rest-names))
       (rest-history h (cdr rest-history)))
      ((or (null? rest-history) (null? rest-names))
       (cond ((not (null? rest-history))
	      (display* "[" (length rest-history) " more entries]")
	      (newline)))
       #unspecified)
    (display* (car rest-names) "   ")
    (stepper-printer (car rest-history))
    (newline)))
       

(define (stepper-pp-bindings bindings)
  (define printed-so-far '()) ;; omit shadowed variables in bindings
  (print "Local bindings:")
  (print "---------------")
  (for-each (lambda (binding)
	      (let ((var (car binding))
		    (val (cdr binding)))
		(when (not (member var printed-so-far))
		      (display* var "   ")
		      (stepper-printer val)
		      (newline)
		      (push! var printed-so-far))))
 	    bindings))


(define (stepper-interaction form bindings)

  (define (translate-command c)
    (cond ((assq c
		 (cond ((eq? *stepper-interaction-command-style* 'prolog-like)
			'((c . creep) (s . skip) (st . skip-trace)))
		       (else
			'((c . skip) (ct . skip-trace) (s . creep)))))
	   => cdr)
	  (else c) ;; no translation needed
	  ))

  (define creep-this-form? #t)
  (incval! *stepper-level*)
  (push! form *stepper-dynamic-history*)
  (push! bindings *stepper-bindings-history*)
  (case *stepper-mode*
    ((skip) (if (> *stepper-level* *stepper-skip-level*)
		(set! creep-this-form? #f)))
    ((trace) (if (> *stepper-level* *stepper-skip-level*)
		 (begin (newline)
			(stepper-notify "FORM: " form)
			(set! creep-this-form? #f))))
    ;; this can be programmed more compact using match-lambda:
    ((leap) (cond ((and (pair? form)
			(eq? (car form) 'body-of)
			(pair? (cdr form))
			(symbol? (cadr form))
			(getprop (cadr form) 'stepper-break))
		   ;; don't want indentation:
		   (set! *stepper-level* 0))
		  ((and (pair? form) 
			(eq? (car form) 'break)))
		  ((and (pair? form) 
			(eq? (car form) 'assert)))
		  (else (set! creep-this-form? #f))))
    (else (set! creep-this-form? #t)))
  (cond (creep-this-form?
	 (set! *stepper-mode* 'creep)
	 (set! *stepper-skip-level* #unspecified)
	 (let* (response ;; user input on prompt
		;; how many entries to show in history length?
		(this-dynamic-history-length 
		 (min *stepper-offered-dynamic-history-length*
		      (length *stepper-dynamic-history*)))
		(extended-bindings 
		 (stepper-extend-bindings 
		  ;; vars:
		  (list-head *stepper-dynamic-history-names*
			     this-dynamic-history-length)
		  ;; vals:
		  (list-head *stepper-dynamic-history*
			     this-dynamic-history-length)
		  bindings))
		(local-error-handler (lambda (escape proc mes obj)
				       (notify-error proc mes obj)
				       (escape #unspecified))))
	   (let read-loop ()
	     (newline)
	     (stepper-notify "FORM: " form)
	     (newline)
	     (stepper-indent *stepper-level*) 
	     (display "Now? ")
	     (set! response (translate-command (read)))
	     (cond ((symbol? response)
		    (case response
		      ((a) 
		       (stepper-abort-handler #unspecified)
		       ;; if properly initialised, this is not reachable:
		       (read-loop)
		       )
		      ((creep) 
		       #unspecified
		       )
		      ((skip-trace)
		       (set! *stepper-mode* 'trace)
		       (set! *stepper-skip-level* *stepper-level*)
		       #unspecified)
		      ((skip)
		       (set! *stepper-mode* 'skip)
		       (set! *stepper-skip-level* *stepper-level*)
		       #unspecified)
		      ((l)
		       (set! *stepper-mode* 'leap)
		       #unspecified)
		      ((gt)
		       (set! *stepper-mode* 'trace)
		       (set! *stepper-skip-level* -1)
		       #unspecified)
		      ((g)
		       (set! *stepper-mode* 'skip)
		       (set! *stepper-skip-level* -1)
		       #unspecified)
		      ((b) 
		       (eval (stepper-bindings-to-let extended-bindings 
						      '(repl)))
		       (read-loop))
		      ;; vpp replaced by v; former v is obsolete
		      ((e) 
		       (let ((exp (read))
			     (sp stepper-printer))
			 (set! stepper-printer pp)
			 (try (stepper-notify  "=> " 
					       (eval (stepper-bindings-to-let 
						      extended-bindings exp)))
			      local-error-handler)
			 (set! stepper-printer sp))
		       (read-loop))
		      ((pp) 
		       (let ((exp (read)))
			 (cond ((and (symbol? exp)
				     (getprop exp 'stepper-source))
				=> (lambda (definition)
				     (display* "File ")
				     (write (cadr (cer definition)))
				     (display* 
				      ", character "
				      (caddr (cer definition))
				      ": ")
				     (newline)
				     (pp definition)))
			       (else
				(try (pp (eval (stepper-bindings-to-let 
						extended-bindings exp)))
				     local-error-handler))))
		       (read-loop))
		      ((h)
		       (stepper-pp-dynamic-history
			*stepper-dynamic-history*)
		       (read-loop))
		      ((v)
		       (stepper-pp-bindings bindings)
		       (read-loop))
		      ((?)
		       (stepper-help)
		       (read-loop))
		      (else
		       (print "Unknown command for stepper")
		       (stepper-help)
		       (read-loop))))
		   (else
		    (print "Unknown command for stepper")
		    (stepper-help)
		    (read-loop))))
	   )))
  #unspecified)

(define (stepper-help)
  (print "Stepper commands:")
  (print "-----------------")
  (print "?       - this text.")
  (print "a       - abort - return to read-eval-print loop")
  (print "b       - break - enter a read-eval-print loop")
  (print "h       - history of pending forms in evaluation")
  (print "v       - current local bindings")
  (cond ((eq? *stepper-interaction-command-style* 'prolog-like)
	 (print "c       - creep - step into current form")
	 (print "s       - skip - eval current form without stepping")
	 (print "st      - skip trace, like skip but trace evaluation of form"))
  (else ;; assume gdb-like
	 (print "s       - step into current form")
	 (print "c       - continue - eval current form without stepping")
	 (print "ct      - continue trace, like continue but trace evaluation of form")))
  (print "l       - leap - continue until next break point is reached")
  (print "g       - go - resume execution without stepping")
  (print "gt      - go trace - like go but trace evaluation")
  (print "e FORM  - eval FORM and print result")
  (print "pp FORM - eval FORM and pretty print result")
)

(define (stepper-remember-source form)
  (when (and (pair? form)
	     (member (car form) '(define define-step)))
	(let ((name (if (symbol? (cadr form))
			(cadr form)     ;; (define a ...)
			(caadr form)))) ;; (define (a ...) ...)
	  (putprop! name 'stepper-source form))))


(define (loads filename)
  (if (not (string? filename))
      (error 'loads "Type 'BSTRING' expected" filename))
  (if (not (file-exists? filename))
      (error 'loads "File does not exist" filename))
  (let* ((outfilename 
	  (string-append (prefix filename) 
			 "-stepper." (suffix filename))
	  )
	 (file (open-input-file filename))
	 (outfile (open-output-file outfilename)))
    (do ((next (read file #t) (read file #t)))
	((eof-object? next)
	 (close-input-port file)
	 (close-output-port outfile)
	 (load outfilename))
      (cond ((and (pair? next)
		  (member (car next) '(define define-step)))
	     ;; make a define-step of it
	     (write (cons 'define-step (cdr next)) outfile)
	     (newline outfile)
	     (stepper-remember-source next))
	    (else
	     ;; put it to output as read in
	     (write next outfile)
	     (newline outfile))))))
	
;; some helpers:

(define (all pred? list)
  (let loop ((rest list))
    (cond ((null? rest) '())
	  ((pred? (car rest))
	   (cons (car rest) 
		 (loop (cdr rest))))
	  (else (loop (cdr rest))))))

(define (but-last-pair l) ;; complement to last-pair
  ;; assume that l ist non-nil
  (if (null? (cdr l)) 
      '()
      (cons (car l) (but-last-pair (cdr l)))))

(define (list-head l n)
  ;; complement to list-tail - return sublist of l consisting of n first 
  ;; elements 
  (if (eqv? n 0) '() (cons (car l) (list-head (cdr l) (- n 1)))))

(begin 
  (print "Happy stepping (stepper version " *stepper-version* ")")
  #t)
