package biss.awt;

import biss.Command;
import biss.CommandQueue;
import biss.ObserverSocket;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Graphics;
import java.util.Observable;
import java.util.Observer;

/**
 * A SpinButton widget (TextField with two Buttons to circulate through
 * a list of predefined values)
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
public class SpinButton
  extends Compound
  implements Command, FocusSetInterface, Observer, HotKeyProcessor
{
	int CurVal = 0;
	TextField Entry = new TextField();
	int Max = 100;
	int Min = 0;
	int LoopDir = 0;
	int ScrollOp = 0;
/**
 * triggered by spin-up transition (final spin position reached)
 * or the TextFields Enter key.
 * parameter: -
 */
	public ObserverSocket OsSpinChanged = new ObserverSocket( this);
/**
 * triggered by any contents change (spin operation or setContents() ) 
 * parameter: -
 */
	public ObserverSocket OsValueChanged = new ObserverSocket( this);
	Object[] Values = null;
	int SpinStep = 1;
	boolean Delay = false;

public SpinButton(){
	Entry.setReadOnly( true);
	Entry.setRightAlign( true);

	Entry.setBorder( Border.BS_NONE);
	FBorder.setBorder( TextField.DefaultBorderStyle);

	int bw = FBorder.Ext;
	Lo.add( Entry, 0, 0, 1000, 1000, false, bw, bw, -Awt.ScrollBarWidth-2-bw, -bw, 0 );

	registerObserver();
}

public SpinButton ( Object[] values) {
	this();
	setContents( values);
}

public SpinButton ( int min, int max) {
	this();
	setRange( min, max);
}

public boolean atLimit() {
	if ( CurVal == Max) return true;
	if ( CurVal == Min) return true;
	return false;
}

public boolean canGetFocus(){
	return Entry.canGetFocus();
}

public void clear() {
	Entry.setContents( "");
	CurVal = 0;
}

public boolean execute(){
	if ( LoopDir == 1){
		if ( CurVal < Max)
			spinUp( SpinStep);
	}
	else if ( LoopDir == 2){
		if ( CurVal > Min)
			spinDown( SpinStep);
	}

	if ( ScrollOp > 0){
		int delay = Awt.SpinDelay;
		try {
			if ( Delay){
				delay *= 4;
				Delay = false;
			}
			Thread.sleep( delay);
		}
		catch ( Exception x ) {}

		if ( ScrollOp > 0)
			CommandQueue.append( this);
	}

	if ( ScrollOp == 0)
		OsSpinChanged.notifyObservers();;

	return true;
}

public int getContents() {
	return CurVal;
}

public int getMaximum() {
	return Max;
}

public int getMinimum() {
	return Min;
}

public int getSpinStep() {
	return SpinStep;
}

public int getValueIdx( Object obj) {
	if ( Values == null)
		return -1;
	for ( int i=0; i<Values.length; i++){
		if ( obj.equals( Values[i] ))
			return i;
	}
	return -1;
}

public Object[] getValues() {
	return Values;
}

public boolean hasFocus(){
	return Entry.hasFocus();
}

public boolean mouseDown( Event evt, int x, int y) {
	GUIHandler.disposeSingleInstWnd( this);
	if ( ! Enabled)
		return true;
	if ( x > Entry.Width + FBorder.Ext) {
		if ( y < FBorder.Ext+Entry.Height/2 ){
			if ( CurVal < Max)
				ScrollOp = 1;
		}
		else if ( CurVal > Min)
			ScrollOp = 2;
	}
	if ( ScrollOp > 0) {
		Delay = true;
		LoopDir = ScrollOp;
		redrawSpins();
		CommandQueue.append( this);
	}
	else
		LoopDir = 0;
	return true;
}

public boolean mouseMove( Event evt, int x, int y) {
	if ( Enabled && (x > Entry.Width) )
		Entry.setCursor( Awt.ActionCursor);
	return true;
}

public boolean mouseUp( Event evt, int x, int y) {
	if ( Enabled) {
		ScrollOp = 0;
		redrawSpins();
	}
	return true;
}

public Dimension preferredSize(){
	return new Dimension( 7*Awt.SysFontWidth, 3*Awt.SysFontHeight/2);
}

public void processKey( Object sel) {
	if ( sel.equals( "spinUp"))
		spinUp( SpinStep);
	else if ( sel.equals( "spinDown"))
		spinDown( SpinStep);
}

public void redraw( Graphics g) {
	redrawSpins( g);
}

void redrawSpins () {
	Graphics g = getGraphics();
	if ( g != null ) {
		redrawSpins( g);
		g.dispose();
	}
}

void redrawSpins( Graphics g) {
	int x = Entry.Width + FBorder.Ext;
	int w = Awt.ScrollBarWidth + 2;

	GraphicsLib.paintSpinButton( g, x, FBorder.Ext, w, Entry.Height, ScrollOp);
}

public void registerObserver() {
	Entry.OsFocusGot.addObserver( this);
	Entry.OsFocusLost.addObserver( this);
	Entry.OsEnter.addObserver( this);

	Entry.addHotKey( 'u', HotKey.Control, this, "spinUp");
	Entry.addHotKey( 'd', HotKey.Control, this, "spinDown");
}

public synchronized void setBackground( Color c){
	Entry.setBackground( c);
}

public void setContents( Object[] va){
	Values = va;
	Min    = 0;
	Max    = Values.length-1;
	CurVal = 0;

	Entry.setContents( Values[CurVal].toString());
}

public void setContents( int val){

	if ( val > Max)       val = Max;
	else if ( val < Min)  val = Min;

	CurVal = val;

	if ( Values != null)
		Entry.setContents( Values[CurVal].toString() );
	else
		Entry.setContents( Integer.toString( CurVal));

	OsValueChanged.notifyObservers();
}

public boolean setFocus(){
	return Entry.setFocus();
}

public synchronized void setFont( Font f){
	Entry.setFont( f);
}

public synchronized void setForeground( Color c){
	Entry.setForeground( c);
}

public void setRange( int min, int max){
	Values = null;
	Min    = min;
	Max    = max;
	CurVal = min;

	Entry.setContents( Integer.toString( CurVal));  
}

public void setRange( int min, int max, int cur){
	Values = null;
	Min    = min;
	Max    = max;
	CurVal = cur;

	Entry.setContents( Integer.toString( CurVal));  
}

public void setReadOnly( boolean state){
	Entry.setReadOnly( state);
}

public void setSpinStep( int st){
	SpinStep = st;
}

public void setStepContents( int val){
	setContents( val*SpinStep);
}

public void setStepRange( int min, int max){
	setRange( min*SpinStep, max*SpinStep);
}

public void spinDown( int steps) {
	setContents( CurVal - steps);
}

public void spinUp( int steps) {
	setContents( CurVal + steps);
}

public void update( Observable obs, Object arg) {
	if ( obs == Entry.OsFocusGot)
		OsFocusGot.notifyObservers();
	else if ( obs == Entry.OsFocusLost)
		OsFocusLost.notifyObservers();
	else if ( obs == Entry.OsEnter){
		try {			
			setContents( Integer.parseInt( Entry.getContents() ) );
		}
		catch ( NumberFormatException nfe) {
			int vi = getValueIdx( Entry.getContents() );
			if ( vi > -1) CurVal = vi;
		}
		OsSpinChanged.notifyObservers();
	}
}

public void updateRange( int min, int max){
	Min    = min;
	Max    = max;

	if ( CurVal > max)	  CurVal = max;
	else if ( CurVal < min) CurVal = min;

	Entry.setContents( Integer.toString( CurVal));  
}
}
