package biss.awt.kernel;

import biss.Queue;
import biss.awt.Awt;
import biss.awt.GUIHandler;
import biss.awt.GifProducer;
import java.awt.Button;
import java.awt.Canvas;
import java.awt.Checkbox;
import java.awt.CheckboxMenuItem;
import java.awt.Choice;
import java.awt.Color;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.FileDialog;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Frame;
import java.awt.Image;
import java.awt.Label;
import java.awt.List;
import java.awt.Menu;
import java.awt.MenuBar;
import java.awt.MenuItem;
import java.awt.Panel;
import java.awt.Scrollbar;
import java.awt.TextArea;
import java.awt.TextField;
import java.awt.Window;
import java.awt.image.ColorModel;
import java.awt.image.ImageObserver;
import java.awt.image.ImageProducer;
import java.awt.peer.*;
import java.io.File;
import java.net.URL;

/**
 * kernel implementation of java.awt.Toolkit.
 * Its main purposes are to create native window objects via Peer instances,
 * to transform native events into java.awt.Events and to dispatch them.
 * In doing so, it has to be careful to follow the GUI threading rules of
 * the platform it runs on (creating windows / dispatching events in the
 * same thread).
 * What really is shining is the fact that it uses ordinary biss.awt
 * classes to implement the specific java.awt user widgets (Lists,
 * TextFields etc.). Only the true generic windowing objects (like Canvas,
 * Frame..) are directly mapped to biss.awt.kernel classes.
 * What seems to be confusing to most people is the fact that biss.awt
 * objects acting as peers are on their part implemented by (generic)
 * peers (yes, peers which have peers). Take breath and sketch it down:
 *
 * java.awt.List -> biss.awt.List -> (Compound -> Panel) -> Peer
 *                                     (ListCanvas -> Canvas) -> Peer
 *                                     (Scrollbars -> Canvases) -> Peers
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class Toolkit
  extends java.awt.Toolkit
  implements Runnable
{
	NativeLib Lib;
	Queue CreateQueue;
	biss.awt.kernel.Graphics PaintGraphics;
	Peer[] PeerTable = new Peer[509];
	String Name;
	Thread EventThread;
	final static int PSEUDO_CLR = 3;

Toolkit () {
	this( Kernel.Display);
}

Toolkit ( String name ) {
	/*
	 * be careful to avoid any direct or indirect (via object creates of
	 * Peer etc.) reference of java.awt.getDefaultToolkit (biss.awt.Awt, for
	 * instance, does it from inside <cinit>). The private static field
	 * "toolkit" is not yet set (will be done on return of this method)
	 */
	Lib = new NativeLib( this, name, PeerTable, new Peer());

	if ( Dbg.Level > 0 ) {
		Lib.setDebugLevel( Dbg.Level );
		//		Dbg.log( "Toolkit ctor: " + name);
	}

	Name = name;
	PaintGraphics = new biss.awt.kernel.Graphics( Lib );

	Lib.initialize();
	initStdColors();
	initStdFonts();

	if ( Lib.HasKernelThreads )
		CreateQueue = new Queue(50);

	EventThread = new Thread( this);
	EventThread.setPriority( Thread.MAX_PRIORITY - 1);
	EventThread.start();
}

public int checkImage (Image image, int width, int height, ImageObserver observer) {
	return ((biss.awt.kernel.Image)image).State;
}

public ButtonPeer createButton ( Button target ) {
	if ( Lib.getTargetFont( target) == null )
		target.setFont( Awt.ButtonFont);

	return new biss.awt.Button( target);
}

public CanvasPeer createCanvas ( Canvas target ) {

	//	Dbg.log( "createCanvas: " + target);
	Peer wnd = new Peer( Lib, target, Peer.CANVAS);

	createPeer( wnd);
	return wnd;  
}

public CheckboxPeer createCheckbox ( Checkbox target ) {
	if ( Lib.getTargetFont( target) == null )
		target.setFont( Awt.ButtonFont);

	return new biss.awt.Checkbox( target);
}

public CheckboxMenuItemPeer createCheckboxMenuItem (CheckboxMenuItem target) {
	return null;
}

public ChoicePeer createChoice (Choice target) {
	return new biss.awt.Choice( target);
}

public DialogPeer createDialog (Dialog target) {
	return null;
}

public FileDialogPeer createFileDialog (FileDialog target) {
	return null;
}

public FramePeer createFrame ( Frame target ) {
	//	Dbg.log( "Display.createFrame: " + target);

	Peer wnd = new Peer( Lib, target, Peer.FRAME);
	createPeer( wnd);
	return wnd;
}

public Image createImage (ImageProducer producer) {
	return new biss.awt.kernel.Image( Lib, producer);
}

public LabelPeer createLabel ( Label target ) {
	if ( Lib.getTargetFont( target) == null )
		target.setFont( Awt.StaticFont);

	if ( Lib.getTargetForeground( target) == null )
		target.setForeground( Awt.StaticForeClr);

	if ( Lib.getTargetBackground( target) == null )
		target.setBackground( Awt.StaticBackClr);	

	return new biss.awt.Label( target);
}

public ListPeer createList (List target) {
	return new biss.awt.List( target);
}

public MenuPeer createMenu (Menu target) {
	return new biss.awt.Menu();
}

public MenuBarPeer createMenuBar (MenuBar target) {
	return null;
}

public MenuItemPeer createMenuItem (MenuItem target) {
	return null;
}

public PanelPeer createPanel ( Panel target ) {
	//	Dbg.log( "createPanel: " + target);
	Peer wnd = new Peer( Lib, target, Peer.PANEL);

	createPeer( wnd);
	return wnd;  
}

void createPeer ( Peer wnd ) {
	if ( !Lib.HasKernelThreads || Thread.currentThread() == EventThread ) {
		//		Dbg.log( "createPeer()");
		Lib.createWindow( wnd);
	}
	else {
		try {
			CreateQueue.append( this);
			//			Dbg.log( "createPeer queued..");
			while ( wnd.PData == 0 )
				wait();
			//			Dbg.log( "createPeer processed");
		}
		catch ( InterruptedException x ) {}
	}
}

public ScrollbarPeer createScrollbar (Scrollbar target) {
	return null;
}

public TextAreaPeer createTextArea (TextArea target) {
	if ( Lib.getTargetFont( target) == null )
		target.setFont( Awt.FixedFont);

	if ( Lib.getTargetForeground( target) == null )
		target.setForeground( Awt.EntryForeClr);

	if ( Lib.getTargetBackground( target) == null )
		target.setBackground( Awt.EntryBackClr);	

	return new biss.awt.TextArea( target);
}

public TextFieldPeer createTextField (TextField  target) {
	if ( Lib.getTargetFont( target) == null )
		target.setFont( Awt.SysFont);

	if ( Lib.getTargetForeground( target) == null )
		target.setForeground( Awt.EntryForeClr);

	if ( Lib.getTargetBackground( target) == null )
		target.setBackground( Awt.EntryBackClr);	

	return new biss.awt.TextField( target);
}

public WindowPeer createWindow ( Window  target ) {
	//	Dbg.log( "Display.createWindow: " + target);
	int type = Peer.WINDOW;

	// trigger (optional) native popup support (works in combination with
	// NativeLib.popDownHint evaluation in run()
	if ( Kernel.HasNativePopUps ) {
		if ( (target instanceof biss.awt.ChoiceWindow) ||
		     (target instanceof biss.awt.MenuWindow && !GUIHandler.hasSingleInstWnd()) )
			type |= Peer.POPUP;
	}

	Peer wnd = new Peer( Lib, target, type);

	createPeer( wnd);
	return wnd;
}

void dumpPeers() {
	for ( int i=0; i<PeerTable.length; i++ ) {
		if ( (PeerTable[i] != null) && (PeerTable[i] != Lib.RecycledPeer) )
			System.out.println( "[" + i + "] " + PeerTable[i]);
	}
}

public ColorModel getColorModel () {
	return null;
}

public String[] getFontList () {
	String ret[] = {"Helvetica", "TimesRoman", "Courier", "Symbol"};
	return ret;
}

public FontMetrics getFontMetrics (Font font) {
	return new biss.awt.kernel.FontMetrics( Lib, font);
}

public java.awt.Image getImage ( String gifFile ) {
	GifProducer gp = new GifProducer( new File( gifFile));

	return new biss.awt.kernel.Image( Lib, gp);
}

public Image getImage ( URL url ) {
	GifProducer gp = new GifProducer( url);

	return new biss.awt.kernel.Image( Lib, gp);
}

public int getScreenResolution () {
	return Lib.Resolution;
}

public Dimension getScreenSize () {
	return new Dimension( Lib.Width, Lib.Height);
}

void initStdColors () {
	Lib.initStdColor( Color.white);
	Lib.initStdColor( Color.lightGray);
	Lib.initStdColor( Color.gray);
	Lib.initStdColor( Color.darkGray);
	Lib.initStdColor( Color.black);
	Lib.initStdColor( Color.red);
	Lib.initStdColor( Color.pink);
	Lib.initStdColor( Color.orange);
	Lib.initStdColor( Color.yellow);
	Lib.initStdColor( Color.green);
	Lib.initStdColor( Color.magenta);
	Lib.initStdColor( Color.cyan);
	Lib.initStdColor( Color.blue);
}

void initStdFonts () {

	// numeric key - font mapping is assumed to be fix, don't change it !
	if ( Kernel.Courier != null )    Lib.setStdFontSpec( 0, Kernel.Courier);
	if ( Kernel.Helvetica != null )  Lib.setStdFontSpec( 1, Kernel.Helvetica);
	if ( Kernel.TimesRoman != null ) Lib.setStdFontSpec( 2, Kernel.TimesRoman);
	if ( Kernel.Dialog != null )     Lib.setStdFontSpec( 3, Kernel.Dialog);
	if ( Kernel.Symbol != null )     Lib.setStdFontSpec( 4, Kernel.Symbol);
}

public boolean prepareImage (Image image, int width, int height, ImageObserver observer) {
	((biss.awt.kernel.Image)image).queueProduction( observer);
	return true;
}

public void run () {
	Peer wnd;
	int  idx;

	//  Dbg.log( "Toolkit.run");

	while ( true ) {
		try {
			while ( true ) {
				while ( (idx = Lib.getNextEvent()) >= 0 ) {
					wnd = PeerTable[idx];

					// I don't consider this the right way since it think the
					// paint(.) call is the default system behavior for WINDOW_EXPOSE
					// event processing. Therefor it should be possible to override it.
					// However, the SDK behaves different, it calls paint(.) regardless
					// of the handleEvent return value
					if ( Lib.NextEvent.id == Event.WINDOW_EXPOSE ) {
						PaintGraphics.Wnd = wnd;
						PaintGraphics.ClipRect = null;
						Lib.initPaintGraphics( PaintGraphics);
						wnd.Target.paint( PaintGraphics);
						Lib.releasePaintGraphics( PaintGraphics);
					}

					wnd.handleEvent( Lib.NextEvent);
				}

				if ( Lib.SyncHint ) {
					Lib.sync();
					Lib.SyncHint = false;
				}

				if ( Lib.PopDownHint ) {
					GUIHandler.disposeSingleInstWnd( null);
					Lib.PopDownHint = false;
				}

				if ( Lib.HasKernelThreads ) {
					while ( !CreateQueue.empty() ) {
						wnd = (Peer) CreateQueue.getNext();
						Lib.createWindow( wnd );
						notify();
					}
				}

				if ( !Lib.AttentionHint )
					Thread.sleep( 30);
			}
		}
		catch ( Exception x ) {
			x.printStackTrace();
		}
	}
}

public void sync () {
	Lib.sync();
}
}
