package biss.jde;

import biss.Scanner;
import biss.awt.Awt;
import biss.awt.List;
import biss.awt.SelfDrawingObject;
import java.awt.Color;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.io.PrintStream;

/**
 * auxiliary class used by the Designer to represent designable widgets 
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
class DesignObject
  implements SelfDrawingObject
{
	Rectangle Rect = new Rectangle();
	String ClsName;
	final static int CPW = 7;
	String FldName;
	Point RatioOrigin = new Point(0,0);
	Point RatioCorner = new Point(0,0);
	Rectangle[] RectHandles = new Rectangle[8];
	Rectangle[] RatioHandles = new Rectangle[2];
	DesignObjLayout LoSpec;
	String Text;
	final static FontMetrics Fm = Awt.DefToolkit.getFontMetrics( Awt.StaticFont);
	boolean HasInners = false;
	static DesignDrawInterface ExternalDraw;
	String InitParams;
	String PkgName;
	String ClsPath;

static {
	if ( Jde.DesignDrawCls != null) {
		try {
			Class ddc = Class.forName( Jde.DesignDrawCls);
			ExternalDraw = (DesignDrawInterface)ddc.newInstance();
		}
		catch( Throwable t){
			System.out.println( "invalid Jde.DesignDrawCls specification");
		}
	}
}

public DesignObject () {
}

public DesignObject ( Scanner s) {
	char[]  da = { ',', ' ', ')'};
	int[]   iv = new int[10];
	boolean duo;
	int     i;

	s.skipTo( '(');
	s.skip( 1); s.skipBlanks(); s.nextToken( da);
	FldName = s.string();

	for ( i=0; i<4; i++){
		s.skip( 1); s.skipBlanks(); s.nextToken( da);
		iv[i] = Integer.parseInt( s.string());
	}

	s.skip( 1); s.skipBlanks(); s.nextToken( da);
	duo = s.string().equals( "true");

	for ( i=4; i<9; i++){
		s.skip( 1); s.skipBlanks(); s.nextToken( da);
		iv[i] = Integer.parseInt( s.string());
	}

	LoSpec = new DesignObjLayout( iv[0], iv[1], iv[2], iv[3],
	                              duo, iv[4], iv[5], iv[6], iv[7], iv[8]); 

	initHandles();
}

public DesignObject ( String cName, String fName, String txt, int tw, int th) {
	this( cName, fName, txt, 30, 30, 100, 30, tw, th);
}

public DesignObject (String cName, String fName, String params,
              int x, int y, int width, int height, int tw, int th) {
	LoSpec = new DesignObjLayout();
	initHandles();

	Rect.reshape( x, y, width, height);
	RatioOrigin.x = 0; RatioOrigin.y = 0;
	RatioCorner.x = 0; RatioCorner.y = 0;

	FldName = fName;

	updateHandles( tw, th);
	setClass( cName);
	setInitParams( params);
}

void adjustHandle ( int x, int y, Rectangle r, int width, int height) {
	int cw = CPW/2;
	int nx = x - cw;
	int ny = y - cw;

	if ( nx < 0)								nx = 0;
	else if ( nx > width - CPW)	nx = width - CPW;
	if ( ny < 0) 								ny = 0;
	else if ( ny > height -CPW) ny = height - CPW;

	r.reshape( nx, ny, CPW, CPW);
}

public Object clone(){
	DesignObject dob = new DesignObject();
	dob.Rect = new Rectangle( Rect.x, Rect.y, Rect.width, Rect.height);
	dob.RatioOrigin = new Point( RatioOrigin.x, RatioOrigin.y);
	dob.RatioCorner = new Point( RatioCorner.x, RatioCorner.y);
	dob.ClsName     = new String( ClsName);
	dob.PkgName     = new String( PkgName);
	dob.FldName     = new String( FldName);

	dob.LoSpec = (DesignObjLayout)LoSpec.clone();

	if ( Text != null)
		dob.Text = new String( Text);
	if ( InitParams != null)
		dob.InitParams = new String( InitParams);

	dob.initHandles();
	return (Object)dob;
}

int detectHandlePos ( int x, int y) {
	for ( int i=0; i<RectHandles.length; i++){
		if ( RectHandles[i].inside( x, y))
			return i;
	}
	for ( int i=RatioHandles.length-1; i>=0; i--){
		if ( RatioHandles[i].inside( x, y))
			return i+9;
	}
	if ( Rect.inside( x, y))
		return 8;

	return -1;
}

void drawExtended (Graphics g, boolean sel, boolean multiSel) {
	if ( ExternalDraw != null)
		ExternalDraw.drawDesignObject( ClsName, Text, Rect, HasInners);
	else if ( ! HasInners) {
		g.setColor( Color.white);
		g.fillRect( Rect.x+1, Rect.y+1, Rect.width-2, Rect.height-2);

		if ( hasText() ) g.setColor( Color.blue);
		else             g.setColor( Color.green);

		g.drawString( getIdString( 3), Rect.x + 3, Rect.y + Rect.height -
		              (Rect.height - Fm.getHeight()) / 2 - Fm.getDescent() );
	}
	drawMinimal( g, sel, multiSel);
}

public int drawHeight(){
	return Awt.SysFontHeight;
}

void drawMinimal (Graphics g, boolean sel, boolean multiSel) {
	g.setColor( Color.black);
	g.drawRect( Rect.x, Rect.y, Rect.width-1, Rect.height-1);
	if ( multiSel)
		drawRectHandles( g, true);
	else if ( sel) {
		drawRectHandles( g, false);
		drawRatioHandles( g);
	}
}

void drawRatioHandles (Graphics g) {
	g.setColor( Color.blue);
	g.drawLine( RatioOrigin.x, RatioOrigin.y, Rect.x, Rect.y);
	g.drawLine( RatioCorner.x, RatioCorner.y, Rect.x + Rect.width, Rect.y + Rect.height);

	for ( int i=0; i<RatioHandles.length; i++){
		Rectangle r = RatioHandles[i];
		g.fillRect( r.x, r.y, r.width, r.height);
	}
}

void drawRectHandles (Graphics g, boolean multiSel) {
	if ( multiSel) g.setColor( Color.green);
	else           g.setColor( Color.red);

	for ( int i=0; i<RectHandles.length; i++){
		Rectangle r = RectHandles[i];
		g.fillRect( r.x, r.y, r.width, r.height);
	}
}

public void drawSelfIn( Object pane){
	List lst = (List)pane;
	Graphics g = lst.DrawGraphics;
	Rectangle r = lst.DrawRect;

	g.setFont( Awt.SysFont);
	g.drawString( FldName, r.x + 5,
	              r.y + r.height -
	              (r.height - Awt.SysFontHeight) / 2 -
	              Awt.SysFontDesc );
}

public boolean equals ( Object o) {
	if ( ! ( o instanceof DesignObject))
		return false;
	DesignObject dob = (DesignObject) o;
	if ( dob.ClsName.equals( ClsName) &&
	     dob.PkgName.equals( PkgName) &&
	     dob.FldName.equals( FldName) &&
	     dob.Rect.equals( Rect) )
		return true;
	return false;
}

String getIdString ( int xOffs) {
	int    maxL = Rect.width - xOffs;
	String ids = hasText() ? Text : "<" + ClsName + ">";

	if ( Fm.stringWidth( ids) < maxL )
		return ids;

	int cLen = xOffs;
	for ( int i=0; i<ids.length(); i++){
		int cl = Fm.charWidth( ids.charAt( i));
		if ( cLen + cl > maxL)
			return ids.substring( 0, i) + ".";
		cLen += cl;
	}

	return null;
}

String getInitParams () {
	if (InitParams == null)
		return "";
	return InitParams;
}

boolean hasText () {
	return ( (Text != null) && ( Text.length() > 0) );
}

void initHandles () {
	for ( int i=0; i<RectHandles.length; i++)
		RectHandles[i] = new Rectangle();
	for ( int i=0; i<RatioHandles.length; i++)
		RatioHandles[i] = new Rectangle();

	//  updateHandles();
}

public String label(){
	return FldName;
}

public int maxDrawWidth(){
	return 30*Awt.SysFontWidth;
}

void printLayoutOn ( PrintStream s) {
	s.print( FldName + ", ");
	LoSpec.printLayoutOn( s);
}

void setClass ( String cls) {
	int pkLen = cls.lastIndexOf( '.');

	if ( pkLen > -1){
		PkgName = cls.substring( 0, pkLen);
		ClsName = cls.substring( pkLen + 1, cls.length());
		if ( PkgName.equals( "biss.awt"))
			ClsPath = ClsName;
		else
			ClsPath = cls;
	}
	else {
		PkgName = "biss.awt";
		ClsName = cls;
		ClsPath = cls;
	}

}

void setInitParams ( String params) {
	// set ct parameters from file or dialog
	if ( params.length() == 0) {
		Text = "";
		InitParams = "";
		return;
	}

	InitParams = params.trim();

	char lc = '"';
	int fi = InitParams.indexOf( lc);
	int ni = InitParams.indexOf( lc, fi + 1);

	// get text representation, first literal ...
	if ( (fi > -1) && (ni > -1) ){
		Text = InitParams.substring( fi + 1, ni);
	}
	else if ( (fi == -1) && (ni == -1) ){
		if ( InitParams.indexOf( '~') > -1){
			Text = InitParams;
			InitParams = "\"" + InitParams + "\""; 
		}
	}
	else{ // or nothing
		Text = null;
	}

}

String sizeString () {
	return new String( Rect.width + "x" + Rect.height);
}

void updateHandles (int width, int height) {
	int cw = CPW/2;
	int hh = Rect.height/2;
	int ww = Rect.width/2;

	adjustHandle( Rect.x,      Rect.y + hh, RectHandles[0], width, height);
	adjustHandle( Rect.x,      Rect.y,      RectHandles[1], width, height);
	adjustHandle( Rect.x + ww, Rect.y,      RectHandles[2], width, height);

	adjustHandle( Rect.x + Rect.width - 1, Rect.y,       RectHandles[3], width, height);
	adjustHandle( Rect.x, Rect.y + Rect.height - 1,      RectHandles[4], width, height);
	adjustHandle( Rect.x + ww, Rect.y + Rect.height - 1, RectHandles[5], width, height);
	adjustHandle( Rect.x + Rect.width - 1, Rect.y + hh,  RectHandles[7], width, height);
	adjustHandle( Rect.x + Rect.width - 1, Rect.y + Rect.height - 1, RectHandles[6], width, height);

	adjustHandle( RatioOrigin.x, RatioOrigin.y, RatioHandles[0], width, height);
	adjustHandle( RatioCorner.x, RatioCorner.y, RatioHandles[1], width, height);
}

void updateHandlesX () {
	int cw = CPW/2;

	RectHandles[0].reshape( Rect.x - cw, Rect.y + Rect.height/2 - cw, CPW, CPW);
	RectHandles[1].reshape( Rect.x - cw, Rect.y - cw, CPW, CPW);
	RectHandles[2].reshape( Rect.x + Rect.width/2 - cw, Rect.y - cw, CPW, CPW);
	RectHandles[3].reshape( Rect.x + Rect.width - cw - 1, Rect.y - cw, CPW, CPW);
	RectHandles[4].reshape( Rect.x - cw, Rect.y + Rect.height - cw - 1, CPW, CPW);
	RectHandles[5].reshape( Rect.x + Rect.width/2 - cw, Rect.y + Rect.height - cw - 1, CPW, CPW);
	RectHandles[6].reshape( Rect.x + Rect.width - cw - 1, Rect.y + Rect.height - cw - 1, CPW, CPW);
	RectHandles[7].reshape( Rect.x + Rect.width - cw - 1, Rect.y + Rect.height/2 - cw, CPW, CPW);

	RatioHandles[0].reshape( RatioOrigin.x - cw, RatioOrigin.y - cw, CPW, CPW);
	RatioHandles[1].reshape( RatioCorner.x - cw, RatioCorner.y - cw, CPW, CPW);
}

void updateLayoutSpec ( int width, int height) {
	// Rect changed ( by DesignPane), update LayoutSpec
	LoSpec.ratioFromPelPoints( width, height, RatioOrigin, RatioCorner);
	LoSpec.offsetFromPelPoints( Rect, RatioOrigin, RatioCorner);
}

void updateRect ( int width, int height) {
	// LayoutSpec changed, update Rect
	LoSpec.ratioToPelPoints( width, height, RatioOrigin, RatioCorner);
	LoSpec.rectFromPelPoints( Rect, RatioOrigin, RatioCorner);
	updateHandles( width, height);
}
}
