/*
 * leap-frog point to allow dmalloc on/off via relink.
 *
 * Copyright 1995 by Gray Watson
 *
 * This file is part of the dmalloc package.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * NON-COMMERCIAL purpose and without fee is hereby granted, provided
 * that the above copyright notice and this permission notice appear
 * in all copies, and that the name of Gray Watson not be used in
 * advertising or publicity pertaining to distribution of the document
 * or software without specific, written prior permission.
 *
 * Please see the PERMISSIONS file or contact the author for information
 * about commercial licenses.
 *
 * Gray Watson makes no representations about the suitability of the
 * software described herein for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * The author may be contacted at gray.watson@letters.com
 */

/*
 * If anyone can think of a better way to do this *please* let me
 * know.
 *
 * The goal of the routines in this file is to allow people to use the
 * debug library during development by including the dmalloc.h file in
 * their C source files, and then disable and return to the system (or
 * more efficient) malloc functions easily.
 *
 * The dmalloc.h file provides the library with file/line information
 * with each call.  Backtracing the stack-frame is neither portable or
 * necessarily easy so the cpp __FILE__ and __LINE__ directives are
 * used instead to do this.
 *
 * dmalloc.h contains macros which override the malloc, calloc, free,
 * etc. functions so that they call _malloc_leap passing the file and
 * line information to these routines.  But we need to provide a
 * "quick-release" functionality so that people will not have to
 * remove the dmalloc.h includes and recompile to use other malloc
 * libraries.
 *
 * I have decided on the leap-frog routines in this file that will have
 * to *always* be compiled in if you include dmalloc.h in your source.
 * When your source wants to call malloc, it will first make a call to
 * _malloc_leap which will in turn call the malloc of your choice.
 * This does mean that an extra function call per memory interaction
 * will occur, but on most systems this is reasonably cheap.  To fully
 * disable the library, you will need to remove the dmalloc.h include
 * and recompile your source files.
 *
 * Please mail me with any reasonable ideas.
 */

#define DMALLOC_DISABLE

#include "dmalloc.h"				/* for DMALLOC_SIZE... */
#include "conf.h"				/* for const */

#include "dmalloc_loc.h"			/* for EXPORT */
#include "dmalloc_lp.h"

#if INCLUDE_RCS_IDS
static	char	*rcs_id =
  "$Id: dmalloc_lp.c,v 1.33 1997/12/22 00:24:38 gray Exp $";
#endif

/*
 * exported variables
 */
/* to inform the dmalloc library from which file the call comes from */
char		*_dmalloc_file = DMALLOC_DEFAULT_FILE;

/* to inform the library from which line-number the call comes from */
unsigned int	_dmalloc_line = DMALLOC_DEFAULT_LINE;

/* pre-set dmalloc_debug() value before the library is setup */
int		_dmalloc_debug_preset = DEBUG_PRE_NONE;

/* pointers to shutdown function to allow calls without linked routine */
void		(*_dmalloc_shutdown_func)() = NULL;

/* pointers to log_heap_map function to allow calls without linked routine */
void		(*_dmalloc_log_heap_map_func)() = NULL;

/* pointers to log_stats function to allow calls without linked routine */
void		(*_dmalloc_log_stats_func)() = NULL;

/* pointers to log_unfreed function to allow calls without linked routine */
void		(*_dmalloc_log_unfreed_func)() = NULL;

/* pointers to verify function to allow calls without linked routine */
int		(*_dmalloc_verify_func)() = NULL;

/* pointers to verify function to allow calls without linked routine */
int		(*_malloc_verify_func)() = NULL;

/* pointers to debug function to allow calls without linked routine */
void		(*_dmalloc_debug_func)() = NULL;

/* pointers to debug_current function to allow calls without linked routine */
int		(*_dmalloc_debug_current_func)() = NULL;

/* pointers to examine function to allow calls without linked routine */
int		(*_dmalloc_examine_func)() = NULL;

/* pointers to strerror function to allow calls without linked routine */
const char	*(*_dmalloc_strerror_func)(const int error_num) = NULL;

#undef malloc
/*
 * leap routine to malloc
 */
DMALLOC_PNT	_malloc_leap(const char *file, const int line,
			     DMALLOC_SIZE size)
{
  void	*ret;
  
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
  ret = malloc(size);
  
  return ret;
}

#undef calloc
/*
 * leap routine to calloc
 */
DMALLOC_PNT	_calloc_leap(const char *file, const int line,
			     DMALLOC_SIZE ele_n, DMALLOC_SIZE size)
{
  void	*ret;
  
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
  ret = calloc(ele_n, size);
  
  return ret;
}

#undef realloc
/*
 * leap routine to realloc
 */
DMALLOC_PNT	_realloc_leap(const char *file, const int line,
			      DMALLOC_PNT old_p, DMALLOC_SIZE new_size)
{
  void	*ret;
  
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
  ret = realloc(old_p, new_size);
  
  return ret;
}

#undef free
/*
 * leap routine to free
 */
DMALLOC_FREE_RET	_free_leap(const char *file, const int line,
				   DMALLOC_PNT pnt)
{
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
#if defined(__STDC__) && __STDC__ == 1
  free(pnt);
#else
  return free(pnt);
#endif
}

/***************************** xmalloc functions *****************************/

/*
 * leap routine to calloc with error checking
 */
DMALLOC_PNT	_xcalloc_leap(const char *file, const int line,
			      DMALLOC_SIZE ele_n, DMALLOC_SIZE size)
{
  void	*ret;
  
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
  ret = calloc(ele_n, size);
  if (ret == NULL) {
    char	str[256];
    (void)sprintf(str,
		  "Out of memory while allocating %d bytes from '%s:%d'\n",
		  size, file, line);
    (void)write(STDERR, str, strlen(str));
    _exit(1);
  }
  
  return ret;
}

/*
 * leap routine to free
 */
DMALLOC_FREE_RET	_xfree_leap(const char *file, const int line,
				    DMALLOC_PNT pnt)
{
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
#if defined(__STDC__) && __STDC__ == 1
  free(pnt);
#else
  return free(pnt);
#endif
}

/*
 * leap routine to malloc with error checking
 */
DMALLOC_PNT	_xmalloc_leap(const char *file, const int line,
			      DMALLOC_SIZE size)
{
  void	*ret;
  
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
  ret = malloc(size);
  if (ret == NULL) {
    char	str[256];
    (void)sprintf(str,
		  "Out of memory while allocating %d bytes from '%s:%d'\n",
		  size, file, line);
    (void)write(STDERR, str, strlen(str));
    _exit(1);
  }
  
  return ret;
}

/*
 * leap routine to realloc with error checking
 */
DMALLOC_PNT	_xrealloc_leap(const char *file, const int line,
			       DMALLOC_PNT old_p, DMALLOC_SIZE new_size)
{
  void	*ret;
  
  _dmalloc_file = (char *)file;
  _dmalloc_line = line;
  
  ret = realloc(old_p, new_size);
  if (ret == NULL) {
    char	str[256];
    (void)sprintf(str,
		  "Out of memory while allocating %d bytes from '%s:%d'\n",
		  new_size, file, line);
    (void)write(STDERR, str, strlen(str));
    _exit(1);
  }
  
  return ret;
}

/*
 * leap routine for strdup with error checking
 */
char 	*_xstrdup_leap(const char *file, const int line,
		       const char *str)
{
  char 	*buf;
  int	len;
  
  /* len + NULLC */
  len = strlen(str) + 1;
  
  /* xmalloc checks for NULL */
  buf = (char *)_xmalloc_leap(file, line, len);
  (void)strcpy(buf, str);
  
  return buf;
}

/*********************** routines when running dmalloc ***********************/

/*
 * routine to call dmalloc_shutdown when linked in
 */
void	dmalloc_shutdown(void)
{
  if (_dmalloc_shutdown_func != NULL) {
    _dmalloc_shutdown_func();
  }
}

/*
 * routine to call dmalloc_log_unfreed when linked in
 */
void	dmalloc_log_unfreed(void)
{
  if (_dmalloc_log_unfreed_func != NULL) {
    _dmalloc_log_unfreed_func();
  }
}

/*
 * routine to call dmalloc_log_stats when linked in
 */
void	dmalloc_log_stats(void)
{
  if (_dmalloc_log_stats_func != NULL) {
    _dmalloc_log_stats_func();
  }
}

/*
 * routine to call dmalloc_log_heap_map when linked in
 */
void	dmalloc_log_heap_map(void)
{
  if (_dmalloc_log_heap_map_func != NULL) {
    _dmalloc_log_heap_map_func();
  }
}

/*
 * routine to call dmalloc_verify when linked in
 */
int	dmalloc_verify(const DMALLOC_PNT pnt)
{
  if (_dmalloc_verify_func != NULL) {
    return _dmalloc_verify_func(pnt);
  }
  else {
    return DMALLOC_VERIFY_NOERROR;
  }
}

/*
 * routine to call malloc_verify when linked in
 */
int	malloc_verify(const DMALLOC_PNT pnt)
{
  if (_malloc_verify_func != NULL) {
    return _malloc_verify_func(pnt);
  }
  else {
    return DMALLOC_VERIFY_NOERROR;
  }
}

/*
 * routine to call dmalloc_debug when linked in
 */
void	dmalloc_debug(const int flags)
{
  if (_dmalloc_debug_func == NULL) {
    _dmalloc_debug_preset = flags;
  }
  else {
    _dmalloc_debug_func(flags);
  }
}

/*
 * routine to call dmalloc_debug_current when linked in
 */
int	dmalloc_debug_current(void)
{
  if (_dmalloc_debug_current_func != NULL) {
    return _dmalloc_debug_current_func();
  }
  else {
    return 0;
  }
}

/*
 * routine to call dmalloc_examine when linked in
 */
int	dmalloc_examine(const DMALLOC_PNT pnt, DMALLOC_SIZE *size_p,
			char **file_p, unsigned int *line_p,
			DMALLOC_PNT *ret_attr_p)
{
  if (_dmalloc_examine_func != NULL) {
    return _dmalloc_examine_func(pnt, size_p, file_p, line_p, ret_attr_p);
  }
  else {
    return ERROR;
  }
}

/*
 * routine to call dmalloc_strerror when linked in
 */
const char	*dmalloc_strerror(const int error_num)
{
  if (_dmalloc_strerror_func != NULL) {
    return _dmalloc_strerror_func(error_num);
  }
  else {
    return "unknown";
  }
}
