/*
** Copyright (c) Massachusetts Institute of Technology 1994-1998.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. Works derived from this code are not distributed for
**        commercial gain without explicit permission from MIT 
**        (for permission contact lclint-request@sds.lcs.mit.edu).
*/
/*
** tokentable.c
**
** Larch processors token table
** This table stores predefined tokens for LCL.
*/

# include "lclintMacros.nf"
# include "llbasic.h"
# include "lcltokentable.h"

static long unsigned MaxToken; 
static /*@only@*/ /*@null@*/ o_ltoken *LCLTokenTable = NULL;    

static void AllocTokenTable (void) 
   /*@globals LCLTokenTable@*/
   /*@modifies LCLTokenTable, MaxToken@*/;

ltoken 
  LCLInsertToken (ltokenCode cod, lsymbol sym, lsymbol rTxt,
		  bool isPredefined)
{
  /*
  ** If the token is already in the token table, it is returned.  Otherwise,
  ** the token is inserted into the table and then returned.
  **
  ** A new TokenTable is allocated when:
  **   . The TokenTable[] is empty (initial case)
  **   . The location where to insert the token is not in TokenTable[]
  */

  setCodePoint ();

  while (sym >= MaxToken)
    {
      setCodePoint ();
      /* No more space available.  Allocate more. */
      AllocTokenTable ();
    }
  
  llassert (LCLTokenTable != NULL);

  if (ltoken_isUndefined (LCLTokenTable[sym]))
    {
      LCLTokenTable[sym] = ltoken_create (cod, sym);      
      ltoken_setRawText (LCLTokenTable[sym], rTxt);
      ltoken_setDefined (LCLTokenTable[sym], isPredefined);
            return LCLTokenTable[sym];
    }

    return LCLTokenTable[sym];
}

void LCLUpdateToken (ltokenCode cod, lsymbol sym, bool def)
{
  llassert (LCLTokenTable != NULL);

  if (!ltoken_isUndefined (LCLTokenTable[sym]))
    {
      ltoken_setCode (LCLTokenTable[sym], cod);
      ltoken_setDefined (LCLTokenTable[sym], def);
    }
  else
    {
      llfatalbug (message ("LCLUpdateToken: %s", 
			   cstring_fromChars (lsymbol_toChars (sym))));
    }
}

void LCLSetTokenHasSyn (lsymbol sym, bool syn)
{
  llassert (LCLTokenTable != NULL);

  if (!ltoken_isUndefined (LCLTokenTable[sym]))
    {
      ltoken_setHasSyn (LCLTokenTable[sym], syn);
    }
  else
    {
      llfatalbug (message ("LCLSetTokenHasSyn: null token (%d)", (int)sym));
    }
}

ltoken LCLGetToken (lsymbol sym)
{
  llassert (LCLTokenTable != NULL);
  llassert (sym < MaxToken);

  return LCLTokenTable[sym];
}

#if 0
bool LCLTokenTableContainsToken (ltoken tok)
{
  unsigned long i;

  if (LCLTokenTable != NULL) {
    for (i = 0; i < MaxToken; i++) {
      if (LCLTokenTable[i] == tok) {
	return TRUE;
      }
    }
  }

  return FALSE;
}
# endif

/*@exposed@*/ ltoken
LCLReserveToken (ltokenCode cod, char *txt)
{
  /*
  ** The same context that was active when the string-handle
  ** was derived by a previous call to lsymbol_fromChars (),
  ** must be established.
  */
  lsymbol sym;

  setCodePoint ();
  sym = lsymbol_fromChars (txt);
  
  /* 
  ** Reserved tokens never have raw text like synonyms.
  */

    return (LCLInsertToken (cod, sym, lsymbol_undefined, TRUE));
}

static void
  AllocTokenTable (void) /*@globals LCLTokenTable; @*/ 
{
  long unsigned oldSize, newSize;
  long unsigned int i;
  
  oldSize = MaxToken;

  if (oldSize == 0)
    {
      newSize = INITTOKENTABLE;
      llassert (LCLTokenTable == NULL);
      LCLTokenTable = (ltoken *) dmalloc (newSize * sizeof (*LCLTokenTable));
    }
  else
    {
      o_ltoken *oldLCLTokenTable = LCLTokenTable;

      llassert (oldLCLTokenTable != NULL);

      newSize = (long unsigned) (DELTATOKENTABLE * oldSize);
      LCLTokenTable = (ltoken *) dmalloc (newSize * sizeof (*LCLTokenTable));

      for (i = 0; i < oldSize; i++)
	{
	  LCLTokenTable[i] = oldLCLTokenTable[i];
	}
      
      sfree (oldLCLTokenTable);
    }

  MaxToken = newSize;
  
  /*@+loopexec@*/
  for (i = oldSize; i < newSize; i++)
    {
      LCLTokenTable[i] = ltoken_undefined;
    }
  /*@=loopexec@*/
/*@-compdef@*/ } /*=compdef@*/

void
LCLTokenTableInit (void)
{
    MaxToken = 0;
}

void
LCLTokenTableCleanup (void)
{
    
  if (LCLTokenTable != NULL)
    {
      long unsigned i;
      
      for (i = 0; i < MaxToken; i++)
	{
	  ltoken tok = LCLTokenTable[i];
	  
	  LCLTokenTable[i] = NULL;
	  /*@-dependenttrans@*/ ltoken_free (tok);
	  /*@=dependenttrans@*/
	}
      
      sfree (LCLTokenTable); 
      LCLTokenTable = NULL;
    }

  }










