/* 
 * Based on README.aztcd from the linux-sources (Tiny Audio CD Player)
 *       and CDPlay 0.31 from Sariel Har-Peled
 * There is not much left from the original code.
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <linux/cdrom.h>
#include <errno.h>

#include "cthugha.h"
#include "cd_player.h"
#include "keys.h"
#include "display.h"
#include "action.h"
#include "sound.h"


static struct cdrom_tochdr   tocHdr;
struct cdrom_subchnl  subchnl; 
static struct cdrom_tocentry entry;
static struct cdrom_msf      msf; 
union  { 
    struct cdrom_msf msf;
    unsigned char buf[2336];
} azt;
struct cdrom_volctrl  volctrl;
static int cd_playing = 0;			/* already playing ? */
int first, last;
  
int cd_handle = -1;				/* handle of /dev/cdrom */
int cd_in_pause = 0;				/* now in pause-mode */
int cd_first_track = -1;			/* start with track nr */
int cd_loop = 0;
int cd_randomplay = 0;
int cd_eject_on_end = 0;
int *random_played;
int cd_stop_on_exit = 0;				
int ejected=0;
int no_disk=0;
static int rand_ini=0;
static int stop_track=-1;

/*
 * Initialization. If cd_first_track is >= 0 then start playing at the
 *  specified track.
 */
int init_cd() {

    printfv(1,"Initializing CD...\n");

    cd_handle=open(DEV_CDROM, O_RDONLY);
  
    if (cd_handle <= 0) {
	printfee("CD: already playing, no audio disk, door open");
	no_disk = 1;
	return 1;
    }
	
    if ( cd_first_track >= 0) {
	if ( cd_randomplay) {
	    printfv(1,"  Starting random play of CD\n");
	    cd_next(0);
	} else {
	    printfv(1,"  Starting CD at track: %d.\n", cd_first_track);
	    cd_track(cd_first_track);
	}
    }

    if ( cd_loop)
	cd_eject_on_end = 0;

    return 0;
}

/* 
 * Close the CD-Player. 
 * Stop a possibly running CD if that is desired.
 */
int exit_cd() {
    if ( cd_handle == -1)		/* not initialized */
	return 0;

    if ( cd_stop_on_exit && cd_playing)
	cd_stop();
		
    if (close(cd_handle)) {
	printfe("Can not close CD.\n");
	return 1;
    }
    return 0;
}

/*
 *  CD-player check for complete routine
 */
int cd_check() {

    if ( ! cd_playing )			/* player is not played -> OK */
	return 0;

    cd_subchnl();			/* get information from CD */
	
    /* Test wether CD-ROM has completed operation, or if the desired
       track has been reached. Not all CD-ROMs give the CDROM_AUDIO_COMPL.
       status (like mine)
    */
    if ( (subchnl.cdsc_audiostatus == CDROM_AUDIO_COMPLETED) /* 'nice' CD */
	|| (subchnl.cdsc_trk >= stop_track)	/* reached next track */
	|| ( (subchnl.cdsc_trk == 0)		/* again from start, */
	    && (stop_track > last) ) ) {	/*  while plaing last track*/
       	
	if(cd_randomplay)
	    cd_next(0);
	else if(cd_loop)
	    cd_track(0);
	else if(cd_eject_on_end)
	    cd_eject();
	else
	    cd_stop();
    }
    return 0;
}

/*****************************************************************************
 *  Primitives to access the CD:
 */

int rand_track(void)
{
    int num,loop=0;
    
    if(!rand_ini){
	srand((unsigned int)time((long *)NULL));
	rand_ini++;
    }

    num=rand()%last;
    while(random_played[num]){
	if(++num >= last){
	    num=first-1;
	    loop++;
	}
	if(loop>2)
	    break;
    }
    return loop>2?last+1:num+1;
}

#define CD_INIT    if ( cd_handle == -1) if( init_cd() ) return 1;


/* eject the CD */
int cd_eject() {
    int ret;
    CD_INIT

    if(cd_randomplay && random_played) {
	free(random_played);
	random_played = NULL;
    }
    cd_playing=0;

    if(ejected) {
	no_disk=ejected=0;
	ret = ioctl(cd_handle,CDROMSTART);	/* close the door ;) */
	if ( !ret && cd_first_track >= 0) {
	    if(!cd_randomplay)
		cd_track(cd_first_track);
	    else
		cd_next(0);
	}
    }else{
	cd_stop();
	ret = ioctl(cd_handle,CDROMEJECT);
	ejected = ! ret;
    }
    return ret;
}

/* pause CD */
int cd_pause() {
	
    if(ejected || no_disk)
	return 0;

    if (!cd_playing || (cd_handle == -1))
	return 0;			/* ignore if not started playing */

    cd_subchnl();

    if (subchnl.cdsc_audiostatus == CDROM_AUDIO_PAUSED ) {
	cd_in_pause = 0;
	if (ioctl(cd_handle,CDROMRESUME)) 
	    return 1;
    } else {
       	cd_in_pause = 1;
	if (ioctl(cd_handle,CDROMPAUSE)) 
	    return 1;
    }
    return 0;
}

/* stop CD */
int cd_stop() {
    CD_INIT;

    if(ejected || no_disk)
	return 0;

    cd_playing = 0;

    ioctl(cd_handle, CDROMSTOP);

    cd_subchnl();			/* get information from CD */

    return 0;
}

int cd_readtoc() {
    int cmd;

    if(ejected || no_disk)
	return 0;

    cmd=CDROMREADTOCHDR;
    if (ioctl(cd_handle,cmd,&tocHdr)) {
	printfe("Can't read TOC\n");
	return 1;
    }
    first=tocHdr.cdth_trk0;
    last= tocHdr.cdth_trk1;
    if ((first==0)||(first>last)) {
	printfe("Illegal TOC\n");
	return 1;
    } 
    /* set end-time in msf for cd_fast() */
    entry.cdte_track = CDROM_LEADOUT;
    entry.cdte_format = CDROM_MSF;
    if (ioctl (cd_handle, CDROMREADTOCENTRY, &entry)) {
	printfe("Can't read TOC\n");
	return 1;
    }
    msf.cdmsf_min1   = entry.cdte_addr.msf.minute;
    msf.cdmsf_sec1   = entry.cdte_addr.msf.second;
    msf.cdmsf_frame1 = entry.cdte_addr.msf.frame;

    return 0;
}

/* 
 * start play starting at track 
 */
int cd_track(int track) {

    if(ejected || no_disk)
	return 0;

    cd_stop();
		
    cd_readtoc();

    if (track < first)
	track=first;
    if (track > last) 
	track=last;

    entry.cdte_track = track;
    entry.cdte_format = CDROM_MSF;
    if (ioctl (cd_handle, CDROMREADTOCENTRY, &entry)) {
	printfee("Can't read TOC\n");
	return 1;
    }

    /* subtract the skip-value from the current position */
    msf.cdmsf_min0   = entry.cdte_addr.msf.minute;
    msf.cdmsf_sec0   = entry.cdte_addr.msf.second;
    msf.cdmsf_frame0 = entry.cdte_addr.msf.frame;

    /* set stop_track */
    stop_track = last + 1;
    
    /* set end-time to end of CD */
    entry.cdte_track = CDROM_LEADOUT;
    entry.cdte_format = CDROM_MSF;
    if (ioctl (cd_handle, CDROMREADTOCENTRY, &entry)) {
	printfe("Can't read TOC\n");
	return 1;
    }
    msf.cdmsf_min1   = entry.cdte_addr.msf.minute;
    msf.cdmsf_sec1   = entry.cdte_addr.msf.second;
    msf.cdmsf_frame1 = entry.cdte_addr.msf.frame;

    if( ioctl(cd_handle, CDROMSTART)) {
	printfee("Can't start CD");
	return 1;
    }
    if (ioctl(cd_handle,CDROMPLAYMSF,&msf)) {
	printfe("Can't play CD.\n");
	return 1;
    }
    cd_playing=1;

    return 0;
}

/*
 *  Read information from CD-ROM (audio-status, trk, addr)
 */
int cd_subchnl() {

    if ( cd_handle == -1)
	if(init_cd())
	    return 0;
    if ( ! cd_playing )
	cd_readtoc();

    subchnl.cdsc_format=CDROM_MSF;
    if (ioctl(cd_handle,CDROMSUBCHNL,&subchnl)) {
	subchnl.cdsc_audiostatus=CDROM_AUDIO_NO_STATUS;
	if(ejected)
	    return 0;
	else if(no_disk)
	    return 0;
	printfee("Can't SUBCHNL CD-Rom.");
	no_disk=1;
	cd_playing=0;
	return 1;
    }
    if(ejected || no_disk){
	no_disk=ejected=0;
	if(cd_first_track >= 0)
	    if(!cd_randomplay)
		cd_track(cd_first_track);
	    else
		cd_next(0);
    }

    if( (subchnl.cdsc_audiostatus != CDROM_AUDIO_PLAY) &&
       (subchnl.cdsc_audiostatus != CDROM_AUDIO_PAUSED) )
	subchnl.cdsc_trk = 0;
    else if( !cd_playing)
	cd_playing++;
	
    return 0;
}

/* go to next track */
int cd_next(int skip) {
    int track;
    CD_INIT;

    if(ejected || no_disk)
	return 0;

    if (!cd_playing++) {		/* CD not playing */
	cd_readtoc();				/* get table of contents */

	track=cd_randomplay ? rand_track()	/* random track */
	    : first;				/* start at first track */

	if(cd_randomplay) {
	    if ( random_played)
		free( random_played);
	    random_played=calloc(last-first+1,sizeof(int));
	}
    } else {				/* CD is already playing */
	cd_subchnl();				/* get information from CD */

	track = cd_randomplay ? rand_track() 	/* next random track  */
	    : (subchnl.cdsc_trk + skip);	/* increment or drecrement */

	cd_stop();
    }
    if (cd_randomplay) {
	if(track>last)
	    if(cd_loop){
		register z;
		for(z=0;z<last;z++)
		    random_played[z]=0;
		track=rand_track();				
	    }
	    else if(cd_eject_on_end){
		cd_eject();
		return 0;
	    }
	    else{
		cd_stop();
		return 0;
	    }
	
	random_played[track-1]=1;
    }
    /* Check for first and last track */
    if ( track > last) 
	track=first;
    if ( track < first)
	track=last;

    entry.cdte_track = track;
    entry.cdte_format = CDROM_MSF;
    if (ioctl (cd_handle, CDROMREADTOCENTRY, &entry)) {
	printfe("Can't read TOC entry\n");
	return 1;
    }

    /* subtract the skip-value from the current position */
    msf.cdmsf_min0   = entry.cdte_addr.msf.minute;
    msf.cdmsf_sec0   = entry.cdte_addr.msf.second;
    msf.cdmsf_frame0 = entry.cdte_addr.msf.frame;

    entry.cdte_format = CDROM_MSF;
    stop_track = entry.cdte_track = 
    	( !cd_randomplay || track==last ) ?CDROM_LEADOUT:track+1;
    if (ioctl (cd_handle, CDROMREADTOCENTRY, &entry)) {
	printfe("Can't read TOC entry\n");
	return 1;
    }
    msf.cdmsf_min1   = entry.cdte_addr.msf.minute;
    msf.cdmsf_sec1   = entry.cdte_addr.msf.second;
    msf.cdmsf_frame1 = entry.cdte_addr.msf.frame;

    if( ioctl(cd_handle, CDROMSTART)) {
	printfee("Can't start CD");
	return 1;
    }

    printf("Playing from %d:%d to %d:%d\n", 
	   msf.cdmsf_min0,msf.cdmsf_sec0,
	   msf.cdmsf_min1,msf.cdmsf_sec1);

    if (ioctl(cd_handle,CDROMPLAYMSF,&msf))
	printf("Drive Error\n");
    cd_playing=1;
    
    return 0;
}

int cd_fast(int skip) {
    int pos;

    CD_INIT;

    cd_subchnl();			/* get information from CD */

    if(ejected || no_disk)
	return 0;

    /* calculate the current position in frames */
    pos = subchnl.cdsc_absaddr.msf.minute * CD_SECS * CD_FRAMES +
	  subchnl.cdsc_absaddr.msf.second * CD_FRAMES +
	  subchnl.cdsc_absaddr.msf.frame;

    /* Skip by seconds */
    pos += skip * CD_FRAMES;
    if ( pos < 0)			/* play first if a begin */
	return cd_randomplay?cd_next(0):cd_track(0);

    /* subtract the skip-value from the current position */
    msf.cdmsf_min0   = pos / (CD_SECS*CD_FRAMES);
    msf.cdmsf_sec0   = (pos / CD_FRAMES) % CD_SECS;
    msf.cdmsf_frame0 = pos % CD_FRAMES;

    /* we don't set end values - it is already set before, but check */
    if(((msf.cdmsf_min1 * 60 + msf.cdmsf_sec1)*CD_FRAMES+msf.cdmsf_frame1) 
       < pos ){
	printfe("Can't skip, CD untouched");
	return 0;
    }
    /* Play it */
    if( ioctl(cd_handle, CDROMSTART)) {
	printfee("Can't start CD");
	return 1;
    }
    if (ioctl(cd_handle,CDROMPLAYMSF,&msf)) { 
	printfee("Drive error or invalid address\n");
    }

    cd_subchnl();			/* get information from CD */
    return 0;
}

/*
int cd_volume(int volume) {
    cmd=CDROMVOLCTRL;
    printf("--Channel 0 Left  (0-255): ");
    scanf("%d",&arg1);
    printf("--Channel 1 Right (0-255): ");
    scanf("%d",&arg2);
    volctrl.channel0=arg1;
    volctrl.channel1=arg2;
    volctrl.channel2=0;
    volctrl.channel3=0;
    if (ioctl(handle,cmd,&volctrl)) 
	{ printf("Drive error or unsupported command\n");
      }
    break;  
}
*/
