#include "cthugha.h"
#include "translate.h"
#include "display.h"
#include "interface.h"
#include "options.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

int use_translations = 1;			/* allow translations */
feature_entry _translation =  
    { NULL, 1, "none", "No Translation" };	/* trans. 0 is identiy */

feature translations = {
    "translate",
    NULL, 0,
    CHANGE_RANDOM,
    f_translation
};
char translate_first[256];			/* first translation */
char lib_size[512];
char * translation_path[] = {
    "./",
    "./tab/",
    lib_size,
    INST_LIB "/tab/",
    ""
};
int trans_stretch = 1;				/* allow stretching */

int done_translation=0;				/* trans. already done for
						   this iteration */

feature_entry* read_trans(FILE * file, char * name, char * dir);
feature_entry* read_cmdtrans(FILE * file, char * name, char * dir);

/*
 * Initialize the translation-tables.
 */
int init_translate() {

    add_feature(&translations, &_translation);

    if (!use_translations) 
	return 0;

    printfv(2,"  loading translations...\n");
    
    sprintf(lib_size, INST_LIB "/tab/%dx%d/", 
            BUFF_WIDTH,BUFF_HEIGHT);
	
    load( translation_path, "/tab/", ".tab", read_trans, &translations);
    load( translation_path, "/tab/", ".cmd", read_cmdtrans, &translations);

    printfv(2,"  number of loaded translations: %d\n", 
	    translations.nr_entries);

    return 0;
}

/*
 * Read a translation file
 * 
 * The file should contain BUFF_WIDTH * BUFF_HEIGHT utint-values.
 * These values specifiy the translation as described in the doc-file.
 */

static int read_trans_data(FILE * file, tab_header header, int BSize, int * dst, char * name) {
    union data {
	unsigned long * l;
	unsigned short * s;
    } D;
    int i,j;

    int size = (BSize > 65535) ? sizeof(long) : sizeof(short);

    /* allocate memory for read buffer */
    D.l = (unsigned long*)cth_memory(NULL, size*header.size_x, NULL);
    if ((void*)D.l != (void*)D.s) {
        printfe("Wackiness afoot at %d in %s\n",__LINE__,__FILE__);
        exit(1);
    }

    /* read data */
    for(i=0; i < header.size_y; i++) {
	if ( (j=fread( D.l, size, header.size_x, file)) < header.size_x ) {
	    printfe("  Can't read at line: %d, read: %d (%s)\n", i,j, name);
	    free(D.l);
	    return 1;
	}	
	for(j=0; j < header.size_x; j++) {
	    if( BSize > 65535) {
		if ( D.l[j] >= BSize) {
		    printfe("  High-translation (value: %ld) in %s.\n", D.l[j], name);
		    free(D.l);
		    return 1;
		}
		*dst++ = (int)D.l[j];
	    } else {
		if ( D.s[j] >= BSize) {
		    printfe("  High-translation (value: %d) in %s.\n",  D.s[j], name);
		    free(D.l);
		    return 1;
		}
		*dst++ = (int)D.s[j];
	    }
	}
    }

    free(D.l);

    /* Check for too much data */
    if ( fread( &D, size, 1, file) ) {
	printfe("  Extra data at end of file %s\n", name);
	return 1;
    }

    return 0;
}

static int stretch_trans(int * dst, int * src, tab_header header) {
    double xs,ys;
    int x,y,tp,ox,oy,dx,dy;
    int * tabin = src;
    int i,j;
    
    printfv(3," ... stretching");
    
    xs = (double)header.size_x / BUFF_WIDTH;
    ys = (double)header.size_y / BUFF_HEIGHT;
    
    for (j=0;j<BUFF_HEIGHT;j++) {
	
	y = (double)j*ys;
	if (y >= header.size_y)
	    y = header.size_y - 1;
	
	for (i=0;i<BUFF_WIDTH;i++) {
	    
	    x = (double)i*xs;
	    if (x >= header.size_x)
		x = header.size_x - 1;
	    
	    tp = tabin[x + y*header.size_x];
	    ox = tp % header.size_x;
	    oy = tp / header.size_x;
	    dx = (double)(ox - x)/xs;
	    dy = (double)(oy - y)/ys;
	    dst[i+j*BUFF_WIDTH] = abs(i + dx + (j + dy) * BUFF_WIDTH)%BUFF_SIZE;
	    
	}
    }
    return 0;
}

feature_entry* read_trans(FILE * file, char * name, char * dir) {
    tab_header header;
    static feature_entry new_trans;

    int BSize = BUFF_SIZE;
    int * tmpBuffer = NULL;
    int stretch = 0;

    /* read header */
    if ( fread( &header, sizeof(tab_header), 1, file) != 1) {
	printfe("  Can't read header in file %s.\n", name);
	return NULL;
    }

    /* check header ID */
    if ( header.id != *((long*)"HDKB")) {
	printfv(2,"\n  Header ID-mismatch. Trying without header.");

	/* go back to start of file */
	rewind(file);
	/* no description in old style */
	new_trans.desc[0] = '\0'; 

	/* fill in header */
	header.size_x = BUFF_WIDTH;
	header.size_y = BUFF_HEIGHT;
    } else {
	/* ID OK - now test size */
	if( (header.size_x != BUFF_WIDTH) || (header.size_y != BUFF_HEIGHT)) {
	    printfv(3,"\n    Size mismatch (%dx%d instead of %dx%d)", 
		    header.size_x, header.size_y,
		    BUFF_WIDTH, BUFF_HEIGHT);
	    if ( trans_stretch) {		/* allow stretching */
		stretch = 1;
	    } else {
		return NULL;
	    }
	}
	   
	/* translation table may use a different size */
	BSize = header.size_x * header.size_y;
	/* for stretching we need a temp buffer */
	if ( stretch ) 
	    tmpBuffer = cth_memory(NULL, BSize*sizeof(int),
		  "Can't allocate memory for temporary translation.\n");
	    
	/* now copy description */
	strncpy(new_trans.desc, header.description, MAX_DESC_LEN);
    }

    new_trans.data = cth_memory(NULL, BUFF_SIZE * sizeof(int),
				"Can't allocate memory for translation.\n");
    
    /* read data */
    if(read_trans_data(file, header, BSize, stretch ? tmpBuffer : new_trans.data, name)) {
	free( new_trans.data );
	if ( tmpBuffer) free(tmpBuffer);
	return NULL;
    }
    
    /* do the stretching if necessary (from: Rus Maxham) */
    if(stretch) {
	stretch_trans(new_trans.data, tmpBuffer, header);
	free(tmpBuffer);	/* no longer needed */
    }

    /* fill name and usage for translation */
    strncpy(new_trans.name, name, MAX_NAME_LEN);
    new_trans.use = 1;

    return &new_trans;
}			


/*
 * read a file in the form:
 *  cmdtab
 *  <descrption>
 *  <commd> [args] %d %d
 */
feature_entry* read_cmdtrans(FILE * file, char * name, char * dir) {
    char line[512];
    static feature_entry new_trans;
    char command[1024];
    FILE * cmd_file;
    unsigned long * dst;
    int i;

    /* check ID */
    fgets(line, 512, file);
    if ( strncmp(line, "cmdtab", 6) != 0) {
	printfe("  Not a command translation file: %s.\n", name);
	return NULL;
    }
    /* get name and description, delete trailing \n */
    strncpy(new_trans.name, name, MAX_NAME_LEN);

    fgets(new_trans.desc, MAX_DESC_LEN, file);
    if(strlen(new_trans.desc) > 0)
	new_trans.desc[ strlen(new_trans.desc) - 1] = '\0';

    /* get command */
    fgets(line, 512, file);

    /* generate the command line */
    if(strchr(line, '/') == NULL) {
	char cmd2[1024];
	strncpy(cmd2, dir, 1024);
	strncat(cmd2, "/", 1024);
	strncat(cmd2, line, 1024);
	sprintf(command, cmd2, BUFF_WIDTH, BUFF_HEIGHT);
    } else {
	sprintf(command, line, BUFF_WIDTH, BUFF_HEIGHT);
    }
    if(strlen(command) > 0)
	command[strlen(command) - 1] = '\0';

    /* start the command */
    printfv(3, "\n    starting: %s", command);
    if( (cmd_file = popen(command, "r")) == NULL) {
	printfee("  Can't run command '%s'.\n", command);
	return NULL;
    }

    new_trans.data = cth_memory(NULL, BUFF_SIZE * sizeof(int),
				"Can't allocate memory for translation.\n");
    dst = new_trans.data;
	
    /* read data from translation table program */
    for(i=0; i < BUFF_SIZE; i++, dst ++) {
	fread(dst, sizeof(long), 1, cmd_file);
	if( *dst >= (unsigned int)BUFF_SIZE) {
	    printfe("  High-translation (value: %ld) in %s.\n", *dst, name);
	    pclose(cmd_file);
	    free( new_trans.data );
	    return NULL;
	}
    }
    if(fread(&i, 1, 1, cmd_file)) {		/* something more to read */
	printfe("  Extra data at end of table in %s.\n", name);
	pclose(cmd_file);
	free(new_trans.data);
	return NULL;
    }

    pclose(cmd_file);

    new_trans.use = 1;

    return &new_trans;
}



/*
 * Do the translation.
 *
 * if active_translation is 0 then nothing is done.
 */			
void translate() {
    int * trans;
    int i;
    unsigned int * dst; 
    unsigned char * src;

    if( done_translation)
	return;

    if( active_translation == 0)		/* 0 is identity */
	return;

    dst = (unsigned int *)passive_buffer;	/* destination */
    src = active_buffer;
    active_buffer = passive_buffer; passive_buffer = src;
			
    trans = (int*)(active_translation);		/* transl.-map */
    src[0] = 0;

    /* always write 4 values at once */
    for(i=BUFF_SIZE/4; i != 0; i--) {  		/* do the translation */
	unsigned int D = src[*trans++];
	D <<= 8;
	D = D + src[*trans++];
	D <<= 8;
	D = D + src[*trans++];
	D <<= 8;
	*dst++ = D + src[*trans++];
    }
}












