#include "../cthugha.h"
#include "../display.h"
#include "../disp-sys.h"
#include "../cthugha-X11.h"
#include "../imath.h"
#include "dga.h"

#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/Intrinsic.h>
#include <X11/extensions/XShm.h>

int ncurses_use = 0;

unsigned char * display_bitmap = NULL;		/* memory for X11-bitmap */
int display_mit_shm = 1;			/* use MIT-SHM if possible */
int display_on_root = 0;			/* display on root window */

int display_syncwait = 0;

static Status mit_shm;
static XShmSegmentInfo shminfo;
static XImage * image = NULL;

xy window_size = {0,0};

int nobuff = 0;

#include "../font8x8.c"


int init_display_sys() {
    static int X[] = {320,320,320,260,640,800,1024,1280,1600};
    static int Y[] = {200,240,400,480,480,600, 768,1024,1200};
    static int nr_screen_modes = sizeof(X) / sizeof(int);

    if ( display_mode == -1) {
	/* a special display resolution is given */
	disp_size.x = disp_size.x;
	disp_size.y = disp_size.y;
    } else {
	/* use one of the default resolutions */
	if ( (display_mode >= nr_screen_modes) || (display_mode < 0))
	    display_mode = 0;
	
	disp_size.x = X[display_mode];
	disp_size.y = Y[display_mode];
    }

    return 0;
}


/*
 * Allocate the XImage
 */
int alloc_image() {

#if USE_DGA == 1
    if( xcth_dga)
	return alloc_image_dga();
#endif

    /* Check if MIT-SHM is available */
    if ( display_mit_shm) {
	int d1,d2;    Bool d3;		/* dummy variables */
	mit_shm = XShmQueryVersion(xcth_display,&d1,&d2,&d3);
    }

    if ( mit_shm) {
	/* create XImage */
	if( (image = XShmCreateImage(xcth_display,
				     xcth_visual,
				     xcth_planes,
				     ZPixmap,		/* format */
				     NULL,		/* data */
				     &shminfo, 
				     disp_size.x, disp_size.y)) == NULL) {
	    printfe("Can not create XImage.\n");
	    return 1;
	}

	/* maybe the size changed */
	bytes_per_line = image->bytes_per_line;
	disp_size.y = image->height;

	/* create Shared Memory */
	if ( (shminfo.shmid = shmget(IPC_PRIVATE, SCREEN_SIZE,
				     IPC_CREAT|0777)) == -1 ) {
	    printfee("Can not create shared memory segment");
	    return 1;
	}
	/* attach Shard Memory */
	if( (shminfo.shmaddr = image->data = shmat(shminfo.shmid, 0,0)) == (void*)-1) {
	    printfee("Can not attach shared memory segment");
	    return 1;
	}
	    
	/* allow X11 to write */
	shminfo.readOnly = False;

	/* Attach X11 with Shared Memory */
	if( XShmAttach(xcth_display, &shminfo) == 0) {
	    printfe("Can not X-attach shared memory segment.\n");
	    return 1;
	}

	/* keep pointer to data */
	display_bitmap = image->data;

    } else {	/* don't use MIT-SHM Extensions */

	/* allocate memory for bitmap without Shared Memory*/
	if( (display_bitmap = (unsigned char *)malloc(SCREEN_SIZE)) == NULL) {
	    printfe("Can not allocate memory for bitmap.\n");
	    return 1;
	}
	/* create Image */
	if( (image = XCreateImage(xcth_display, xcth_visual, xcth_planes, 
				  ZPixmap, 0, display_bitmap, 
				  disp_size.x, disp_size.y,
				  XBitmapPad(xcth_display),  
				  disp_size.x*bypp)) == NULL) {
	    printfe("Can not create XImage.\n");
	    return 1;
	}
	
	/* maybe the size changed */
	bytes_per_line = image->bytes_per_line;
	disp_size.y = image->height;
    }
    return 0;
}

/*
 * free the XImage
 */
int free_image() {

    XSync(xcth_display, True);	

#if USE_DGA == 1
    if(xcth_dga) 
	return free_image_dga();
#endif

    if(image == NULL)
	return 0;

    if ( mit_shm) {
	/* remove Shared Memory if needed */
	XShmDetach(xcth_display, &shminfo);
	XDestroyImage(image);
	shmdt(shminfo.shmaddr);
	shmctl(shminfo.shmid,IPC_RMID,0);
    } else {
	XDestroyImage(image);
    }
    image = NULL;

    return 0;
}

/*
 * Resize window
 * Reallocates the XImage
 */
int resize_display(int new_width, int new_height) {

    disp_size.x = max(new_width, BUFF_WIDTH);
    disp_size.y = max(new_height, BUFF_HEIGHT);

    if( free_image() )
	return 1;

    if( alloc_image() )
	return 1;

    return 0;
}


int init_graph_mode_sys() {
    alloc_image();
    if(window_size.x > 0)
	printfv(3, "  window size : %dx%d\n", window_size.x, window_size.y);
    return 0;
}

int exit_graph_mode_sys() {
    free_image();    
    return 0;
}


int disp_clear_box_sys(int x, int y, int width, int height) {
    int i;
    char * dst = display_bitmap + y * bytes_per_line + x * bypp;
    
    for(i=0; i < height; i++, dst += bytes_per_line)
	bzero(dst, bypp*width);
    
    return 0;
}

int disp_copy_box_sys(int x, int y, int width, int height, 
		      int destx, int desty) {
    int i;
    char * src = display_bitmap + y * bytes_per_line + x * bypp;
    char * dst = display_bitmap + desty * bytes_per_line + destx * bypp;

    for(i=0; i < height; i++, src += bytes_per_line, dst += bytes_per_line)
	memcpy(dst, src, width*bypp);
    
    return 0;
}

char * pre_draw_sys() {

    /* make sure, the image is big enough */
    if( (disp_size.x < BUFF_WIDTH) || (disp_size.y < BUFF_HEIGHT)) {
	resize_display(BUFF_WIDTH, BUFF_HEIGHT);
    }

#if USE_DGA == 1
    if(xcth_dga) {
	if(!nobuff) 
	    while(!XF86DGAViewPortChanged(xcth_display, xcth_screen, 2));

	if(display_direct)				/* draw directly to video memory */
	    display_bitmap = DGAMemBase +				
		(window_size.y - disp_size.y)/2 * bytes_per_line +	
		(window_size.x - disp_size.x)/2 * bypp;			
	else
	    display_bitmap = DGAMemPtr +				
		(window_size.y - disp_size.y)/2 * bytes_per_line +	
		(window_size.x - disp_size.x)/2 * bypp;			
    }
#endif

    return display_bitmap;
}


int disp_copy_sys(int clear) {
    int ss_offset_x = max((window_size.x - disp_size.x)/2, 0);
    int ss_offset_y = max((window_size.y - disp_size.y)/2, 0);

#if USE_DGA == 1
    if( xcth_dga) {
	if(!display_direct)	
	    DOUBLEBUFFER(1);

 	return 0;
    }
#endif
    
    if(clear)		/* draw full screen */
	if ( mit_shm) {
	    XShmPutImage(xcth_display, xcth_window, xcth_gc, image,0,0,
			 ss_offset_x,  ss_offset_y, 
			 disp_size.x, disp_size.y, 
			 0);
	} else {
	    XPutImage(xcth_display, xcth_window, xcth_gc, image, 0,0,
		      ss_offset_x, ss_offset_y,
		      disp_size.x, disp_size.y);
	}
    else			/* or only a part */
	if ( mit_shm) {
	    XShmPutImage(xcth_display, xcth_window, xcth_gc, image,
			 SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
			 ss_offset_x + SCREEN_OFFSET_X,
			 ss_offset_y + SCREEN_OFFSET_Y,
			 draw_size.x, draw_size.y,
			 0);
	} else {
	    XPutImage(xcth_display, xcth_window, xcth_gc, image, 
		      SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
		      ss_offset_x + SCREEN_OFFSET_X,
		      ss_offset_y + SCREEN_OFFSET_Y,
		      draw_size.x, draw_size.y);
	}
    
    return 0;
}



/*
 * display text
 */

void display_write_character(int x,int y,int text, int color)  {
    unsigned char i,j,index;

    char * d = display_bitmap + y*bytes_per_line + x*bypp;
    
    text <<= 3;
    for(j = 0; j < 8; j++, d += bytes_per_line - 8*bypp) {
	for(i = 0,index = 0x80; i < 8; i++,index >>= 1, d += bypp) {

	    if(display_font[text + j] & index) {

		if(colormapped) {
		    switch(draw_mode) {
		    case DM_direct: case DM_tmp_mapped:
			*d = color;
			break;
		    case DM_mapped1:
			*d = bitmap_colors[color];
		    default:
			/* this can not occur */
		    }
		} else {
		    int c = 
			(red_mask   & shift(display_text_color[color][0], red_shift  )) |
			(green_mask & shift(display_text_color[color][1], green_shift)) |
			(blue_mask  & shift(display_text_color[color][2], blue_shift ));

		    switch(draw_mode) {
		    case DM_mapped1:
			*d = c;
		    case DM_mapped2:
			*((unsigned short*)d) = c;
			break;
		    case DM_mapped4:
			*(unsigned long*)d = c;
			break;
		    default:
			/* this can not occur */
		    }
		}
	    }
	}
    }
}


int display_print_string_sys(int x, int y, char* text, int color) {
    if(colormapped)
	color = 255 - color;

    while((*text) != '\0') {  
	if((x >= 0) && (x <= (disp_size.x - 8)) && 
	   (y >= 0) && (y <= (disp_size.y - 8)))
	    display_write_character(x,y,(int)*text, color);
	x += 8;
	text++;
    } 
    return 0;
}

int display_sync_sys() {
    XSync(xcth_display, False);
    return 0;
}

