module HTMLWizard where

import Common
import HTML

{-
HTML utility combinators (Friday, March 07, 1997)

Erik Meijer (erik@cs.ruu.nl)
-}

-- Interface HTMLWizard --------------------------------------
page     :: String -> [(Name,Value)] -> [HTML] -> HTML

format   :: Tag -> String -> HTML

h        :: Int -> String -> HTML
p        :: [HTML] -> HTML
font     :: Color -> [Face] -> Size -> [HTML] -> HTML

href     :: URL -> [HTML] -> HTML
name     :: String -> [HTML] -> HTML
image    :: String -> URL -> HTML

ul       :: [[HTML]] -> HTML
ol       :: [[HTML]] -> HTML
dl       :: [(String,[HTML])] -> HTML

table    :: String -> [String] -> [[ [HTML] ]] -> HTML

gui      :: URL -> [HTML] -> HTML
widget   :: Widget -> Name -> HTML
menu     :: Name -> [Value] -> HTML
textarea :: Name -> Int -> Int -> Wrap -> Value -> HTML
object   :: CLSID -> String -> [(Name,Value)] -> HTML
--------------------------------------------------------------

type Color = String
type Face  = String
type Size  = String

type Widget = String
type Wrap   = String
type CLSID  = String

{-
page title attributes html
 = <HTML>
     <HEAD>
       <TITLE>title</TITLE>
     </HEAD>
     <BODY attributes>
       html
     </BODY>
   </HTML>
-}

page title attributes html
 = element "HTML"
     [ element "HEAD" [element "TITLE" [prose title]]
     , attributedElement "BODY" attributes html
     ]

vanillaPage                   = page "" []
backgroundPage url title body = page title [("BACKGROUND",url)] body

{-
format tag text
 = <tag>text</tag>
-}

format tag text = element tag [prose text]

{-
h i heading
 = <Hi>heading</Hi>
-}

h n heading = element ("H"++show n) [prose heading]
h1 = h 1; h2 = h 2; h3 = h 3; h4 = h 4; h5 = h 5; h6 = h 6; h7 = h 7

{-
p html
 = <P>html</P>
-}

p = element "P"

{-
font color [face1,...,facen] size html
 = <FONT
      COLOR = color
      FACE  = face1,...,facen
      SIZE  = size
   >
     html
   </FONT>
-}

font color face size html
 = attributedElement "FONT"
     [ ("COLOR", color)
     , ("FACE" ,(tail.init.show) face)
     , ("SIZE" , size)
     ] html

{-
href url html
 = <A HREF = url>
     html
   </A>
-}

href url html   = attributedElement "A" [("HREF",url)] html

{-
name label html
 = <A HREF = name>
     html
   </A>
-}

name label html = attributedElement "A" [("NAME",label)] html

{-
image alt url
 = <IMG
      SRC = src
      ALT = alt
   >
-}

image alt src
 = attributedElement "IMAGE" [("SRC",src), ("ALT",alt)] []

{-
ul [item1,...,itemn]
 = <UL>
     <LI>item1</LI>
     ...
     <LI>itemn</LI>
   </UL>
-}

ul is = element "OL" [ element "LI" i | i <- is ]

{-
ol [item1,...,itemn]
 = <OL>
     <LI>item1</LI>
     ...
     <LI>itemn</LI>
   </OL>
-}

ol is = element "OL" [ element "LI" i | i <- is ]

{-
dl [(label1,item1),...(labeln,itemn)]
 = <DL>
     <DT><B>label1</B></DT>
     <DD>item1</DD>
     ...
     <DT><B>labeln</B></DT>
     <DD>itemn</DD>
   </DL>
-}

dl tds
 = element "DL" (concat [ [ element "DT" [format "B" dt]
                          , element "DD" dd
                          ]
                        | (dt,dd) <- tds
                        ]
                )

{-
table caption [th1,..., thn]
             [[td11,...,td1n]
             ,...
             ,[tdm1,...,tdmn]
             ]
 = <TABLE BORDER>
     <CAPTION>caption</CAPTION>
     <TR>
       <TH>th1</TH>...<TH>thn</TH>
     </TR>
     <TR>
       <TD>td11</TD>...<TD>td1n</TD>
     </TR>
     ...
     <TR>
       <TD>tdm1</TD>...<TD>tdmn</TD>
     </TR>
   </TABLE>
-}

table caption header rows
 = attributedElement 
     "TABLE" [("BORDER","")]
     ( element "CATION" [prose caption]
     :   element "TR" [ element "TH" [prose th] | th <- header ]
     : [ element "TR" [ element "TD" td         | td <- row    ] | row <- rows ]
     )

{-
widget w name
 = <INPUT
      TYPE = w
      NAME = name
   >
   </INPUT>
-}

widget w name
 = attributedElement "INPUT" [("TYPE",w), ("NAME",name)] []

checkbox name value
 = set [("VALUE", value)] (widget "CHECKBOX" name)

hidden name value
 = set [("VALUE",value)] (widget "HIDDEN" name)

password name
 = widget "PASSWORD" name

radio name value
 = set [("VALUE", value)] (widget "RADIO" name)

reset name value
 = set [("VALUE",value)] (widget "RESET" name)

submit name value
 = set [("VALUE",value)] (widget "SUBMIT" name)

textfield name
 = widget "TEXT" name

file name
 = widget "FILE" name

clickmap name
 = widget "IMAGE" name

group w name values
 = let buttons = [ [ [w name value], [prose value] ]
                 | value <- values
                 ]
   in table name [] buttons
{-
menu name [choice1,...,choicen]
 = <SELECT NAME=name>
     <OPTION>choice1</OPTION>
     ...
     ,OPTION>choicen</OPTION>
   </SELECT>
-}

menu name choices
 = attributedElement "SELECT" [("NAME",name)]
     [ element "OPTION" [prose choice]
     | choice <- choices
     ]

{-
textarea name rows cols wrap text
 = <TEXTAREA
      NAME = name
      ROWS = rows
      COLS = cols
      WRAP = wrap
   >
     text
   </TEXTAREA>
-}

textarea name rows cols wrap text
 = attributedElement "TEXTAREA"
     [ ("NAME",name), ("ROWS", show rows), ("COLLS", show cols), ("WRAP", wrap)]
     [prose text]

{-
gui action html
 = <FORM
      ACTION = action
      METHOD = "POST"
   >
     html
   </FORM>
-}

gui action html
 = attributedElement "FORM"
     [ ("ACTION",action), ("METHOD","POST") ]
     html
{-
object classid id [(name1,value1),...,(namen,valuen)]
 = <OBJECT
      CLASSID = "clsid:"++classid
      ID      = id
   >
     <PARAM name1=value1></PARAM>
     ...
     <PARAM namen = valuen></PARAM>
   </OBJECT>
-}

object classid id params
 = attributedElement "OBJECT"
     [ ("CLASSID", "clsid:"++classid), ("ID", id) ]
     [ attributedElement "PARAM" [("NAME",n),("VALUE",v)] [] | (n,v) <- params ]
