{-

This program generates a packing list for InstallShield5.  The input
is a file list (from the tar file).  The generated output is a .fgl file 
that should be placed in the "File Groups" folder of the IShield project.
The fileRoot variable indicates where the Hugs files are stored.

-}

module Main where

import System

----------------------------------------------------------------

main = getArgs >>= main'
    
main' [fileRoot, inputFile, outputFile] = do 
  input <- readFile inputFile
  writeFile outputFile (process fileRoot input)
main' args = fail (userError ("Wrong arguments: " ++ unwords args))

----------------------------------------------------------------

process fileRoot input = unlines list
 where
  files = map parseSlashes (lines input)
  Dir _ [(_,hugsDir)] = toDir files
  list = makeFileList fileRoot hugsDir finalLines

finalLines = ["[General]", "Type=FILELIST", "Version=1.00.000"]

makeFileList fileRoot hugsDir = makeFileList' "" hugsDir 
 where
  labName :: String -> L 
  labName "" = showL "[TopDir]"
  labName s  = showL ("[" ++ s ++ "]")
  
  doExtra "" = noL
  doExtra l  = showL "fulldirectory="

  makeFileList' dirName (Dir files subDirs) =
	  labName dirName 
	  . concatL [ eqL ("file" ++ show i) (fileRoot ++ dirName `slash` f)
                    | (i,f) <- zip [0..] files
                    ]
	  . concatL [ eqL ("SubDir" ++ show i) (dirName `slash` n)
                    | (i,(n,_)) <- zip [0..] subDirs
                    ]
	  . doExtra dirName
	  . concatL [ makeFileList' (dirName `slash` n) sd
                    | (n,sd) <- subDirs
                    ]

----------------------------------------------------------------

slash "" x = x
slash x y = x ++ "\\" ++ y

eqL x y = showL (x ++ "=" ++ y)

----------------------------------------------------------------

noL        :: L
showL      :: String -> L 
concatL    :: [L] -> L
concatMapL :: (a -> L) -> [a] -> L

type L = [String] -> [String]
noL        = id
showL l ls = l : ls
concatL = foldr (.) id

concatMapL f []     = noL
concatMapL f (x:xs) = f x . concatMapL f xs

----------------------------------------------------------------

-- A directory is a list of filenames and a list of subdirectories
data Dir a = Dir [a] [(a, Dir a)] deriving Show

emptyDir :: Dir a
emptyDir = Dir [] []

toDir :: Eq a => [[a]] -> Dir a
toDir = foldr insertFile emptyDir 

fromDir :: Dir a -> [[a]]
fromDir (Dir fs ds) =  [] 
                    :  [ [f] | f <- fs ] 
                    ++ [ (n:f) | (n,sd) <- ds, f <- fromDir sd ]

insertFile :: Eq a => [a] -> Dir a -> Dir a
insertFile [f]   (Dir fs sd) = Dir (f:fs) sd   -- assume no duplicates
insertFile (d:f) (Dir fs sd) = Dir fs (addSubdir d f sd)
insertFile []    d           = d

addSubdir :: Eq a => a -> [a] -> [(a,Dir a)] -> [(a,Dir a)]
addSubdir d f [] = [(d,insertFile f (Dir [] []))]
addSubdir d f ((v@(d',sd)):r) 
  | d == d'   = (d', insertFile f sd) : r
  | otherwise = v : addSubdir d f r

----------------------------------------------------------------

parseSlashes :: String -> [String]
parseSlashes "" = []
parseSlashes ('/':l) = parseSlashes l
parseSlashes l = (f:parseSlashes r) where (f,r) = break (== '/') l 

