#include	"qpage.h"


/*
** global variables
*/
#ifndef lint
static char	sccsid[] = "@(#)util.c  1.22  04/27/97  tomiii@mtu.edu";
#endif
#ifndef CLIENT_ONLY
jmp_buf		TimeoutEnv;


/*
** qpage_log()
**
** This function logs messages for qpage either via syslog() or by
** printing them to stdout or stderr, depending on the Interactive
** and Debug flags.
**
**	Input:
**		pri - the syslog priority (if used)
**		fmt - a printf format string
**		... - a variable argument list (see "man va_start")
**
**	Returns:
**		nothing
**
**	Note:
**		Some messages should never be logged to stdout.  We
**		will abuse LOG_ALERT to kludge this condition.
*/
void
qpage_log(int pri, char *fmt, ...)
{
	FILE		*where;
	va_list		ap;
	char		buff[1024];
	int		tmp;


	/*
	** KLUDGE ALERT -- DANGER WILL ROBINSON
	**
	** There are some messages we always want logged.  The calling
	** functions will use LOG_ALERT to signal us of this condition,
	** and we'll just remap it to LOG_INFO.
	*/
	tmp = Interactive;
	if (pri == LOG_ALERT) {
		Interactive = FALSE;
		pri = LOG_INFO;
	}

	va_start(ap, fmt);

	/*
	** There are several possibilities here, depending on the
	** Interactive and Debug flags.
	**
	**	I D | output
	**	----+-------
	**	0 0 | sylog()
	**	0 1 | stderr
	**	1 0 | stdout
	**	1 1 | stdout
	*/
	if (Debug || Interactive) {
		where = Interactive ? stdout : stderr;

		(void)vfprintf(where, fmt, ap);
		fprintf(where, "\n");
		(void)fflush(where);
	}
	else {
#if 0
		vsyslog(pri, fmt, ap);
#else
		/*
		** Unfortunately we can't use vsyslog() here because
		** it's not supported on all platforms.  Same thing
		** for vsnprintf().  It looks like we'll have to
		** take a chance that we won't have buffer overruns
		** and risk using vsprintf().  Yuck.  Maybe one of
		** these days I'll get GNU autoconf worked out.
		*/
		(void)vsprintf(buff, fmt, ap);
		syslog(pri, buff);
#endif
	}

	va_end(ap);

	Interactive = tmp;
}


/*
** sigalrm()
**
** This function catches SIGALRM.  The alarm is used to implement
** timeouts for incoming SNPP commands.
*/
void
sigalrm(void)
{
	longjmp(TimeoutEnv, 1);
}


/*
** sigchld()
**
** This function is a signal handler for SIGCHLD.  It is
** called to clean up after a child process exits.
*/
void
sigchld(void)
{
	int	status;
	int	pid;


	if ((pid = wait(&status)) < 0) {
		qpage_log(LOG_ERR, "wait() failed: %s", strerror(errno));

		return;
	}

	if (Debug)
		qpage_log(LOG_DEBUG, "Child pid %u exited with status 0x%x",
			pid, WEXITSTATUS(status));
}


/*
** sigterm()
**
** This function is a signal handler for SIGTERM.  When SIGTERM is
** caught by this function, a SIGTERM signal is sent to the rest of
** the processes in this process group.  This allows an administrator
** to kill all active qpage processes with one signal.
*/
void
sigterm(void)
{
	(void)signal(SIGTERM, SIG_IGN);
	(void)kill(0, SIGTERM);
	qpage_log(LOG_INFO, "caught SIGTERM, exiting...");
	exit(0);
}


/*
** sigusr1()
**
** This function is a signal handler for SIGUSR1.  It does nothing,
** but we need it so that SIGUSR1 wakes up processes which are blocking
** on a pause() call.
*/
void
sigusr1(void)
{
}
#endif /* CLIENT_ONLY */


/*
** my_realloc()
**
** This function is exactly like realloc() but it allows the incoming
** pointer to be NULL, in which case malloc() is called.
**
**	Input:
**		ptr - the buffer to reallocate
**		len - the new size
**
**	Returns:
**		a pointer to the new buffer (or NULL on error)
*/
void *
my_realloc(void *ptr, int len)
{
	if (ptr == NULL)
		return(malloc(len));
	else
		return(realloc(ptr, len));
}


/*
** my_ctime()
**
** This function returns the string the real ctime() returns, except
** without the trailing newline.  What idiot decided it would be nice
** to supply that extra newline character, anyway?
**
**	Input:
**		clock - the number of seconds since 1970
**
**	Returns:
**		a string representation of time, or NULL on failure
*/
char *
my_ctime(time_t *clock)
{
	char	*ptr;
	char	*tmp;


	ptr = ctime(clock);

	if (ptr && ((tmp = strchr(ptr, '\n')) != NULL))
		*tmp = '\0';

	return(ptr);
}


/*
** getinput()
**
** This function reads input characters from the specified stream.
** If the "oneline" flag is TRUE, characters are read until CRLF
** (or just LF) is received.  Otherwise, characters are read one
** at a time until the sequence "CRLF.CRLF" is received (the CR
** characters are optional).  All contiguous whitespace is reduced
** to a single space character.  Leading and trailing whitespace
** is removed.
**
**	Input:
**		fp - a pointer to the input stream
**		oneline - whether to return only a single line
**
**	Returns:
**		a null terminated message with all occurrences of
**		space characters compressed down to a single space.
**
**	Note:
**		The buffer returned by this function is dynamically
**		allocated via malloc().  It is the responsibility of
**		the caller to free this memory when it is no longer
**		needed.
*/
char *
getinput(FILE *fp, int oneline)
{
	char		*buf;
	char		*ptr;
	char		lastc;
	int		buflen;
	int		almosteom;	/* true if "CRLF." was seen */
	int		len;
	int		c;


	buflen = 0;
	buf = NULL;
	ptr = NULL;
	len = 0;
	lastc = '\n';
	almosteom = 0;

#ifndef CLIENT_ONLY
	/*
	** Set the alarm so we don't hang.  Since this function is
	** also used by the client to read the initial message from
	** the user, we'll only set the timer if we're not reading
	** from standard input.
	*/
	if (fp != stdin)
		(void)alarm(SNPPTimeout);

	/*
	** check to see if a timeout occurred
	*/
	if (setjmp(TimeoutEnv) > 0) {
		if (buf)
			free(buf);

		return("");
	}
#endif

	while (!feof(fp)) {
		/*
		** make sure the buffer is big enough
		*/
		if (buflen-len < 10) {
			buf = (char *)my_realloc(buf, buflen+BUFCHUNKSIZE);
			buflen += BUFCHUNKSIZE;
			ptr = buf + len;
		}

		if ((c = getc(fp)) == EOF)
			break;

		/*
		** if this is CR and the next char is LF, skip the CR
		*/
		if (c == '\r') {
			if ((c = getc(fp)) == EOF)
				break;

			if (c != '\n')
				(void)ungetc(c, fp);
		}

		/*
		** check for the end of the message
		*/
		if ((almosteom || oneline) && c == '\n') {
			/*
			** if this is a message, nuke the trailing dot
			*/
			if (almosteom)
				len--;

			break;
		}
		else
			if (c == '.' && lastc == '\n')
				almosteom = 1;
			else
				almosteom = 0;

		if (isspace(c)) {
			if (!isspace(lastc)) {
				*ptr++ = ' ';
				len++;
			}
		}
		else {
			*ptr++ = c;
			len++;
		}

		lastc = c;
	}

#ifndef CLIENT_ONLY
	/*
	** we're out of danger; turn off the alarm
	*/
	(void)alarm(0);
#endif

	/*
	** clean up the end of the message
	*/
	if (buf)
		buf[len] = '\0';

	if (len > 0 && buf[len-1] == ' ')
		buf[len-1] = '\0';

	if (buf && *buf == '\0') {
		free(buf);
		buf = NULL;
	}

	return(buf);
}


/*
** msgcpy()
**
** Like strncpy() but break at whitespace.
**
**	Input:
**		dst - the destination string
**		src - the source string
**		n - the maximum number of bytes to copy
**
**	Returns:
**		a pointer to the first byte not copied or NULL if there
**		is nothing left to copy
*/
char *
msgcpy(char *dst, char *src, int n)
{
	int		seenspace;


	seenspace = 0;

	/* nuke leading whitespace */
	while (isspace(*src))
		src++;

	while (*src && n) {
		if (isspace(*src)) {
			seenspace = 1;
			*dst++ = ' ';
			n--;

			while (isspace(*src))
				src++;
		}
		else {
			*dst++ = *src++;
			n--;
		}
	}

	*dst = '\0';

	/*
	** Now make sure we didn't chop a word in the middle.
	** If we did then we need to back up to the previous
	** whitespace character and copy the rest of the
	** message in a future call to this function.
	*/
	if (*src && !isspace(*src) && !isspace(*(src-1))) {
		/* back up only if we've seen a space */
		if (seenspace) {
			while (!isspace(*src)) {
				src--;
				*dst-- = '\0';
			}
			src++;
		}
	}

	/*
	** It's possible that the only characters left over at this
	** point are whitespace characters.  We should get rid of them
	** here so we don't end up sending an "empty" page next time.
	*/
	while (isspace(*src))
		src++;

	if (*src)
		return(src);
	else
		return(NULL);
}


/*
** snpptime()
**
** This function checks to see if its argument is a valid SNPP time
** specification.  If so, an equivalent local time value is returned.
**
**	Input:
**		arg - the argument string
**
**	Returns:
**		the equivalent local time
*/
time_t
snpptime(char *arg)
{
	struct tm	tm;
	struct tm	*tmp;
	time_t		now;
	time_t		timespec;
	time_t		gmtoffset;
	int		i;


	tzset();

	/*
	** check for "YYMMDDHHMMSS" length
	*/
	if (strlen(arg) < 12) {
		return(INVALID_TIME);
	}

	for (i=0; i<12; i++)
		if (!isdigit(arg[i]))
			return(INVALID_TIME);

	if ((i != 12) || (arg[i] && !isspace(arg[i])))
		return(INVALID_TIME);

	(void)memset((char *)&tm, 0, sizeof(tm));
	tm.tm_isdst = -1;
	tm.tm_year += (*arg++ - '0') * 10;
	tm.tm_year += (*arg++ - '0');
	tm.tm_mon  += (*arg++ - '0') * 10;
	tm.tm_mon  += (*arg++ - '0');
	tm.tm_mday += (*arg++ - '0') * 10;
	tm.tm_mday += (*arg++ - '0');
	tm.tm_hour += (*arg++ - '0') * 10;
	tm.tm_hour += (*arg++ - '0');
	tm.tm_min  += (*arg++ - '0') * 10;
	tm.tm_min  += (*arg++ - '0');
	tm.tm_sec  += (*arg++ - '0') * 10;
	tm.tm_sec  += (*arg++ - '0');

	/*
	** they must have meant next century
	*/
	if (tm.tm_year < 50)
		tm.tm_year += 100;

	/*
	** do some sanity checking on the user-supplied values
	*/
	tm.tm_mon--;
	if ((tm.tm_mon < 0) || (tm.tm_mon > 11))
		return(INVALID_TIME);

	if ((tm.tm_mday < 1) || (tm.tm_mday > 31))
		return(INVALID_TIME);

	if ((tm.tm_hour < 0) || (tm.tm_hour > 24))
		return(INVALID_TIME);

	if ((tm.tm_min < 0) || (tm.tm_min > 60))
		return(INVALID_TIME);

	if ((tm.tm_sec < 0) || (tm.tm_sec > 60))
		return(INVALID_TIME);

	/*
	** verify that we're at the end of a token
	*/
	if (*arg && !isspace(*arg))
		return(INVALID_TIME);

	/*
	** skip whitespate
	*/
	while (*arg && isspace(*arg))
		arg++;

	/*
	** check for optional GMT offset
	*/
	if (*arg) {
		i = atoi(arg);

		/*
		** all GMT offsets are multiples of 100
		*/
		if (i % 100)
			return(INVALID_TIME);

		now = time(NULL);
		tmp = gmtime(&now);
		tmp->tm_isdst = -1;
		gmtoffset = (now - mktime(tmp)) / 3600;
		tm.tm_hour += gmtoffset - i/100;
	}

	timespec = mktime(&tm);
	return(timespec);
}


/*
** parse_time()
**
** Convert a string into the number of seconds represented by the string.
** If the first character is a '+' then the result is added to the current
** time.  If the specified time already happened today, assume the user
** means the same time tomorrow.
**
**	Input:
**		str - the string to parse
**
**	Returns:
**		the time in seconds since 1970 that the page should be sent
*/
time_t
parse_time(char *str)
{
	struct tm	*tm;
	time_t		result;
	time_t		offset;
	time_t		now;
	char		*ptr;
	char		*tmp;
	int		days;
	int		i;


	days = 0;
	offset = 0;
	now = time(NULL);

	/*
	** As a convenience to the user, first check to see if they
	** specified the time in SNPP format.
	**
	** (Hi Bill, I hope this makes you happy now!)
	*/
	if ((result = snpptime(str)) != INVALID_TIME)
		return(result);


	if (*str == '+') {
		/*
		** They specified a time relative to the current time.
		** No problem, this is easy.
		*/
		str++;
		result = now;
	}
	else {
		/*
		** They specified an absolute time.  This means we must
		** figure out what time() would have returned at 12am
		** this morning.
		*/
		if ((tm = localtime(&now)) == NULL)
			return(INVALID_TIME);

		tm->tm_hour = 0;
		tm->tm_min = 0;
		tm->tm_sec = 0;

		if ((result = mktime(tm)) == INVALID_TIME)
			return(INVALID_TIME);
	}

	/*
	** see if they specified a number of days
	*/
	if ((tmp = strchr(str, '+')) != NULL && isdigit(*str)) {
		*tmp++ = '\0';
		days = atoi(str);
		str = tmp;
	}

	i = strlen(str);

	if (i > 4)
		return(INVALID_TIME);

	/*
	** reverse the string and take it apart one digit at a time
	*/
	ptr = (void *)malloc(i+1);
	tmp = ptr;
	while (i--) {
		if (!isdigit(*str)) {
			free(ptr);
			return(INVALID_TIME);
		}

		*tmp++ = *(str+i);
	}
	*tmp = '\0';

	tmp = ptr;
	if (*tmp) offset += ((*tmp++) - '0') * 60;
	if (*tmp) offset += ((*tmp++) - '0') * 600;
	if (*tmp) offset += ((*tmp++) - '0') * 3600;
	if (*tmp) offset += ((*tmp++) - '0') * 36000;
	free(ptr);

	result += (offset + (days * 24 * 60 * 60));

	/*
	** If the specified time already happened, they probably
	** meant the same time tomorrow.
	*/
	if (result < now)
		result += (24 * 60 * 60);

	return(result);
}


/*
** get_user()
**
** Return the username of the person running this program.
**
**	Input:
**		none
**
**	Returns:
**		the userid of the person running this program
*/
char *
get_user(void)
{
	static char	username[255];
	struct passwd	*p;


	if ((p = getpwuid(geteuid())) != NULL)
		(void)strcpy(username, p->pw_name);
	else
		(void)sprintf(username, "UID=%lu", geteuid());


	return(username);
}


#ifndef CLIENT_ONLY
/*
** on_duty()
**
** This function returns a boolean status of whether the specified
** time falls inside the specified schedule.  A schedule has the
** format DaylistStarttime-Endtime (the same syntax as the UUCP
** /etc/uucp/Systems file).  For example, MoThFr1600-1800 specifies
** a two hour shift three days a week starting at 4pm.
** 
**	Input:
**		schedule - the schedule to check
**		when - the time to use when checking the schedule
**
**	Returns:
**		an integer status code
*/
int
on_duty(char *schedule, time_t when)
{
	struct tm	*tm;
	char		*days[] = {"Su", "Mo", "Tu", "We", "Th", "Fr", "Sa"};
	int		start;
	int		end;
	int		now;


	if (schedule == NULL)
		return(TRUE);

	if (when == 0)
		when = time(NULL);

	/*
	** It's an error if localtime() fails, but we'll send
	** the page out anyway.  It's probably better to page
	** someone by mistake than to drop the page on the floor.
	*/
	if ((tm = localtime(&when)) == NULL)
		return(-1);

	now = (tm->tm_hour * 100) + tm->tm_min;

	if (sscanf(schedule, "%*[^0123456789]%d-%d", &start, &end) != 2) {
		qpage_log(LOG_ERR, "invalid schedule: %s", schedule);
		return(-1);
	}

	if (start == 2400)
		start = 0;

	if (end == 0)
		end = 2400;

	if (!strstr(schedule, days[tm->tm_wday]) && !strstr(schedule, "Any"))
		return(FALSE);

	if (now >= start && now <= end)
		return(TRUE);

	return(FALSE);
}


/*
** safe_string()
**
** Make a string safe for printing (i.e. change control chars to \nnn).
**
**	Input:
**		str - the string to translate
**
**	Returns:
**		a buffer containing the safe string
**
**	Side effects:
**		The buffer is declared static and is overritten each time
**		this function is called.
*/
char *
safe_string(char *str)
{
	static char     buf[4096];
	char            *ptr = buf;
	int             i;


	*ptr = '\0';

	while (*str) {
		if (isgraph(*str))
			*ptr++ = *str++;
		else {
			switch(*str) {
				case ' ':
					*ptr++ = *str;
					break;

				case 27:
					*ptr++ = '\\';
					*ptr++ = 'e';
					break;

				case '\t':
					*ptr++ = '\\';
					*ptr++ = 't';
					break;

				case '\n':
					*ptr++ = '\\';
					*ptr++ = 'n';
					break;

				case '\r':
					*ptr++ = '\\';
					*ptr++ = 'r';
					break;

				default:
					i = *str;
					(void)sprintf(ptr, "\\%03o", i);
					ptr += 4;
					break;
			}

			str++;
		}

		*ptr = '\0';

	}

	return(buf);
}


/*
** lock_file()
**
** This function creates a kernel lock on a file descriptor.
**
**	Input:
**		fd - the file descriptor
**		mode - O_RDONLY or O_RDWR
**		block - block if the lock cannot be granted immediately
**
**	Returns:
**		an integer status code (0=success, -1=failure)
*/
int
lock_file(int fd, int mode, int block)
{
	struct flock	lock;
	int		i;


	(void)memset((char *)&lock, 0, sizeof(lock));

	lock.l_type = (mode == O_RDONLY ? F_RDLCK : F_WRLCK);
	lock.l_pid = getpid();

	i = fcntl(fd, (block ? F_SETLKW : F_SETLK), &lock);

	return(i);
}


/*
** lock_queue()
**
** This function creates a kernel lock on a lock file in the current
** working directory (assumed to be the the page queue).  The purpose
** of this function is to ensure that only one qpage process has the
** queue under its control at any given time.
**
**	Input:
**		nothing
**
**	Returns:
**		a locked open file descriptor, or -1 on failure
*/
int
lock_queue(void)
{
	int	fd;


	if ((fd = open("lock", O_CREAT|O_WRONLY, 0666)) < 0) {
		qpage_log(LOG_ERR, "cannot open queue lockfile: %s",
			strerror(errno));

		return(-1);
	}

	if (lock_file(fd, O_RDWR, FALSE) < 0) {
		qpage_log(LOG_ERR, "cannot lock queue lockfile");
		(void)close(fd);
		return(-1);
	}

	return(fd);
}


/*
** lookup()
**
** This function searches a linked list for the node matching the
** specified name.  A NULL pointer is returned if no such node exists.
**
**	Input:
**		list - a linked list (service_t *) or (pager_t *).
**		name - the name to search for
**
**	Returns:
**		a pointer to the found structure or NULL if not found
**
**	Note:
**		The linked list passed to this function must have the
**		firsts two fields be a pointer to the next node and
**		a pointer to a name, respectively.
*/
void *
lookup(void *list, char *name)
{
	struct {
		void	*next;
		char	*name;
	} *ptr;


	for (ptr=list; ptr; ptr=ptr->next) {
		if (!strcmp(ptr->name, name))
			return(ptr);
	}

	return(NULL);
}


/*
** get_lockname()
**
** This function determines the name of the lockfile to use when
** locking the modem.
**
**	Input:
**		buff - the buffer to store the filename
**		device - the name of the modem device
**
**	Returns:
**		an integer status code (0=success, -1=failure)
*/
int
get_lockname(char *buff, char *device)
{
#ifdef SOLARIS
	struct stat     statbuf;
#else
	char		*ptr;
#endif


#ifdef SOLARIS
	if (stat(device, &statbuf) < 0) {
		qpage_log(LOG_WARNING, "cannot stat() modem: %s",
			strerror(errno));

		return(-1);
	}

	(void)sprintf(buff, "%s/LK.%3.3lu.%3.3lu.%3.3lu", LOCKDIR,
		(unsigned long)major(statbuf.st_dev),
		(unsigned long)major(statbuf.st_rdev),
		(unsigned long)minor(statbuf.st_rdev));
#else
	if ((ptr = strrchr(device, '/')) == NULL)
		return(-1);

	ptr++;
	(void)sprintf(buff, "%s/LCK..%s", LOCKDIR, ptr);
#endif

	return(0);
}


/*
** lock_modem()
**
** This function locks the modem so that "tip" and friends don't
** stomp on us.
**
**	Input:
**		device - the name of the device to lock
**
**	Returns:
**		an integer status code (0=success, -1=error)
*/
int
lock_modem(char *device)
{
	FILE		*fp;
	pid_t		pid;
	char		buff[100];
	char		tmpfile[255];
	char		lockfile[255];
	int		fd;


	if (Debug || Interactive)
		qpage_log(LOG_DEBUG, "locking modem");

	if (get_lockname(lockfile, device) < 0) {
		qpage_log(LOG_WARNING, "cannot determine lock name for modem");
		return(-1);
	}

	(void)sprintf(tmpfile, "%s/LTMP.%lu", LOCKDIR, getpid());

	if ((fd = creat(tmpfile, 0444)) < 0) {
		qpage_log(LOG_WARNING, "cannot create lockfile: %s",
			strerror(errno));

		return(-1);
	}

	(void)sprintf(buff, "%10lu\n", getpid());
	(void)write(fd, buff, strlen(buff));
	(void)close(fd);

	if (link(tmpfile, lockfile) < 0) {
		if (errno != EEXIST) {
			qpage_log(LOG_WARNING, "cannot create lockfile: %s",
				strerror(errno));

			(void)unlink(tmpfile);
			return(-1);
		}

		if ((fp = fopen(lockfile, "r")) == NULL) {
			qpage_log(LOG_WARNING, "cannot open existing lockfile");
			(void)unlink(tmpfile);
			return(-1);
		}

		buff[0] = '\0';
		(void)fgets(buff, sizeof(buff), fp);
		(void)fclose(fp);

		pid = atoi(buff);

		/*
		** Does the lock owner exist?
		*/
		errno = 0;
		if (pid > 0 && ((kill(pid, 0) == 0) || (errno != ESRCH))) {
			qpage_log(LOG_INFO, "modem already in use");
			(void)unlink(tmpfile);
			return(-1);
		}

		/*
		** The process that owns the lock must have died.
		** Blast the existing lock file and create a new one.
		*/
		if (unlink(lockfile) < 0) {
			qpage_log(LOG_WARNING, "cannot remove lockfile: %s",
				strerror(errno));

			(void)unlink(tmpfile);
			return(-1);
		}

		if (link(tmpfile, lockfile) < 0) {
			qpage_log(LOG_WARNING, "cannot create lockfile: %s",
				strerror(errno));

			(void)unlink(tmpfile);
			return(-1);
		}
	}

	(void)unlink(tmpfile);
	return(0);
}


/*
** unlock_modem()
**
** This function unlocks the modem.
**
**	Input:
**		device - the name of the device to unlock
**
**	Returns:
**		nothing
*/
void
unlock_modem(char *device)
{
	char		lockfile[255];


	if (Debug || Interactive)
		qpage_log(LOG_DEBUG, "unlocking modem");

	if (get_lockname(lockfile, device) < 0) {
		qpage_log(LOG_WARNING, "cannot determine lock name for modem");
		return;
	}

	if (unlink(lockfile) < 0)
		qpage_log(LOG_WARNING, "cannot unlock lock modem: %s",
			strerror(errno));
}


void
get_sysinfo(void)
{
	struct utsname	utsname;
	char		buff[257];


	(void)(uname(&utsname));

#ifdef SI_SYSNAME
	if (sysinfo(SI_SYSNAME, buff, sizeof(buff)) > 0)
		qpage_log(LOG_INFO, "SI_SYSNAME = %s", buff);

#ifdef SI_HOSTNAME
	if (sysinfo(SI_HOSTNAME, buff, sizeof(buff)) > 0)
		qpage_log(LOG_INFO, "SI_HOSTNAME = %s", buff);
#endif

#ifdef SI_RELEASE
	if (sysinfo(SI_RELEASE, buff, sizeof(buff)) > 0)
		qpage_log(LOG_INFO, "SI_RELEASE = %s", buff);
#endif

#ifdef SI_VERSION
	if (sysinfo(SI_VERSION, buff, sizeof(buff)) > 0)
		qpage_log(LOG_INFO, "SI_VERSION = %s", buff);
#endif

#ifdef SI_MACHINE
	if (sysinfo(SI_MACHINE, buff, sizeof(buff)) > 0)
		qpage_log(LOG_INFO, "SI_MACHINE = %s", buff);
#endif

#ifdef SI_ARCHITECTURE
	if (sysinfo(SI_ARCHITECTURE, buff, sizeof(buff)) > 0)
		qpage_log(LOG_INFO, "SI_ARCHITECTURE = %s", buff);
#endif

#ifdef SI_ISALIST
	if (sysinfo(SI_ISALIST, buff, sizeof(buff)) > 0)
		qpage_log(LOG_INFO, "SI_ISALIST = %s", buff);
#endif

#ifdef SI_PLATFORM
	if (sysinfo(SI_PLATFORM, buff, sizeof(buff)) > 0)
		qpage_log(LOG_INFO, "SI_PLATFORM = %s", buff);
#endif

#ifdef SI_HW_PROVIDER
	if (sysinfo(SI_HW_PROVIDER, buff, sizeof(buff)) > 0)
		qpage_log(LOG_INFO, "SI_HW_PROVIDER = %s", buff);
#endif
#else
	/*
	** apparently sysinfo() isn't supported on this platform
	*/
	qpage_log(LOG_INFO, "utsname.sysname  = %s", utsname.sysname);
	qpage_log(LOG_INFO, "utsname.nodename = %s", utsname.nodename);
	qpage_log(LOG_INFO, "utsname.release  = %s", utsname.release);
	qpage_log(LOG_INFO, "utsname.version  = %s", utsname.version);
	qpage_log(LOG_INFO, "utsname.machine  = %s", utsname.machine);
#endif /* SI_SYSNAME */

	printf("\n");
}
#endif /* CLIENT_ONLY */
