/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1995-1998.  The Regents of the University of California.  All     */
/*   rights reserved.                                                      */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XDIR shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XDIR for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <Xm/SelectioB.h>
#include <Xm/Separator.h>
#include <Xm/ToggleBG.h>
#include <Xm/RowColumn.h>
#include <Xm/Form.h>
#include <Xm/Text.h>
#include <Xm/Frame.h>
#include <Xm/PushB.h>
#include <Xm/Label.h>
#include "xdir.h"
#include "str.h"
#include "pixmaps.h"
#include "okflag.h"
#include "errorflag.h"

int diagnostics;

static struct {
	Widget w_shell;
	Widget w_form;
	Widget w_controlArea;
	Widget w_actionArea;
	Widget w_closeButton;
	Widget w_saveButton;
	Widget w_clearButton;
	Widget w_helpButton;
	Widget w_separator1;
	Widget w_separator2;
	Widget w_diagnosticsRowcolumn;
	Widget w_diagnosticsLabel1;
	Widget w_diagnosticsLabel2;
	Widget w_diagnosticsFrame;
	Widget w_diagnosticsRadioBox;
	Widget w_quietToggle;
	Widget w_normalToggle;
	Widget w_verboseToggle;
	Widget w_debugToggle;
	Widget w_resultFlagFrame;
	Widget w_log;
	Widget w_resultFlag;
} diag;

static Pixmap okflag_pixmap;
static Pixmap errorflag_pixmap;
static int diag_visible;

static char *diagnostics_help[] = {
	"This window displays error messages and status information.  You",
	"can control the level of verbosity of messages via the four",
	"\"Diagnostics Level\" radio buttons at the top of the window.  The",
	"contents of the message log can be saved to a file by pressing the",
	"SAVE button.  Clear the message log by pressing the CLEAR",
	"button.  Dismiss this window by pressing the CLOSE button.\n",
	"\n",
	"The status flag indicates whether an operation completes",
	"successfully or not.  If an operation fails then (1) a beep",
	"is emitted, (2) the flag changes from green to red",
	"(white to black on monochrome displays), (3) the",
	"\"OK\" in the flag changes to \"E\", and (4) an error",
	"message is written to the message log.  The",
	"next operation will automatically clear the error flag.",
	NULL
};

extern Widget w_toplev;
extern Display *display;
extern int print_password_in_debug;
extern int diagnostics;
extern struct st_host_info hinfo[];
extern int initial_diagnostics;
extern int screen;
extern int sys_nerr;
#if !defined(__FreeBSD__) && !defined(__NetBSD__) && !defined(__GNU_LIBRARY__) && !defined(__GLIBC__) && !defined(__EMX__)
extern char *sys_errlist[];
#endif

void cb_diag_clear_log();
void cb_diag_save_log();
void cb_diag_close();
void cb_diag_help();
void cb_diag_save_log_ok();
void cb_diag_save_log_help();
void cb_change_diagnostics_level();
Pixmap create_pixmap();
char *get_filename();


/*
 * create_diagnostics_window - Create the diagnostics window.
 */
create_diagnostics_window()
{
	int i;
	Arg args[5];

	/* Done for naming consistency */
	diag.w_shell = w_toplev;

	/* Attach custom icon */
	attach_wm_icon(diag.w_shell);

	/* Create form */
	diag.w_form = XtVaCreateWidget("form", xmFormWidgetClass,
		diag.w_shell, NULL);

	/* Add callback for the WM_DELETE_WINDOW protocol */
	add_wm_delete_window_cb(diag.w_shell, cb_diag_close, NULL, True);

	/* Create form for action area */
    diag.w_actionArea = XtVaCreateWidget(
        "actionArea",
        xmFormWidgetClass,
        diag.w_form,
		XmNmarginHeight,		10,
		XmNbottomAttachment,	XmATTACH_FORM,
		XmNleftAttachment,		XmATTACH_FORM,
		XmNrightAttachment,		XmATTACH_FORM,
        NULL
    );

	/* Create Close pushbutton */
    diag.w_closeButton = XtVaCreateManagedWidget(
        "closeButton",
        xmPushButtonWidgetClass,
        diag.w_actionArea,
		XmNdefaultButtonShadowThickness,	1,
		XmNmarginHeight,					4,
		XmNtopAttachment,					XmATTACH_FORM,
        XmNbottomAttachment,    			XmATTACH_FORM,
        XmNleftAttachment,      			XmATTACH_POSITION,
        XmNleftPosition,        			5,
        XmNrightAttachment,     			XmATTACH_POSITION,
        XmNrightPosition,       			23,
        NULL
    );
    XtAddCallback(diag.w_closeButton, XmNactivateCallback, cb_diag_close,
        (XtPointer)NULL);
	XtVaSetValues(diag.w_form, XmNdefaultButton, diag.w_closeButton, NULL);

	/* Create Save pushbutton */
    diag.w_saveButton = XtVaCreateManagedWidget(
        "saveButton",
        xmPushButtonWidgetClass,
        diag.w_actionArea,
		XmNdefaultButtonShadowThickness,	1,
		XmNmarginHeight,					4,
		XmNtopAttachment,					XmATTACH_FORM,
        XmNbottomAttachment,    			XmATTACH_FORM,
        XmNleftAttachment,      			XmATTACH_POSITION,
        XmNleftPosition,        			29,
        XmNrightAttachment,     			XmATTACH_POSITION,
        XmNrightPosition,       			47,
        NULL
    );
    XtAddCallback(diag.w_saveButton, XmNactivateCallback, cb_diag_save_log,
		(XtPointer)NULL);

	/* Create Clear pushbutton */
    diag.w_clearButton = XtVaCreateManagedWidget(
        "clearButton",
        xmPushButtonWidgetClass,
        diag.w_actionArea,
		XmNdefaultButtonShadowThickness,	1,
		XmNmarginHeight,					4,
		XmNtopAttachment,					XmATTACH_FORM,
        XmNbottomAttachment,    			XmATTACH_FORM,
        XmNleftAttachment,      			XmATTACH_POSITION,
        XmNleftPosition,        			53,
        XmNrightAttachment,     			XmATTACH_POSITION,
        XmNrightPosition,       			71,
        NULL
    );
    XtAddCallback(diag.w_clearButton, XmNactivateCallback, cb_diag_clear_log,
		(XtPointer)NULL);

	/* Create Help pushbutton */
    diag.w_helpButton = XtVaCreateManagedWidget(
        "helpButton",
        xmPushButtonWidgetClass,
        diag.w_actionArea,
		XmNdefaultButtonShadowThickness,	1,
		XmNmarginHeight,					4,
		XmNtopAttachment,					XmATTACH_FORM,
        XmNbottomAttachment,    			XmATTACH_FORM,
        XmNleftAttachment,      			XmATTACH_POSITION,
        XmNleftPosition,        			77,
        XmNrightAttachment,     			XmATTACH_POSITION,
        XmNrightPosition,       			95,
        NULL
    );
    XtAddCallback(diag.w_helpButton, XmNactivateCallback, cb_diag_help,
		(XtPointer)NULL);

	XtManageChild(diag.w_actionArea);

    /* Create separator */
    diag.w_separator1 = XtVaCreateManagedWidget(
        "separator1",
        xmSeparatorWidgetClass,
        diag.w_form,
        XmNbottomAttachment,    XmATTACH_WIDGET,
        XmNbottomWidget,        diag.w_actionArea,
        XmNleftAttachment,      XmATTACH_FORM,
        XmNrightAttachment,     XmATTACH_FORM,
        NULL
    );

    /* Create form for control area */
    diag.w_controlArea = XtVaCreateWidget(
        "controlArea",
        xmFormWidgetClass,
        diag.w_form,
		XmNtopAttachment,		XmATTACH_FORM,
		XmNleftAttachment,		XmATTACH_FORM,
		XmNrightAttachment,		XmATTACH_FORM,
		XmNbottomAttachment,	XmATTACH_WIDGET,
		XmNbottomWidget,		diag.w_separator1,
		XmNmarginWidth,			10,
        NULL
    );

	/* Create rowcolumn to hold "Diagnostics Level" label */
    diag.w_diagnosticsRowcolumn = XtVaCreateManagedWidget(
        "diagnosticsRowcolumn",
        xmRowColumnWidgetClass,
        diag.w_controlArea,
		XmNentryAlignment,		XmALIGNMENT_CENTER,
		XmNtopAttachment,		XmATTACH_FORM,
		XmNtopOffset,			10,
		XmNleftAttachment,		XmATTACH_FORM,
        NULL
    );

    /* Create label for first line of diagnostics level label */
    diag.w_diagnosticsLabel1 = XtVaCreateManagedWidget(
        "diagnosticsLabel1",
        xmLabelWidgetClass,
        diag.w_diagnosticsRowcolumn,
        NULL
    );

    /* Create label for second line of diagnostics level label */
    diag.w_diagnosticsLabel2 = XtVaCreateManagedWidget(
        "diagnosticsLabel2",
        xmLabelWidgetClass,
        diag.w_diagnosticsRowcolumn,
        NULL
    );

	/* Create frame for diagnostics radio box*/
	diag.w_diagnosticsFrame = XtVaCreateManagedWidget(
		"diagnosticsFrame",
		xmFrameWidgetClass,
		diag.w_controlArea,
		XmNtopAttachment,		XmATTACH_FORM,
		XmNtopOffset,			17,
		XmNleftAttachment,		XmATTACH_WIDGET,
		XmNleftWidget,			diag.w_diagnosticsRowcolumn,
		XmNleftOffset,			10,
		NULL
	);

	/* Create radio box for diagnostics level */
	 i = 0;
	 XtSetArg(args[i], XmNorientation, XmHORIZONTAL); i++;
	 diag.w_diagnosticsRadioBox = XmCreateRadioBox(diag.w_diagnosticsFrame,
		"diagnosticsRadioBox", args, i);

	/* Create "Quiet" toggle */
	diag.w_quietToggle = XtVaCreateManagedWidget(
		"quietToggle",
		xmToggleButtonGadgetClass,
		diag.w_diagnosticsRadioBox,
		XmNmarginHeight,	0,
		NULL
	);
	XtAddCallback(diag.w_quietToggle, XmNvalueChangedCallback,
		cb_change_diagnostics_level, (XtPointer)NULL);

	/* Create "Normal" toggle */
	diag.w_normalToggle = XtVaCreateManagedWidget(
		"normalToggle",
		xmToggleButtonGadgetClass,
		diag.w_diagnosticsRadioBox,
		XmNmarginHeight,	0,
		NULL
	);
	XtAddCallback(diag.w_normalToggle, XmNvalueChangedCallback,
		cb_change_diagnostics_level, (XtPointer)NULL);

	/* Create "Verbose" toggle */
	diag.w_verboseToggle = XtVaCreateManagedWidget(
		"verboseToggle",
		xmToggleButtonGadgetClass,
		diag.w_diagnosticsRadioBox,
		XmNmarginHeight,	0,
		NULL
	);
	XtAddCallback(diag.w_verboseToggle, XmNvalueChangedCallback,
		cb_change_diagnostics_level, (XtPointer)NULL);

	/* Create "Debug" toggle */
	diag.w_debugToggle = XtVaCreateManagedWidget(
		"debugToggle",
		xmToggleButtonGadgetClass,
		diag.w_diagnosticsRadioBox,
		XmNmarginHeight,	0,
		NULL
	);
	XtAddCallback(diag.w_debugToggle, XmNvalueChangedCallback,
		cb_change_diagnostics_level, (XtPointer)NULL);

	/* Initialize diagnostics level */
	diagnostics = initial_diagnostics;
	switch (diagnostics) {
	case QUIET:
		XmToggleButtonGadgetSetState(diag.w_quietToggle, True, True);
		break;
	case NORMAL:
		XmToggleButtonGadgetSetState(diag.w_normalToggle, True, True);
		break;
	case VERBOSE:
		XmToggleButtonGadgetSetState(diag.w_verboseToggle, True, True);
		break;
	case DEBUG:
		XmToggleButtonGadgetSetState(diag.w_debugToggle, True, True);
	}

	XtManageChild(diag.w_diagnosticsRadioBox);

    /* Create separator */
    diag.w_separator2 = XtVaCreateManagedWidget(
        "separator2",
        xmSeparatorWidgetClass,
        diag.w_controlArea,
        XmNtopAttachment,	XmATTACH_WIDGET,
		XmNtopWidget,		diag.w_diagnosticsRowcolumn,
		XmNtopOffset,		10,
        XmNleftAttachment,	XmATTACH_FORM,
        XmNrightAttachment,	XmATTACH_FORM,
        NULL
    );

	/* Create frame for result flag */
	diag.w_resultFlagFrame = XtVaCreateManagedWidget(
		"resultFlagFrame",
		xmFrameWidgetClass,
		diag.w_controlArea,
		XmNshadowType,		XmSHADOW_IN,
		XmNtopAttachment,	XmATTACH_WIDGET,
		XmNtopWidget,		diag.w_separator2,
		XmNtopOffset,		10,
		XmNleftAttachment,	XmATTACH_FORM,
		NULL
	);

    /* Create result flag */
    diag.w_resultFlag = XtVaCreateManagedWidget(
        "resultFlag",
        xmLabelWidgetClass,
        diag.w_resultFlagFrame,
        XmNlabelType,       XmPIXMAP,
        NULL
    );

	/* Initialize result flag to "OK" */
	init_result_flag();

	/* Create diagnostics log */
	i = 0;
	XtSetArg(args[i], XmNeditMode, XmMULTI_LINE_EDIT); i++;
	XtSetArg(args[i], XmNeditable, False); i++;
	XtSetArg(args[i], XmNmarginHeight, 3); i++;
	diag.w_log = XmCreateScrolledText(diag.w_controlArea, "log", args, i);
	XtVaSetValues(XtParent(diag.w_log),
        XmNtopAttachment,		XmATTACH_OPPOSITE_WIDGET,
		XmNtopWidget,			diag.w_resultFlagFrame,
		XmNtopOffset,			0,
		XmNbottomAttachment,	XmATTACH_FORM,
		XmNbottomOffset,		6,
		XmNleftAttachment,		XmATTACH_WIDGET,
		XmNleftWidget,			diag.w_resultFlagFrame,
		XmNleftOffset,			7,
		XmNrightAttachment,		XmATTACH_FORM,
		NULL
	);
	XtManageChild(diag.w_log);

	XtManageChild(diag.w_controlArea);
	XtManageChild(diag.w_form);

	add_dialog_to_list(diag.w_shell);
}


/*
 * cb_diag_close - Callback to hide diagnostics window.
 */
void
cb_diag_close(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	XtUnmapWidget(diag.w_shell);
	diag_visible = False;
}


void
cb_diag_help(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	help_dialog(widget, False, "Diagnostics", diagnostics_help);
}


/*
 * cb_popup_diagnostics - Callback to pop diagnostics window to top.
 */
cb_popup_diagnostics(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	XMapRaised(display, XtWindow(diag.w_shell));
	traverse_to_widget(diag.w_closeButton);
	diag_visible = True;
}


/*
 * write_log - Write the string "msg" to the log window.
 */
write_log(msg)
char *msg;
{
	/* Write message to log */
    XmTextInsert(diag.w_log, XmTextGetLastPosition(diag.w_log), msg);
    XmTextShowPosition(diag.w_log, XmTextGetLastPosition(diag.w_log));
	XFlush(display);
}


/*
 * cb_diag_clear_log - Callback to clear the log window.
 */
void
cb_diag_clear_log(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	/* Clear error flag */
	raise_okflag();

	XmTextSetString(diag.w_log, "");
}


/*
 * report_perror - Write "msg" to log followed by error message corresponding
 *                 to "errno".
 */
report_perror(msg)
char *msg;
{
	char buf[10];
	int saved_errno = errno;

	write_log("Local: ");
	if (msg != NULL && strlen(msg) != 0) {
		write_log(msg);
		write_log(" - ");
	}
	if (saved_errno < sys_nerr) {
		write_log(sys_errlist[saved_errno]);
		write_log("\n");
	} else {
		sprintf(buf, "errno = %d\n", saved_errno);
		write_log(buf);
	}
}


/*
 * report_ftp_reply - Write the string "reply" to the log window.
 */
report_ftp_reply(host, reply)
int host;
char *reply;
{
	write_log(hinfo[host].hostname);
	write_log(": ");
    if (diagnostics == DEBUG)
        write_log("<-- ");
    write_log(reply);
}


/*
 * report_ftp_cmd - Write the string "cmd" to the log window.
 */
report_ftp_cmd(host, cmd)
int host;
char *cmd;
{
	char *temp = XtNewString(cmd);
	char *q;

	write_log(hinfo[host].hostname);
	write_log(": --> ");

	/* Should the password be displayed? */
	if (!print_password_in_debug && (q = strstr(temp, "PASS")))
		q[4] = '\0';

	write_log(temp);
	write_log("\n");
	XtFree(temp);
}


/*
 * cb_diag_save_log - Callback to save the contents of the log window into
 *                    a disk file.
 */
void
cb_diag_save_log(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	char *logfile_name;
	FILE *fp;
	char *buf;
	char msg[MAXPATHLEN+25];

    /* Clear error flag */
    raise_okflag();

	/* Get logfile name */
	if (!(logfile_name = get_filename("Save Diagnostics Log",
			"Name for log file:", diag.w_shell)))
		return;

	/* Open file for writing */
    if ((fp = fopen(logfile_name, "w")) == NULL) {
		sprintf(msg, "Trouble saving log to %s\n", logfile_name);
		record_and_alert(msg, diag.w_shell);
		return;
	}

    /* Get copy of log buffer */
    buf = XmTextGetString(diag.w_log);

    /* Write buffer to file */
    fprintf(fp, "%s", buf);
    fclose(fp);

    /* Free memory */
    XtFree(buf);
    XtFree(logfile_name);
}


/*
 * show_top_of_log - Scroll to top of log.
 */
show_top_of_log()
{
	XtVaSetValues(diag.w_log, XmNtopCharacter, 0, NULL);
}


/*
 * record_warning - Writes string "msg" to diagnostics log and prepends
 *                  a warning message.
 */
record_warning(msg)
char *msg;
{
	char *buf;

	buf = XtMalloc(strlen(msg)+30);
	sprintf(buf, "*** WARNING: %s\n", msg);
	write_log(buf);
	XtFree(buf);
	raise_errorflag();
}


/*
 * record_abort - Writes to log a message that "operation" was aborted.
 */
record_abort(operation)
char *operation;
{
	char *buf;

	buf = XtMalloc(strlen(operation)+30);
	sprintf(buf, "*** OPERATION ABORTED: %s\n", operation);
	write_log(buf);
	XtFree(buf);
}


/*
 * report_retry - Write retry message to the log for "operation".
 */
report_retry(operation)
char *operation;
{
	char *msg = XtMalloc(30+strlen(operation));

	sprintf(msg, "*** %s failed. Will retry.\n", operation);
	write_log(msg);
	XtFree(msg);
}


/*
 * init_result_flag - Create the pixmaps for the "OK" and "Error" flags,
 *                    and initialize the result flag to "OK".
 */
init_result_flag()
{
	Pixel foreground;
	Pixel background;

	/* Get foreground and background colors of diagnostics window */
	XtVaGetValues(diag.w_form,
		XmNforeground,	&foreground,
		XmNbackground,	&background,
		NULL
	);

	/* Create flag pixmaps */
	okflag_pixmap = create_pixmap(&okflag, (unsigned long)foreground,
		(unsigned long)background);
	errorflag_pixmap = create_pixmap(&errorflag, (unsigned long)foreground,
		(unsigned long)background);

	/* Initialize result flag to "OK" */
    XtVaSetValues(diag.w_resultFlag,
		XmNlabelPixmap, okflag_pixmap,
		XmNuserData,	True,
		NULL
	);
}


/*
 * raise_okflag - Set the result flag to the "OK" flag.
 */
raise_okflag()
{
	int okflag_raised;

	/* See if "OK" flag is already raised */
	XtVaGetValues(diag.w_resultFlag, XmNuserData, &okflag_raised, NULL);

    if (!okflag_raised)
        XtVaSetValues(diag.w_resultFlag,
			XmNlabelPixmap, okflag_pixmap,
			XmNuserData,	True,
			NULL
		);
}


/*
 * raise_errorflag - Set the result flag to the "Error" flag.
 */
raise_errorflag()
{
	int okflag_raised;

	/* See if error flag is already raised */
	XtVaGetValues(diag.w_resultFlag, XmNuserData, &okflag_raised, NULL);

    if (okflag_raised)
        XtVaSetValues(diag.w_resultFlag,
			XmNlabelPixmap, errorflag_pixmap,
			XmNuserData,	False,
			NULL
		);
}

/*
 * cb_change_diagnostics_level - Callback to process changed diagnostics
 *                               level.
 */
void
cb_change_diagnostics_level(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	if (XmToggleButtonGadgetGetState(diag.w_quietToggle))
		diagnostics = QUIET;
	else if (XmToggleButtonGadgetGetState(diag.w_normalToggle))
		diagnostics = NORMAL;
	else if (XmToggleButtonGadgetGetState(diag.w_verboseToggle))
		diagnostics = VERBOSE;
	else if (XmToggleButtonGadgetGetState(diag.w_debugToggle))
		diagnostics = DEBUG;
	else
		fatal_error("Bug in cb_change_diagnostics_level()");
}


/*
 * iconify_diag_window - Iconify diagnostics window.
 */
iconify_diag_window()
{
	XIconifyWindow(display, XtWindow(diag.w_shell), screen);
}


/*
 * deiconify_diag_window - Deiconify diagnostics window.
 */
deiconify_diag_window()
{
	if (diag_visible)
		XMapWindow(display, XtWindow(diag.w_shell));
}

