//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  The 12K cartridge class used by CBS' RAM Plus cartridges. There
  are 3 4K banks and 256-bytes of RAM.

  @author  Bradford W. Mott
  @version $Id: CartFASC.cxx,v 1.2 1997/05/17 19:00:04 bwmott Exp $
*/

#include "CartFASC.hxx"
#include "System.hxx"

//============================================================================
// Constructor
//============================================================================
CartridgeFASC::CartridgeFASC(System& system, uByte* image)
    : Cartridge(system),
      myImageOffset(0),
      myRAMOffset(0)
{
  // Copy the ROM image into my buffer
  for(uWord bank = 0; bank < 3; ++bank)
  {
    for(uWord addr = 0; addr < 4096; ++addr)
    {
      myImage[(bank * 4096) + addr] = image[(bank * 4096) + addr];
    }
  }

  // Map all of my addresses in the system
  for(uWord addr = 0; addr < 8192; ++addr)
  {
    if(addr & 0x1000)
    {
      // Is this read address mapped to the cartridge RAM
      if((addr & 0x0f00) == 0x0100)
      {
        mySystem.mapPeek(addr, *this, &myRAM[addr & 0x00ff], &myRAMOffset);
      }
      // Is this read address a "hot spot" or a read from "no where"
      else if(((addr & 0x0fff) == 0x0ff8) || ((addr & 0x0fff) == 0x0ff9) ||
          ((addr & 0x0fff) == 0x0ffA) || ((addr & 0x0f00) == 0x0000))
      {
        mySystem.mapPeek(addr, *this);
      }
      // Is this read address mapped to the standard ROM image
      else
      {
        mySystem.mapPeek(addr, *this, 
            &myImage[addr & 0x0fff], &myImageOffset);
      }

      // Is this write address mapped to the cartridge RAM
      if((addr & 0x0f00) == 0x0000)
        mySystem.mapPoke(addr, *this, &myRAM[addr & 0x00ff]);
      else
        mySystem.mapPoke(addr, *this);
    }
  }

  // Reset myself
  reset();
}
 
//============================================================================
// Destructor
//============================================================================
CartridgeFASC::~CartridgeFASC()
{
}

//============================================================================
// Reset to my power on state
//============================================================================
void CartridgeFASC::reset()
{
  // RAM's addressing offset is always zero
  myRAMOffset = 0;

  // Start execution in bank #2
  myImageOffset = 2 * 4096;
}

//============================================================================
// Answer the byte at the given address
//============================================================================
uByte CartridgeFASC::peek(uWord addr)
{
  // Switch banks if necessary
  switch (addr & 0x0fff)
  {
    case 0x0ff8:    // Bank #0
      myImageOffset = 0 * 4096;
      break;

    case 0x0ff9:    // Bank #1
      myImageOffset = 1 * 4096;
      break;

    case 0x0ffA:    // Bank #2
      myImageOffset = 2 * 4096;
      break;

    default:
      break; 
  }
  return myImage[myImageOffset + (addr & 0x0fff)];
}

//============================================================================
// Store value in the given address
//============================================================================
void CartridgeFASC::poke(uWord addr, uByte)
{
  // Switch banks if necessary
  switch (addr & 0x0fff)
  {
    case 0x0ff8:    // Bank #0
      myImageOffset = 0 * 4096;
      break;

    case 0x0ff9:    // Bank #1
      myImageOffset = 1 * 4096;
      break;

    case 0x0ffA:    // Bank #2
      myImageOffset = 2 * 4096;
      break;

    default:
      break; 
  }
}

