//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  The 2600's paddle controller

  @author  Bradford W. Mott
  @version $Id: Paddle.cxx,v 1.2 1997/05/17 19:00:06 bwmott Exp $
*/

#include "BasTerm.hxx"
#include "Paddle.hxx"
#include "System.hxx"

//============================================================================
// Constructor
//============================================================================
Paddle::Paddle(const System& system, Paddle::PhysicalPort paddleZeroPort,
    Paddle::PhysicalPort paddleOnePort)
    : Controller(system),
      myPaddleZeroPort(paddleZeroPort),
      myPaddleOnePort(paddleOnePort)
{
  myDumping = true;
  myBitZeroTimer = myBitOneTimer = 0; 
}
 
//============================================================================
// Destructor
//============================================================================
Paddle::~Paddle()
{
}

//============================================================================
// Read the TIA information from the controller
//============================================================================
uByte Paddle::readTIA() const
{
  uByte tia = 0x03;

  // Set the dumped inputs
  if((mySystem.m6507().cycles() > myBitZeroTimer) && (!myDumping))
    tia |= 0x01;
  else
    tia &= 0x06;

  if((mySystem.m6507().cycles() > myBitOneTimer) && (!myDumping))
    tia |= 0x02;
  else
    tia &= 0x05;

  return tia;
}

//============================================================================
// Invoked before each frame is generated
//============================================================================
void Paddle::update()
{
  switch(myPaddleZeroPort)
  {
    case Zero:
      // Set the fire button's 
      if(mySystem.terminal().eventState(BasicTerminal::PaddleZeroFire))
        myPIA &= 0x07;
      else
        myPIA |= 0x08;
      break;

    case One:
      // Set the fire button's 
      if(mySystem.terminal().eventState(BasicTerminal::PaddleOneFire))
        myPIA &= 0x07;
      else
        myPIA |= 0x08;
      break;

    case Two:
      // Set the fire button's 
      if(mySystem.terminal().eventState(BasicTerminal::PaddleTwoFire))
        myPIA &= 0x07;
      else
        myPIA |= 0x08;
      break;

    case Three:
      // Set the fire button's 
      if(mySystem.terminal().eventState(BasicTerminal::PaddleThreeFire))
        myPIA &= 0x07;
      else
        myPIA |= 0x08;
      break;

    default:
      break;
  }
  
  switch(myPaddleOnePort)
  {
    case Zero:
      // Set the fire button's 
      if(mySystem.terminal().eventState(BasicTerminal::PaddleZeroFire))
        myPIA &= 0x0b;
      else
        myPIA |= 0x04;
      break;

    case One:
      // Set the fire button's 
      if(mySystem.terminal().eventState(BasicTerminal::PaddleOneFire))
        myPIA &= 0x0b;
      else
        myPIA |= 0x04;
      break;

    case Two:
      // Set the fire button's 
      if(mySystem.terminal().eventState(BasicTerminal::PaddleTwoFire))
        myPIA &= 0x0b;
      else
        myPIA |= 0x04;
      break;

    case Three:
      // Set the fire button's 
      if(mySystem.terminal().eventState(BasicTerminal::PaddleThreeFire))
        myPIA &= 0x0b;
      else
        myPIA |= 0x04;
      break;

    default:
      break;
  }
}

//============================================================================
// Configure the TIA portion of the controller
//============================================================================
void Paddle::configureTIA(TIACommand command)
{
  if(command == EnableDump)
  {
    // Enable the dump
    myDumping = true;
  }
  else if(command == DisableDump)
  {
    // Disable the dump
    myDumping = false;

    // Compute the CPU cycle when the TIA's dumped bits should be set
    uLong cycles = mySystem.m6507().cycles();

    switch(myPaddleZeroPort)
    {
      case Zero:
        myBitZeroTimer = cycles + 1000 + 30 *
            mySystem.terminal().eventState(BasicTerminal::PaddleZeroPosition);
        break;

      case One:
        myBitZeroTimer = cycles + 1000 + 30 *
            mySystem.terminal().eventState(BasicTerminal::PaddleOnePosition);
        break;

      case Two:
        myBitZeroTimer = cycles + 1000 + 30 *
            mySystem.terminal().eventState(BasicTerminal::PaddleTwoPosition);
        break;

      case Three:
        myBitZeroTimer = cycles + 1000 + 30 *
            mySystem.terminal().eventState(BasicTerminal::PaddleThreePosition);
        break;

      default:
        break;
    }

    switch(myPaddleOnePort)
    {
      case Zero:
        myBitOneTimer = cycles + 1000 + 30 *
            mySystem.terminal().eventState(BasicTerminal::PaddleZeroPosition);
        break;

      case One:
        myBitOneTimer = cycles + 1000 + 30 *
            mySystem.terminal().eventState(BasicTerminal::PaddleOnePosition);
        break;

      case Two:
        myBitOneTimer = cycles + 1000 + 30 *
            mySystem.terminal().eventState(BasicTerminal::PaddleTwoPosition);
        break;

      case Three:
        myBitOneTimer = cycles + 1000 + 30 *
            mySystem.terminal().eventState(BasicTerminal::PaddleThreePosition);
        break;

      default:
        break;
    }
  }
}

