//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  Sound class for the MS-DOS operating system with sound-blaster cards.

  @author  Bradford W. Mott
  @version $Id: SndDOS.cxx,v 1.4 1997/06/07 00:46:00 bwmott Exp $
*/

#include <go32.h>
#include <dos.h>
#include <dpmi.h>
#include <sys/movedata.h>
#include <pc.h>

#include <assert.h>
#include <ctype.h>
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>

#include "sound/TIASound.h"
#include "SndDOS.hxx"

#define DMA_BASE         0x00
#define DMA_COUNT        0x01
#define DMA_MASK         0x0a
#define DMA_MODE         0x0b
#define DMA_FF           0x0c

// Globals for the SB configuration
uLong thePort = 0x220;
uLong theIRQ = 5;
uLong theDMA = 1;
uWord theDSPVersion = 0;
uWord theSampleRate = 0;
uWord theDMABufferSize = 512;

unsigned char theOriginalMasterPICState;
unsigned char theOriginalSlavePICState;
 
int theSelector[2];                          // Selectors for DMA buffers
unsigned long theBuffer[2];                  // Pointers to DMA buffers
unsigned char theTempBuffer[2][2048];        // Temp buffers
unsigned int theCurrentBuffer = 0;           // Current buffer in use
bool theBufferUsed[2];

_go32_dpmi_seginfo theOldInterruptVector;
_go32_dpmi_seginfo theInterruptVector;

//============================================================================
// Read a byte from the SB DSP chip
//============================================================================
volatile uByte readDSP()
{
  while((inportb(thePort + 0x0E) & 0x80) == 0);
  return inportb(thePort + 0x0A);
}

//============================================================================
// Write a byte to the SB DSP chip
//============================================================================
volatile void writeDSP(uByte c)
{
  while((inportb(thePort + 0x0C) & 0x80) != 0);

  outportb(thePort + 0x0C, c);
}

//============================================================================
// Resets the SB DSP chip and answer true iff successful
//============================================================================
bool resetDSP()
{
  // Set the reset flag
  outportb(thePort + 0x06, 1);

  // Wait for at least 3 microseconds
  delay(3);

  // Clear the reset flag
  outportb(thePort + 0x06, 0);

  bool success = false;
  for(int t = 0; (t < 1000) && !success; ++t)
  {
    if(inportb(thePort + 0x0E) & 0x80)
      success = true;
  }

  if(success)
  {
    for(int t = 0; t < 1000; ++t)
    {
      if(inportb(thePort + 0x0A) == 0xAA)
      {
        // Get the DSP version number
        writeDSP(0xE1);
        theDSPVersion = readDSP();
        theDSPVersion <<= 8;
        theDSPVersion |= readDSP();

        return true;
      }
    }
  }

  return false;
}

//============================================================================
// Detect the sound blaster setup answering true iff SB exists
//============================================================================
bool detectSB()
{
  // Start with some reasonable values
  thePort = 0x220;
  theIRQ = 5;
  theDMA = 1;

  // Parse the BLASTER environment variable
  char* blaster = getenv("BLASTER");

  while(*blaster != 0)
  {
    // Skip white space
    while((*blaster == ' ') || (*blaster == '\t'))
      blaster++;

    if(tolower(*blaster) == 'a') 
    {
      thePort = strtol(blaster + 1, 0, 16);
    }
    else if(tolower(*blaster) == 'd') 
    {
      theDMA = strtol(blaster + 1, 0, 10);
    }
    else if(tolower(*blaster) == 'i') 
    {
      theIRQ = strtol(blaster + 1, 0, 10);
    }

    // Skip over information we've parsed
    while((*blaster != 0) && (*blaster != ' ') && (*blaster != '\t'))
      ++blaster;
  }

  return resetDSP();
}

//============================================================================
// Set the sample rate
//============================================================================
uWord setSampleRate(uWord rate)
{
  // Make sure sample rate isn't to low
  if(rate < 5000)
  {
    rate = 5000;
  }

  // Set the maximum sample rate based on the DSP version in the sound card
  if(theDSPVersion > 0x0202)
  {
    if(rate > 44100)
      rate = 44100;
  }
  else
  {
    if(rate > 23000)
      rate = 23000;
  }

  // Set the sample rate in the DSP
  if(theDSPVersion > 0x0400)
  {
    // Set the output sample rate in the SB16
    writeDSP(0x41);
    writeDSP((rate >> 8) & 0x00FF);
    writeDSP(rate & 0x00FF);
  }
  else
  {
    uByte constant = (65536L - (256000000L / rate)) >> 8;
    writeDSP(0x40);
    writeDSP(constant);
    rate = 256000000 / (65536L - (uLong)constant * 256);
  }

  return rate;
}

//============================================================================
// Compute the buffer size to use based on the given sample rate
//============================================================================
unsigned long computeBufferSize(int sampleRate)
{
  int t;

  for(t = 7; t <= 12; ++t)
  {
    if((1 << t) > (sampleRate / 60))
    {
      return (1 << (t - 1));
    }
  }

  return 256;
}

//============================================================================
// Allocate a block of memory for DMA access of the given size
//============================================================================
bool allocateDMA(uLong size, int& selector, unsigned long& addr)
{
  int segment = __dpmi_allocate_dos_memory((size * 2 + 15) >> 4, &selector);

  if(segment < 0)
  {
    selector = 0;
    addr = 0;
    return false;
  }

  addr = segment << 4;

  // If it crosses a page boundary then bump it up to the next page boundary
  if((addr >> 16) != ((addr + size) >> 16))
    addr = ((addr + size) & 0xffffff00);

  return true;
}

//============================================================================
// Send the contents of the indexed buffer to the DSP
//============================================================================
void sendAudio(int buffer)
{
  unsigned int pagePort[] = {0x87, 0x83, 0x81, 0x82};

  unsigned long addr = theBuffer[buffer];
  unsigned long page = addr >> 16;
  unsigned long offset = addr & 0xFFFF;
  unsigned long size = theDMABufferSize - 1;

  // Setup DMA to do the transfer
  disable();
  outportb(DMA_MASK, 0x04 | theDMA);
  outportb(DMA_FF, 0);
  outportb(DMA_MODE, 0x48 | theDMA);
  outportb(DMA_BASE + (theDMA << 1), offset & 0xFF);
  outportb(DMA_BASE + (theDMA << 1), offset >> 8);
  outportb(pagePort[theDMA], page);
  outportb(DMA_COUNT + (theDMA << 1), size & 0xFF);
  outportb(DMA_COUNT + (theDMA << 1), size >> 8);
  outportb(DMA_MASK, theDMA);
  enable();

  // Tell DSP to start playing 
  if(theDSPVersion > 0x0400)
  {
    writeDSP(0xC0);          // 8-bit output
    writeDSP(0x00);          // mono unsigned PCM
    writeDSP(size & 0xFF);
    writeDSP(size >> 8);
  }
  else if((theDSPVersion > 0x0201) && (theSampleRate >= 23000))
  {
    writeDSP(0x48);          // Set block size
    writeDSP(size & 0xFF);
    writeDSP(size >> 8);
    writeDSP(0x91);          // 8-bit PCM high-speed output
  }
  else
  { 
    writeDSP(0x14);          // 8-bit PCM output
    writeDSP(size & 0xFF);
    writeDSP(size >> 8);
  }
}

//============================================================================
// Called to process the next buffer
//============================================================================
void fillNextBuffer()
{
  int buffer = 1 - theCurrentBuffer;

  if(!theBufferUsed[buffer])
  {
    Tia_process(theTempBuffer[buffer], theDMABufferSize);
    theBufferUsed[buffer] = true;

    // Copy from temp buffer into DMA buffer
    dosmemput(theTempBuffer[buffer], theDMABufferSize, theBuffer[buffer]);
  }
}

//============================================================================
// SB end-of-buffer interrupt handler
//============================================================================
static void interruptServiceRoutine()
{
  // Indicate that the current buffer is free
  theBufferUsed[theCurrentBuffer] = false;

  // Move to the next buffer
  theCurrentBuffer = 1 - theCurrentBuffer;

  // Transmit buffer if it's ready
  if(theBufferUsed[theCurrentBuffer])
  {
    sendAudio(theCurrentBuffer);
  }

  static volatile bool semaphore = false;

  if(!semaphore)
  {
    semaphore = true;

    // Fill the unused buffer before we leave
    fillNextBuffer();

    semaphore = false;
  }

  // Acknowledge DSP interrupt
  inportb(thePort + 0x0E);

  // Acknowledge slave interrupt
  if(theIRQ >= 8)
    outportb(0xA0, 0x20);

  // Acknowledge master interrupt
  outportb(0x20, 0x20);
}  

//============================================================================
// Enable the specified interrupt
//============================================================================
static void enableInterrupt(int irq)
{ 
  if(irq < 8)
  {
    outportb(0x21, inportb(0x21) & (~ (1 << irq)));
  }
  else
  {
    outportb(0x21, inportb(0x21) & 0xFB);
    outportb(0xA1, inportb(0xA1) & (~ (1 << (irq - 8))));
  }
} 

//============================================================================
// Disable the specified interrupt
//============================================================================
void disableInterrupt(int irq)
{ 
  if(irq < 8)
  {
    outportb(0x21, inportb(0x21) | (1 << irq));
  }
  else
  {
    outportb(0xA1, inportb(0xA1) | (1 << (irq - 8)));
  }
} 

//============================================================================
// Constructor
//============================================================================
SoundDOS::SoundDOS()
  : myEnabled(true)
{
  // Detect the SB
  if(!detectSB())
  {
    myEnabled = false;
    cerr << "ERROR: Couldn't reset DSP!" << endl;
    return;
  }

  // Turn speaker on
  writeDSP(0xD1);

  // Set sample rate constant and buffer size
  theSampleRate = setSampleRate(31400); 
  theDMABufferSize = computeBufferSize(theSampleRate);

  // Initialize TIA Sound Library
  Tia_sound_init(31400, theSampleRate);

  // Allocate buffers for audio
  if(!allocateDMA(theDMABufferSize, theSelector[0], theBuffer[0]))
  {
    myEnabled = false;
    cerr << "ERROR: Couldn't allocate DMA buffer!" << endl;
    return;
  }

  if(!allocateDMA(theDMABufferSize, theSelector[1], theBuffer[1]))
  {
    myEnabled = false;
    cerr << "ERROR: Couldn't allocate DMA buffer!" << endl;
    return;
  }

  theBufferUsed[0] = false;
  theBufferUsed[1] = false;
  theCurrentBuffer = 0;

  // Figure out the vector for the IRQ
  int vectorNumber;
  if(theIRQ < 8)
    vectorNumber = theIRQ + 8;
  else
    vectorNumber = theIRQ + 104;

  // Save the original state of the PICs
  theOriginalMasterPICState = inportb(0x21);
  theOriginalSlavePICState = inportb(0xA1);

  disable();
  disableInterrupt(theIRQ);

  _go32_dpmi_get_protected_mode_interrupt_vector(vectorNumber,
      &theOldInterruptVector);
  theInterruptVector.pm_selector = _my_cs();
  theInterruptVector.pm_offset = (int)interruptServiceRoutine;
  _go32_dpmi_allocate_iret_wrapper(&theInterruptVector);
  _go32_dpmi_set_protected_mode_interrupt_vector(vectorNumber,
      &theInterruptVector);

  enableInterrupt(theIRQ);
  enable();

  fillNextBuffer();
  sendAudio(theCurrentBuffer);
}
 
//============================================================================
// Destructor
//============================================================================
SoundDOS::~SoundDOS()
{
  if(myEnabled)
  {
    disable();

    // Halt DMA
    writeDSP(0xD0);

    // Reset DSP just in case
    resetDSP(); 

    // Turn speaker off
    writeDSP(0xD3);

    // Restore master PIC and slave PIC to original state
    outportb(0x21, theOriginalMasterPICState);
    outportb(0xA1, theOriginalSlavePICState);

    // Figure out the vector for the IRQ
    int vectorNumber;
    if(theIRQ < 8)
      vectorNumber = theIRQ + 8;
    else
      vectorNumber = theIRQ + 104;

    // Restore original interrupt routine
    _go32_dpmi_set_protected_mode_interrupt_vector(vectorNumber,
        &theOldInterruptVector);
    _go32_dpmi_free_iret_wrapper(&theInterruptVector);

    enable();

    // Free DMA buffers
    // TODO!
  }
}

//============================================================================
// Set the given sound register to the specified value
//============================================================================
void SoundDOS::set(Sound::Register reg, uByte value)
{
  if(!myEnabled)
    return;

  switch(reg) 
  {
    case AUDC0:
      Update_tia_sound(0x15, value);
      break;
    
    case AUDC1:
      Update_tia_sound(0x16, value);
      break;

    case AUDF0:
      Update_tia_sound(0x17, value);
      break;
    
    case AUDF1:
      Update_tia_sound(0x18, value);
      break;

    case AUDV0:
      Update_tia_sound(0x19, value);
      break;

    case AUDV1:
      Update_tia_sound(0x1A, value);
      break;

    default:
      break;
  }
}

