//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  This class provides a standard API for the screen.  This file contains
  the Linux SVGALIB implementation of the API.

  @author  Bradford W. Mott
  @version $Id: TermSVGA.cxx,v 1.2 1997/05/17 19:00:09 bwmott Exp $
*/

#include <assert.h>
#include <stdio.h>
#include <iostream.h>

#include "colors.h"
#include "Term.hxx"
#include "TIA.hxx"

// Indicates if an instance is alive
bool Terminal::ourInstanceCreated = false;

// Contains the state of each of the events based on the keyboard
static volatile uLong theKeyboardEventState[TERMINAL_NUMBER_OF_EVENTS];

// Contains the state of each of the events based on the joystick
static volatile uLong theJoystickEventState[TERMINAL_NUMBER_OF_EVENTS];

static uWord thePixelDataTable[256];

// Indicates the width and height based on the property information
static uLong theWidth;
static uLong theHeight;
static uLong theXStart;

#ifdef LINUX_JOYSTICK
  #include <unistd.h>
  #include <fcntl.h>
  #include <linux/joystick.h>

  // File descriptors for the joystick devices
  int theLeftJoystickFd;
  int theRightJoystickFd;
#endif

#include <termios.h>
#include <vga.h>
#include <vgamouse.h>
#include <vgakeyboard.h>

// Define scancodes for 1,2,3 and 4 if they aren't defined
#ifndef SCANCODE_1
  #define SCANCODE_1                      2
  #define SCANCODE_2                      3
  #define SCANCODE_3                      4
  #define SCANCODE_4                      5
#endif

// Buffer for the terminal settings
struct termios theOldTerminalSettings;

//============================================================================
// Called whenever a key is pressed or released
//============================================================================
static void HandleKeyEvent(int scanCode, int press)
{
  struct Switches
  {
    int scanCode;
    Terminal::Event eventCode;
  };

  static Switches list[] = {
    { SCANCODE_CURSORBLOCKDOWN,   Terminal::LeftJoystickDown },
    { SCANCODE_CURSORBLOCKUP,     Terminal::LeftJoystickUp },
    { SCANCODE_CURSORBLOCKLEFT,   Terminal::LeftJoystickLeft },
    { SCANCODE_CURSORBLOCKRIGHT,  Terminal::LeftJoystickRight },
    { SCANCODE_SPACE,             Terminal::LeftJoystickFire }, 
    { SCANCODE_ENTER,             Terminal::LeftJoystickFire }, 
    { SCANCODE_H,                 Terminal::RightJoystickDown },
    { SCANCODE_Y,                 Terminal::RightJoystickUp },
    { SCANCODE_G,                 Terminal::RightJoystickLeft },
    { SCANCODE_J,                 Terminal::RightJoystickRight },
    { SCANCODE_Z,                 Terminal::RightJoystickFire }, 
    { SCANCODE_C,                 Terminal::Color },
    { SCANCODE_B,                 Terminal::BlackAndWhite },
    { SCANCODE_1,                 Terminal::LeftDifficultyA },
    { SCANCODE_2,                 Terminal::LeftDifficultyB },
    { SCANCODE_3,                 Terminal::RightDifficultyA },
    { SCANCODE_4,                 Terminal::RightDifficultyB },
    { SCANCODE_S,                 Terminal::Select },
    { SCANCODE_R,                 Terminal::Reset },
    { SCANCODE_P,                 Terminal::Pause },
    { SCANCODE_Q,                 Terminal::Quit }
  };

  for(unsigned int i = 0; i < sizeof(list) / sizeof(Switches); ++i)
  {
    if(list[i].scanCode == scanCode)
    {
      theKeyboardEventState[(int)list[i].eventCode] = 
          (press == KEY_EVENTPRESS) ? 1 : 0;
    }
  }
}

//============================================================================
// Constructor
//============================================================================
Terminal::Terminal(const Properties& properties)
    : myProperties(&properties)
{
  assert(!ourInstanceCreated);
  ourInstanceCreated = true;

  // Clear the terminal event state
  for(uLong i = 0; i < TERMINAL_NUMBER_OF_EVENTS; i++) 
  {
    theKeyboardEventState[i] = 0;
    theJoystickEventState[i] = 0;
  }

  // Setup the pixel data table
  for(uLong j = 0; j < 256; ++j)
  {
    thePixelDataTable[j] = j | (j << 8);
  }

  // Enable automatic mouse setup
  vga_setmousesupport(1);

  // Init the SVGALIB library
  vga_init();
  vga_setmode(G320x200x256);
  vga_waitretrace();

  for(int index = 0; index < 256; ++index)
  {
    vga_setpalette(index, (theColorTable[index] & 0x00ff0000) >> 18,
                          (theColorTable[index] & 0x0000ff00) >> 10,
                          (theColorTable[index] & 0x000000ff) >> 2);
  }

  // Set the mouse range from 0 to 255
  mouse_setxrange(0, 512);

  tcgetattr(0, &theOldTerminalSettings);
  keyboard_init();
  keyboard_seteventhandler(HandleKeyEvent);

#ifdef LINUX_JOYSTICK
  // Open the joystick devices 
  theLeftJoystickFd = open("/dev/js0", O_RDONLY);
  theRightJoystickFd = open("/dev/js1", O_RDONLY);
#endif

  // Get the desired width and height of the display
  theWidth = myProperties->integer("Display.Width");
  theXStart = myProperties->integer("Display.XStart");
  theHeight = myProperties->integer("Display.Height");
}

//============================================================================
// Destructor
//============================================================================
Terminal::~Terminal()
{
  keyboard_close();

  // Restore text mode
  tcsetattr(0, 0, &theOldTerminalSettings);
  vga_setmode(TEXT);

#ifdef LINUX_JOYSTICK
  if(theLeftJoystickFd >= 0)
    close(theLeftJoystickFd);

  if(theRightJoystickFd >= 0)
    close(theRightJoystickFd);
#endif

  ourInstanceCreated = false;
}

//============================================================================
// Update the screen with the TIA frame
//============================================================================
void Terminal::update(TIA& tia)
{
  uByte* currentFrame = tia.currentFrameBuffer();
  uByte* previousFrame = tia.previousFrameBuffer();

  uLong width = theWidth / 4;
  uLong height = (theHeight > 200) ? 200 : theHeight;

  int offset = ((200 - height) / 2) * 320;
  uWord* data = (uWord*)(vga_getgraphmem() + offset) + ((160 - theWidth) / 2);

  for(uLong y = 0; y < height; ++y)
  {
    uLong* current = (uLong*)(currentFrame + theXStart);
    uLong* previous = (uLong*)(previousFrame + theXStart);
    uWord* screen = data;
    for(uLong x = 0; x < width; ++x, ++current, ++previous)
    {
      if(*current != *previous)
      {
        uByte* frame = (uByte*)current;
        *(screen++) = thePixelDataTable[*frame++];
        *(screen++) = thePixelDataTable[*frame++];
        *(screen++) = thePixelDataTable[*frame++];
        *(screen++) = thePixelDataTable[*frame];
      }
      else
      {
        screen += 4;
      }
    }
    currentFrame += 160;
    previousFrame += 160;
    data += 160;
  }

  // Handle any events that have occured
  handleEvents();
}

//============================================================================
// Answer the state of the given event
//============================================================================
uLong Terminal::eventState(Event event)
{
  // If the value is not set by joystick then return the keyboard's value
  if(theJoystickEventState[(int)event] == 0)
    return theKeyboardEventState[(int)event];
  else
    return theJoystickEventState[(int)event];
}

//============================================================================
// Process any events related to the terminal
//============================================================================
void Terminal::handleEvents()
{
  // Update events based on any keys that have been pressed or released
  keyboard_update();

  // Handle any mouse events
  mouse_update();

  theJoystickEventState[(int)Terminal::PaddleZeroPosition] = 512 - mouse_getx();
  theJoystickEventState[(int)Terminal::PaddleZeroFire] = 
      mouse_getbutton() & MOUSE_LEFTBUTTON;
  
#ifdef LINUX_JOYSTICK
  // Read joystick values and modify event states
  if(theLeftJoystickFd >= 0)
  {
    struct JS_DATA_TYPE joystick;
    read(theLeftJoystickFd, &joystick, JS_RETURN);

    theJoystickEventState[(int)Terminal::LeftJoystickFire] = 
        (joystick.buttons & 0x01) ? 1 : 0;
    theJoystickEventState[(int)Terminal::LeftJoystickLeft] = 
        (joystick.x < 250) ? 1 : 0;
    theJoystickEventState[(int)Terminal::LeftJoystickRight] = 
        (joystick.x > 750) ? 1 : 0;
    theJoystickEventState[(int)Terminal::LeftJoystickUp] = 
        (joystick.y < 250) ? 1 : 0;
    theJoystickEventState[(int)Terminal::LeftJoystickDown] = 
        (joystick.y > 750) ? 1 : 0;
  }

  if(theRightJoystickFd >= 0)
  {
    struct JS_DATA_TYPE joystick;
    read(theRightJoystickFd, &joystick, JS_RETURN);

    theJoystickEventState[(int)Terminal::RightJoystickFire] = 
        (joystick.buttons & 0x01) ? 1 : 0;
    theJoystickEventState[(int)Terminal::RightJoystickLeft] = 
        (joystick.x < 250) ? 1 : 0;
    theJoystickEventState[(int)Terminal::RightJoystickRight] = 
        (joystick.x > 750) ? 1 : 0;
    theJoystickEventState[(int)Terminal::RightJoystickUp] = 
        (joystick.y < 250) ? 1 : 0;
    theJoystickEventState[(int)Terminal::RightJoystickDown] = 
        (joystick.y > 750) ? 1 : 0;
  }
#endif
}

