/********************************************************************************
 * Copyright (c) Des Herriott 1993, 1994
 *               Erik Kunze   1995 - 1998
 *
 * Permission to use, distribute, and sell this software and its documentation
 * for any purpose is hereby granted without fee, provided that the above
 * copyright notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that the name
 * of the copyright holder not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.  The
 * copyright holder makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or implied
 * warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
 *
 * THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Des Herriott
 *         Erik Kunze
 *
 * changed by EKU
 *******************************************************************************/
#ifndef lint
static char rcsid[] = "$Id: resource.c,v 4.3 1998/01/06 21:21:30 erik Rel $";
#endif

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/param.h>
#ifdef NEED_UNAME
#include <sys/utsname.h>
#endif
#include "debug.h"
#include "mem.h"
#include "util.h"
#include "main.h"
#include "resource.h"

#define LOCAL_SERVER	0
#define REMOTE_SERVER	1
#define MAX_CHARS		80
#define MAX_DISP_LEN	256
#define MAX_STRING_LEN	1024

#define NELEM(a)		(sizeof(a) / sizeof(a[0]))
#define ARG(x)			x, sizeof(x)

enum {
	KM_JOY_UP,
	KM_JOY_DOWN,
	KM_JOY_LEFT,
	KM_JOY_RIGHT,
	KM_JOY_FIRE,
#ifdef SLOWDOWN
	KM_KEY_FASTER,
	KM_KEY_SLOWER,
#endif
	NUM_KEYMAPPINGS
};

static int getResource(XrmDatabase, char *, char *, char *, unsigned);
static int getString(XrmDatabase, char *, char *, char **);
static int getBool(XrmDatabase, char *, char *, int *);
static int getInteger(XrmDatabase, char *, int, int *);
static void getFileDefaults(Display *, XrmDatabase *);
#ifdef MITSHM
static int isLocalServer(char *);
#endif
static void copyright(void);
static void usage(void);
static void version(void);
#ifdef DEBUG
static void printResources(void);
#endif

Config Conf;


static XrmOptionDescRec options[] = {

	{ "-help",      "*help",          XrmoptionNoArg,  "True" },
	{ "-version",   "*version",       XrmoptionNoArg,  "True" },
	{ "-libDir",    "*libDir",        XrmoptionSepArg, NULL   },
	{ "-quiet",     "*quiet",         XrmoptionSepArg, NULL   },

	{ "-display",   "*display",       XrmoptionSepArg, NULL   },
	{ "-geometry",  "*geometry",      XrmoptionSepArg, NULL   },
#ifndef EXCLUDE_PIXMAP
	{ "-pixmap",    "*pixmap",        XrmoptionSepArg, NULL   },
#endif
	{ "-mono",      "*monocrome",     XrmoptionSepArg, NULL   },
	{ "-private",   "*private",       XrmoptionSepArg, NULL   },
#ifdef MITSHM
	{ "-mitshm",    "*mitshm",        XrmoptionSepArg, NULL   },
#endif

#ifdef DEBUG
	{ "-debug",     "*debug",         XrmoptionSepArg, NULL   },
#endif
	{ "-machine",   "*machine",       XrmoptionSepArg, NULL   },
	{ "-ms",        "*ms",            XrmoptionSepArg, NULL   },
	{ "-fast",      "*fast",          XrmoptionSepArg, NULL   },
#ifdef SLOWDOWN
	{ "-slowdown",  "*slowdown",      XrmoptionSepArg, NULL   },
	{ "-keyFaster", "*keyFaster",     XrmoptionSepArg, NULL   },
	{ "-keySlower", "*keySlower",     XrmoptionSepArg, NULL   },
#endif

	{ "-scale",     "*scale",         XrmoptionSepArg, NULL   },
	{ "-flashing",  "*flashing",      XrmoptionSepArg, NULL   },
	{ "-border",    "*border",        XrmoptionSepArg, NULL   },
	{ "-rr",        "*rr",            XrmoptionSepArg, NULL   },
#ifdef MITSHM
	{ "-rrShm",     "*rrShm",         XrmoptionSepArg, NULL   },
#endif
	{ "-rrNoshm",   "*rrNoshm",       XrmoptionSepArg, NULL   },

#if defined(SUN_AUDIO) || defined(PCSPKR_AUDIO)
	{ "-noise",     "*noise",         XrmoptionSepArg, NULL   },
#endif

	{ "-joyUp",     "*joyUp",         XrmoptionSepArg, NULL   },
	{ "-joyDown",   "*joyDown",       XrmoptionSepArg, NULL   },
	{ "-joyLeft",   "*joyLeft",       XrmoptionSepArg, NULL   },
	{ "-joyRight",  "*joyRight",      XrmoptionSepArg, NULL   },
	{ "-joyFire",   "*joyFire",       XrmoptionSepArg, NULL   },
#ifdef JOY
	{ "-joystick",  "*joystick",      XrmoptionSepArg, NULL   },
	{ "-joyDev",    "*joyDev",        XrmoptionSepArg, NULL   },
	{ "-tolerance", "*tolerance",     XrmoptionSepArg, NULL   },
#endif

#ifdef AUDIO
	{ "-sound",     "*sound",         XrmoptionSepArg, NULL   },
#if defined(SUN_AUDIO) || defined(AYCHIP_AUDIO)
	{ "-audioDev",  "*audioDev",      XrmoptionSepArg, NULL   },
#endif
#if defined(SUN_AUDIO) || defined(XBELL_AUDIO)
	{ "-volume",    "*volume",        XrmoptionSepArg, NULL   },
#endif
#endif

	{ "-rom48",     "*rom48",         XrmoptionSepArg, NULL   },
	{ "-issue",     "*issue",         XrmoptionSepArg, NULL   },
	{ "-kbdlayout", "*kbdlayout",     XrmoptionSepArg, NULL   },

	{ "-rom1280",   "*rom1280",       XrmoptionSepArg, NULL   },
	{ "-rom1281",   "*rom1281",       XrmoptionSepArg, NULL   },

#ifdef XZX_PLUS3
	{ "-rompl30",   "*rompl30",       XrmoptionSepArg, NULL   },
	{ "-rompl31",   "*rompl31",       XrmoptionSepArg, NULL   },
	{ "-rompl32",   "*rompl32",       XrmoptionSepArg, NULL   },
	{ "-rompl33",   "*rompl33",       XrmoptionSepArg, NULL   },
	{ "-fda",       "*fda",           XrmoptionSepArg, NULL   },
	{ "-fdb",       "*fdb",           XrmoptionSepArg, NULL   },
#endif

#ifdef XZX_IF1
	{ "-if1",       "*if1",           XrmoptionSepArg, NULL   },
	{ "-if1Rom",    "*if1Rom",        XrmoptionSepArg, NULL   },
	{ "-crlf",      "*crlf",          XrmoptionSepArg, NULL   },
	{ "-strcr",     "*strcr",         XrmoptionSepArg, NULL   },
	{ "-m1",        "*m1",	          XrmoptionSepArg, NULL   },
#if IF1_DRIVES > 1
	{ "-m2",        "*m2",	          XrmoptionSepArg, NULL   },
#endif
#if IF1_DRIVES > 2
	{ "-m3",        "*m3",	          XrmoptionSepArg, NULL   },
#endif
#if IF1_DRIVES > 3
	{ "-m4",        "*m4",	          XrmoptionSepArg, NULL   },
#endif
#if IF1_DRIVES > 4
	{ "-m5",        "*m5",	          XrmoptionSepArg, NULL   },
#endif
#if IF1_DRIVES > 5
	{ "-m6",        "*m6",	          XrmoptionSepArg, NULL   },
#endif
#if IF1_DRIVES > 6
	{ "-m7",        "*m7",	          XrmoptionSepArg, NULL   },
#endif
#if IF1_DRIVES > 7
	{ "-m8",        "*m8",	          XrmoptionSepArg, NULL   },
#endif
#endif

#ifdef XZX_MF128
	{ "-mf128",     "*mf128",         XrmoptionSepArg, NULL   },
	{ "-mf128Rom",  "*mf128Rom",      XrmoptionSepArg, NULL   },
#endif
};

static struct {
	char *res_name;
	char *def_keysym;
} mapTab[NUM_KEYMAPPINGS] = {
	{ "joyUp",     "q"           },
	{ "joyDown",   "a"           },
	{ "joyLeft",   "o"           },
	{ "joyRight",  "p"           },
	{ "joyFire",   "space"       },
#ifdef SLOWDOWN
	{ "keySlower", "KP_Add"      },
	{ "keyFaster", "KP_Subtract" },
#endif
};
static char *myName  = "xzx";
static char *myClass = "Xzx";

static int
getResource(XrmDatabase db, char *resource, char *fallBack, char *result,
			unsigned size)
{
	int	 i, len;
	char strName[80], strClass[80], *strType[10];
	XrmValue rmValue;
	(void)sprintf(strName, "%s.%s", myName, resource);
	(void)sprintf(strClass, "%s.%c%s",
				  myClass, isupper(*resource) ?
						   toupper(*resource) : *resource, resource + 1);
	if (XrmGetResource(db, strName, strClass, strType, &rmValue) == True)
	{
		if (!rmValue.addr)
		{
			len = 0;
		}
		else
		{
			len = rmValue.size < size - 1 ? rmValue.size : size - 1;
			(void)strncpy(result, rmValue.addr, len);
		}
		result[len] = '\0';
		for (i = 0; i < NELEM(options); i++)
		{
			if (options[i].argKind == XrmoptionIsArg
				&& (!strcmp(result, options[i].option)
					|| !strcmp(result, options[i].specifier)))
			{
				(void)strncpy(result, "true", size);
				result[size - 1] = '\0';
				break;
			}
		}
		return 1;
	}
	if (fallBack)
	{
		(void)strncpy(result, fallBack, size - 1);
		result[size - 1] = '\0';
	}
	else
	{
		result[0] = '\0';
	}
	return 0;
}

static int
getString(XrmDatabase db, char *resource, char *fallBack, char **result)
{
	char *src, *dst, buf[MAX_STRING_LEN];
	int inDB;
	inDB = getResource(db, resource, fallBack, buf, sizeof(buf));
	src = dst = buf;
	while ((*src & 0x7f) == *src && !isgraph(*src) && *src)
	{
		src++;
	}
	while ((*src & 0x7f) != *src || isgraph(*src))
	{
		*dst++ = *src++;
	}
	*dst = '\0';
	*result = Strdup(buf, "getString");
	return inDB;
}

static int
getBool(XrmDatabase db, char *resource, char *fallBack, int *result)
{
	char resValue[MAX_CHARS];
	int inDB;
	inDB = getResource(db, resource, fallBack, ARG(resValue));
	*result = (!strncasecmp(resValue, "true", 4)
			   || !strncasecmp(resValue, "on", 2)
			   || !strncasecmp(resValue, "yes", 3));
	return inDB;
}

static int
getInteger(XrmDatabase db, char *resource, int fallBack, int *result)
{
    char resValue[MAX_CHARS];
	int inDB;
	if (!(inDB = getResource(db, resource, NULL, ARG(resValue)))
		|| sscanf(resValue, "%d", result) <= 0)
		*result = fallBack;
	return inDB;
}

static void
getFileDefaults(Display *dpy, XrmDatabase *rDBptr)
{
	int len;
	char *ptr;
	char path[MAXPATHLEN];
	XrmDatabase	tmpDB;
#ifdef NEED_UNAME
	struct utsname un;
#endif
	char *lang = getenv("LANG");
	char *home = getenv("HOME");
	(void)sprintf(path, "%s%s", LIBDIR, myClass);
	*rDBptr = XrmGetFileDatabase(path);
#ifdef VMS
	tmpDB = XrmGetFileDatabase("SYS$LOGIN:decw$xdefaults.dat");
	XrmMergeDatabases(tmpDB, rDBptr);
	tmpDB = XrmGetFileDatabase("DECW$USER_DEFAULTS:xzx.dat");
	XrmMergeDatabases(tmpDB, rDBptr);
#else
	if (lang)
	{
		(void)sprintf(path, "%s/%s/app-defaults/%s", USRLIBDIR, lang, myClass);
		if (access(path, F_OK) == -1)
		{
			(void)sprintf(path, "%s/app-defaults/%s", USRLIBDIR, myClass);
		}
	}
	else
	{
		(void)sprintf(path, "%s/app-defaults/%s", USRLIBDIR, myClass);
	}
	tmpDB = XrmGetFileDatabase(path);
	XrmMergeDatabases(tmpDB, rDBptr);
	if ((ptr = getenv("XUSERFILESEARCHPATH")))
	{
		(void)sprintf(path, "%s/%s", ptr, myClass);
		tmpDB = XrmGetFileDatabase(path);
		XrmMergeDatabases(tmpDB, rDBptr);
	}
	else if ((ptr = getenv("XAPPLRESDIR")))
	{
		if (lang)
		{
			(void)sprintf(path, "%s/%s/%s", ptr, lang, myClass);
			if (access(path, F_OK) == -1)
			{
				(void)sprintf(path, "%s/%s", ptr, myClass);
			}
		}
		else
		{
			(void)sprintf(path, "%s/%s", ptr, myClass);
		}
		tmpDB = XrmGetFileDatabase(path);
		XrmMergeDatabases(tmpDB, rDBptr);
	}
	else if (home)
	{
		if (lang)
		{
			(void)sprintf(path, "%s/app-defaults/%s/%s", home, lang, myClass);
			if (access(path, F_OK) == -1)
			{
				(void)sprintf(path, "%s/app-defaults/%s", home, myClass);
			}
		}
		else
		{
			(void)sprintf(path, "%s/app-defaults/%s", home, myClass);
		}
		tmpDB = XrmGetFileDatabase(path);
		XrmMergeDatabases(tmpDB, rDBptr);
	}
	if ((ptr = XResourceManagerString(dpy)))
	{
		tmpDB = XrmGetStringDatabase(ptr);
		XrmMergeDatabases(tmpDB, rDBptr);
	}
	else if (home)
	{
		(void)sprintf(path, "%s/.Xdefaults", home);
		tmpDB = XrmGetFileDatabase(path);
		XrmMergeDatabases(tmpDB, rDBptr);
	}
	if ((ptr = getenv("XENVIRONMENT")))
	{
		tmpDB = XrmGetFileDatabase(ptr);
		XrmMergeDatabases(tmpDB, rDBptr);
	}
	else if (home)
	{
		(void)sprintf(path, "%s/.Xdefaults-", home);
		len = strlen(path);
#ifdef NEED_UNAME
		(void)uname(&un);
		(void)strncpy(&path[len], un.sysname, sizeof(path - len));
#else
		(void)gethostname(&path[len], sizeof path - len);
		path[sizeof path - 1] = '\0';
#endif
		tmpDB = XrmGetFileDatabase(path);
		XrmMergeDatabases(tmpDB, rDBptr);
	}
#endif
}

Display *
GetResources(int *argcp, char **argvp)
{
	int	 i;
	char *ptr;
	char resValue[MAX_CHARS];
	char displayName[MAX_DISP_LEN];
	XrmDatabase argDB, rDB;
	Display *display;
	XrmInitialize();
	argDB = 0;
	XrmParseCommand(&argDB, options, NELEM(options), myName, argcp, argvp);

	for (i = 1; i < *argcp; i++)
	{
		if (argvp[i][0] == '-')
		{
			(void)fprintf(stderr, "Unknown option '%s'\n", argvp[i]);
			(void)fprintf(stderr, "Type: %s -help to see a list of options\n",
						  ProgName);
			Quit(1);
		}
	}
	if (getResource(argDB, "help", NULL, ARG(resValue)))
	{
		usage();
		Quit(1);
	}
	copyright();
	if (getResource(argDB, "version", NULL, ARG(resValue)))
	{
		version();
		Quit(1);
	}
	if (!getString(argDB, "display", NULL, &ptr) || !*ptr)
	{
		free(ptr);
#ifdef VMS
		if ((ptr = getenv("DECW$DISPLAY")))
#else
		if ((ptr = getenv("DISPLAY")))
#endif
		{
			(void)strncpy(displayName, ptr, MAX_DISP_LEN);
			displayName[MAX_DISP_LEN - 1] = '\0';
		}
		else
		{
#ifdef VMS
			(void)strcpy(displayName, "::0.0");
			(void)sprintf(displayName, "%s::0.0", host);
#else
			(void)strcpy(displayName, ":0.0");
#endif
		}
		display = XOpenDisplay(displayName);
	}
	else
	{
		display = XOpenDisplay(ptr);
		free(ptr);
	}
	if (!display)
	{
		Msg(M_PERR, "couldn't open display <%s>", displayName);
		return NULL;
	}
	getFileDefaults(display, &rDB);
	XrmMergeDatabases(argDB, &rDB);

#ifdef DEBUG
	(void)getInteger(rDB, "debug", 0, &GETCFG(debug));
#endif

	ptr = (char *)Malloc(strlen(LIBDIR) + 3, "GetResources");
	(void)sprintf(ptr, ".:%s", LIBDIR);
	(void)getString(rDB, "libDir", ptr, &GETCFG(libDir));
	free(ptr);
	(void)getBool(rDB, "quiet", "false", &GETCFG(quiet));

	(void)getString(rDB, "geometry", "", &GETCFG(geometry));
#ifndef EXCLUDE_PIXMAP
	(void)getBool(rDB, "pixmap", "false", &GETCFG(pixmaps));
#endif
	(void)getBool(rDB, "monocrome", "false", &GETCFG(mono));
	(void)getBool(rDB, "private", "false", &GETCFG(private));
#ifdef MITSHM
	(void)getBool(rDB, "mitshm", "true", &GETCFG(mitshm));
#ifndef EXCLUDE_PIXMAP
	if (GETCFG(pixmaps))
	{
		SETCFG(mitshm, 0);
	}
#endif

	if (GETCFG(mitshm) && isLocalServer(displayName) == REMOTE_SERVER)
	{
		Msg(M_INFO, "MIT-SHM not possible on remote X server, disabling");
		SETCFG(mitshm, 0);
	}
#endif

	(void)getInteger(rDB, "machine", 48, &GETCFG(machine));
	switch (GETCFG(machine))
	{
		case 48:
			SETCFG(machine, SP_48_3);
			break;
		case 128:
			SETCFG(machine, SP_128);
			break;
#ifdef XZX_PLUS3
		case 3:
			SETCFG(machine, SP_3);
			break;
#endif
		default:
			Msg(M_WARN, "invalid emulation mode %d, using 48", GETCFG(machine));
			SETCFG(machine, SP_48_3);
			break;
	}
	(void)getInteger(rDB, "ms", 20, &GETCFG(ms));
	(void)getBool(rDB, "fast", "false", &GETCFG(fast));
	if (GETCFG(ms) <= 0)
	{
		Msg(M_WARN, "timer value out of range, setting to default 20");
		SETCFG(ms, 20);
	}
#ifdef SLOWDOWN
	(void)getInteger(rDB, "slowdown", 0, &GETCFG(slowdown));

	for (i = KM_KEY_FASTER; i <= KM_KEY_SLOWER; i++)
	{
		(void)getString(rDB, mapTab[i].res_name, mapTab[i].def_keysym, &ptr);
		SETCFG(slowdownKeys[i - KM_KEY_FASTER], XStringToKeysym(ptr));
		if (GETCFG(slowdownKeys[i - KM_KEY_FASTER]) == NoSymbol)
		{
			Msg(M_WARN, "unknown keysym <%s> found, using <%s>", ptr,
				mapTab[i].def_keysym);
			SETCFG(slowdownKeys[i - KM_KEY_FASTER],
				   XStringToKeysym(mapTab[i].def_keysym));
		}
		free(ptr);
	}
#endif

	(void)getInteger(rDB, "scale", 1, &GETCFG(scale));
	if (GETCFG(scale) < 1 || GETCFG(scale) > 3)
	{
		Msg(M_WARN, "scale factor out of range, setting to default 1");
		SETCFG(scale, 1);
	}
	(void)getBool(rDB, "flashing", "true", &GETCFG(flashing));
	(void)getBool(rDB, "border", "true", &GETCFG(border));
#ifdef MITSHM
	(void)getInteger(rDB, "rrShm", 3, &GETCFG(rrShm));
#endif
	(void)getInteger(rDB, "rrNoshm", 6, &GETCFG(rrNoShm));
	if (getInteger(rDB, "rr", 0, &i))
	{
#ifdef MITSHM
		SETCFG(rrShm, i);
#endif
		SETCFG(rrNoShm, i);
	}

#if defined(SUN_AUDIO) || defined(PCSPKR_AUDIO)
	(void)getBool(rDB, "noise", "false", &GETCFG(noise));
#endif

#ifdef JOY
	(void)getBool(rDB, "joystick", "false", &GETCFG(useJoy));
	(void)getString(rDB, "joyDev", "/dev/js0", &GETCFG(joyDevice));
	(void)getInteger(rDB, "tolerance", 20, &GETCFG(joyTolerance));
#endif

	for (i = KM_JOY_UP; i <= KM_JOY_FIRE; i++)
	{
		(void)getString(rDB, mapTab[i].res_name, mapTab[i].def_keysym, &ptr);
		SETCFG(joyKeys[i], XStringToKeysym(ptr));
		if (GETCFG(joyKeys[i]) == NoSymbol)
		{
			Msg(M_WARN, "unknown keysym <%s> found, using <%s>", ptr,
				mapTab[i].def_keysym);
			SETCFG(joyKeys[i], XStringToKeysym(mapTab[i].def_keysym));
		}
		free(ptr);
	}

#ifdef AUDIO
	(void)getBool(rDB, "sound", "false", &GETCFG(useSound));
#if defined(SUN_AUDIO) || defined(AYCHIP_AUDIO)
#ifdef __linux__
	(void)getString(rDB, "audioDev", "/dev/dsp", &GETCFG(audioDev));
#else
	(void)getString(rDB, "audioDev", "/dev/audio", &GETCFG(audioDev));
#endif
#endif
#if defined(SUN_AUDIO) || defined(XBELL_AUDIO)
	(void)getInteger(rDB, "volume", 50, &GETCFG(volume));
	if (GETCFG(volume) > 100 || GETCFG(volume) < 0)
	{
		Msg(M_WARN, "audio volume out of range, setting to default 50");
		SETCFG(volume, 50);
	}
#endif
#endif

	(void)getString(rDB, "rom48", "spectrum.rom", &GETCFG(rom48));
	(void)getInteger(rDB, "issue", 3, &GETCFG(issue));
	if (GETCFG(issue) != 2 && GETCFG(issue) != 3)
	{
		Msg(M_WARN, "invalid issue type %d, using 3", GETCFG(issue));
		SETCFG(issue, 3);
	}
	if (GETCFG(issue) == 2 && GETCFG(machine) == SP_48_3)
	{
		SETCFG(machine, SP_48_2);
	}
	(void)getString(rDB, "kbdlayout", "keyboard.scr", &GETCFG(kbdlayout));

	(void)getString(rDB, "rom1280", "128-0.rom", &GETCFG(rom128_0));
	(void)getString(rDB, "rom1281", "128-1.rom", &GETCFG(rom128_1));

#ifdef XZX_PLUS3
	(void)getString(rDB, "rompl30", "pl3-0.rom", &GETCFG(rompl3_0));
	(void)getString(rDB, "rompl31", "pl3-1.rom", &GETCFG(rompl3_1));
	(void)getString(rDB, "rompl32", "pl3-2.rom", &GETCFG(rompl3_2));
	(void)getString(rDB, "rompl33", "pl3-3.rom", &GETCFG(rompl3_3));
	(void)getString(rDB, "fda", "diska.dsk", &GETCFG(disks[0]));
	(void)getString(rDB, "fdb", "diskb.dsk", &GETCFG(disks[1]));
#endif

#ifdef XZX_IF1
	(void)getBool(rDB, "if1", "true", &GETCFG(if1_active));
	(void)getBool(rDB, "crlf", "false", &GETCFG(translate_nl));
	(void)getBool(rDB, "strcr", "false", &GETCFG(strip_nl));
	(void)getString(rDB, "if1Rom", "if1.rom", &GETCFG(if1_rom));
	for (i = 0; i < IF1_DRIVES; i++)
	{
		char resname[3]; char resvalue[5];
		(void)sprintf(resname, "m%d", i + 1);
		(void)sprintf(resvalue, "cart%d.mdr", i + 1);
		(void)getString(rDB, resname, resvalue, &GETCFG(cartfiles[i]));
	}
#ifdef XZX_PLUS3
	if (GETCFG(if1_active) && GETCFG(machine) == SP_3)
	{
		Msg(M_INFO, "Interface I not supported by the +3, disabling");
		SETCFG(if1_active, 0);
	}
#endif
#endif

#ifdef XZX_MF128
	(void)getBool(rDB, "mf128", "false", &GETCFG(mf128_active));
	(void)getString(rDB, "mf128Rom", "mf128.rom", &GETCFG(mf128_rom));
#ifdef XZX_PLUS3
	if (GETCFG(mf128_active) && GETCFG(machine) == SP_3)
	{
		Msg(M_INFO, "Multiface 128 not supported by the +3, disabling");
		SETCFG(mf128_active, 0);
	}
#endif
#endif
#ifdef DEBUG
	if (GETCFG(debug) & D_X11)
	{
		printResources();
	}
#endif
	XrmDestroyDatabase(rDB);
	return display;
}

#ifdef MITSHM
static int
isLocalServer(char *displayName)
{
#ifdef NEED_UNAME
	struct utsname un;
#else
	char sysname[MAX_DISP_LEN];
#endif
#ifdef DEBUG
	if (GETCFG(debug) & D_X11)
	{
		Msg(M_DEBUG, "isLocalServer : %s", displayName);
	}
#endif
	if (displayName[0] == ':'
		|| !strncmp(displayName, "unix", 4)
		|| !strncmp(displayName, "localhost", 9))
	{
		return LOCAL_SERVER;
	}

#ifdef NEED_UNAME
	(void)uname(&un);
	if (!strncmp(displayName, un.sysname, strlen(un.sysname))
		|| !strncmp(displayName, un.nodename, strlen(un.nodename)))
	{
		return LOCAL_SERVER;
	}
#else
	(void)gethostname(sysname, MAX_DISP_LEN);
	if (!strncmp(displayName, sysname, strlen(sysname)))
	{
		return LOCAL_SERVER;
	}
#endif
	return REMOTE_SERVER;
}
#endif

static void
copyright(void)
{
	(void)fprintf(stderr,
	"XZX is copyright (c) 1993,1994 Des Herriott\n"
	"                 (c) 1995-1998 Erik Kunze\n"
	"XZX comes with no warranty; see the file COPYRIGHT for details.\n\n"
	"If you use XZX regulary please send a postcard to:\n"
	"    Erik Kunze, Nebelhornstrasse 30, D-82223 Eichenau, Germany\n\n");
}

static void
usage(void)
{
	(void)fprintf(stderr, "Usage: %s [options] [snapshot file]\n", ProgName);
	(void)fprintf(stderr, "Available options:\n"

	"  -help                 display this message\n"
	"  -version              display version number and compile-time options\n"
	"  -libDir <path>        specify library search path\n"
	"  -quiet <bool>         suppress all non-error messages\n"

	"  -display <display>    specify X display\n"
	"  -geometry <geospec>   specify window position\n"
#ifndef EXCLUDE_PIXMAP
	"  -pixmap <bool>        to use pixmaps\n"
#endif
	"  -mono <bool>          use only black and white\n"
	"  -private <bool>       use a private colormap\n"
#ifdef MITSHM
	"  -mitshm <bool>        enable MIT-SHM\n"
#endif

#ifdef DEBUG
	"  -debug <n>            specify the debug level\n"
#endif
	"  -machine [48|128|3]   emulate the specified Spectrum type\n"
	"  -ms <n>               specify number of milliseconds per frame\n"
	"  -fast <bool>          do not run at real Spectrum speed\n"
#ifdef SLOWDOWN
	"  -slowdown <int>       specify slowdown factor\n"
	"  -keyFaster <keysym>   speed up the emulator\n"
	"  -keySlower <keysym>   slow down the emulator\n"
#endif

	"  -scale <n>            Spectrum to X11 screen scaling factor\n"
	"  -flashing <bool>      enable flashing attributes\n"
	"  -border <bool>        enable/disable border emulation\n"
	"  -rr <n>               refresh screen every <n> frames\n"
#ifdef MITSHM
	"  -rrShm <n>            refresh screen every <n> frames (MITSHM)\n"
#endif
	"  -rrNoshm <n>          refresh screen every <n> frames (no MITSHM)\n"

#if defined(SUN_AUDIO) || defined(PCSPKR_AUDIO)
	"  -noise <bool>         produce noise while loading from tape\n"
#endif

	"  -joyUp <keysym>       define key for joystick up\n"
	"  -joyDown <keysym>     define key for joystick down\n"
	"  -joyLeft <keysym>     define key for joystick left\n"
	"  -joyRight <keysym>    define key for joystick right\n"
	"  -joyFire <keysym>     define key for joystick fire\n"
#ifdef JOY
	"  -joystick <bool>      enable/disable Kempston joystick\n"
	"  -joyDev <path>        specify joystick device\n"
	"  -tolerance <int>      specify joystick displacement tolerance\n"
#endif

#ifdef AUDIO
	"  -sound <bool>         enable/disable sound\n"
#if defined(SUN_AUDIO) || defined(AYCHIP_AUDIO)
	"  -audioDev <path>      specify the audio device file name\n"
#endif
#if defined(SUN_AUDIO) || defined(XBELL_AUDIO)
	"  -volume <n>           specify audio volume as a percentage\n"
#endif
#endif

	"  -rom48 <fname>        specify the file to use as 48K ROM image\n"
	"  -issue [2|3]          emulate specified issue on 48K Spectrums\n"
	"  -kbdlayout <fname>    specify the file with keybord layout to use\n"

	"  -rom128{0,1} <fname>  use the specified file as 128K ROM image\n"

#ifdef XZX_PLUS3
	"  -rompl3{0-3} <fname>  specify the file to use as +3 ROM image\n"
	"  -fd[ab] <fname>       link the specified file to virtual disk drives\n"
#endif

#ifdef XZX_IF1
	"  -if1 <bool>           enable Interface 1\n"
	"  -if1Rom <path>        specify the file to use as Interface 1 ROM image\n"
	"  -crlf <bool>          enable CR/LF translation on RS232 input\n"
	"  -strcr <bool>         strip CR on RS232 output\n"
	"  -m<n> <path>          put Microdrive cartridge file in drive <n>\n"
#endif

#ifdef XZX_MF128
	"  -mf128 <bool>         enable Multiface 128\n"
	"  -mf128Rom <path>      specify the file to use as Multiface 128 ROM image\n"
#endif
#ifdef XZX_IF1
	"\nMicrodrive emulation reads and writes .MDR files.\n"
#endif
	"Tape emulation reads .TAP and .TZX files and writes .TAP files.\n"
#ifdef XZX_PLUS3
	"Disk emulation reads and writes .DSK files.\n"
#endif
	"XZX reads and writes .SNA, .Z80, .SLT and .DAT format snapshots.\n");
}

static void
version(void)
{
	(void)fprintf(stderr, "This is %s", Version);
	(void)fprintf(stderr, " with the following compile time options:\n"
	" *  XZX is built for a"
#ifdef LITTLE_ENDIAN
	" little-endian architecture.\n"
#else
	" big-endian architecture.\n"
#endif
#ifdef SLOWDOWN
    " *  Speed control is enabled.\n"
#endif
#ifdef MITSHM
	" *  MIT-SHM image transfer is possible.\n"
#endif
#ifdef JOY
	" *  Linux analogue joystick emulation is possible.\n"
#endif
#ifdef AYCHIP_AUDIO
	" *  AY-8912 audio emulation is enabled.\n"
#endif
#ifdef SUN_AUDIO
	" *  Sun SPARC audio emulation (/dev/audio) is enabled.\n"
#endif
#ifdef PCSPKR_AUDIO
	" *  Linux PC Speaker style audio emulation is possible.\n"
#endif
#ifdef XZX_PLUS3
	" *  Spectrum +3 emulation is enabled.\n"
#endif
#ifdef XZX_IF1
	" *  Interface 1, microdrive & RS232 emulation are enabled.\n"
#endif
#ifdef XZX_MF128
	" *  Multiface 128 emulation is enabled.\n"
#endif
#ifdef PSEUDO_IO
	" *  Character input/output is emulated via ports.\n"
#endif
#ifdef DEBUG
	" *  Debugging is possible.\n"
#endif
	);
}

#ifdef DEBUG
static void
printResources(void)
{
#ifdef XZX_IF1
	int i;
#endif
#define BOOL(x)	 (x ? "true" : "false")
#define KEY(x)	 XKeysymToString(x)
	Msg(M_DEBUG, "libDir        : %s", GETCFG(libDir));
	Msg(M_DEBUG, "quiet         : %s", BOOL(GETCFG(quiet)));
	Msg(M_DEBUG, "geometry      : %s", GETCFG(geometry));
#ifndef EXCLUDE_PIXMAP
	Msg(M_DEBUG, "pixmap        : %s", BOOL(GETCFG(pixmaps)));
#endif
	Msg(M_DEBUG, "mono          : %s", BOOL(GETCFG(mono)));
	Msg(M_DEBUG, "private       : %s", BOOL(GETCFG(private)));
#ifdef MITSHM
	Msg(M_DEBUG, "mitshm        : %s", BOOL(GETCFG(mitshm)));
#endif
#ifdef DEBUG
	Msg(M_DEBUG, "debug         : %d", GETCFG(debug));
#endif
	Msg(M_DEBUG, "machine       : %d", GETCFG(machine));
	Msg(M_DEBUG, "ms            : %d", GETCFG(ms));
	Msg(M_DEBUG, "fast          : %s", BOOL(GETCFG(fast)));
#ifdef SLOWDOWN
	Msg(M_DEBUG, "slowdown      : %s", BOOL(GETCFG(slowdown)));
	Msg(M_DEBUG, "keyFaster     : %s", KEY(GETCFG(slowdownKey)));
	Msg(M_DEBUG, "keySlower     : %s", KEY(GETCFG(speedupKey)));
#endif
	Msg(M_DEBUG, "scale         : %d", GETCFG(scale));
	Msg(M_DEBUG, "flashing      : %s", BOOL(GETCFG(flashing)));
	Msg(M_DEBUG, "border        : %s", BOOL(GETCFG(border)));
#ifdef MITSHM
	Msg(M_DEBUG, "rrShm         : %d", GETCFG(rrShm));
#endif
	Msg(M_DEBUG, "rrNoshm       : %d", GETCFG(rrNoShm));
#if defined(SUN_AUDIO) || defined(PCSPKR_AUDIO)
	Msg(M_DEBUG, "noise         : %s", BOOL(GETCFG(noise)));
#endif
	Msg(M_DEBUG, "joyUp         : %s", KEY(GETCFG(joyUp)));
	Msg(M_DEBUG, "joyDown       : %s", KEY(GETCFG(joyDown)));
	Msg(M_DEBUG, "joyLeft       : %s", KEY(GETCFG(joyLeft)));
	Msg(M_DEBUG, "joyRight      : %s", KEY(GETCFG(joyRight)));
	Msg(M_DEBUG, "joyFire       : %s", KEY(GETCFG(joyFire)));
#ifdef JOY
	Msg(M_DEBUG, "joystick      : %s", BOOL(GETCFG(useJoy)));
	Msg(M_DEBUG, "joyDev        : %s", GETCFG(joyDevice));
	Msg(M_DEBUG, "joyTolerance  : %d", GETCFG(joyTolerance));
#endif
#ifdef AUDIO
	Msg(M_DEBUG, "sound         : %s", BOOL(GETCFG(useSound)));
#if defined(SUN_AUDIO) || defined(AYCHIP_AUDIO)
	Msg(M_DEBUG, "audioDev      : %s", GETCFG(audioDev));
#endif
#if defined(SUN_AUDIO) || defined(XBELL_AUDIO)
	Msg(M_DEBUG, "volume        : %d", GETCFG(volume));
#endif
#endif
	Msg(M_DEBUG, "rom48         : %s", GETCFG(rom48));
	Msg(M_DEBUG, "issue         : %d", GETCFG(issue));
	Msg(M_DEBUG, "kbdlayout     : %s", GETCFG(kbdlayout));
	Msg(M_DEBUG, "rom1280       : %s", GETCFG(rom128_0));
	Msg(M_DEBUG, "rom1281       : %s", GETCFG(rom128_1));
#ifdef XZX_PLUS3
	Msg(M_DEBUG, "rompl30       : %s", GETCFG(rompl3_0));
	Msg(M_DEBUG, "rompl31       : %s", GETCFG(rompl3_1));
	Msg(M_DEBUG, "rompl32       : %s", GETCFG(rompl3_2));
	Msg(M_DEBUG, "rompl33       : %s", GETCFG(rompl3_3));
	Msg(M_DEBUG, "fda           : %s", GETCFG(disks[0]));
	Msg(M_DEBUG, "fdb           : %s", GETCFG(disks[1]));
#endif
#ifdef XZX_IF1
	Msg(M_DEBUG, "if1           : %s", BOOL(GETCFG(if1_active)));
	Msg(M_DEBUG, "if1rom        : %s", GETCFG(if1_rom));
	Msg(M_DEBUG, "crlf          : %s", BOOL(GETCFG(translate_nl)));
	Msg(M_DEBUG, "strcr         : %s", BOOL(GETCFG(strip_nl)));
	for (i = 0; i < IF1_DRIVES; i++)
	{
		Msg(M_DEBUG, "m%d            : %s", i,  GETCFG(cartfiles[i]));
	}
#endif
#ifdef XZX_MF128
	Msg(M_DEBUG, "mf128         : %s", BOOL(GETCFG(mf128_active)));
	Msg(M_DEBUG, "mf128rom      : %s", GETCFG(mf128_rom));
#endif
}
#endif

