/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                                                                       */
/*                 Author: Paul Taylor                                   */
/*                   Date: 6 Jan 1998                                    */
/* --------------------------------------------------------------------- */
/*            LPC residual synthesis alternative version                 */
/*                                                                       */
/*************************************************************************/

#include "UniSyn.h"
#include "module_support.h"
#include "siod.h"
#include "EST_sigpr.h"
#include "EST_FileType.h"
#include "EST_THash.h"
#include "us_diphone.h"

Declare_TVector(EST_Item);


#if defined(INSTANTIATE_TEMPLATES)
#include "../base_class/EST_TVector.cc"

Instantiate_TVector(EST_Item);

#endif

static USDiphIndex *diph_index = 0;
static int tc_us_db = -1;
static LISP us_dbs = NIL;

static void parse_diphone_times(EST_Relation &diphone_stream, 
				EST_Relation &source_lab);
static void us_get_all_diphones(EST_Relation &diphone);
static void load_separate_diphone(int unit);
static void load_grouped_diphone(int unit);
static void get_diphone(EST_Item &d);
static int find_diphone_index_simple(const EST_String &d,USDiphIndex &di);
static int find_diphone_index(const EST_Item &d);
static void us_check_db();
static void us_add_diphonedb(USDiphIndex *db);

USDiphIndex::USDiphIndex() : dihash(1500)
{ 
    gc_protect(&params); 
}

USDiphIndex::~USDiphIndex() 
{ 
    gc_unprotect(&params);
}

void dur_to_end(EST_Relation &r)
{
    float prev_end = 0;

    for (EST_Item *p = r.head(); p ; p = next(p))
    {
	p->fset("end", p->fF("dur") + prev_end);
	prev_end = p->fF("end");
    }
}

static USDiphIndex *get_c_us_db(LISP x)
{
    if (TYPEP(x,tc_us_db))
	return (class USDiphIndex *)USERVAL(x);
    else
	err("wta to get_c_us_db",x);

    return NULL;  // err doesn't return but compilers don't know that
}

static LISP siod_make_us_db(USDiphIndex *s)
{
    if (s==0)
	return NIL;
    else
	return siod_make_typed_cell(tc_us_db,s);
}


void add_end_silences(EST_Relation &segment, EST_Relation &target)
{
    EST_Item *t, *n;
    float shift = 0.0;
    const float pause_duration = 0.1;

    t = segment.head();
    if (!ph_is_silence(t->f("name")))
    {
	n = t->insert_before();
	n->fset("name", ph_silence());
	n->fset("dur", pause_duration);
	shift += pause_duration;
    }

    t = segment.tail();
    if (!ph_is_silence(t->f("name")))
    {
	n = t->insert_after();
	n->fset("name", ph_silence());
	n->fset("dur", pause_duration);
	shift += pause_duration;
    }
    dur_to_end(segment);

    target.tail()->fset("pos", (target.tail()->fF("pos") + shift));
}

void add_end_silences(EST_Relation &segment)
{
    EST_Item *t, *n;

    t = segment.head();
    if (!ph_is_silence(t->f("name")))
    {
	n = t->insert_before();
	n->fset("name", ph_silence());
    }

    t = segment.tail();
    if (!ph_is_silence(t->f("name")))
    {
	n = t->insert_after();
	n->fset("name", ph_silence());
    }
}

static EST_String get_diphone_name(EST_Item *item,const EST_String dir)
{
    // Get diphone name which may differ from phone name
    // Looks for us_diphone_<dir>, us_diphone, or name in that order
    EST_String d1;
    static EST_String dname = "us_diphone";

    if (!item)
	return "";
    else if ((d1 = item->fS(dname+"_"+dir)) != "0")
	return d1;
    else if ((d1 = item->fS(dname)) != "0")
	return d1;
    else
	return item->f("name").string();
}

void us_get_diphones(EST_Utterance &utt)
{
    // Create unit stream with coefficients and signals for each
    // diphone
    EST_Item *p, *d;
    EST_String name1, name2, file;

    us_check_db();
    utt.create_relation("Unit");

    p = utt.relation("Segment")->head();
    name1 = get_diphone_name(p,"left");  // left part of first diphone

    for (p=next(p); p; p = next(p))
    {
	d = utt.relation("Unit")->append();
	name2 = get_diphone_name(p,"right");
	d->fset("name", (name1 + "-" + name2));
	get_diphone(*d);
	name1 = get_diphone_name(p,"left");
    }

    utt.create_relation("SourceSegments");
    for (p = utt.relation("Segment")->head(); p; p = next(p))
    {
	d = utt.relation("SourceSegments")->append();
	d->set_name(p->name());
    }

    parse_diphone_times(*(utt.relation("Unit")), 
			*(utt.relation("SourceSegments")));
}

static void parse_diphone_times(EST_Relation &diphone_stream, 
				EST_Relation &source_lab)
{
    EST_Item *s, *u;
    EST_Track *pm;
    int e_frame, m_frame = 0;
    float dur_1 = 0.0, dur_2 = 0.0, p_time;
    float t_time = 0.0, end;
    p_time = 0.0;
    
    for (s = source_lab.head(), u = diphone_stream.head(); u; u = next(u), 
	 s = next(s))
    {
	pm = (EST_Track *) u->f("coefs").ptr();
	if (pm == 0)
	{
	    cerr << "Couldn't get pitchmarks for " << u->name() << endl;
	    festival_error();
	}
	
	e_frame = pm->num_frames() - 1;
	m_frame = u->fI("middle_frame");

	dur_1 = pm->t(m_frame);
	dur_2 = pm->t(e_frame) - dur_1;
	
	s->fset("end", (dur_1 + p_time));
	p_time = s->fF("end") + dur_2;

	end = dur_1 + dur_2 + t_time;
	t_time = end;
	u->fset("end", t_time);
    }
    if (s)
	s->fset("end", (dur_2 + p_time));
}

LISP us_make_group_file(LISP lname, LISP params)
{
    EST_String group_file, index_file;
    EST_String track_file_format, sig_file_format, sig_sample_format;
    EST_Relation diphone;
    EST_TokenStream ts;
    EST_Item *d;
    EST_Wave *sig;
    EST_Track *tr;
    FILE *fp, *fp_group;
    const int block_size = 1024;
    int pos;

    us_check_db();

    track_file_format = get_param_str("track_file_format",params,"est_binary");
    sig_file_format = get_param_str("sig_file_format",params,"snd");
    sig_sample_format = get_param_str("sig_sample_format",params,"mulaw");

    group_file = make_tmp_filename();
    group_file += ".group";
    index_file = get_c_string(lname);
    us_get_all_diphones(diphone);
    
    if ((fp = fopen(group_file, "wb")) == NULL)
    {
	cerr << "US DB: failed to open group file as temporary file\n";
	festival_error();
    }

    for (d = diphone.head(); d; d = next(d))
    {
	sig = (EST_Wave *)d->f("sig").ptr();
	tr = (EST_Track *)d->f("coefs").ptr();

	pos = ftell(fp);
	d->fset("track_start", pos);
	tr->save(fp, track_file_format);

	pos = ftell(fp);
	d->fset("wave_start", pos);
	sig->save_file(fp, sig_file_format, sig_sample_format, EST_NATIVE_BO);
    }
    fclose(fp);

    if ((fp = fopen(index_file, "wb")) == NULL)
    {
	cerr << "US DB: failed to final group file \"" << index_file
	    << "\"" << endl;
	festival_error();
    }

    fprintf(fp, "EST_File index\n");
    fprintf(fp, "DataType ascii\n");
    fprintf(fp, "NumEntries %d\n", diphone.length());
    fprintf(fp, "IndexName %s\n", (const char *)diph_index->name);
    fprintf(fp, "DataFormat grouped\n");
    fprintf(fp, "Version 2\n");
    fprintf(fp, "track_file_format %s\n",(const char *)track_file_format);
    fprintf(fp, "sig_file_format %s\n",(const char *)sig_file_format);
    fprintf(fp, "EST_Header_End\n");

    for (d = diphone.head(); d; d = next(d))
	fprintf(fp, "%s %d %d %d\n", (const char *)d->fS("name"), 
		d->fI("track_start"), d->fI("wave_start"),
		d->fI("middle_frame"));

    // Copy binary data from temporary group file to end of
    // real group file
    char buf[block_size];
    int r;

    if ((fp_group = fopen(group_file, "rb")) == NULL)
	return NIL;

    while ((r = fread(buf, sizeof(char), block_size, fp_group)) != 0)
	fwrite(buf, sizeof(char), r, fp);

    fclose(fp);
    fclose(fp_group);
    unlink(group_file);

    return NIL;
}

static void us_get_all_diphones(EST_Relation &diphone)
{
    EST_Item *d;
    EST_String name1;

    for (int i = 0; i < diph_index->diphone.n(); ++i)
    {
	cout << diph_index->diphone[i].f("name") << endl;
	d = diphone.append();
	d->fset("name", diph_index->diphone[i].fS("name"));
	get_diphone(*d);
    }
}

int read_diphone_index(const EST_String &filename, 
			  USDiphIndex &di)
{
    EST_TokenStream ts;
    int i, ref;
    int num_entries;
    EST_Option hinfo;
    EST_EstFileType t;
    EST_String v, pointer, n;
    bool ascii;
    EST_read_status r;
    EST_Item d;

    di.index_offset = 0;
    
    if (ts.open(filename) != 0)
    {
	cerr << "US DB: can't open index file " << filename << endl;
	return misc_read_error;
    }
    // set up the character constant values for this stream
    ts.set_SingleCharSymbols(";");
    
    if ((r = read_est_header(ts, hinfo, ascii, t)) != format_ok)
	return r;
    if (t != est_file_index)
	return misc_read_error;
    
    num_entries = hinfo.ival("NumEntries");
    di.grouped = (hinfo.val("DataFormat") == "grouped") ? true : false;
    di.diphone.resize(num_entries);
    
    if (di.grouped)
    {
	di.track_file_format = hinfo.val_def("track_file_format","est");
	di.sig_file_format = hinfo.val_def("sig_file_format","est");
	for (i = 0; i < num_entries; ++i)
	{
	    di.diphone[i].fset("name", ts.get().string());
	    di.diphone[i].fset("count", 0);
	    di.diphone[i].fset("track_start", atoi(ts.get().string()));
	    di.diphone[i].fset("wave_start", atoi(ts.get().string()));
	    di.diphone[i].fset("middle_frame", atoi(ts.get().string()));
	    di.dihash.add_item(di.diphone[i].f("name"),i);
	}
	di.index_offset = ts.tell();
	// cout << "index offset = " << di.index_offset << endl;
    }
    else
    {
	int n_num_entries = num_entries;
	for (i = 0; i < n_num_entries; ++i)
	{
	    if (ts.eof())
	    {
		cerr << "US DB: unexpected EOF in \"" << filename << "\": " <<
		    i << " entries instead of " << num_entries << endl;
		festival_error();
	    }
	    n = ts.get().string();
	    di.diphone[i].fset("name", n);
	    di.diphone[i].fset("filename", ts.get().string());
	    di.diphone[i].fset("count", 0);
	    if (!di.diphone[i].fS("filename").contains("&", 0))
	    {
		di.diphone[i].fset("start", atof(ts.get().string()));
		di.diphone[i].fset("middle", atof(ts.get().string()));
		di.diphone[i].fset("end", ts.get().string());
		if ((di.diphone[i].fF("start")>=di.diphone[i].fF("middle"))||
		    (di.diphone[i].fF("middle") >= di.diphone[i].fF("end")))
		{
		    cerr << "US DB: diphone index for " << n << 
			" start middle end not in order, ignored " << endl;
		    i--;
		    n_num_entries--;
		}
	    }
	    di.dihash.add_item(n,i);
	}
    
	// now copy reference entries
	for (i = 0; i < num_entries; ++i)
	    if (di.diphone[i].fS("filename").contains("&", 0))
	    {
		pointer = di.diphone[i].fS("filename").after("&", 0);
//		cout << "pointer: = " << pointer << endl;
		if ((ref = find_diphone_index_simple(pointer,di)) == -1)
		{
		    cerr << "US DB: Illegal diphone pointer in index file: " 
			<< i << " " 
			<< di.diphone[i].fS("name") << " -> " << 
			    di.diphone[i].fS("filename") << endl;
		    festival_error();
		}
		di.diphone[i].fset("filename",
				    di.diphone[ref].fS("filename"));
		di.diphone[i].fset("start", di.diphone[ref].fS("start"));
		di.diphone[i].fset("middle", di.diphone[ref].fS("middle"));
		di.diphone[i].fset("end", di.diphone[ref].fS("end"));
	    }
    }

    return format_ok;
}

LISP us_diphone_init(LISP args)
{
    EST_String x;
    USDiphIndex *diph_index = new USDiphIndex;
    diph_index->grouped = false;
    diph_index->params = args;  
    diph_index->name = get_param_str("name",args,"name");
    diph_index->index_file = get_param_str("index_file",args,"");

    read_diphone_index(diph_index->index_file, *diph_index);

    // This is needed because there is no get_param_EST_String function
    x = get_param_str("grouped",args,"");
    if (x == "true")
    {
	diph_index->grouped = true;
	if (diph_index->ts.open(diph_index->index_file) != 0)
	{
	    cerr << "US DB: can't open grouped diphone file " 
		<< diph_index->index_file << endl;
	    festival_error();
	}
	// set up the character constant values for this stream
	diph_index->ts.set_SingleCharSymbols(";");
    }
    else
    {
	*cdebug << ":" << get_param_str("grouped",args,"") << ":" << endl;
	*cdebug << "index grouped:" << diph_index->grouped << endl;
	*cdebug << "true:" << true << endl;
	*cdebug << "false:" << false << endl;
	
	diph_index->coef_dir = get_param_str("coef_dir",args,"");
	diph_index->sig_dir = get_param_str("sig_dir",args,"");
	
	diph_index->coef_ext = get_param_str("coef_ext",args,"");
	diph_index->sig_ext = get_param_str("sig_ext",args,"");
    }

    us_add_diphonedb(diph_index);

    return rintern(diph_index->name);
}

static void get_diphone(EST_Item &d)
{
    int unit;
    
    unit = find_diphone_index(d);
    
    if (diph_index->diphone[unit].f("count") == 0)
    {
	if (diph_index->grouped)
	    load_grouped_diphone(unit);
	else
	    load_separate_diphone(unit);
	diph_index->diphone[unit].fset("count", d.fI("count") + 1);
    }
    
    d.fset_val("sig", diph_index->diphone[unit].f("sig"));
    d.fset_val("coefs", diph_index->diphone[unit].f("coefs"));
    d.fset_val("middle_frame", diph_index->diphone[unit].f("middle_frame"));
}

static void load_grouped_diphone(int unit)
{
    int middle_frame;
    EST_Track *coefs;
    EST_Wave *sig;
    int wave_start, track_start;
    
    coefs = new EST_Track;
    sig = new EST_Wave;
    
    track_start = diph_index->diphone[unit].fI("track_start");
    wave_start = diph_index->diphone[unit].fI("wave_start");
    middle_frame = diph_index->diphone[unit].fI("middle_frame");
    
    diph_index->ts.seek(track_start + diph_index->index_offset);
    coefs->load(diph_index->ts); // type is self determined at present
    
    diph_index->ts.seek(wave_start + diph_index->index_offset);
    sig->load(diph_index->ts,diph_index->sig_file_format);
    
    diph_index->diphone[unit].fset("coefs", coefs, gc_track);
    diph_index->diphone[unit].fset("middle_frame", middle_frame);
    
    diph_index->diphone[unit].fset("sig", sig, gc_wave);
}

static void load_separate_diphone(int unit)
{
    // Load in the coefficents and signame for this diphone
    // It caches the results in the diphone index entry, though
    // someone else may clear them.  Note the full file is loaded
    // each time which isn't optimal if there are multiple diphones
    // is the same file
    int samp_start, samp_end;
    int pm_start, pm_end, pm_middle;
    EST_Track full_coefs, dcoefs, *coefs;
    
    if (full_coefs.load(diph_index->coef_dir + "/" 
			+ diph_index->diphone[unit].f("filename")
			+ diph_index->coef_ext) != format_ok)
    {
	cerr << "US DB: failed to read coefs file from " <<
	    diph_index->coef_dir + "/" 
		+ diph_index->diphone[unit].f("filename")
		    + diph_index->coef_ext << endl;
	festival_error();
    }
    
    pm_start = full_coefs.index(diph_index->diphone[unit].f("start"));
    pm_middle = full_coefs.index(diph_index->diphone[unit].f("middle"));
    pm_end = full_coefs.index(diph_index->diphone[unit].f("end"));
    
    // find time of mid-point, i.e. boundary between phones
    full_coefs.sub_track(dcoefs, pm_start, pm_end - pm_start + 1);
    // Copy coefficents so the full coeffs can be safely deleted
    coefs = new EST_Track(dcoefs);
    for (int j = 0; j < dcoefs.num_frames(); ++j)
	coefs->t(j) = dcoefs.t(j) - full_coefs.t(Gof((pm_start - 1), 0));

    diph_index->diphone[unit].fset("coefs",coefs,gc_track);
    diph_index->diphone[unit].fset("middle_frame", pm_middle - pm_start -1);
    
    EST_Wave full_sig, sub_sig;
    if (full_sig.load(diph_index->sig_dir + "/" 
		      + diph_index->diphone[unit].f("filename")
		      + diph_index->sig_ext) != format_ok)
    {
	cerr << "US DB: failed to read signal file from " <<
	    diph_index->sig_dir + "/" 
		+ diph_index->diphone[unit].f("filename")
		    + diph_index->sig_ext << endl;
	festival_error();
    }
    
    // go to the periods before and after
    samp_start = (int)(full_coefs.t(Gof((pm_start - 1), 0))
		       * (float)full_sig.sample_rate());
    if (pm_end+1 < full_coefs.num_frames())
	pm_end++;
    samp_end = (int)(full_coefs.t(pm_end) * (float)full_sig.sample_rate());
    full_sig.sub_wave(sub_sig, samp_start, samp_end - samp_start + 1);
    EST_Wave *sig = new EST_Wave(sub_sig);

    diph_index->diphone[unit].fset("sig", sig, gc_wave);
}

static int find_diphone_index_simple(const EST_String &d,USDiphIndex &di)
{
    int found,r;
    
    r = di.dihash.val(d,found);
    if (found)
	return r;
    else
	return -1;
}

static int find_diphone_index(const EST_Item &d)
{
    // Find the approrpiate entry in the diphone index table
    // mapping to alternates if required.
    int index;
    EST_String diname = d.f("name");

    // If all goes well the diphone will be found directly in the index
    index=find_diphone_index_simple(diname,*diph_index);
    if ((index=find_diphone_index_simple(diname,*diph_index)) != -1)
	return index;

    // But for various reasons it might not be there so allow
    // a run time specification of alternates.  This isn't optimal
    // but is better than falling immediately back on just a default
    // diphone
    LISP alt_left = get_param_lisp("alternates_left",diph_index->params,NIL);
    LISP alt_right = get_param_lisp("alternates_right",diph_index->params,NIL);
    EST_String di_left = diname.before("-");
    EST_String di_right = diname.after("-");
    EST_String di_left_alt = get_param_str(di_left,alt_left,di_left);
    EST_String di_right_alt = get_param_str(di_right,alt_right,di_right);
    EST_String di_alt = di_left_alt+"-"+di_right_alt;

    if ((index=find_diphone_index_simple(di_alt,*diph_index)) != -1)
    {
//	cout << "UniSyn: using alternate diphone " << di_alt << " for " <<
//	    diname << endl;
	return index;
    }
    
    // It really isn't there so return the default one and print and error
    // now
    EST_String default_diphone = 
	get_param_str("default_diphone",diph_index->params,"");
    
    if (default_diphone != "")
    {
	index = find_diphone_index_simple(default_diphone,*diph_index);
	if (index == -1)
	{
	    cerr << "US DB: can't find diphone " << d.f("name") 
		<< " and even default diphone (" << default_diphone 
		    << ") doesn't exist" << endl;
	    festival_error();
	}
	else
	    cout << "UniSyn: using default diphone " << default_diphone << 
		" for " << diname << endl;
	return index;
    }
    else
    {
	cerr << "US DB: can't find diphone " << d.f("name") << 
	    " nor alternatives" << endl;
	festival_error();
    }
    return -1;
}

static void us_add_diphonedb(USDiphIndex *db)
{
    // Add this to list of loaded diphone dbs and select it
    LISP lpair;
    USDiphIndex *ddb;

    if (us_dbs == NIL)
    {
	gc_protect(&us_dbs);
	tc_us_db = siod_register_user_type("us_db");
    }
	
    lpair = siod_assoc_str(db->name,us_dbs);

    if (lpair == NIL)
    {   // new diphone db of this name
	us_dbs = cons(cons(rintern(db->name),
			   cons(siod_make_us_db(db),NIL)),
			   us_dbs);
    }
    else
    {	// already one of this name, don't know howto free it
	cerr << "US_db: warning redefining diphone database "
	    << db->name << endl;
	ddb = get_c_us_db(car(cdr(lpair)));
	delete ddb;
	setcar(cdr(lpair),siod_make_us_db(db));
    }
    
    diph_index = db;
}

LISP us_select_db(LISP name)
{
    // Select diphone set
    LISP lpair;
    
    lpair = siod_assoc_str(get_c_string(name),us_dbs);
    
    if (lpair == NIL)
    {
	cerr << "US DB: no diphone database named " << get_c_string(name)
	    << " defined\n";
	festival_error();
    }
    else
	diph_index = get_c_us_db(car(cdr(lpair)));

    return name;
}

LISP us_list_dbs(void)
{
    // List names of currently loaded dbs 
    LISP names,n;

    for (names=NIL,n=us_dbs; n != NIL; n=cdr(n))
	names = cons(car(car(n)),names);
    return reverse(names);
}

LISP us_db_params(void)
{
    // Return parameters of current db

    us_check_db();
    return diph_index->params;
}

static void us_check_db()
{
    if (diph_index == 0)
    {
	cerr << "US DB: no diphone database loaded\n";
	festival_error();
    }
}
