/* Copyright (C) 1991, 1992, 1994 Aladdin Enterprises.  All rights reserved.

  This file is part of Aladdin Ghostscript.

  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.

  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* gsdpsx.c */
/* Display PostScript graphics additions for Ghostscript library */

#include "math_.h"
#include "gx.h"
#include "gserrors.h"
#include "gsmatrix.h"                   /* for gscoord.h */
#include "gscoord.h"
#include "gspaint.h"
#include "gxfixed.h"
#include "gzstate.h"
#include "iref.h"
#include "interp.h"
#include "imain.h"
#include "iminst.h"

/* for the DPS/X11 streams */
#include "stream.h"
#include "gxiodev.h"

/* the X people can't make their headers define what they need automatically... */
#define NEED_REPLIES
#define NEED_EVENTS
#include "x_.h"                 /* standard X include files */
#include "gxdevice.h"
#include "X11/Xproto.h"
#include "X11/Xlibint.h"        /* we need internals, too */
#include "dpsNXprops.h" /* has the property definitions */
#include "DPS/XDPSproto.h"
#include "DPSCAPproto.h"
#include "DPSCAP.h"
#include "DPS/dpsNXargs.h"

#include "gdevx.h"

extern gx_device_X gs_x11_device;

#include "gsdpsx.h"

/* network headers for the agent */
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <stdlib.h>

#define BUFLEN 65536

/* ------ static data ------ */
typedef struct gsdpsx_conn_info_s {
    /* socket is defined by the array position */
    gx_device_X *xdev;
    Window       clientWindow; /* relative to xdev's Display */
    gs_state    *pgs;
    gs_memory_t *mem;
    gs_main_instance *minst;
    unsigned char *buf;
    unsigned char *curpos;
    unsigned int   len; /* if we need to leave messages here, the remaining length */
    unsigned char *inputstart; /* these are for when we get an input message in here */
    unsigned char *inputend;
    unsigned int  serial;
    CARD16       displayNum; /* don't know, but we get it... */
    CARD16       screen;
    XID          cxid;
    int          sock;
    int          isfrag; /* first item in the buffer is a partial fragment */
} gsdpsx_conn_info;

private gsdpsx_conn_info gsdpsx_conninfo[64];
private gsdpsx_conn_info *current_conn; /*! THIS IS BAD ****/
private Atom resume_atom;
private Atom billboard_atom;
private int billboard_atom_needs_cleanup;
private Display *main_dpy; /*! BAD */
private Window main_win; /*! BAD */

/* ------ forward prototypes ------ */
/* message handlers */
/* returns the number of bytes message used.  0 if it is just a fragment. */
/* returns negative on a different error. */
private int dpscap_ConnSetup(gsdpsx_conn_info *conn, int sock, unsigned char *pmsg, int remaining, Window main_win);
private int dpscap_Notify(gsdpsx_conn_info *conn, unsigned char *pmsg, int remaining);
private int dps_CreateContext(gsdpsx_conn_info *conn, unsigned char *pmsg, int remaining);
/* utility functions */

/* ------ DPS/X11 streams ------ */
private void gsdpsx_std_init(void);

/* ------ Agent state ------ */

/* Starts the X DPS agent */
int
gs_startXdpsagent(gs_state *pgs, gs_memory_t *mem)
{
    Atom willingness_atom, open_license_atom;
    int server_socket, server_port, base_port, i;
    struct sockaddr_in sock;
    struct servent *pent;
    char tmp[64];

    /* open the display, this will last the entire session and be used for
       every window used */
    main_dpy = XOpenDisplay(NULL);
    if (!main_dpy) {
	fprintf(stderr, "Cannot open X Display: %s\nABORTING!\n", 
	        XDisplayName(NULL));
        return -1;
    }

    /******* create us a socket to listen on                      ********/
    pent = getservbyname(DPS_NX_SERV_NAME, "tcp");
    if (pent) {
	base_port = ntohs(pent->s_port);
    } else {
	base_port = ntohs(CSDPSPORT);
    }

    server_socket = socket(AF_INET, SOCK_STREAM, 0);
    if (server_socket < 0) {
	perror("server_socket's socket(): ");
        XCloseDisplay(main_dpy);
        return -1;
    }

    /* Get rid of TCP buffering */
#ifdef TCP_NODELAY
    {
      int tmp = 1;
      setsockopt (server_socket, IPPROTO_TCP, TCP_NODELAY,
		  (char *)&tmp, sizeof (int));
    }
#endif

    /* find an open port */
    sock.sin_family = AF_INET;
    sock.sin_addr.s_addr = INADDR_ANY;
    for (server_port = 0; server_port < 100; server_port++) {
        sock.sin_port = htons(base_port + server_port);
        if (bind(server_socket, (struct sockaddr*)&sock, (int)sizeof(sock)) == 0) {
            /* we found a socket */
            break;
        }
    }
    if (server_port == 100) {   /* we couldn't find a port... */
	fprintf(stderr, "Unable to find a port for the server socket, tried %d through %d\n", base_port, base_port + server_port - 1);
        XCloseDisplay(main_dpy);
        close(server_socket);
    }

    /******* Create a base window                                 ********/
    main_win = XCreateSimpleWindow(main_dpy, DefaultRootWindow(main_dpy),
                                   0, 0, 1, 1, /* min X window size */
                                   0, /* no border either */
                                   BlackPixel(main_dpy, DefaultScreen(main_dpy)),
                                   WhitePixel(main_dpy, DefaultScreen(main_dpy)));
    if (!main_win) {
	fprintf(stderr, "Unable to create main comm window.\n");
        XCloseDisplay(main_dpy);
        close(server_socket);
    }
    /* we want basically no input, but that does give us Client messages */
    XSelectInput(main_dpy, main_win, NoEventMask);

    /******* now that we are ready to accept connections, we must ********/
    /******* inform the server                                    ********/

    /* retrieve all the Atoms for the properites we need to set beforehand */
    billboard_atom   = XInternAtom(main_dpy, XDPSNX_BILLBOARD_PROP, FALSE);
    willingness_atom = XInternAtom(main_dpy, XDPSNX_WILLINGNESS_PROP, FALSE);
    resume_atom      = XInternAtom(main_dpy, DPSCAP_TYPE_RESUME, FALSE);
    if (!billboard_atom || !willingness_atom) {
	fprintf(stderr, "unable to get X11 Atoms.\n");
        XCloseDisplay(main_dpy);
        close(server_socket);
        return -1;
    }

    /* put the needed properties into main_win */
    /* set willingness... 100 is very willing */
    i = 100;
    XChangeProperty(main_dpy, main_win,
                             willingness_atom, XA_INTEGER, 32,
                             PropModeReplace,
                             (unsigned char *)&i, 1);

    /* we only support open license, but that should work with any client */
    sprintf(tmp, "%s:%d", LICENSE_METHOD_OPEN, OPEN_LICENSE_VERSION);
    open_license_atom = XInternAtom(main_dpy, tmp, False);
    XChangeProperty(main_dpy, main_win,
                    XInternAtom(main_dpy, XDPSNX_LICENSE_METHOD_PROP, False),
                    XA_ATOM, 32,
                    PropModeReplace,
                    (unsigned char *)&open_license_atom, 1);

    /* set the transport information */
    {
        int info[2];
        info[0] = XDPSNX_TRANS_TCP;
        info[1] = base_port + server_port;
        /*! need to support UNIX sockets here, too! */
        XChangeProperty(main_dpy, main_win,
                        XInternAtom(main_dpy, XDPSNX_TRANSPORT_INFO_PROP, False),
                        XA_INTEGER, 32,
                        PropModeReplace,
                        (unsigned char *)&info, 2);
    }

    /* set the hostname */
    i = gethostname(tmp, sizeof(tmp));
    if (i) {
	perror("gethostname: ");
        XDestroyWindow(main_dpy, main_win);
        XCloseDisplay(main_dpy);
        close(server_socket);
        return -1;
    }
    /*! if we use the full hostname, the client library will */
    /*! switch us to UNIX socket */
    strcpy(tmp, "localhost");
    XChangeProperty(main_dpy, main_win,
                    XInternAtom(main_dpy, XDPSNX_HOST_NAME_PROP, False),
                    XA_STRING, 8,
                    PropModeReplace,
                    (unsigned char *)tmp, strlen(tmp));

    /* and append us into the property list */
    XChangeProperty(main_dpy, DefaultRootWindow(main_dpy),
                        billboard_atom, XA_WINDOW, 32,
                        PropModePrepend,
                        (unsigned char *)&main_win, 1);
    billboard_atom_needs_cleanup = 1;

    /* flush all remaining X events */
    XSync(main_dpy, False);

    /******* alright, its time to really accept a connection and get to work ********/
    /* this code will all have to be rewritten with multiple contexts in mind */
    while (listen(server_socket, 1) >= 0) {
        struct sockaddr_in in_sock;
        int sock, addrlen, len;

        addrlen = sizeof(in_sock);
        sock = accept(server_socket, (struct sockaddr *)&in_sock, &addrlen);
        if (sock < 0 || sock >= (sizeof(gsdpsx_conninfo)/sizeof(gsdpsx_conninfo[0])))
            break;

        /* setup basic connection info */
        gsdpsx_conninfo[sock].buf = (unsigned char *)
            gs_alloc_bytes(mem, BUFLEN, "gs_startXdpsagent");
        if (!gsdpsx_conninfo[sock].buf) {
            XDestroyWindow(main_dpy, main_win);
            XCloseDisplay(main_dpy);
            close(server_socket);
            close(sock);
            return -1;
        }
        gsdpsx_conninfo[sock].curpos = gsdpsx_conninfo[sock].buf;
        gsdpsx_conninfo[sock].pgs = pgs;
        gsdpsx_conninfo[sock].mem = mem;
        gsdpsx_conninfo[sock].serial = 1;


        /* get the connection setup request */
        len = recv(sock, gsdpsx_conninfo[sock].buf, BUFLEN, 0);
        if (gsdpsx_conninfo[sock].buf[0] != 0x6C
            || dpscap_ConnSetup(&(gsdpsx_conninfo[sock]), sock, gsdpsx_conninfo[sock].buf, len, main_win) <= 0) {
            dprintf("dpscap_ConnSetup failed!\n");
            /* ERROR! */
            XDestroyWindow(main_dpy, main_win);
            XCloseDisplay(main_dpy);
            close(server_socket);
            close(sock);
            return -1;
        }

        {
        int exit_code;
        ref error_object;
        gs_main_run_string_with_length(gsdpsx_conninfo[sock].minst,
                                       "(%stdin) run  ",
                                       14,
                                       gsdpsx_conninfo[sock].minst->user_errors,
                                       &exit_code,
                                       &error_object);
        }

        return -1;
    }

    XDestroyWindow(main_dpy, main_win);
    XCloseDisplay(main_dpy);
    close(server_socket);

    return -1; /* if we got here, it is an error */
}

private int
dpscap_ConnSetup(gsdpsx_conn_info *conn, int sock, unsigned char *msg, int len, Window main_win)
{
    xCAPConnSetupReq *req = (xCAPConnSetupReq*)msg;
    int code, used;
    gx_device_X *new_dev;
    xCAPConnSuccess reply;

    used = sz_xCAPConnSetupReq;
    if (len-used < 0)
        return 0;

    /* check that the display name is there */
    used += req->displayStringLength;
    if (len-used < 0 || req->displayStringLength > 255) /* we can't deal with that */
        return 0;

    code = gs_copydevice((gx_device**)&new_dev,
                         (gx_device*)&gs_x11_device,
                         conn->mem);
    if (code < 0)
        return code;

    /* The display name might not be there and it certainly won't be NULL
       terminated.  We don't need to worry about overwriting what is after it */
    if (req->displayStringLength) {
        int off = sz_xCAPConnSetupReq +
          req->authProtoNameLength +
          req->authProtoDataLength;
        char displayname[256];
        memcpy(displayname, msg+off, req->displayStringLength);
        displayname[req->displayStringLength] = '\0';
        dprintf1("setting DISPLAY to '%s'\n", displayname);
        setenv("DISPLAY", displayname, 1);
    }

    new_dev->owner = gs_x_dpsagent;
    new_dev->pwin = req->clientWindow;

    /* now we instantiate the device */
    code = gs_setdevice(conn->pgs, (gx_device*)new_dev);
    if (code < 0) {
        /*** HOW DO WE FREE THIS? ***/
        /* gs_destroydevice(pgs, (gx_device*)new_dev, mem); */
        return code;
    }

    conn->sock = sock; /* seems odd, but we won't keep this order forever */
    conn->clientWindow = req->clientWindow;
    conn->minst = gs_main_instance_default();
    conn->xdev = new_dev;

    /* alright, time to send a reply back */
    reply.success = TRUE;
    reply.reasonLength = sizeof(reply);
    reply.additionalLength = 0;
    reply.serverVersion = DPSPROTOCOLVERSION;
    reply.dpscapVersion = DPSCAPPROTOVERSION;
    reply.reserved = 0;
    reply.flagsUsed = 0;
    reply.preferredNumberFormat = 0; /*!*/
    reply.floatingNameLength = 0; /*! whats this? */
    reply.agentWindow = main_win; /* only used to resume suspended contexts */
    send(sock, (void*)&reply, reply.reasonLength, 0);

    current_conn = conn;

    /* initialize outputs */
    gsdpsx_std_init();


    return used;
}

/* checks to see if the event is a resume on the cxid */
private Bool WaitForResumeProc(Display *xdpy, XEvent *event, XPointer arg)
{
  if (event->xany.window == main_win && event->type == ClientMessage
      && event->xclient.message_type == resume_atom && event->xclient.format == 32 && event->xclient.data.l[0] == (int)arg)
    /* the commented out is context X id... we don't support multiple contexts! */
  {       /*dprintf("got RESUME Message\n");*/
          return True;
  }
  return False;
}

int dpscap_Notify(gsdpsx_conn_info *conn, unsigned char *msg, int remaining)
{
  /*    int code; */
    xCAPNotifyReq *req = (xCAPNotifyReq*)msg;

    /* check for the whole message... must do first check to ensure
       that you can do the second */
    if (remaining < sz_xCAPNotifyReq || ((req->length)*4 > remaining))
        return 0;

    switch (req->notification) {
    case DPSCAPNOTE_SYNC:
    {
        XEvent event;
        event.xclient.type = ClientMessage;
        event.xclient.serial = req->data;
        event.xclient.send_event = True;
        event.xclient.message_type = XInternAtom(conn->xdev->dpy, DPSCAP_TYPE_SYNC,False);
        event.xclient.format = 32;
        event.xclient.data.l[0] = req->data;
        event.xclient.display = conn->xdev->dpy;
        event.xclient.window = conn->clientWindow;
        XSendEvent(event.xclient.display, event.xclient.window, False, 0, &event);
        XSync(event.xclient.display, False);
        break;
    }

    case DPSCAPNOTE_PAUSE:
    {
        /* a PAUSE requires that we wait until we receive an X event (on the X comm link) */
        /* to resume.  This ensures that X has caught up with the window creation or other */
        /* changes */
        XEvent event;
        /*dprintf("Waiting for resume X event\n");*/
        /*! yielding to other threads here would be nice, but this will require some fancy work */
        /*! this could eat the event from another context... we need to be careful of that,
          either by creating a different window per connection, or waking the other up */
        /*! we might actually be okay, it turns out */
        XIfEvent(main_dpy, &event, WaitForResumeProc, (XPointer)req->cxid);
        break;
    }
    
    default:
        dprintf1("ignoring unknown xCAPNotify %X\n", (unsigned)req->notification);
    }

    return (req->length)*4;
}

int dps_CreateContext(gsdpsx_conn_info *conn, unsigned char *msg, int remaining)
{
    float xppp, yppp;
    gs_matrix *initial_matrix;
    
    /*    int code; */
    xPSCreateContextReq *req = (xPSCreateContextReq*)msg;
    xPSCreateContextReply reply;

    /* check for the whole message... must do first check to ensure
       that you can do the second */
    if (remaining < sz_xPSCreateContextReq || ((req->length)*4 > remaining))
        return 0;

    conn->cxid = req->cxid;
    gdev_x_set_drawable(conn->xdev, req->drawable);

    /* initial matrix, should it be needed */
    xppp = conn->xdev->x_pixels_per_inch / 72.0;
    yppp = conn->xdev->y_pixels_per_inch / 72.0;
    initial_matrix = xdev_initial_matrix_inline(conn->xdev);
    initial_matrix->xx = xppp;
    initial_matrix->xy = 0.0;
    initial_matrix->yx = 0.0;
    initial_matrix->yy = -yppp;
    initial_matrix->tx = 0;
    initial_matrix->ty = 512;
    gs_initmatrix(conn->pgs);

    /* send the expected reply */
    reply.type = X_Reply;
    reply.sequenceNumber = 0; /* whoop.  not in a server */
    reply.length = (sizeof(reply)-sz_xGenericReply)/4;
    reply.cpsid = 1; /* we dont support more contexts yet */
    send(conn->sock, (void*)&reply, sizeof(reply), 0);

#if 1
    if (billboard_atom_needs_cleanup) { 
      /* remove ourselves from the availability list */
      /* we *MUST* do this after the ConnSetup, because the client has
         the display locked at this time, it will deadlock X if we
         XGrabServer here */
        Window *data;
        Atom returnType;
        unsigned long numAgents, bytesLeft;
        int format, i;

	billboard_atom_needs_cleanup = 0;
        fprintf(stderr, "attempting to grab server.\n");
        fflush(stderr);
        XGrabServer(main_dpy);
        fprintf(stderr, "grabbed server.\n");
        fflush(stderr);
        i = XGetWindowProperty(main_dpy, 
                               DefaultRootWindow(main_dpy),
                               billboard_atom,
			       0, /* starting offset */
			       ((((unsigned long) -1) >> 1) / 4), /* maximum to return */
			       False, /* don't delete */
			       XA_WINDOW,
			       &returnType,
			       &format,
			       &numAgents,
			       &bytesLeft,
			       (unsigned char **)&data);
        if (i == Success && returnType == XA_WINDOW) {
	    if (numAgents <= 1) {
	        fprintf(stderr, "deleteing property\n");
	        XDeleteProperty(main_dpy,
	                        DefaultRootWindow(main_dpy),
	    		        billboard_atom);
	    } else { 
	        fprintf(stderr, "found %lu agents\n", numAgents);
	        /* remove the agent from the list */
                for (i = 0; i < numAgents; i++) {
	            if (data[i] == main_win) {
		        fprintf(stderr, "found our window at %d\n", i);
		        numAgents--;
		        break;
		    }
		}
	                
	        for (i++; i < numAgents; i++) {
		    data[i-1] = data[i];
	        }
	        XChangeProperty(main_dpy, DefaultRootWindow(main_dpy),
	                        billboard_atom, XA_WINDOW, 32,
                                PropModeReplace,
		                (unsigned char *)data, numAgents);
	        fprintf(stderr, "Removed agent from list\n");
	    }
	}
	XUngrabServer(main_dpy);
	XFlush(main_dpy);
	fprintf(stderr, "done with property munging\n");
	fflush(stderr);
    } /* end removing property */
#endif

    return (req->length)*4;
}

/* code to deal with DPS/X11 input streams! */
private int
gsdpsx_std_read_process(stream_state *st, stream_cursor_read *ignore_pr,
  stream_cursor_write *pw, bool last)
{
	int count = pw->limit - pw->ptr;
        int tocopy, len;
        stream *s = (stream*)st;
        gsdpsx_conn_info * conn = (gsdpsx_conn_info *)(s->file);

	if ( count == 0 ) { 		/* empty buffer */
                dprintf("returning 1 from count == 0\n");
		return 1;
        }

        /* flush out any drawing commands in the queue */
        XFlush(conn->xdev->dpy);
        
        while (!(conn->inputstart)) {
            unsigned char *cur;
            int used = 0;

            if (conn->len && !conn->isfrag) {
                len = conn->len;
                cur = conn->curpos;
            } else {
                len = recv(conn->sock,
                           conn->curpos,
                           BUFLEN - (conn->curpos - conn->buf), 0);
                if (-1 == len) {
                    break;
                }
                len += conn->len;
                cur = conn->buf;
            }

            if (0) dprintf1("actually got data block of %d bytes.\n", len);

            if (!len)
               break;

            while (!(conn->inputstart) && len) {
                int dealt_with = 1;

                if (cur[0] == DPSXOPCODEBASE) {
                    switch(cur[1]) {
                    case X_PSCreateContext:
                    {
                        used = dps_CreateContext(conn, cur, len);
                        break;
                    }
                    case X_PSGiveInput:
                    {
                        xPSGiveInputReq *req = (xPSGiveInputReq*)cur;
                        used = (req->length)*4;
                        if (0) dprintf1("input %d bytes\n", req->nunits);
                        if (len < used) {
                            /* get the rest of the data */
                            if (1) dprintf2("incomplete message, len is %d, needed %d\n", (int)len, (int)used);
                            used = 0;
                            break;
                        }
                        conn->inputstart = (char*)(req+1);
                        conn->inputend = conn->inputstart + req->nunits;
                        break;
                    }
                    default:
                        dprintf1("received unknown DPSX message %X.\n",
                                 (unsigned)cur[1]);
                        dealt_with = 0;
                    }
                } else if (cur[0] == DPSCAPOPCODEBASE) {
                    switch(cur[1]) {
                    case X_CAPNotify:
                        used = dpscap_Notify(conn, cur, len);
                        break;
                    default:
                        dprintf1("received unknown DPSCAP message %X.\n",
                                 (unsigned)cur[1]);
                        dealt_with = 0;
                    }
                } else if (cur[0] == X_ChangeGC) {
                    xChangeGCReq *req = (xChangeGCReq*)cur;
                    if (0)
                      dprintf1("ignoring X_ChangeGC with mask %X\n",
                               (unsigned)req->mask);
                    /*xdev_gc_inline(conn->xdev)->gid = req->gc;*/
                    used = (req->length)*4;
                } else {
                    dealt_with = 0;
                }
                //if (1) dprintf3("dealt with: %d, used: %d, len: %d\n", dealt_with, used, len);
#if 1
                if (!dealt_with) {
//if(1) {
  if(!dealt_with)                    used = (((xPSGiveInputReq*)cur)->length)*4; /* remove all data */
                    /*! send error */
                    dprintf1("got data of %d bytes!\n", len);
                    {
                        int j,k;
                        for (j=0; j < used;) {
                            for (k = 0; k < 16; k++) {
                                if ((j+k)>=len)
                                    dprintf("   ");
                                else
                                    dprintf1("%02X ", cur[j+k]);
                                if (k == 7) dprintf("- ");
                                else if (k == 3 || k == 11) dprintf(" ");
                            }
                            dprintf("    ");
                            for (k = 0; k < 16; k++) {
                                unsigned char c = cur[j+k];
                                if ((j+k)<len)
                                    dprintf1("%c", (c >= 32 && c < 128) ? c : '.');
                                if (k == 7) dprintf("-");
                            }
                            dprintf("\n");
                            j += 16;
                        }
                    }
#endif /* dump unhandled messages */
                } /* if (!dealt_with) */

                /* if !used, then we had a controlled underflow, end of loop */
                /* if used < 0, then an error */
                if (used <= 0) {
                    break;
                }

                if (len-used < 0) {
                    dprintf("ERROR: buffer underflow! something went haywire!");
                    used = len;
                }

                cur += used;
                len -= used;
            } /* while len */

            /* error? */
            if (used < 0) {
                 /*** WRONG BEHAVIOR ***/
                dprintf1("error occured (used = %d).  Trying to continue.\n", (int)used);
                len = 0;
            }

            /* there was some leftover data, shuffle it around */
            if (len) {
                if (!(conn->inputstart)) {
                    dprintf1("Fragment of %d bytes\n", len);
                    memmove(conn->buf, cur, len);
                    conn->curpos = conn->buf + len;
                    conn->len = len;
                    conn->isfrag = 1;
                } else {
                    conn->len = len;
                    conn->curpos = cur;
                    conn->isfrag = 0;
                    if (0) dprintf1("queing %d bytes of messages for next trip!\n", len);
                }
            } else {
                conn->curpos = conn->buf;
                conn->len = 0;
                conn->isfrag = 0;
            }

            /* flush out any drawing commands in the queue */
	    XFlush(conn->xdev->dpy);
        } /* while data */

        if (!(conn->inputstart)) {
            dprintf("returning EOFC\n");
            /* EOF */
            return EOFC;
        }

        tocopy = min((conn->inputend)-(conn->inputstart), count);
        //tocopy = 1;
        memmove(pw->ptr+1, (conn->inputstart), tocopy);
        conn->inputstart += tocopy;
        if (conn->inputstart >= conn->inputend) {
            conn->inputstart = conn->inputend = NULL;
        }
        
	pw->ptr += tocopy;
        //dprintf1("returning %d\n", (count == tocopy) ? 1 : 0);
        return 1;
	//return (count == tocopy) ? 1 : 0;
}

/* code to deal with DPS/X11 result streams! */
private int
gsdpsx_std_write_process(stream_state *st, stream_cursor_read *pr,
  stream_cursor_write *ignore_pw, bool last)
{	uint count = pr->limit - pr->ptr;
        stream *s = (stream*)st;
        gsdpsx_conn_info * conn = (gsdpsx_conn_info *)(s->file);
        static Atom outputAtom = 0;

        if (!conn->xdev->dpy)
          return -1;

        if (!outputAtom)
          outputAtom = XInternAtom(conn->xdev->dpy, DPSCAP_TYPE_PSOUTPUT_LEN, False);

	while ( count != 0 )
	  {	int towrite = min(count, DPSCAP_DATA_LEN);
                XEvent event;
                DPSCAPOutputEvent *out;
                event.xclient.type = ClientMessage;
                event.xclient.serial = 0;
                event.xclient.send_event = True;
                event.xclient.message_type = outputAtom;
                event.xclient.format = 32;
                out = (DPSCAPOutputEvent*)event.xclient.data.l;
                out->cxid = conn->cxid;
                memcpy(out->data, pr->ptr+1, towrite);
                out->data[DPSCAP_DATA_LEN] = towrite;
                event.xclient.display = conn->xdev->dpy;
                event.xclient.window = conn->clientWindow;
                if (0) {
                    char data[21];
                    memcpy(data,pr->ptr+1,towrite);
                    data[towrite] = '\0';
                    dprintf2("sent %d bytes of results\n  ->%s<-\n", towrite, data);
                }
                XSendEvent(event.xclient.display, event.xclient.window, False, 0, &event);
                pr->ptr += towrite;
                count -= towrite;
	  }
        XSync(conn->xdev->dpy, False);
        return 0;
}

extern int iodev_stdin_open(P4(gx_io_device *, const char *, stream **,
				gs_memory_t *));
private int
gsdpsx_stdin_open(gx_io_device *iodev, const char *access, stream **ps,
  gs_memory_t *mem)
{	int code = iodev_stdin_open(iodev, access, ps, mem);
	stream *s = *ps;
	if ( code < 0 )
	  return code;
        //#error FLUSH THE QUEUE, maybe that will help
        //#error also check that we need all the below
	s->procs.process = gsdpsx_std_read_process;
        s->procs.seek = s_std_noseek;
        s->procs.available = s_std_noavailable;
        s->procs.switch_mode = s_std_switch_mode;
        s->procs.close = s_std_close;
        s->procs.flush = s_std_read_flush;
	s->file = (FILE*)current_conn; /***** GET RIGHT INFO! *****/
	return 0;
}

extern int iodev_stderr_open(P4(gx_io_device *, const char *, stream **,
				gs_memory_t *));
private int
gsdpsx_stderr_open(gx_io_device *iodev, const char *access, stream **ps,
  gs_memory_t *mem)
{	int code = iodev_stderr_open(iodev, access, ps, mem);
	stream *s = *ps;
	if ( code < 0 )
	  return code;
	s->procs.process = gsdpsx_std_write_process;
        s->procs.seek = s_std_noseek;
        s->procs.available = s_std_noavailable;
        s->procs.switch_mode = s_std_switch_mode;
        s->procs.close = s_std_close;
        s->procs.flush = s_std_write_flush;
	s->file = (FILE*)current_conn; /***** GET RIGHT INFO! *****/
	return 0;
}

extern int iodev_stdout_open(P4(gx_io_device *, const char *, stream **,
				gs_memory_t *));
private int
gsdpsx_stdout_open(gx_io_device *iodev, const char *access, stream **ps,
  gs_memory_t *mem)
{	int code = iodev_stdout_open(iodev, access, ps, mem);
	stream *s = *ps;
	if ( code < 0 )
	  return code;
	s->procs.process = gsdpsx_std_write_process;
        s->procs.seek = s_std_noseek;
        s->procs.available = s_std_noavailable;
        s->procs.switch_mode = s_std_switch_mode;
        s->procs.close = s_std_close;
        s->procs.flush = s_std_write_flush;
	s->file = (FILE*)current_conn; /***** GET RIGHT INFO! *****/
	return 0;
}

/* Patch stdout/err to use our stream. */
private void
gsdpsx_std_init(void)
{
    gs_findiodevice((const byte *)"%stdout", 7)->procs.open_device =
        gsdpsx_stdout_open;

    gs_findiodevice((const byte *)"%stderr", 7)->procs.open_device =
        gsdpsx_stderr_open;

    gs_findiodevice((const byte *)"%stdin", 6)->procs.open_device =
        gsdpsx_stdin_open;
}
