;;;***************************************************************
;;;An ACL2 Library of Floating Point Arithmetic

;;;David M. Russinoff
;;;Advanced Micro Devices, Inc.
;;;February, 1998
;;;***************************************************************

(in-package "ACL2")

(include-book "sticky")

;(local (in-theory (enable trunc-zero away-zero near-zero)))

(defun inf (x n)
  (if (>= x 0)
      (away x n)
    (trunc x n)))

(defun minf (x n)
  (if (>= x 0)
      (trunc x n)
    (away x n)))

;move back up?
(defthm trunc-x-not-acl2-numberp
  (implies (not (acl2-numberp x))
	   (= (trunc x n) 0))
	:hints (("Goal" :in-theory (enable trunc))))

(defthm away-x-not-acl2-numberp
  (implies (not (acl2-numberp x))
	   (= (away x n) 0))
	:hints (("Goal" :in-theory (enable away))))

(defthm inf-minus
  (= (inf (* -1 x) n) (* -1 (minf x n)))
  :rule-classes nil
  :hints(("goal" :cases ((acl2-numberp x)))
         ("Subgoal 2" :use (away-0 trunc-0))
         ("Subgoal 1" :use 
          (trunc-minus away-minus sig-minus))))

(defthm minf-minus
  (= (minf (* -1 x) n) (* -1 (inf x n)))
  :rule-classes nil
  :hints(("goal" :cases ((acl2-numberp x)))
         ("Subgoal 2" :use (trunc-0))
         ("Subgoal 1" :use 
          (trunc-minus away-minus sig-minus))))

(defthm inf-shift
  (implies (and (rationalp x)
                (integerp n)
                (integerp k))
           (= (inf (* x (expt 2 k)) n)
              (* (inf x n) (expt 2 k))))
  :hints (("Goal" :use (trunc-shift away-shift))))

(defthm minf-shift
  (implies (and (rationalp x)
                (integerp n)
                (integerp k))
           (= (minf (* x (expt 2 k)) n)
              (* (minf x n) (expt 2 k))))
  :hints (("Goal" :use (trunc-shift away-shift))))


(defun IEEE-MODE-P (mode)
  (member mode '(trunc inf minf near)))

(defun rounding-mode-p (mode)
  (or (IEEE-mode-p mode) (equal mode 'away)))

(defthm ieee-mode-p-implies-rounding-mode-p
  (implies (IEEE-mode-p mode)
           (rounding-mode-p mode))
  :rule-classes (:rewrite; :forward-chaining
))

(defun RND (x mode n)
  (case mode
    (away (away x n)) ;added by Eric in August, 2001
    (trunc (trunc x n))
    (inf (inf x n))
    (minf (minf x n))
    (near (near x n))))

(defun flip (m)
  (case m
    (inf 'minf)
    (minf 'inf)
    (t m)))

(defthm rnd-flip
  (implies (rounding-mode-p m)
           (= (rnd (* -1 x) (flip m) n)
              (* -1 (rnd x m n))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable rnd ieee-mode-p)
           :use (near-minus trunc-minus away-minus minf-minus inf-minus))))

(defun RND-CONST (e mode n)
  (case mode
    (near (expt 2 (- e n)))
    (inf (1- (expt 2 (1+ (- e n)))))
    (otherwise 0)))

(local (defthm rnd-const-thm-1
    (implies (and (integerp n)
		  (> n 1)
		  (integerp x)
		  (> x 0)
		  (>= (expo x) n))
	     (= (near x n)
		(if (and (exactp x (1+ n))
			 (not (exactp x n)))
		    (trunc (+ x (rnd-const (expo x) 'near n)) (1- n))
		  (trunc (+ x (rnd-const (expo x) 'near n)) n))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance near-trunc))))))

(local (defthm hack1
    (equal (+ (- (EXPO X)) -1 1 (EXPO X))
	   0)))

(local (defthm rnd-const-thm-2
    (implies (and (integerp n)
		  (> n 1)
		  (integerp x)
		  (> x 0)
		  (>= (expo x) n))
	     (= (away x n)
		(trunc (+ x (rnd-const (expo x) 'inf n)) n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable exactp2)
		  :use ((:instance away-imp (m (1+ (expo x)))))))))

(defthm RND-CONST-THM
    (implies (and (ieee-mode-p mode)
		  (integerp n)
		  (> n 1)
		  (integerp x)
		  (> x 0)
		  (>= (expo x) n))
	     (= (rnd x mode n)
		(if (and (eql mode 'near)
			 (exactp x (1+ n))
			 (not (exactp x n)))
		    (trunc (+ x (rnd-const (expo x) mode n)) (1- n))
		  (trunc (+ x (rnd-const (expo x) mode n)) n))))
  :rule-classes ()
  :hints (("Goal" :use (rnd-const-thm-1 rnd-const-thm-2))))



(defthm RND-STICKY
    (implies (and (rounding-mode-p mode)
		  (rationalp x) (> x 0)
		  (integerp k) (> k 0)
		  (integerp n) (> n (1+ k)))
	     (= (rnd x mode k)
		(rnd (sticky x n) mode k)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable sticky)
		  :use (sticky-pos
			(:instance trunc-sticky (m k))
			(:instance away-sticky (m k))
			(:instance near-sticky (m k))))))


(defthm rnd-shift
    (implies (and (rationalp x)
		  (integerp n)
		  (rounding-mode-p mode)
		  (integerp k))
	     (= (rnd (* x (expt 2 k)) mode n)
		(* (rnd x mode n) (expt 2 k))))
  :rule-classes ()
  :hints (("goal" :use (trunc-shift
			away-shift
			near-shift
                        inf-shift
                        minf-shift
;			(:instance expt-pos (x k))
                        ))))
;elim
(defthm expo-rnd
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0)
		  (rounding-mode-p mode)
		  (not (= (abs (rnd x mode n))
			  (expt 2 (1+ (expo x))))))
	     (= (expo (rnd x mode n))
		(expo x)))
  :rule-classes ()
  :hints (("goal" :in-theory (enable ieee-mode-p near rnd)
		  :use (expo-trunc expo-away))))

(defthm rnd-pos
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (rounding-mode-p mode))
	     (> (rnd x mode n) 0))
  :rule-classes ()
  :hints (("goal" :in-theory (enable ieee-mode-p near rnd)
		  :use (trunc-pos away-pos))))


(defthm rnd-0
  (implies (rounding-mode-p m)
           (equal (rnd 0 m n) 0))
  :hints (("Goal" :in-theory (enable rnd ieee-mode-p)
           :use (trunc-0 away-0))))


;from merge 4 below
(defthm rnd-neg
    (implies (and (rationalp x)
		  (< x 0)
		  (integerp n)
		  (> n 0)
		  (rounding-mode-p mode))
	     (< (rnd x mode n) 0))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable rnd ieee-mode-p)
		  :use (trunc-neg
			away-neg
			near-neg))))

(defthm rnd-exactp
    (implies (and (rationalp x)
		  (rounding-mode-p m)
		  (integerp n) 
		  (> n 0))
	     (iff (= x (rnd x m n))
		  (exactp x n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable rnd near inf minf ieee-mode-p)
		  :use (trunc-exactp-a
			away-exactp-a))))

;elim?
(defthm sgn-rnd
    (implies (and (rationalp x)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0))
	     (equal (sgn (rnd x mode n))
		    (sgn x)))
  :hints (("Goal" :in-theory (enable ieee-mode-p rnd)
		  :use (sgn-trunc
			sgn-away
			sgn-near-2))))


(defthm rnd-exactp-a
    (implies (and (rationalp x)
		  (rounding-mode-p mode)
		  (integerp n) 
		  (> n 0))
	     (iff (= x (rnd x mode n))
		  (exactp x n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable ieee-mode-p rnd)
		  :use (near-exactp-a
			trunc-exactp-a
			away-exactp-a))))

(defthm rnd-exactp-b
    (implies (and (rationalp x)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0))
	     (exactp (rnd x mode n) n))
  :hints (("Goal" :in-theory (enable ieee-mode-p rnd)
		  :use (near-exactp-b
			trunc-exactp-b
			away-exactp-b))))

(defthm rnd-exactp-c
    (implies (and (rationalp x)
		  (> x 0)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0)
		  (rationalp a)
		  (exactp a n)
		  (>= a x))
	     (>= a (rnd x mode n)))
  :hints (("Goal" :in-theory (enable ieee-mode-p rnd)
		  :use (near-exactp-c
			away-exactp-c
			trunc-upper-pos))))

(defthm rnd-exactp-d
    (implies (and (rationalp x)
		  (> x 0)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0)
		  (rationalp a)
		  (exactp a n)
		  (<= a x))
	     (<= a (rnd x mode n)))
  :hints (("Goal" :in-theory (enable ieee-mode-p rnd)
		  :use (near-exactp-c
			trunc-exactp-c
			away-lower-pos))))


(defthm rnd<=away
    (implies (and (rationalp x)
		  (> x 0)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0))
	     (<= (rnd x mode n) (away x n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable ieee-mode-p rnd)
		  :use (trunc-upper-pos
			away-lower-pos
			near-choice))))

(defthm rnd>=trunc
    (implies (and (rationalp x)
		  (> x 0)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0))
	     (>= (rnd x mode n) (trunc x n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable ieee-mode-p rnd)
		  :use (trunc-upper-pos
			away-lower-pos
			near-choice))))

(defthm monotone-rnd
    (implies (and (rationalp x)
		  (rationalp y)
		  (< 0 x)
		  (<= x y)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0))
	     (<= (rnd x mode n) (rnd y mode n)))
  :hints (("Goal" :in-theory (enable ieee-mode-p rnd)
		  :use (trunc-monotone
			away-monotone
			monotone-near))))

(defthm exactp-rnd
  (implies (and (rationalp x)
                (rounding-mode-p mode)
                (integerp n)
                (> n 0))
           (exactp (rnd x mode n) n))
  :hints (("Goal" :in-theory (enable ieee-mode-p rnd inf minf))))

(in-theory (disable rnd))

;drnd


(defun drnd (x mode n k)
  (- (rnd (+ x (* (sgn x) (expt 2 (- 2 (expt 2 (1- k)))))) mode n)
     (* (sgn x) (expt 2 (- 2 (expt 2 (1- k)))))))

(defthm drnd-flip
  (implies (and (rationalp x)
                (rounding-mode-p m)
                (integerp n)
                (integerp k))
           (= (drnd (* -1 x) (flip m) n k)
              (* -1 (drnd x m n k))))
  :rule-classes ()
  :hints (("Goal" :in-theory
           (set-difference-theories (enable rnd sgn)
                                    ;; disable forcing
                                    '((:executable-counterpart force)))
           :use ((:instance rnd-flip 
                            (x (+ x (* (sgn x) (expt 2 (- 2 (expt 2 (1- k))))))))))))

(in-theory (disable ieee-mode-p flip))

(defthm drnd-sticky-pos
    (implies (and (rounding-mode-p mode)
		  (natp n)
		  (> n 0)
		  (natp m)
		  (> m 1)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (> x 0)
		  (<= (expo x) (- 1 (expt 2 (1- k))))
		  (<= (expo x) (- m (+ n (expt 2 (1- k))))))
	     (equal (drnd (sticky x m) mode n k)
		    (drnd x mode n k)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable sgn)
		  :use (expo-upper-bound
			expo-lower-bound
			(:instance sticky-pos (n m))
			(:instance sticky-plus
				   (x (expt 2 (- 2 (expt 2 (1- k)))))
				   (y x)
				   (k m)
				   (k1 (- (+ m 2) (+ (expt 2 (1- k)) (expo x))))
				   (k2 (- (+ m 2) (+ (expt 2 (1- k)) (expo x)))))
			(:instance exactp-2**n 
				   (n (- 2 (expt 2 (1- k))))
				   (m (- (+ m 1) (+ (expt 2 (1- k)) (expo x)))))
			(:instance rnd-sticky
				   (x (+ x (expt 2 (- 2 (expt 2 (1- k))))))
				   (n (- (+ m 2) (+ (expt 2 (1- k)) (expo x))))
				   (k n))))))



(defthm drnd-0
  (implies (rounding-mode-p m)
           (equal (drnd 0 m n k) 0)))

(in-theory (disable drnd))

(defthm drnd-sticky-1
    (implies (and (rounding-mode-p mode)
		  (natp n)
		  (> n 0)
		  (natp m)
		  (> m 1)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (< x 0)
		  (<= (expo x) (- 1 (expt 2 (1- k))))
		  (<= (expo x) (- m (+ n (expt 2 (1- k))))))
	     (equal (drnd (sticky x m) mode n k)
		    (- (drnd (sticky (- x) m) (flip mode) n k))))
    :rule-classes ()
    :hints (("Goal" :in-theory (disable REARRANGE-NEGATIVE-COEFS-EQUAL)
		    :use ((:instance drnd-flip (x (sticky x m)) (m mode))
			  (:instance sticky-minus (n m))))))



(defthm ieee-mode-p-flip
    (implies (ieee-mode-p m)
	     (ieee-mode-p (flip m)))
    :hints (("Goal" :in-theory (enable ieee-mode-p flip))))

(defthm rounding-mode-p-flip
    (implies (rounding-mode-p m)
	     (rounding-mode-p (flip m)))
    :hints (("Goal" :in-theory (enable ieee-mode-p flip))))

(defthm drnd-sticky-2
    (implies (and (rounding-mode-p mode)
		  (natp n)
		  (> n 0)
		  (natp m)
		  (> m 1)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (< x 0)
		  (<= (expo x) (- 1 (expt 2 (1- k))))
		  (<= (expo x) (- m (+ n (expt 2 (1- k))))))
	     (equal (drnd (sticky x m) mode n k)
		    (- (drnd (- x) (flip mode) n k))))
    :rule-classes ()
    :hints (("Goal" :in-theory (disable REARRANGE-NEGATIVE-COEFS-EQUAL)
		    :use (expo-minus
			  drnd-sticky-1
			  (:instance drnd-sticky-pos (x (- x)) (mode (flip
                                                                      mode)))))))


(defthm drnd-sticky-3
    (implies (and (rounding-mode-p mode)
		  (natp n)
		  (> n 0)
		  (natp m)
		  (> m 1)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (< x 0)
		  (<= (expo x) (- 1 (expt 2 (1- k))))
		  (<= (expo x) (- m (+ n (expt 2 (1- k))))))
	     (equal (drnd (sticky x m) mode n k)
		    (drnd x mode n k)))
    :rule-classes ()
    :hints (("Goal" :in-theory (disable REARRANGE-NEGATIVE-COEFS-EQUAL)
		    :use ((:instance drnd-flip (m mode))
			  drnd-sticky-2))))

(defthm drnd-sticky
    (implies (and (rounding-mode-p mode)
		  (natp n)
		  (> n 0)
		  (natp m)
		  (> m 1)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (<= (expo x) (- 1 (expt 2 (1- k))))
		  (<= (expo x) (- m (+ n (expt 2 (1- k))))))
	     (equal (drnd (sticky x m) mode n k)
		    (drnd x mode n k)))
    :rule-classes ()
    :hints (("Goal" :use (drnd-sticky-pos
			  drnd-sticky-3))))


(defthm expo-rnd-bnd
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0)
		  (rounding-mode-p mode))
	     (>= (expo (rnd x mode n))
		 (expo x)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo-minus)
           :use (expo-rnd
			(:instance expo-minus (x (rnd x mode n)))))))

(defthm natp-compound-recognizer
  (equal (natp x)
         (and (integerp x)
              (>= x 0)))
  :rule-classes :compound-recognizer)

(local-defthm drnd-bnd-1
    (implies (and (natp n)
		  (> n 0)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (>= x (expt 2 (- 2 (expt 2 (1- k)))))
		  (< x (+ (expt 2 (- 2 (expt 2 (1- k)))) 
			  (expt 2 (- 3 (+ n (expt 2 (1- k))))))))
	     (equal (trunc x n)
		    (expt 2 (- 2 (expt 2 (1- k))))))
  :hints (("Goal" :in-theory (disable trunc-exactp-b trunc-exactp-c)
		  :use (trunc-exactp-b
			trunc-upper-pos
			(:instance expo+ (n (- 1 n)) (m (- 2 (expt 2 (1- k)))))
			(:instance trunc-exactp-c (a (expt 2 (- 2 (expt 2 (1- k))))))
			(:instance exactp-2**n (n (- 2 (expt 2 (1- k)))) (m n))
			(:instance fp+1
				   (y (trunc x n))
				   (x (expt 2 (- 2 (expt 2 (1- k))))))))))



(local-defthm drnd-bnd-2
    (implies (and (natp n)
		  (> n 0)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (>= x 0)
		  (< x (expt 2 (- 3 (+ n (expt 2 (1- k)))))))
	     (equal (drnd x 'trunc n k)
		    0))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable sgn rnd drnd)
           :use trunc-0)))



(local-defthm drnd-bnd-3
    (implies (and (natp n)
		  (> n 0)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (> x (expt 2 (- 2 (expt 2 (1- k)))))
		  (<= x (+ (expt 2 (- 2 (expt 2 (1- k)))) 
			   (expt 2 (- 3 (+ n (expt 2 (1- k))))))))
	     (equal (away x n)
		    (+ (expt 2 (- 2 (expt 2 (1- k)))) 
		       (expt 2 (- 3 (+ n (expt 2 (1- k))))))))
  :hints (("Goal" :in-theory (disable away-exactp-b away-exactp-c)
		  :use (away-exactp-b
			away-lower-pos
			(:instance expo+ (n (- 1 n)) (m (- 2 (expt 2 (1- k)))))
			(:instance away-exactp-c
				   (a (+ (expt 2 (- 2 (expt 2 (1- k))))
					 (expt 2 (- 3 (+ n (expt 2 (1- k))))))))
			(:instance exactp-2**n (n (- 2 (expt 2 (1- k)))) (m n))
			(:instance fp+2
				   (x (expt 2 (- 2 (expt 2 (1- k))))))
			(:instance fp+1
				   (y (away x n))
				   (x (expt 2 (- 2 (expt 2 (1- k))))))))))

(in-theory (enable inf))

(local-defthm drnd-bnd-4
    (implies (and (natp n)
		  (> n 0)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (> x 0)
		  (<= x (expt 2 (- 3 (+ n (expt 2 (1- k)))))))
	     (equal (drnd x 'inf n k)
		    (expt 2 (- 3 (+ n (expt 2 (1- k)))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable sgn rnd drnd))))

(local-defthm drnd-bnd-5
    (implies (and (natp n)
		  (> n 0)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (> x (expt 2 (- 2 (expt 2 (1- k)))))
		  (< x (+ (expt 2 (- 2 (expt 2 (1- k))))
			  (expt 2 (- 3 (+ n (expt 2 (1- k)))))))
		  (< x (+ (expt 2 (- 2 (expt 2 (1- k))))
			  (expt 2 (- 2 (+ n (expt 2 (1- k))))))))
	     (equal (near x n)
		    (expt 2 (- 2 (expt 2 (1- k))))))
  :rule-classes ()
  :hints (("Goal" :use (near1-a
			(:instance expo+ (m (- 2 (+ n (expt 2 (1- k))))) (n 1))))))

(local-defthm drnd-bnd-6
    (implies (and (natp n)
		  (> n 0)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (> x (expt 2 (- 2 (expt 2 (1- k)))))
		  (< x (+ (expt 2 (- 2 (expt 2 (1- k))))
			  (expt 2 (- 2 (+ n (expt 2 (1- k))))))))
	     (equal (near x n)
		    (expt 2 (- 2 (expt 2 (1- k))))))
  :hints (("Goal" :use (drnd-bnd-5
			(:instance expt-monotone
				   (n (- 2 (+ n (expt 2 (1- k)))))
				   (m (- 3 (+ n (expt 2 (1- k))))))))))

(local-defthm drnd-bnd-7
    (implies (and (natp n)
		  (> n 0)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (> x 0)
		  (< x (expt 2 (- 2 (+ n (expt 2 (1- k)))))))
	     (equal (drnd x 'near n k)
		    0))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable sgn rnd drnd))))

(local-defthm drnd-bnd-8
    (implies (and (ieee-mode-p m)
		  (natp n)
		  (> n 0)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (> x 0)
		  (< x (expt 2 (- 2 (+ n (expt 2 (1- k)))))))
	     (equal (drnd x m n k)
		    (if (eql m 'inf)
			(expt 2 (- 3 (+ n (expt 2 (1- k)))))
		      0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable ieee-mode-p sgn rnd drnd)
		  :use (drnd-bnd-2
			drnd-bnd-4
			drnd-bnd-7
			(:instance expt-monotone
				   (n (- 2 (+ n (expt 2 (1- k)))))
				   (m (- 3 (+ n (expt 2 (1- k))))))))))

(local-defthm drnd-bnd-9
    (implies (and (ieee-mode-p m)
		  (natp n)
		  (> n 0)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (> x 0)
		  (< x (expt 2 (- 2 (+ n (expt 2 (1- k))))))
		  (rationalp y)
		  (> y 0)
		  (< y (expt 2 (- 2 (+ n (expt 2 (1- k)))))))
	     (equal (drnd x m n k)
		    (drnd y m n k)))
  :rule-classes ()
  :hints (("Goal" :use (drnd-bnd-8
			(:instance drnd-bnd-8 (x y))))))

(defthm drnd-tiny-equal
    (implies (and (ieee-mode-p m)
		  (natp n)
		  (> n 0)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (< (abs x) (expt 2 (- 2 (+ n (expt 2 (1- k))))))
		  (rationalp y)
		  (< (abs y) (expt 2 (- 2 (+ n (expt 2 (1- k))))))
		  (equal (sgn x) (sgn y)))
	     (equal (drnd x m n k)
		    (drnd y m n k)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable sgn)
		  :use (drnd-flip
			(:instance drnd-flip (x y))
			(:instance drnd-bnd-9 (m (if (< x 0) (flip m) m)) (x (abs x)) (y (abs y)))))))