#!/bin/sh
#
# File: make/Makefile-libs.bash
#
# RAMS Library Path Computation - sets up library paths (creates Makefile.acl2
# files and Makefile.dirs files in RAMS project directories)
#
# Author: Jared Davis 

readonly SCRIPT="Makefile-libs.bash"
source $MAKEDIR/Makefile-checks.bash

readonly ACL2_OUT_FILE=`pwd`/"Makefile.acl2"
readonly MAKE_OUT_FILE=`pwd`/"Makefile.dirs"

Debug "Using ACL2_OUT_FILE=$ACL2_OUT_FILE"
Debug "Using MAKE_OUT_FILE=$MAKE_OUT_FILE"

if [[ "$QUIET" != "1" ]]
then
    echo -n "Setting up libraries, checking external dependencies..."
    DebugNewline
fi


# First accumulate and sort all DIRS entries, removing any duplicates, and
# standardizing them into lower case.  The results are left in $ENTRIES,
# separated by newlines.

TEMP=`mktemp /tmp/rams.XXXXXX` || exit 1
echo -n "" > $TEMP

if [ -e $MAKEDIR/DIRS ]
then
    if [ -e DIRS ]
    then
	LINES=`grep "^:" $MAKEDIR/DIRS && grep "^:" DIRS`
    else
	LINES=`grep "^:" $MAKEDIR/DIRS`
    fi
else
    LINES=""
fi

IFS="
"
for line in $LINES
do  
    COLON_PART=`echo $line | sed "s/[ \t]*=[^=]*//" | tr -t [:upper:] [:lower:]`
    echo $COLON_PART >> $TEMP
done


ENTRIES=`sort --unique $TEMP`

Debug "Entries are $ENTRIES"



# Now check to make sure that each library in LIBS has a corresponding entry in
# DIRS, signalling an error and quitting if this isn't the case.

IFS=" "
for lib in $LIBS
do
    if [ -n "`echo $ENTRIES | grep $lib`" ]
    then
	Debug "  $lib checks out ok."
    else
	echo ""
	echo ""
	echo "Error: $lib not found in DIRS or $MAKEDIR/DIRS."
	echo ""
	echo "  The Makefile indicates that $lib is a necessary external library,"
	echo "  but $lib is never mentioned in DIRS or $MAKEDIR/DIRS."
	echo ""	
	rm $TEMP
	exit 1
    fi
done


# Now go through each entry we had found, figure out what the "real" path of it
# is (i.e., the "final" line of the mythical MAKEDIR/DIRS + DIRS combination
# file), and resolve this path, storing it into a temporary file.

TEMP_ACL2OUT=`mktemp /tmp/rams.XXXXXX` || exit 1
TEMP_MAKEOUT=`mktemp /tmp/rams.XXXXXX` || exit 1

echo ";; $ACL2_OUT_FILE" > $TEMP_ACL2OUT
echo ";; " >> $TEMP_ACL2OUT
echo ";; This file is automatically generated.  Do not hand edit!" >> $TEMP_ACL2OUT
echo ";; Use \"make help\" for more information." >> $TEMP_ACL2OUT
echo "" >> $TEMP_ACL2OUT

echo ";; THE FOLLOWING IS BZO BZO BZO TERRIBLE FIXME BAD!!" >> $TEMP_ACL2OUT
echo ";; We add bozo-ld so that we can fake having ld with :dir support" >> $TEMP_ACL2OUT
echo ";; remove this in v2.9.3 and change all bozo-ld's to ld's." >> $TEMP_ACL2OUT
echo "" >> $TEMP_ACL2OUT
cat $MAKEDIR/bozo-ld.lisp >> $TEMP_ACL2OUT
echo "" >> $TEMP_ACL2OUT
echo "" >> $TEMP_ACL2OUT

echo "# $MAKE_OUT_FILE" > $TEMP_MAKEOUT
echo "# " >> $TEMP_MAKEOUT
echo "# This file is automatically generated.  Do not hand edit!" >> $TEMP_MAKEOUT
echo "# If you need to add dependencies, write them in the file " >> $TEMP_MAKEOUT
echo "# MOREDEPS instead!  See \"make help\" for more information." >> $TEMP_MAKEOUT
echo "" >> $TEMP_MAKEOUT

# The MAKE_OUT_FILE will always begin with its own directory name.
if [ -n "$NAME" ]
then
    MYNAME="ACL2_`echo $NAME | sed "s/://" | tr -t [:lower:] [:upper:]`_BOOKS"

    # We do this crazy trick now, so that when we process our own Makefile.deps,
    # our name goes away.  But when other people process our Makefile.deps, our
    # name stays there.  Thus, we get out-of-directory dependencies.

    echo "$MYNAME = `pwd -P`" >> $TEMP_MAKEOUT
else
    echo ""
    echo ""
    echo "No NAME found.  Please add a NAME line to your Makefile, e.g., "
    echo "NAME = :lists"
    echo ""
    exit 1
fi


IFS="
"
for entry in $ENTRIES 
do
    if [ -n "`echo $LIBS | grep $entry`" ]
    then
	Debug "Processing $entry"

	# Figure out where this entry should come from and set $LAST to be 
	# the line, e.g., :bags = ../bags, and set $BASE_PATH to be the path
	# to the DIRS file that $LAST comes from.

	if [ -e DIRS ]
        then
	    echo -n "" > $TEMP
	    grep -i "^$entry" DIRS >> $TEMP
	    LAST=`tail -n 1 $TEMP`
        else
	    LAST=""
	fi

	if [ -n "$LAST" ] 
	then
	    Debug "LAST=$LAST comes from DIRS"
	    BASE_PATH=`pwd`
	else
	    if [ -e $MAKEDIR/DIRS ]
	    then
		echo -n "" > $TEMP
		grep -i "^$entry" $MAKEDIR/DIRS >> $TEMP
		LAST=`tail -n 1 $TEMP`
		Debug "LAST=$LAST comes from $MAKEDIR/DIRS"	    
		BASE_PATH=$MAKEDIR
	    fi
	fi

	if [ ! "$LAST" ]
	then
	    echo ""
	    echo "Fatal Internal error in RAMS Makefile system."
	    echo "Entry $entry has not been found in DIRS or $MAKEDIR/DIRS."
	    echo "This should never happen."
	    echo ""
	    rm -f $TEMP $TEMP_ACL2OUT $TEMP_MAKEOUT
	    exit 1
	fi
	

	# Split $LAST into a colon part (e.g., :bags) and a path part, (e.g.,
	# ../bags), and then check to make sure that the path exists.
	
        COLON_PART=`echo $LAST | sed "s/[ \t]*=[^=]*//"`
        PATH_PART=`echo $LAST | sed "s/[^=]*=[ \t]*//"`

	if [ ! -d $BASE_PATH/$PATH_PART ]
	then
	    echo ""
	    echo "Error: $BASE_PATH/$PATH_PART is not a directory."
	    echo "  According to the DIRS files, this is the location of the "
	    echo "  $COLON_PART library, which the Makefile indicates is necessary."
	    echo "  Perhaps you have not checked out the $COLON_PART module from CVS?"
	    echo ""
	    rm -f $TEMP $TEMP_ACL2OUT $TEMP_MAKEOUT
	    exit 1
	fi

	# Recursively go into $BASE_PATH/$PATH_PART and invoke the make script
	# to ensure that everything will build.

	if [[ "$BASE_PATH/$PATH_PART" == "`pwd -P`" ]]
        then
	    echo "Error: Circular dependence to $COLON_PART detected"
	    exit 1
	elif [ -z "$NORECUR" ]
	    then
	    Debug "Recurring into $BASE_PATH/$PATH_PART."
	    cd $BASE_PATH/$PATH_PART; make -s deps QUIET=1	    
	fi

	



        # We now resolve the path: we remove all indirection and eliminate
        # symlinks in order to get the real location of this directory.
        #
        # Continuing our example where line is ":bags = ../bags",
        # The path part is ../bags, so if we are in, say, /home/jared/foo,
        # then the RESOLVED_PATH should be /home/jared/bags.
        
	RESOLVED_PATH="`cd $BASE_PATH/$PATH_PART; pwd -P`"
	    
        # We now compute the book name as it will appear in Makefiles.  We just
        # drop the colon from the COLON_PART and replace it with its upper-case
        # counterpart, e.g., "BAGS" in our example.

	BOOK_NAME="ACL2_`echo $COLON_PART | sed "s/://" | tr -t [:lower:] [:upper:]`_BOOKS"

        # Write the line to our temporary ACL2 and MAKE files.

        # Note: it's important that this path ends in a slash, so we manually
        # insert one.  Otherwise, ACL2 will be upset when it processes the 
        # add-include-book-dir commands.

	echo "(add-include-book-dir $COLON_PART \"$RESOLVED_PATH/\")" >> $TEMP_ACL2OUT

        # Also echo the line into what will become a Makefile.dirs file.  Make
	# absolutely sure you include the trailing slash here!

	echo "$BOOK_NAME = $RESOLVED_PATH" >> $TEMP_MAKEOUT

        Debug "Added $BOOK_NAME --> $RESOLVED_PATH"

    else
	Debug "Ignoring '$entry' since it is not in LIBS."
    fi
done

rm -f $TEMP



# We can now compare our temporary file to the real ACL2_OUT_FILE and see if
# any changes are necessary.  If so, we will install our new temporary file.

if [ -e $ACL2_OUT_FILE ]
then
    Debug "Comparing temp file with existing $ACL2_OUT_FILE"

    if [ "`diff $ACL2_OUT_FILE $TEMP_ACL2OUT`" ]
    then
	mv $TEMP_ACL2OUT $ACL2_OUT_FILE
	mv $TEMP_MAKEOUT $MAKE_OUT_FILE

	if [[ "$QUIET" != "1" ]]
	then
	    echo " Done (Some locations have changed)."
	else
	    echo ""
	    echo -n "   Note: Updated library links for $NAME"
	fi
    else
	rm -f $TEMP_ACL2OUT $TEMP_MAKEOUT

	if [[ "$QUIET" != "1" ]]
	then
	    echo " Done (No changes.)"
	fi
    fi
else
    Debug "Installing temp file since no pre-existing $ACL2_OUT_FILE"
    mv $TEMP_ACL2OUT $ACL2_OUT_FILE
    mv $TEMP_MAKEOUT $MAKE_OUT_FILE

    if [[ "$QUIET" != "1" ]]
    then
	echo " Done (First make)."
    else
	echo ""
	echo -n "   Note: Created initial library links for $NAME"
    fi
fi

