(in-package "ACL2")

(include-book "regex-parse")
(include-book "regex-exec")
(include-book "regex-chartrans")


;; (defun convert-matches (matches str)
;;   (if (endp matches)
;;       nil
;;     (cons (cons (coerce (firstn (- (len str) 
;;                                    (len (strip-begin (caar matches))))
;;                                 str)
;;                         'string)
;;                 (coerce-strings (cddar matches)))
;;           (convert-matches (cdr matches) str))))

(defmacro case-sens-bindings (body)
  `(let ((untrans-reg reg)
         (untrans-str str)
         (reg (if (parse-options-case-insensitive opts)
                  (apply-translation reg *case-insens-translation*)
                reg))
         (str (if (parse-options-case-insensitive opts)
                  (apply-translation str *case-insens-translation*)
                str)))
     ,body))

(set-ignore-ok t)

(defun all-matches-fn (reg str opts)
  (case-sens-bindings
    (run-regex (regex-do-parse reg opts) str 0 nil)))

(defun longest-matches (matches)
  (if (endp matches)
      nil
    (let ((rest (longest-matches (cdr matches))))
      (cond ((= (length (caar matches))
                (length (caar rest)))
             (cons (car matches) rest))
            ((< (length (caar matches))
                (length (caar rest)))
             rest)
            (t (list (car matches)))))))

;; Return all matches to prefixes of str.
;; Like raw run-regex but parses the regex and 
;; takes and returns strings instead of lists.
(defmacro all-matches (reg str &optional opts)
  (let ((opts (if opts opts ''(t t nil nil))))
         `(all-matches-fn ,reg ,str ,opts)))
         
(defmacro all-longest-matches (reg str &optional opts)
  `(longest-matches (all-matches ,reg ,str ,opts)))

;; Returns the "best" match of reg in str.
;; Inputs are real strings, unparsed, unexploded.
;; Returns an mv of
;; 0 and the best match if there's a match
;; 1 and nil if no match
;; 2 and the error message on error.
(defun run-regex-test (str reg opts)
  (declare (xargs :guard (and (stringp str)
                              (stringp reg)
                              (parse-opts-p opts))))
  (case-sens-bindings
   (let ((regex (regex-do-parse reg opts)))
     (if (stringp regex) 
         (mv 2 regex nil)
       (mv-let (succ match brs) (match-regex regex str untrans-str)
               (if succ
                   (mv 0 match brs)
                 (mv 1 nil brs)))))))


(in-theory (disable run-regex-test))


;; Returns the prefix of the list l
;; up to the first occurrence of element e,
;; and the rest of the list
(defun up-to-first (l e)
  (declare (xargs :guard (true-listp l)))
  (if (endp l)
      (mv nil nil)
    (if (equal (car l) e)
        (mv nil l)
      (mv-let (pre rest)
              (up-to-first (cdr l) e)
              (mv (cons (car l) pre) rest)))))


;;Return the list of sublists of l
;;delimited by element e
(defun separate-by-delim (l e)
  (declare (xargs :guard (true-listp l)
                  :measure (len l)))
  (if (endp l)
      nil
    (mv-let (first rest) (up-to-first l e)
            (cons first
                  (separate-by-delim (cdr rest) e)))))



;; Runs a regex test case and prints the content if it's wrong.
;; num - identifier for the test
;; ans - 0, 1, or 2 depending whether it's supposed to match, not match, or error
;;        (so far error is considered same as no match)
;; str, reg - unparsed, unexploded strings
(defun do-test (num ans str reg opts)
  (declare (xargs :guard (and (natp ans)
                              (stringp reg)
                              (stringp str)
                              (parse-opts-p opts))))
  (mv-let (code match brs) (run-regex-test str reg opts)
          (declare (ignore brs))
          (if (equal code ans)
              0
            (prog2$ (cw "~c0) ~p1 ~p2 ~s3~t4~s5~t6~s7~%" 
                        `(,num . 4) ans code str 20 reg 40 match) 
                    1))))


(defun regex-other-testp (x)
  (declare (xargs :guard t))
  (and (true-listp x)
       (integerp (nth 0 x))
       (integerp (nth 1 x))
       (stringp (nth 2 x))
       (stringp (nth 3 x))
       (parse-opts-p (nth 4 x))
       (implies (nth 5 x) (stringp (nth 5 x)))
       (implies (nth 6 x) (stringp (nth 6 x)))))


(defthm regex-other-testp-def
  (implies (regex-other-testp x)
           (and (true-listp x)
                (integerp (nth 0 x))
                (integerp (nth 1 x))
                (stringp (nth 2 x))
                (stringp (nth 3 x))
                (parse-opts-p (nth 4 x))
                (implies (nth 5 x) (stringp (nth 5 x)))
                (implies (nth 6 x) (stringp (nth 6 x))))))


(in-theory (disable regex-other-testp))

(defun do-other-test (test)
  (declare (xargs :guard (regex-other-testp test)
; Added by Matt K. after ACL2 Version 3.6.1 for change in too-many-ifs
; heuristic:
                  :guard-hints
                  (("Goal" :in-theory (enable regex-other-testp)))))
   (let* ((num (nth 0 test))
          (ans (nth 1 test))
          (reg (nth 2 test))
          (str (nth 3 test))
          (opts (nth 4 test)))
     (case-sens-bindings
      (let* ((ematch (nth 5 test))
             (ebrs  (if (nth 6 test)
                        (separate-by-delim (coerce (nth 6 test) 'list) #\,)
                      nil))
             (regex (regex-do-parse reg opts)))
        (if (stringp regex)
            (if (equal ans 2)
                0
              (prog2$ (cw "~c0) ~s1 ~s2~%" `(,num . 4)  (nth 3 test) reg)
                      1))
          (mv-let (succ match brs) (match-regex regex str untrans-str)
                  (if (and (if succ (equal ans 0) (equal ans 1))
                           (if ematch (equal match ematch) t)
                           (if ebrs (equal brs ebrs) t))
                      0
                    (prog2$ (cw "~c0) ~p1 ~s2~t3~s4~t5~s6~t7~x8~%"
                                `(,num . 4) ans (nth 3 test) 20 reg 40 match 60 
                                brs)
                            1))))))))

(in-theory (disable do-other-test))

(defun regex-other-test-listp (x)
  (declare (xargs :guard t))
  (if (atom x) (equal x nil)
    (and (regex-other-testp (car x))
         (regex-other-test-listp (cdr x)))))

(defun do-other-test-list (testlist)
  (declare (xargs :guard (regex-other-test-listp testlist)))
  (if (endp testlist)
      0
    (+ (do-other-test (car testlist))
       (do-other-test-list (cdr testlist)))))
              


;; is x a valid regex test,
;; i.e. a list consisting of the four arguments to do-test
(defun regex-testp (x)
  (declare (xargs :guard t))
  (and (equal (len x) 4)
       (integerp (car x))
       (natp (cadr x))
       (stringp (caddr x))
       (stringp (cadddr x))))

;; is x a list of regex test cases (recognized by regex-testp)
(defun regex-test-listp (x)
  (declare (xargs :guard t))
  (if (atom x)
      (equal x nil)
    (and (regex-testp (car x))
         (regex-test-listp (cdr x)))))


;; Run do-test on a list of test cases
(defun do-test-list (opts testlist)
  (declare (xargs :guard (and (regex-test-listp testlist)
                              (parse-opts-p opts))
                  :measure (len testlist)))
  (if (endp testlist)
      0
    (+ (do-test (caar testlist)
                (cadar testlist)
                (caddar testlist)
                (car (cdddar testlist))
                opts)
       (do-test-list opts (cdr testlist)))))




(defconst *bre-tests*
  '((1 0 "abc" "a\\(b\\)c")
    (2 0 "a(" "a(")
    (3 2 "EPAREN" "a\\(")
    (4 2 "EPAREN" "a\\(b")
    (5 0 "a(b" "a(b")
    (6 0 "a)" "a)")
    (7 2 "EPAREN" "a\\)")
    (8 2 "EPAREN" "\\)")
    (9 0 "ab" "a\\(\\)b")
    (10 0 "a$b" "a$b")
    (11 0 "b" "a*\\(^b$\\)c*")
    (12 0 "|" "|")
    (13 0 "*" "*")
    (14 0 "abc" "\\(\\)")
    (15 0 "*" "^*")
    (16 0 "{" "\\{")
    (17 1 "abbcbd" "a\\(b*\\)c\\1d")
    (18 1 "abbcbbbd" "a\\(b*\\)c\\1d")
    (19 1 "abc" "^\\(.\\)\\1")
    (20 0 "abbccd" "a\\(\\([bc]\\)\\2\\)*d")
    (21 1 "abbcbd" "a\\(\\([bc]\\)\\2\\)*d")
    (22 0 "abbbd" "a\\(\\(b\\)*\\2\\)*d")
    (23 0 "aabcd" "\\(a\\)\\1bcd")
    (24 0 "aabcd" "\\(a\\)\\1bc*d")
    (25 0 "aabd" "\\(a\\)\\1bc*d")
    (26 0 "aabcccd" "\\(a\\)\\1bc*d")
    (27 0 "aabcccd" "^\\(a\\)\\1b\\(c\\)*cd$")
    (28 0 "a*b" "a\\(*\\)b")
    (29 0 "ab" "a\\(**\\)b")
    (30 0 "*a" "*a")
    (31 0 "a" "**a")
    (32 0 "ab" "a\\{1\\}b")
    (33 0 "ab" "a\\{1,\\}b")
    (34 0 "aab" "a\\{1,2\\}b")
    (35 2 "EBRACE" "a\\{1")
    (36 2 "EBRACE" "a\\{1a")
    (37 2 "BADBR" "a\\{1a\\}")
    (38 2 "BADBR" "a\\{,2\\}")
    (39 2 "BADBR" "a\\{,\\}")
    (40 2 "BADBR" "a\\{1,x\\}")
    (41 2 "EBRACE" "a\\{1,x")
    (42 2 "BADBR" "a\\{32768\\}")
    (43 2 "BADBR" "a\\{1,0\\}")
    (44 0 "abcac" "ab\\{0,0\\}c")
    (45 0 "abcac" "ab\\{0,1\\}c")
    (46 0 "abbcac" "ab\\{0,3\\}c")
    (47 0 "acabc" "ab\\{1,1\\}c")
    (48 0 "acabc" "ab\\{1,3\\}c")
    (49 0 "abcabbc" "ab\\{2,2\\}c")
    (50 0 "abcabbc" "ab\\{2,4\\}c")
    (51 1 "acd" "a\\(b\\)?c\\1d")
    (52 0 "-5" "-\\{0,1\\}[0-9]*$")
    ))

(defconst *ere-tests*
  '((1 0 "a" "a")
    (2 0 "abc" "abc")
    (3 0 "abc" "abc|de")
    (4 0 "abc" "a|b|c")
    (5 0 "abc" "a(b)c")
    (6 2 "EPAREN" "a(")
    (7 0 "a(" "a\\(")
    (8 2 "EPAREN" "a(b")
    (9 0 "ab" "a()b")
    (10 0 "abc" "^abc$")
    (11 1 "a^b" "a^b")
    (12 1 "a$b" "a$b")
    (13 0 "abc" "^")
    (14 0 "abc" "$")
    (15 0 "b" "a*(^b$)c*")
    (16 1 "PASS" "&C")
    (17 0 "abc" "()")
    (18 0 "abc" "a.c")
    (19 0 "abd" "a[bc]d")
    (20 0 "a*c" "a\\*c")
    (21 0 "a[b" "a\\[b")
    (22 2 "EBRACK" "a[b")
    (23 0 "a" "a$")
    (24 1 "a$" "a$")
    (25 1 "a" "a\\\\$")
    (26 0 "abc" "ab*c")
    (27 0 "abc" "ab+c")
    (28 0 "abc" "ab?c")
    (29 0 "{1" "{1")
    (30 0 "ab" "a{1}b")
    (31 0 "ab" "a{1,}b")
    (32 0 "aab" "a{1,2}b")
    (33 0 "a{1" "a{1")
    (34 1 "aa" "a{1a")
    (35 0 "a{1a}" "a{1a}")
    (36 0 "a{,2}" "a{,2}")
    (37 0 "a{,}" "a{,}")
    (38 0 "a{1,,,}" "a{1,*}")
    (39 0 "abcac" "ab{0,0}c")
    (40 0 "abcac" "ab{0,1}c")
    (41 0 "abbcac" "ab{0,3}c")
    (42 0 "acabc" "ab{1,1}c")
    (43 0 "acabc" "ab{1,3}c")
    (44 0 "abcabbc" "ab{2,2}c")
    (45 0 "abcabbc" "ab{2,4}c")
    (46 0 "abc" "a[b]c")
    (47 0 "abc" "a[ab]c")
    (48 0 "adc" "a[^ab]c")
    (49 0 "a]c" "a[]b]c")
    (50 0 "a[c" "a[[b]c")
    (51 0 "a-c" "a[-b]c")
    (52 0 "adc" "a[^]b]c")
    (53 0 "adc" "a[^-b]c")
    (54 0 "a-c" "a[b-]c")
    (55 2 "EBRACK" "a[b")
    (56 2 "EBRACK" "a[]")
    (57 0 "a2c" "a[1-3]c")
    (58 2 "ERANGE" "a[1-")
    (59 2 "EBRACK" "a[[.")
    (60 2 "EBRACK" "a[[.x")
    (61 2 "EBRACK" "a[[.x.")
    (62 0 "abc" "a[[:alpha:]]c")
    (63 2 "ECTYPE" "a[[:notdef:]]c")
    (64 2 "EBRACK" "a[[:")
    (65 2 "EBRACK" "a[[:alpha")
    (66 2 "EBRACK" "a[[:alpha:]")
    (67 2 "ECTYPE" "a[[:alpha,:]")
    (68 2 "ECTYPE" "a[[:]:]]b")
    (69 2 "ECTYPE" "a[[:-:]]b")
    (70 2 "ECTYPE" "a[[:alph:]]")
    (71 2 "ECTYPE" "a[[:alphabet:]]")
    (72 0 "a019b" "[[:digit:]]+")
    (73 0 "AabC" "[[:lower:]]+")
    (74 0 "aBCd" "[[:upper:]]+")
    (75 0 "p0f3Cq" "[[:xdigit:]]+")
    (76 2 "EBRACK" "a[[=")
    (77 2 "EBRACK" "a[[=b")
    (78 2 "EBRACK" "a[[=b=")
    (79 0 "abc" "a(((b)))c")
    (80 0 "abd" "a(b|(c))d")
    (81 0 "abbd" "a(b*|c)d")
    (82 0 "aaaaabaaaabaaaabaaaab" "a[ab]{20}")
    (83 0 "aaaaabaaaabaaaabaaaab" "a[ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab]")
    (84 0 "aaaaabaaaabaaaabaaaabweeknights" "a[ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab](wee|week)(knights|night)")
    (85 0 "a12345678901234567890123456789b" "12345678901234567890123456789")
    (86 0 "a123456789012345678901234567890b" "123456789012345678901234567890")
    (87 0 "a1234567890123456789012345678901b" "1234567890123456789012345678901")
    (88 0 "a12345678901234567890123456789012b" "12345678901234567890123456789012")
    (89 0 "a123456789012345678901234567890123b" "123456789012345678901234567890123")
    (90 0 "a1234567890123456789012345678901234567890123456789012345678901234567890b" "1234567890123456789012345678901234567890123456789012345678901234567890")
    (91 0 "xacegikmoq" "[ab][cd][ef][gh][ij][kl][mn]")
    (92 0 "xacegikmoq" "[ab][cd][ef][gh][ij][kl][mn][op]")
    (93 0 "xacegikmoqy" "[ab][cd][ef][gh][ij][kl][mn][op][qr]")
    (94 0 "xacegikmoqy" "[ab][cd][ef][gh][ij][kl][mn][op][q]")
    (95 0 "xabcy" "abc")
    (96 1 "abc" "a[^b]c")
    (97 0 "adc" "a[^b]c")
    (98 0 "abc" "[a]b[c]")
    (99 0 "aba" "[a]b[a]")
    (100 0 "abc" "[abc]b[abc]")
    (101 0 "abd" "[abc]b[abd]")
    (102 0 "accd" "a(b?c)+d")
    (103 0 "weeknights" "(wee|week)(knights|night)")
    (104 0 "weeknights" "(we|wee|week|frob)(knights|night|day)")
    (105 0 "xyzaaabcaababdacd" "a[bc]d")
    (106 0 "aaabc" "a[ab]c")
    (107 0 "b" "a*")
    (108 0 "/*x*/" "/\\*.*\\*/")
    (109 0 "/*x*/y/*z*/" "/\\*.*\\*/")
    (110 0 "/*x*/" "/\\*([^*]|\\*[^/])*\\*/")
    (111 0 "/*x*/y/*z*/" "/\\*([^*]|\\*[^/])*\\*/")
    (112 0 "/*x**/y/*z*/" "/\\*([^*]|\\*[^/])*\\*/")
    (113 0 "/*x*/" "/\\*([^*]|\\*+[^*/])*\\*+/")
    (114 0 "/*x*/y/*z*/" "/\\*([^*]|\\*+[^*/])*\\*+/")
    (115 0 "/*x**/y/*z*/" "/\\*([^*]|\\*+[^*/])*\\*+/")
    (116 0 "/*x****/y/*z*/" "/\\*([^*]|\\*+[^*/])*\\*+/")
    (117 0 "/*x**x*/y/*z*/" "/\\*([^*]|\\*+[^*/])*\\*+/")
    (118 0 "/*x***x/y/*z*/" "/\\*([^*]|\\*+[^*/])*\\*+/")
    (119 0 "A1" "(A[1])|(A[2])|(A[3])|(A[4])|(A[5])|(A[6])|(A[7])|(A[8])|(A[9])|(A[A])")
    (120 0 "abcdefghijklmnop" "abcdefghijklmnop")
    (121 0 "abcdefghijklmnopqrstuv" "abcdefghijklmnopqrstuv")
    (122 0 "CC11" "CC[13]1|a{21}[23][EO][123][Es][12]a{15}aa[34][EW]aaaaaaa[X]a")
    (123 0 "ab" "a?b")
    ))


(defconst *khadafy-tests*
  (let ((kregex "M[ou]'?am+[ae]r .*([AEae]l[- ])?[GKQ]h?[aeu]+([dtz][dhz]?)+af[iy]"))
    `((1 0 "Muammar Qaddafi" ,kregex)
      (2 0 "Mo'ammar Gadhafi" ,kregex)
      (3 0 "Muammar Kaddafi" ,kregex)
      (4 0 "Muammar Qadhafi" ,kregex)
      (5 0 "Moammar El Kadhafi" ,kregex)
      (6 0 "Muammar Gadafi" ,kregex)
      (7 0 "Mu'ammar al-Qadafi" ,kregex)
      (8 0 "Moamer El Kazzafi" ,kregex)
      (9 0 "Moamar al-Gaddafi" ,kregex)
      (10 0 "Mu'ammar Al Qathafi" ,kregex)
      (11 0 "Muammar Al Qathafi" ,kregex)
      (12 0 "Mo'ammar el-Gadhafi" ,kregex)
      (13 0 "Moamar El Kadhafi" ,kregex)
      (14 0 "Muammar al-Qadhafi" ,kregex)
      (15 0 "Mu'ammar al-Qadhdhafi" ,kregex)
      (16 0 "Mu'ammar Qadafi" ,kregex)
      (17 0 "Moamar Gaddafi" ,kregex)
      (18 0 "Mu'ammar Qadhdhafi" ,kregex)
      (19 0 "Muammar Khaddafi" ,kregex)
      (20 0 "Muammar al-Khaddafi" ,kregex)
      (21 0 "Mu'amar al-Kadafi" ,kregex)
      (22 0 "Muammar Ghaddafy" ,kregex)
      (23 0 "Muammar Ghadafi" ,kregex)
      (24 0 "Muammar Ghaddafi" ,kregex)
      (25 0 "Muamar Kaddafi" ,kregex)
      (26 0 "Muammar Quathafi" ,kregex)
      (27 0 "Muammar Gheddafi" ,kregex)
      (28 0 "Muamar Al-Kaddafi" ,kregex)
      (29 0 "Moammar Khadafy" ,kregex)
      (30 0 "Moammar Qudhafi" ,kregex)
      (31 0 "Mu'ammar al-Qaddafi" ,kregex)
      (32 0 "Mulazim Awwal Mu'ammar Muhammad Abu Minyar al-Qadhafi" ,kregex)
      )))

(defconst *spencer1-tests*
  '((1 0 "abc" "abc")
    (2 1 "xbc" "abc")
    (3 1 "axc" "abc")
    (4 1 "abx" "abc")
    (5 0 "xabcy" "abc")
    (6 0 "ababc" "abc")
    (7 0 "abc" "ab*c")
    (8 0 "abc" "ab*bc")
    (9 0 "abbc" "ab*bc")
    (10 0 "abbbbc" "ab*bc")
    (11 0 "abbc" "ab+bc")
    (12 1 "abc" "ab+bc")
    (13 1 "abq" "ab+bc")
    (14 0 "abbbbc" "ab+bc")
    (15 0 "abbc" "ab?bc")
    (16 0 "abc" "ab?bc")
    (17 1 "abbbbc" "ab?bc")
    (18 0 "abc" "ab?c")
    (19 0 "abc" "^abc$")
    (20 1 "abcc" "^abc$")
    (21 0 "abcc" "^abc")
    (22 1 "aabc" "^abc$")
    (23 0 "aabc" "abc$")
    (24 0 "abc" "^")
    (25 0 "abc" "$")
    (26 0 "abc" "a.c")
    (27 0 "axc" "a.c")
    (28 0 "axyzc" "a.*c")
    (29 1 "axyzd" "a.*c")
    (30 1 "abc" "a[bc]d")
    (31 0 "abd" "a[bc]d")
    (32 1 "abd" "a[b-d]e")
    (33 0 "ace" "a[b-d]e")
    (34 0 "aac" "a[b-d]")
    (35 0 "a-" "a[-b]")
    (36 0 "a-" "a[b-]")
    (37 1 "-" "a[b-a]")
    (38 2 "-" "a[]b")
    (39 2 "-" "a[")
    (40 0 "a]" "a]")
    (41 0 "a]b" "a[]]b")
    (42 0 "aed" "a[^bc]d")
    (43 1 "abd" "a[^bc]d")
    (44 0 "adc" "a[^-b]c")
    (45 1 "a-c" "a[^-b]c")
    (46 1 "a]c" "a[^]b]c")
    (47 0 "adc" "a[^]b]c")
    (48 0 "abc" "ab|cd")
    (49 0 "abcd" "ab|cd")
    (50 0 "def" "()ef")
    (51 0 "-" "()*")
    (52 1 "-" "*a")
    (53 0 "-" "^*")
    (54 0 "-" "$*")
    (55 1 "-" "(*)b")
    (56 1 "b" "$b")
    (57 2 "-" "a\\")
    (58 0 "a(b" "a\\(b")
    (59 0 "ab" "a\\(*b")
    (60 0 "a((b" "a\\(*b")
    (61 1 "a\\x" "a\\x")
    (62 2 "-" "abc)")
    (63 2 "-" "(abc")
    (64 0 "abc" "((a))")
    (65 0 "abc" "(a)b(c)")
    (66 0 "aabbabc" "a+b+c")
    (67 0 "-" "a**")
    (68 0 "-" "a*?")
    (69 0 "-" "(a*)*")
    (70 0 "-" "(a*)+")
    (71 0 "-" "(a|)*")
    (72 0 "-" "(a*|b)*")
    (73 0 "ab" "(a+|b)*")
    (74 0 "ab" "(a+|b)+")
    (75 0 "ab" "(a+|b)?")
    (76 0 "cde" "[^ab]*")
    (77 0 "-" "(^)*")
    (78 0 "-" "(ab|)*")
    (79 2 "-" ")(")
    (80 1 "" "abc")
    (81 1 "" "abc")
    (82 0 "" "a*")
    (83 0 "abbbcd" "([abc])*d")
    (84 0 "abcd" "([abc])*bcd")
    (85 0 "e" "a|b|c|d|e")
    (86 0 "ef" "(a|b|c|d|e)f")
    (87 0 "-" "((a*|b))*")
    (88 0 "abcdefg" "abcd*efg")
    (89 0 "xabyabbbz" "ab*")
    (90 0 "xayabbbz" "ab*")
    (91 0 "abcde" "(ab|cd)e")
    (92 0 "hij" "[abhgefdc]ij")
    (93 1 "abcde" "^(ab|cd)e")
    (94 0 "abcdef" "(abc|)ef")
    (95 0 "abcd" "(a|b)c*d")
    (96 0 "abc" "(ab|ab*)bc")
    (97 0 "abc" "a([bc]*)c*")
    (98 0 "abcd" "a([bc]*)(c*d)")
    (99 0 "abcd" "a([bc]+)(c*d)")
    (100 0 "abcd" "a([bc]*)(c+d)")
    (101 0 "adcdcde" "a[bcd]*dcdcde")
    (102 1 "adcdcde" "a[bcd]+dcdcde")
    (103 0 "abc" "(ab|a)b*c")
    (104 0 "abcd" "((a)(b)c)(d)")
    (105 0 "alpha" "[A-Za-z_][A-Za-z0-9_]*")
    (106 0 "abh" "^a(bc+|b[eh])g|.h$")
    (107 0 "effgz" "(bc+d$|ef*g.|h?i(j|k))")
    (108 0 "ij" "(bc+d$|ef*g.|h?i(j|k))")
    (109 1 "effg" "(bc+d$|ef*g.|h?i(j|k))")
    (110 1 "bcdd" "(bc+d$|ef*g.|h?i(j|k))")
    (111 0 "reffgz" "(bc+d$|ef*g.|h?i(j|k))")
    (112 1 "-" "((((((((((a))))))))))")
    (113 0 "a" "(((((((((a)))))))))")
    (114 1 "uh-uh" "multiple words of text")
    (115 0 "multiple words, yeah" "multiple words")
    (116 0 "abcde" "(.*)c(.*)")
    (117 1 "(.*)\\)" "\\((.*),")
    (118 1 "ab" "[k]")
    (119 0 "abcd" "abcd")
    (120 0 "abcd" "a(bc)d")
    (121 0 "ac" "a[-]?c")
    (122 0 "beriberi" "(....).*\\1")))

(defconst *spencer2-tests*
  '((1 0 "a" "a")
    (2 0 "abc" "abc")
    (3 0 "abc" "abc|de")
    (4 0 "abc" "a|b|c")
    (5 0 "abc" "a(b)c")
    (6 1 "abc" "a\\(b\\)c")
    (7 2 "EPAREN" "a(")
    (8 2 "a(" "a(")
    (9 0 "a(" "a\\(")
    (10 1 "EPAREN" "a\\(")
    (11 1 "EPAREN" "a\\(b")
    (12 2 "EPAREN" "a(b")
    (13 2 "a(b" "a(b")
    (14 2 "a)" "a)")
    (15 2 ")" ")")
    (16 2 "a)" "a)")
    (17 1 "EPAREN" "a\\)")
    (18 1 "EPAREN" "\\)")
    (19 0 "ab" "a()b")
    (20 1 "ab" "a\\(\\)b")
    (21 0 "abc" "^abc$")
    (22 1 "a^b" "a^b")
    (23 1 "a^b" "a^b")
    (24 1 "a$b" "a$b")
    (25 1 "a$b" "a$b")
    (26 0 "abc" "^")
    (27 0 "abc" "$")
    (28 1 "\"\"" "^$")
    (29 1 "\"\"" "$^")
    (30 1 "\"\"" "\\($\\)\\(^\\)")
    (31 0 "\"\"" "^^")
    (32 0 "\"\"" "$$")
    (33 1 "abNc" "b$")
    (34 1 "abNc" "b$")
    (35 1 "aNbNc" "^b$")
    (36 1 "aNbNc" "^b$")
    (37 1 "aNNb" "^$")
    (38 1 "abc" "^$")
    (39 1 "abcN" "^$")
    (40 1 "aNNb" "$^")
    (41 1 "aNNb" "\\($\\)\\(^\\)")
    (42 0 "aNNb" "^^")
    (43 0 "aNNb" "$$")
    (44 0 "a" "^a")
    (45 0 "a" "a$")
    (46 0 "aNb" "^a")
    (47 1 "aNb" "^b")
    (48 0 "bNa" "a$")
    (49 1 "bNa" "b$")
    (50 0 "b" "a*(^b$)c*")
    (51 1 "b" "a*\\(^b$\\)c*")
    (52 0 "EMPTY" "|")
    (53 0 "|" "|")
    (54 0 "BADRPT" "*")
    (55 0 "*" "*")
    (56 0 "BADRPT" "+")
    (57 0 "BADRPT" "?")
    (58 1 "EMPTY" "\"\"")
    (59 0 "abc" "()")
    (60 1 "abc" "\\(\\)")
    (61 0 "EMPTY" "a||b")
    (62 0 "EMPTY" "|ab")
    (63 0 "EMPTY" "ab|")
    (64 1 "EMPTY" "(|a)b")
    (65 1 "EMPTY" "(a|)b")
    (66 1 "BADRPT" "(*a)")
    (67 1 "BADRPT" "(+a)")
    (68 1 "BADRPT" "(?a)")
    (69 1 "BADRPT" "({1}a)")
    (70 1 "BADRPT" "\\(\\{1\\}a\\)")
    (71 1 "BADRPT" "(a|*b)")
    (72 1 "BADRPT" "(a|+b)")
    (73 1 "BADRPT" "(a|?b)")
    (74 1 "BADRPT" "(a|{1}b)")
    (75 0 "BADRPT" "^*")
    (76 0 "*" "^*")
    (77 0 "BADRPT" "^+")
    (78 0 "BADRPT" "^?")
    (79 0 "BADRPT" "^{1}")
    (80 1 "BADRPT" "^\\{1\\}")
    (81 0 "abc" "a.c")
    (82 0 "abd" "a[bc]d")
    (83 0 "a*c" "a\\*c")
    (84 1 "abc" "ac")
    (85 1 "ac" "a\\bc")
    (86 1 "BADRPT" "\\{")
    (87 0 "a[b" "a\\[b")
    (88 2 "EBRACK" "a[b")
    (89 0 "a" "a$")
    (90 1 "a$" "a$")
    (91 1 "a" "a\\$")
    (92 0 "a$" "a\\$")
    (93 1 "a" "a\\$")
    (94 1 "a\\$" "a\\$")
    (95 2 "ESUBREG" "a\\(b\\)\\2c")
    (96 2 "ESUBREG" "a\\(b\\1\\)c")
    (97 2 "abbcbd" "a\\(b*\\)c\\1d")
    (98 2 "abbcbbbd" "a\\(b*\\)c\\1d")
    (99 2 "abc" "^\\(.\\)\\1")
    (100 2 "abbccd" "a\\(\\([bc]\\)\\2\\)*d")
    (101 2 "abbcbd" "a\\(\\([bc]\\)\\2\\)*d")
    (102 2 "abbbd" "a\\(\\(b\\)*\\2\\)*d")
    (103 2 "aabcd" "\\(a\\)\\1bcd")
    (104 2 "aabcd" "\\(a\\)\\1bc*d")
    (105 2 "aabd" "\\(a\\)\\1bc*d")
    (106 2 "aabcccd" "\\(a\\)\\1bc*d")
    (107 2 "aabcccd" "\\(a\\)\\1bc*[ce]d")
    (108 2 "aabcccd" "^\\(a\\)\\1b\\(c\\)*cd$")
    (109 0 "abc" "ab*c")
    (110 0 "abc" "ab+c")
    (111 0 "abc" "ab?c")
    (112 1 "a*b" "a\\(*\\)b")
    (113 1 "ab" "a\\(**\\)b")
    (114 1 "BADRPT" "a\\(***\\)b")
    (115 0 "*a" "*a")
    (116 0 "a" "**a")
    (117 1 "BADRPT" "***a")
    (118 2 "{" "{")
    (119 2 "{abc" "{abc")
    (120 2 "BADRPT" "{1")
    (121 0 "BADRPT" "{1}")
    (122 2 "a{b" "a{b")
    (123 0 "ab" "a{1}b")
    (124 1 "ab" "a\\{1\\}b")
    (125 0 "ab" "a{1,}b")
    (126 1 "ab" "a\\{1,\\}b")
    (127 0 "aab" "a{1,2}b")
    (128 1 "aab" "a\\{1,2\\}b")
    (129 2 "EBRACE" "a{1")
    (130 1 "EBRACE" "a\\{1")
    (131 2 "EBRACE" "a{1a")
    (132 1 "EBRACE" "a\\{1a")
    (133 2 "BADBR" "a{1a}")
    (134 1 "BADBR" "a\\{1a\\}")
    (135 0 "a{,2}" "a{,2}")
    (136 1 "BADBR" "a\\{,2\\}")
    (137 0 "a{,}" "a{,}")
    (138 1 "BADBR" "a\\{,\\}")
    (139 2 "BADBR" "a{1,x}")
    (140 1 "BADBR" "a\\{1,x\\}")
    (141 2 "EBRACE" "a{1,x")
    (142 1 "EBRACE" "a\\{1,x")
    (143 1 "BADBR" "a{300}")
    (144 1 "BADBR" "a\\{300\\}")
    (145 1 "BADBR" "a{1,0}")
    (146 1 "BADBR" "a\\{1,0\\}")
    (147 0 "abcac" "ab{0,0}c")
    (148 1 "abcac" "ab\\{0,0\\}c")
    (149 0 "abcac" "ab{0,1}c")
    (150 1 "abcac" "ab\\{0,1\\}c")
    (151 0 "abbcac" "ab{0,3}c")
    (152 1 "abbcac" "ab\\{0,3\\}c")
    (153 0 "acabc" "ab{1,1}c")
    (154 1 "acabc" "ab\\{1,1\\}c")
    (155 0 "acabc" "ab{1,3}c")
    (156 1 "acabc" "ab\\{1,3\\}c")
    (157 0 "abcabbc" "ab{2,2}c")
    (158 1 "abcabbc" "ab\\{2,2\\}c")
    (159 0 "abcabbc" "ab{2,4}c")
    (160 1 "abcabbc" "ab\\{2,4\\}c")
    (161 0 "BADRPT" "a**")
    (162 1 "BADRPT" "a++")
    (163 0 "BADRPT" "a??")
    (164 0 "BADRPT" "a*+")
    (165 0 "BADRPT" "a*?")
    (166 0 "BADRPT" "a+*")
    (167 0 "BADRPT" "a+?")
    (168 0 "BADRPT" "a?*")
    (169 0 "BADRPT" "a?+")
    (170 1 "BADRPT" "a{1}{1}")
    (171 0 "BADRPT" "a*{1}")
    (172 1 "BADRPT" "a+{1}")
    (173 0 "BADRPT" "a?{1}")
    (174 0 "BADRPT" "a{1}*")
    (175 1 "BADRPT" "a{1}+")
    (176 0 "BADRPT" "a{1}?")
    (177 2 "a{b}" "a*{b}")
    (178 1 "BADRPT" "a\\{1\\}\\{1\\}")
    (179 1 "BADRPT" "a*\\{1\\}")
    (180 1 "BADRPT" "a\\{1\\}*")
    (181 0 "abc" "a[b]c")
    (182 0 "abc" "a[ab]c")
    (183 0 "adc" "a[^ab]c")
    (184 0 "a]c" "a[]b]c")
    (185 0 "a[c" "a[[b]c")
    (186 0 "a-c" "a[-b]c")
    (187 0 "adc" "a[^]b]c")
    (188 0 "adc" "a[^-b]c")
    (189 0 "a-c" "a[b-]c")
    (190 2 "EBRACK" "a[b")
    (191 2 "EBRACK" "a[]")
    (192 0 "a2c" "a[1-3]c")
    (193 1 "ERANGE" "a[3-1]c")
    (194 1 "ERANGE" "a[1-3-5]c")
    (195 1 "a-c" "a[[.-.]--]c")
    (196 2 "ERANGE" "a[1-")
    (197 2 "EBRACK" "a[[.")
    (198 2 "EBRACK" "a[[.x")
    (199 2 "EBRACK" "a[[.x.")
    (200 1 "EBRACK" "a[[.x.]")
    (201 1 "ax" "a[[.x.]]")
    (202 1 "ECOLLATE" "a[[.x,.]]")
    (203 1 "a1b" "a[[.one.]]b")
    (204 1 "ECOLLATE" "a[[.notdef.]]b")
    (205 1 "a]b" "a[[.].]]b")
    (206 0 "abc" "a[[:alpha:]]c")
    (207 2 "ECTYPE" "a[[:notdef:]]c")
    (208 2 "EBRACK" "a[[:")
    (209 2 "EBRACK" "a[[:alpha")
    (210 2 "EBRACK" "a[[:alpha:]")
    (211 2 "ECTYPE" "a[[:alpha,:]")
    (212 2 "ECTYPE" "a[[:]:]]b")
    (213 2 "ECTYPE" "a[[:-:]]b")
    (214 2 "ECTYPE" "a[[:alph:]]")
    (215 2 "ECTYPE" "a[[:alphabet:]]")
    (216 1 "aSSTb" "[[:blank:]]+")
    (217 1 "aNTb" "[[:cntrl:]]+")
    (218 0 "a019b" "[[:digit:]]+")
    (219 0 "Sa%bS" "[[:graph:]]+")
    (220 0 "AabC" "[[:lower:]]+")
    (221 0 "NaSbN" "[[:print:]]+")
    (222 0 "S%-&T" "[[:punct:]]+")
    (223 1 "aSNTb" "[[:space:]]+")
    (224 0 "aBCd" "[[:upper:]]+")
    (225 0 "p0f3Cq" "[[:xdigit:]]+")
    (226 1 "abc" "a[[=b=]]c")
    (227 2 "EBRACK" "a[[=")
    (228 2 "EBRACK" "a[[=b")
    (229 2 "EBRACK" "a[[=b=")
    (230 1 "EBRACK" "a[[=b=]")
    (231 1 "ECOLLATE" "a[[=b,=]]")
    (232 1 "a1b" "a[[=one=]]b")
    (233 0 "abc" "a(((b)))c")
    (234 0 "abd" "a(b|(c))d")
    (235 0 "abbd" "a(b*|c)d")
    (236 0 "aaaaabaaaabaaaabaaaab" "a[ab]{20}")
    (237 0 "aaaaabaaaabaaaabaaaab" "a[ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab]")
    (238 0 "aaaaabaaaabaaaabaaaabweeknights" "a[ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab](wee|week)(knights|night)")
    (239 0 "a12345678901234567890123456789b" "12345678901234567890123456789")
    (240 0 "a123456789012345678901234567890b" "123456789012345678901234567890")
    (241 0 "a1234567890123456789012345678901b" "1234567890123456789012345678901")
    (242 0 "a12345678901234567890123456789012b" "12345678901234567890123456789012")
    (243 0 "a123456789012345678901234567890123b" "123456789012345678901234567890123")
    (244 0 "a1234567890123456789012345678901234567890123456789012345678901234567890b" "1234567890123456789012345678901234567890123456789012345678901234567890")
    (245 0 "xacegikmoq" "[ab][cd][ef][gh][ij][kl][mn]")
    (246 0 "xacegikmoq" "[ab][cd][ef][gh][ij][kl][mn][op]")
    (247 0 "xacegikmoqy" "[ab][cd][ef][gh][ij][kl][mn][op][qr]")
    (248 0 "xacegikmoqy" "[ab][cd][ef][gh][ij][kl][mn][op][q]")
    (249 0 "xabcy" "abc")
    (250 2 "acd" "a\\(b\\)?c\\1d")
    (251 1 "Abc" "aBc")
    (252 1 "abBCcd" "a[Bc]*d")
    (253 1 "0a1" "0[[:upper:]]1")
    (254 1 "0A1" "0[[:lower:]]1")
    (255 1 "abc" "a[^b]c")
    (256 0 "aBc" "a[^b]c")
    (257 0 "adc" "a[^b]c")
    (258 0 "abc" "[a]b[c]")
    (259 0 "aba" "[a]b[a]")
    (260 0 "abc" "[abc]b[abc]")
    (261 0 "abd" "[abc]b[abd]")
    (262 0 "accd" "a(b?c)+d")
    (263 0 "weeknights" "(wee|week)(knights|night)")
    (264 0 "weeknights" "(we|wee|week|frob)(knights|night|day)")
    (265 0 "xyzaaabcaababdacd" "a[bc]d")
    (266 0 "aaabc" "a[ab]c")
    (267 0 "abc" "abc")
    (268 0 "b" "a*")
    (269 0 "/*x*/" "/\\*.*\\*/")
    (270 0 "/*x*/y/*z*/" "/\\*.*\\*/")
    (271 0 "/*x*/" "/\\*([^*]|\\*[^/])*\\*/")
    (272 0 "/*x*/y/*z*/" "/\\*([^*]|\\*[^/])*\\*/")
    (273 0 "/*x**/y/*z*/" "/\\*([^*]|\\*[^/])*\\*/")
    (274 0 "/*x*/" "/\\*([^*]|\\*+[^*/])*\\*+/")
    (275 0 "/*x*/y/*z*/" "/\\*([^*]|\\*+[^*/])*\\*+/")
    (276 0 "/*x**/y/*z*/" "/\\*([^*]|\\*+[^*/])*\\*+/")
    (277 0 "/*x****/y/*z*/" "/\\*([^*]|\\*+[^*/])*\\*+/")
    (278 0 "/*x**x*/y/*z*/" "/\\*([^*]|\\*+[^*/])*\\*+/")
    (279 0 "/*x***x/y/*z*/" "/\\*([^*]|\\*+[^*/])*\\*+/")
    (280 0 "a(b)c" "[abc]")
    (281 0 "a(d)c" "[abc]")
    (282 0 "a(bc)d" "[abc]")
    (283 0 "a(dc)d" "[abc]")
    (284 0 "a()c" ".")
    (285 0 "b(bc)c" "b.*c")
    (286 0 "b(bc)c" "b.*")
    (287 0 "b(bc)c" ".*c")
    (288 0 "abc" "abc")
    (289 0 "xabcy" "abc")
    (290 1 "xyz" "abc")
    (291 0 "aba*b" "a*b")
    (292 0 "ab" "a*b")
    (293 1 "EMPTY" "\"\"")
    (294 1 "a" "aZb")
    (295 1 "a" "aZb")
    (296 0 "(aZb)" "aZb")
    (297 0 "(ab)" "aZ*b")
    (298 0 "(aZb)" "a.b")
    (299 0 "(aZb)c" "a.*")
    (300 2 "a" "[[:<:]]a")
    (301 2 "ba" "[[:<:]]a")
    (302 2 "-a" "[[:<:]]a")
    (303 2 "a" "a[[:>:]]")
    (304 2 "ab" "a[[:>:]]")
    (305 2 "a-" "a[[:>:]]")
    (306 2 "axcd-dayc-dazce-abc" "[[:<:]]a.c[[:>:]]")
    (307 2 "axcd-dayc-dazce-abc-q" "[[:<:]]a.c[[:>:]]")
    (308 2 "axc-dayc-dazce-abc" "[[:<:]]a.c[[:>:]]")
    (309 2 "a_bxc-byc_d-bzc-q" "[[:<:]]b.c[[:>:]]")
    (310 2 "y_xa_-_xb_y-_xc_-axdc" "[[:<:]].x..[[:>:]]")
    (311 2 "x_a_b" "[[:<:]]a_b[[:>:]]")
    (312 0 "A1" "(A[1])|(A[2])|(A[3])|(A[4])|(A[5])|(A[6])|(A[7])|(A[8])|(A[9])|(A[A])")
    (313 0 "abcdefghijklmnop" "abcdefghijklmnop")
    (314 0 "abcdefghijklmnopqrstuv" "abcdefghijklmnopqrstuv")
    (315 0 "CC11" "CC[13]1|a{21}[23][EO][123][Es][12]a{15}aa[34][EW]aaaaaaa[X]a")
    (316 0 "ab" "a?b")
    (317 1 "-5" "-\\{0,1\\}[0-9]*$")))

(defconst *palindrome-tests*
  (let ((reg "^(.?)(.?)(.?)(.?)(.?)(.?)(.?)(.?)(.?).?\\9\\8\\7\\6\\5\\4\\3\\2\\1$"))
    `((1 0 "a" ,reg)
      (2 0 "aa" ,reg)
      (3 0 "aba" ,reg)
      (4 0 "abba" ,reg)
      (5 0 "abcba" ,reg)
      (6 0 "abccba" ,reg)
      (7 0 "abcdcba" ,reg)
      (8 0 "abcddcba" ,reg)
      (9 0 "abcdedcba" ,reg)
      (10 0 "abcdeedcba" ,reg)
      (11 1 "af" ,reg)
      (12 0 "afa" ,reg)
      (13 1 "afba" ,reg)
      (14 1 "afbba" ,reg)
      (15 1 "afbcba" ,reg)
      (16 1 "afbccba" ,reg)
      (17 1 "afbcdcba" ,reg)
      (18 1 "afbcddcba" ,reg)
      (19 1 "afbcdedcba" ,reg)
      (20 1 "afbcdeedcba" ,reg)
      (21 1 "aaf" ,reg)
      (22 1 "abfa" ,reg)
      (23 0 "abfba" ,reg)
      (24 1 "abfcba" ,reg)
      (25 1 "abfccba" ,reg)
      (26 1 "abfcdcba" ,reg)
      (27 1 "abfcddcba" ,reg)
      (28 1 "abfcdedcba" ,reg)
      (29 1 "abfcdeedcba" ,reg)
      (30 1 "abaf" ,reg)
      (31 1 "abbfa" ,reg)
      (32 1 "abcfba" ,reg)
      (33 0 "abcfcba" ,reg)
      (34 1 "abcfdcba" ,reg)
      (35 1 "abcfddcba" ,reg)
      (36 1 "abcfdedcba" ,reg)
      (37 1 "abcfdeedcba" ,reg)
      (38 1 "abbaf" ,reg)
      (39 1 "abcbfa" ,reg)
      (40 1 "abccfba" ,reg)
      (41 1 "abcdfcba" ,reg)
      (42 0 "abcdfdcba" ,reg)
      (43 1 "abcdfedcba" ,reg)
      (44 1 "abcdfeedcba" ,reg)
      (45 1 "abcbaf" ,reg)
      (46 1 "abccbfa" ,reg)
      (47 1 "abcdcfba" ,reg)
      (48 1 "abcddfcba" ,reg)
      (49 1 "abcdefdcba" ,reg)
      (50 0 "abcdefedcba" ,reg)
      (51 1 "abccbaf" ,reg)
      (52 1 "abcdcbfa" ,reg)
      (53 1 "abcddcfba" ,reg)
      (54 1 "abcdedfcba" ,reg)
      (55 1 "abcdeefdcba" ,reg)
      (56 1 "abcdcbaf" ,reg)
      (57 1 "abcddcbfa" ,reg)
      (58 1 "abcdedcfba" ,reg)
      (59 1 "abcdeedfcba" ,reg)
      (60 1 "abcddcbaf" ,reg)
      (61 1 "abcdedcbfa" ,reg)
      (62 1 "abcdeedcfba" ,reg)
      (63 1 "abcdedcbaf" ,reg)
      (64 1 "abcdeedcbfa" ,reg)
      (65 1 "abcdeedcbaf" ,reg)
      (66 0 "abcdefghijihgfedcba" ,reg)
      )
    ))

(defconst *misc-tests*
  (let ((bre-opts (parse-options 'bre t nil nil nil))
        (ere-opts (parse-options 'ere t nil nil nil)))
  `((1 0 "a" "a" ,bre-opts "a")
    (1 0 "a" "a" ,ere-opts "a")
    (2 0 "abc" "abc" ,bre-opts "abc")
    (2 0 "abc" "abc" ,ere-opts "abc")
    (3 0 "abc|de" "abc" ,ere-opts "abc")
    (4 0 "a|b|c" "abc" ,ere-opts "a")
    (5 0 "a(b)c" "abc" ,ere-opts "abc")
    (6 0 "a\\(b\\)c" "abc" ,bre-opts "abc")
    (7 2 "a(" "EPAREN" ,ere-opts)
    (8 0 "a(" "a(" ,bre-opts "a(")
    (9 0 "a\\(" "a(" ,ere-opts "a(")
    (10 2 "a\\(" "EPAREN" ,bre-opts)
    (11 2 "a\\(b" "EPAREN" ,bre-opts)
    (12 2 "a(b" "EPAREN" ,ere-opts)
    (13 0 "a(b" "a(b" ,bre-opts "a(b")
    (14 0 "a)" "a)" ,ere-opts "a)")
    (15 0 ")" ")" ,ere-opts ")")
    (16 0 "a)" "a)" ,bre-opts "a)")
    (17 2 "a\\)" "EPAREN" ,bre-opts)
    (18 2 "\\)" "EPAREN" ,bre-opts)
    (19 0 "a()b" "ab" ,ere-opts "ab")
    (20 0 "a\\(\\)b" "ab" ,bre-opts "ab")
    (21 0 "^abc$" "abc" ,bre-opts "abc")
    (21 0 "^abc$" "abc" ,ere-opts "abc")
    (22 1 "a^b" "a^b" ,ere-opts)
    (23 0 "a^b" "a^b" ,bre-opts "a^b")
    (24 1 "a$b" "a$b" ,ere-opts)
    (25 0 "a$b" "a$b" ,bre-opts "a$b")
    (26 0 "^" "abc" ,bre-opts "")
    (26 0 "^" "abc" ,ere-opts "")
    (27 0 "$" "abc" ,bre-opts "")
    (27 0 "$" "abc" ,ere-opts "")
    (28 0 "^$" "" ,bre-opts "")
    (28 0 "^$" "" ,ere-opts "")
    (29 0 "$^" "" ,ere-opts "")
    (30 0 "\\($\\)\\(^\\)" "" ,bre-opts "")
    (31 0 "^^" "" ,ere-opts "")
    (32 0 "$$" "" ,ere-opts "")
    (33 0 "a*(^b$)c*" "b" ,ere-opts "b")
    (34 0 "a*\\(^b$\\)c*" "b" ,bre-opts "b")
    (35 2 "|" "EMPTY" ,ere-opts)
    (36 0 "|" "|" ,bre-opts "|")
    (37 2 "*" "BADRPT" ,ere-opts)
    (38 0 "*" "*" ,bre-opts "*")
    (39 2 "+" "BADRPT" ,ere-opts)
    (40 2 "?" "BADRPT" ,ere-opts)
    (41 2 "" "EMPTY" ,bre-opts)
    (41 2 "" "EMPTY" ,ere-opts)
    (42 0 "()" "abc" ,ere-opts "")
    (43 0 "\\(\\)" "abc" ,bre-opts "")
    (44 2 "a||b" "EMPTY" ,ere-opts)
    (45 2 "|ab" "EMPTY" ,ere-opts)
    (46 2 "ab|" "EMPTY" ,ere-opts)
    (47 2 "(|a)b" "EMPTY" ,ere-opts)
    (48 2 "(a|)b" "EMPTY" ,ere-opts)
    (49 2 "(*a)" "BADRPT" ,ere-opts)
    (50 2 "(+a)" "BADRPT" ,ere-opts)
    (51 2 "(?a)" "BADRPT" ,ere-opts)
    (52 2 "({1}a)" "BADRPT" ,ere-opts)
    (53 2 "\\(\\{1\\}a\\)" "BADRPT" ,bre-opts)
    (54 2 "(a|*b)" "BADRPT" ,ere-opts)
    (55 2 "(a|+b)" "BADRPT" ,ere-opts)
    (56 2 "(a|?b)" "BADRPT" ,ere-opts)
    (57 2 "(a|{1}b)" "BADRPT" ,ere-opts)
    (58 2 "^*" "BADRPT" ,ere-opts)
    (59 0 "^*" "*" ,bre-opts "*")
    (60 2 "^+" "BADRPT" ,ere-opts)
    (61 2 "^?" "BADRPT" ,ere-opts)
    (62 2 "^{1}" "BADRPT" ,ere-opts)
    (63 2 "^\\{1\\}" "BADRPT" ,bre-opts)
    (64 0 "a.c" "abc" ,bre-opts "abc")
    (64 0 "a.c" "abc" ,ere-opts "abc")
    (65 0 "a[bc]d" "abd" ,bre-opts "abd")
    (65 0 "a[bc]d" "abd" ,ere-opts "abd")
    (66 0 "a\\*c" "a*c" ,bre-opts "a*c")
    (66 0 "a\\*c" "a*c" ,ere-opts "a*c")
    (67 0 "a\\\\b" "a\\b" ,bre-opts "a\\b")
    (67 0 "a\\\\b" "a\\b" ,ere-opts "a\\b")
    (68 0 "a\\\\\\*b" "a\\*b" ,bre-opts "a\\*b")
    (68 0 "a\\\\\\*b" "a\\*b" ,ere-opts "a\\*b")
    (69 0 "a\\bc" "abc" ,bre-opts "abc")
    (69 0 "a\\bc" "abc" ,ere-opts "abc")
    (70 2 "a\\" "EESCAPE" ,bre-opts)
    (70 2 "a\\" "EESCAPE" ,ere-opts)
    (71 0 "a\\\\bc" "a\\bc" ,bre-opts "a\\bc")
    (71 0 "a\\\\bc" "a\\bc" ,ere-opts "a\\bc")
    (72 2 "\\{" "BADRPT" ,bre-opts)
    (73 0 "a\\[b" "a[b" ,bre-opts "a[b")
    (73 0 "a\\[b" "a[b" ,ere-opts "a[b")
    (74 2 "a[b" "EBRACK" ,bre-opts)
    (74 2 "a[b" "EBRACK" ,ere-opts)
    (75 0 "a$" "a" ,bre-opts "a")
    (75 0 "a$" "a" ,ere-opts "a")
    (76 1 "a$" "a$" ,bre-opts)
    (76 1 "a$" "a$" ,ere-opts)
    (77 1 "a\\$" "a" ,bre-opts)
    (77 1 "a\\$" "a" ,ere-opts)
    (78 0 "a\\$" "a$" ,bre-opts "a$")
    (78 0 "a\\$" "a$" ,ere-opts "a$")
    (79 1 "a\\\\$" "a" ,bre-opts)
    (79 1 "a\\\\$" "a" ,ere-opts)
    (80 1 "a\\\\$" "a$" ,bre-opts)
    (80 1 "a\\\\$" "a$" ,ere-opts)
    (81 1 "a\\\\$" "a\\$" ,bre-opts)
    (81 1 "a\\\\$" "a\\$" ,ere-opts)
    (82 0 "a\\\\$" "a\\" ,bre-opts "a\\")
    (82 0 "a\\\\$" "a\\" ,ere-opts "a\\")
    (83 0 "a\\(b*\\)c\\1d" "abbcbbd" ,bre-opts "abbcbbd" "bb")
    (84 1 "a\\(b*\\)c\\1d" "abbcbd" ,bre-opts)
    (85 1 "a\\(b*\\)c\\1d" "abbcbbbd" ,bre-opts)
    (86 1 "^\\(.\\)\\1" "abc" ,bre-opts)
    (87 0 "a\\([bc]\\)\\1d" "abcdabbd" ,bre-opts "abbd" "b")
    (88 0 "a\\(\\([bc]\\)\\2\\)*d" "abbccd" ,bre-opts "abbccd")
    (89 1 "a\\(\\([bc]\\)\\2\\)*d" "abbcbd" ,bre-opts)
    (90 0 "a\\(\\(b\\)*\\2\\)*d" "abbbd" ,bre-opts "abbbd")
    (91 0 "\\(ab*\\)[ab]*\\1" "ababaaa" ,bre-opts "ababaaa" "a")
    (92 0 "\\(a\\)\\1bcd" "aabcd" ,bre-opts "aabcd")
    (93 0 "\\(a\\)\\1bc*d" "aabcd" ,bre-opts "aabcd")
    (94 0 "\\(a\\)\\1bc*d" "aabd" ,bre-opts "aabd")
    (95 0 "\\(a\\)\\1bc*d" "aabcccd" ,bre-opts "aabcccd")
    (96 0 "\\(a\\)\\1bc*[ce]d" "aabcccd" ,bre-opts "aabcccd")
    (97 0 "^\\(a\\)\\1b\\(c\\)*cd$" "aabcccd" ,bre-opts "aabcccd")
    (98 0 "ab*c" "abc" ,bre-opts "abc")
    (98 0 "ab*c" "abc" ,ere-opts "abc")
    (99 0 "ab+c" "abc" ,ere-opts "abc")
    (100 0 "ab?c" "abc" ,ere-opts "abc")
    (101 0 "a\\(*\\)b" "a*b" ,bre-opts "a*b")
    (102 0 "a\\(**\\)b" "ab" ,bre-opts "ab")
    (103 2 "a\\(***\\)b" "BADRPT" ,bre-opts)
    (104 0 "*a" "*a" ,bre-opts "*a")
    (105 0 "**a" "a" ,bre-opts "a")
    (106 2 "***a" "BADRPT" ,bre-opts)
    (107 0 "{" "{" ,bre-opts "{")
    (107 0 "{" "{" ,ere-opts "{")
    (108 0 "{abc" "{abc" ,bre-opts "{abc")
    (108 0 "{abc" "{abc" ,ere-opts "{abc")
    (109 2 "{1" "BADRPT" ,ere-opts)
    (110 2 "{1}" "BADRPT" ,ere-opts)
    (111 0 "a{b" "a{b" ,bre-opts "a{b")
    (111 0 "a{b" "a{b" ,ere-opts "a{b")
    (112 0 "a{1}b" "ab" ,ere-opts "ab")
    (113 0 "a\\{1\\}b" "ab" ,bre-opts "ab")
    (114 0 "a{1,}b" "ab" ,ere-opts "ab")
    (115 0 "a\\{1,\\}b" "ab" ,bre-opts "ab")
    (116 0 "a{1,2}b" "aab" ,ere-opts "aab")
    (117 0 "a\\{1,2\\}b" "aab" ,bre-opts "aab")
    (118 2 "a{1" "EBRACE" ,ere-opts)
    (119 2 "a\\{1" "EBRACE" ,bre-opts)
    (120 2 "a{1a" "EBRACE" ,ere-opts)
    (121 2 "a\\{1a" "EBRACE" ,bre-opts)
    (122 2 "a{1a}" "BADBR" ,ere-opts)
    (123 2 "a\\{1a\\}" "BADBR" ,bre-opts)
    (124 0 "a{,2}" "a{,2}" ,ere-opts "a{,2}")
    (125 2 "a\\{,2\\}" "BADBR" ,bre-opts)
    (126 0 "a{,}" "a{,}" ,ere-opts "a{,}")
    (127 2 "a\\{,\\}" "BADBR" ,bre-opts)
    (128 2 "a{1,x}" "BADBR" ,ere-opts)
    (129 2 "a\\{1,x\\}" "BADBR" ,bre-opts)
    (130 2 "a{1,x" "EBRACE" ,ere-opts)
    (131 2 "a\\{1,x" "EBRACE" ,bre-opts)
    (132 2 "a{300}" "BADBR" ,ere-opts)
    (133 2 "a\\{300\\}" "BADBR" ,bre-opts)
    (134 2 "a{1,0}" "BADBR" ,ere-opts)
    (135 2 "a\\{1,0\\}" "BADBR" ,bre-opts)
    (136 0 "ab{0,0}c" "abcac" ,ere-opts "ac")
    (137 0 "ab\\{0,0\\}c" "abcac" ,bre-opts "ac")
    (138 0 "ab{0,1}c" "abcac" ,ere-opts "abc")
    (139 0 "ab\\{0,1\\}c" "abcac" ,bre-opts "abc")
    (140 0 "ab{0,3}c" "abbcac" ,ere-opts "abbc")
    (141 0 "ab\\{0,3\\}c" "abbcac" ,bre-opts "abbc")
    (142 0 "ab{1,1}c" "acabc" ,ere-opts "abc")
    (143 0 "ab\\{1,1\\}c" "acabc" ,bre-opts "abc")
    (144 0 "ab{1,3}c" "acabc" ,ere-opts "abc")
    (145 0 "ab\\{1,3\\}c" "acabc" ,bre-opts "abc")
    (146 0 "ab{2,2}c" "abcabbc" ,ere-opts "abbc")
    (147 0 "ab\\{2,2\\}c" "abcabbc" ,bre-opts "abbc")
    (148 0 "ab{2,4}c" "abcabbc" ,ere-opts "abbc")
    (149 0 "ab\\{2,4\\}c" "abcabbc" ,bre-opts "abbc")
    (150 0 "((a{1,10}){1,10}){1,10}" "a" ,ere-opts "a" "a,a")
    (151 2 "a**" "BADRPT" ,bre-opts)
    (151 2 "a**" "BADRPT" ,ere-opts)
    (152 2 "a++" "BADRPT" ,ere-opts)
    (153 2 "a??" "BADRPT" ,ere-opts)
    (154 2 "a*+" "BADRPT" ,ere-opts)
    (155 2 "a*?" "BADRPT" ,ere-opts)
    (156 2 "a+*" "BADRPT" ,ere-opts)
    (157 2 "a+?" "BADRPT" ,ere-opts)
    (158 2 "a?*" "BADRPT" ,ere-opts)
    (159 2 "a?+" "BADRPT" ,ere-opts)
    (160 2 "a{1}{1}" "BADRPT" ,ere-opts)
    (161 2 "a*{1}" "BADRPT" ,ere-opts)
    (162 2 "a+{1}" "BADRPT" ,ere-opts)
    (163 2 "a?{1}" "BADRPT" ,ere-opts)
    (164 2 "a{1}*" "BADRPT" ,ere-opts)
    (165 2 "a{1}+" "BADRPT" ,ere-opts)
    (166 2 "a{1}?" "BADRPT" ,ere-opts)
    (167 0 "a*{b}" "a{b}" ,ere-opts "a{b}")
    (168 2 "a\\{1\\}\\{1\\}" "BADRPT" ,bre-opts)
    (169 2 "a*\\{1\\}" "BADRPT" ,bre-opts)
    (170 2 "a\\{1\\}*" "BADRPT" ,bre-opts)
    (171 0 "a[b]c" "abc" ,bre-opts "abc")
    (171 0 "a[b]c" "abc" ,ere-opts "abc")
    (172 0 "a[ab]c" "abc" ,bre-opts "abc")
    (172 0 "a[ab]c" "abc" ,ere-opts "abc")
    (173 0 "a[^ab]c" "adc" ,bre-opts "adc")
    (173 0 "a[^ab]c" "adc" ,ere-opts "adc")
    (174 0 "a[]b]c" "a]c" ,bre-opts "a]c")
    (174 0 "a[]b]c" "a]c" ,ere-opts "a]c")
    (175 0 "a[[b]c" "a[c" ,bre-opts "a[c")
    (175 0 "a[[b]c" "a[c" ,ere-opts "a[c")
    (176 0 "a[-b]c" "a-c" ,bre-opts "a-c")
    (176 0 "a[-b]c" "a-c" ,ere-opts "a-c")
    (177 0 "a[^]b]c" "adc" ,bre-opts "adc")
    (177 0 "a[^]b]c" "adc" ,ere-opts "adc")
    (178 0 "a[^-b]c" "adc" ,bre-opts "adc")
    (178 0 "a[^-b]c" "adc" ,ere-opts "adc")
    (179 0 "a[b-]c" "a-c" ,bre-opts "a-c")
    (179 0 "a[b-]c" "a-c" ,ere-opts "a-c")
    (180 2 "a[b" "EBRACK" ,bre-opts)
    (180 2 "a[b" "EBRACK" ,ere-opts)
    (181 2 "a[]" "EBRACK" ,bre-opts)
    (181 2 "a[]" "EBRACK" ,ere-opts)
    (182 0 "a[1-3]c" "a2c" ,bre-opts "a2c")
    (182 0 "a[1-3]c" "a2c" ,ere-opts "a2c")
    (183 2 "a[3-1]c" "ERANGE" ,bre-opts)
    (183 2 "a[3-1]c" "ERANGE" ,ere-opts)
    (184 2 "a[1-3-5]c" "ERANGE" ,bre-opts)
    (184 2 "a[1-3-5]c" "ERANGE" ,ere-opts)
    (185 0 "a[[.-.]--]c" "a-c" ,bre-opts "a-c")
    (185 0 "a[[.-.]--]c" "a-c" ,ere-opts "a-c")
    (186 2 "a[1-" "ERANGE" ,bre-opts)
    (186 2 "a[1-" "ERANGE" ,ere-opts)
    (187 2 "a[[." "EBRACK" ,bre-opts)
    (187 2 "a[[." "EBRACK" ,ere-opts)
    (188 2 "a[[.x" "EBRACK" ,bre-opts)
    (188 2 "a[[.x" "EBRACK" ,ere-opts)
    (189 2 "a[[.x." "EBRACK" ,bre-opts)
    (189 2 "a[[.x." "EBRACK" ,ere-opts)
    (190 2 "a[[.x.]" "EBRACK" ,bre-opts)
    (190 2 "a[[.x.]" "EBRACK" ,ere-opts)
    (191 0 "a[[.x.]]" "ax" ,bre-opts "ax")
    (191 0 "a[[.x.]]" "ax" ,ere-opts "ax")
    (192 2 "a[[.x,.]]" "ECOLLATE" ,bre-opts)
    (192 2 "a[[.x,.]]" "ECOLLATE" ,ere-opts)
    (193 0 "a[[.one.]]b" "a1b" ,bre-opts "a1b")
    (193 0 "a[[.one.]]b" "a1b" ,ere-opts "a1b")
    (194 2 "a[[.notdef.]]b" "ECOLLATE" ,bre-opts)
    (194 2 "a[[.notdef.]]b" "ECOLLATE" ,ere-opts)
    (195 0 "a[[.].]]b" "a]b" ,bre-opts "a]b")
    (195 0 "a[[.].]]b" "a]b" ,ere-opts "a]b")
    (196 0 "a[[:alpha:]]c" "abc" ,bre-opts "abc")
    (196 0 "a[[:alpha:]]c" "abc" ,ere-opts "abc")
    (197 2 "a[[:notdef:]]c" "ECTYPE" ,bre-opts)
    (197 2 "a[[:notdef:]]c" "ECTYPE" ,ere-opts)
    (198 2 "a[[:" "EBRACK" ,bre-opts)
    (198 2 "a[[:" "EBRACK" ,ere-opts)
    (199 2 "a[[:alpha" "EBRACK" ,bre-opts)
    (199 2 "a[[:alpha" "EBRACK" ,ere-opts)
    (200 2 "a[[:alpha:]" "EBRACK" ,bre-opts)
    (200 2 "a[[:alpha:]" "EBRACK" ,ere-opts)
    (201 2 "a[[:alpha,:]" "ECTYPE" ,bre-opts)
    (201 2 "a[[:alpha,:]" "ECTYPE" ,ere-opts)
    (202 2 "a[[:]:]]b" "ECTYPE" ,bre-opts)
    (202 2 "a[[:]:]]b" "ECTYPE" ,ere-opts)
    (203 2 "a[[:-:]]b" "ECTYPE" ,bre-opts)
    (203 2 "a[[:-:]]b" "ECTYPE" ,ere-opts)
    (204 2 "a[[:alph:]]" "ECTYPE" ,bre-opts)
    (204 2 "a[[:alph:]]" "ECTYPE" ,ere-opts)
    (205 2 "a[[:alphabet:]]" "ECTYPE" ,bre-opts)
    (205 2 "a[[:alphabet:]]" "ECTYPE" ,ere-opts)
    (206 0 "[[:blank:]]+" "a  	b" ,ere-opts "  	")
    (207 0 "[[:digit:]]+" "a019b" ,ere-opts "019")
    (208 0 "[[:lower:]]+" "AabC" ,ere-opts "ab")
    (209 0 "[[:space:]]+" "a 
	b" ,ere-opts " 
	")
    (210 0 "[[:upper:]]+" "aBCd" ,ere-opts "BC")
    (211 0 "[[:xdigit:]]+" "p0f3Cq" ,ere-opts "0f3C")
    (212 0 "a[[=b=]]c" "abc" ,bre-opts "abc")
    (212 0 "a[[=b=]]c" "abc" ,ere-opts "abc")
    (213 2 "a[[=" "EBRACK" ,bre-opts)
    (213 2 "a[[=" "EBRACK" ,ere-opts)
    (214 2 "a[[=b" "EBRACK" ,bre-opts)
    (214 2 "a[[=b" "EBRACK" ,ere-opts)
    (215 2 "a[[=b=" "EBRACK" ,bre-opts)
    (215 2 "a[[=b=" "EBRACK" ,ere-opts)
    (216 2 "a[[=b=]" "EBRACK" ,bre-opts)
    (216 2 "a[[=b=]" "EBRACK" ,ere-opts)
    (217 2 "a[[=b,=]]" "ECOLLATE" ,bre-opts)
    (217 2 "a[[=b,=]]" "ECOLLATE" ,ere-opts)
    (218 0 "a[[=one=]]b" "a1b" ,bre-opts "a1b")
    (218 0 "a[[=one=]]b" "a1b" ,ere-opts "a1b")
    (219 0 "a(((b)))c" "abc" ,ere-opts "abc")
    (220 0 "a(b|(c))d" "abd" ,ere-opts "abd")
    (221 0 "a(b*|c)d" "abbd" ,ere-opts "abbd")
    (222 0 "a[ab]{20}" "aaaaabaaaabaaaabaaaab" ,ere-opts "aaaaabaaaabaaaabaaaab")
    (223 0 "a[ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab]" "aaaaabaaaabaaaabaaaab" ,ere-opts "aaaaabaaaabaaaabaaaab")
    (224 0 "a[ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab][ab](wee|week)(knights|night)" "aaaaabaaaabaaaabaaaabweeknights" ,ere-opts "aaaaabaaaabaaaabaaaabweeknights")
    (225 0 "12345678901234567890123456789" "a12345678901234567890123456789b" ,ere-opts "12345678901234567890123456789")
    (226 0 "123456789012345678901234567890" "a123456789012345678901234567890b" ,ere-opts "123456789012345678901234567890")
    (227 0 "1234567890123456789012345678901" "a1234567890123456789012345678901b" ,ere-opts "1234567890123456789012345678901")
    (228 0 "12345678901234567890123456789012" "a12345678901234567890123456789012b" ,ere-opts "12345678901234567890123456789012")
    (229 0 "123456789012345678901234567890123" "a123456789012345678901234567890123b" ,ere-opts "123456789012345678901234567890123")
    (230 0 "1234567890123456789012345678901234567890123456789012345678901234567890" "a1234567890123456789012345678901234567890123456789012345678901234567890b" ,ere-opts "1234567890123456789012345678901234567890123456789012345678901234567890")
    (231 0 "[ab][cd][ef][gh][ij][kl][mn]" "xacegikmoq" ,ere-opts "acegikm")
    (232 0 "[ab][cd][ef][gh][ij][kl][mn][op]" "xacegikmoq" ,ere-opts "acegikmo")
    (233 0 "[ab][cd][ef][gh][ij][kl][mn][op][qr]" "xacegikmoqy" ,ere-opts "acegikmoq")
    (234 0 "[ab][cd][ef][gh][ij][kl][mn][op][q]" "xacegikmoqy" ,ere-opts "acegikmoq")
    (235 0 "abc" "xabcy" ,bre-opts "abc")
    (235 0 "abc" "xabcy" ,ere-opts "abc")
    (236 1 "a\\(b\\)?c\\1d" "acd" ,bre-opts)
    (237 0 "[a]b[c]" "abc" ,ere-opts "abc")
    (238 0 "[a]b[a]" "aba" ,ere-opts "aba")
    (239 0 "[abc]b[abc]" "abc" ,ere-opts "abc")
    (240 0 "[abc]b[abd]" "abd" ,ere-opts "abd")
    (241 0 "a(b?c)+d" "accd" ,ere-opts "accd")
    (242 0 "(wee|week)(knights|night)" "weeknights" ,ere-opts "weeknights")
    (243 0 "(we|wee|week|frob)(knights|night|day)" "weeknights" ,ere-opts "weeknights")
    (244 0 "a[bc]d" "xyzaaabcaababdacd" ,ere-opts "abd")
    (245 0 "a[ab]c" "aaabc" ,ere-opts "abc")
    (246 0 "a*" "b" ,bre-opts "")
    (246 0 "a*" "b" ,ere-opts "")
    (247 0 "/\\*.*\\*/" "/*x*/" ,ere-opts "/*x*/")
    (248 0 "/\\*.*\\*/" "/*x*/y/*z*/" ,ere-opts "/*x*/y/*z*/")
    (249 0 "/\\*([^*]|\\*[^/])*\\*/" "/*x*/" ,ere-opts "/*x*/")
    (250 0 "/\\*([^*]|\\*[^/])*\\*/" "/*x*/y/*z*/" ,ere-opts "/*x*/")
    (251 0 "/\\*([^*]|\\*[^/])*\\*/" "/*x**/y/*z*/" ,ere-opts "/*x**/y/*z*/")
    (252 0 "/\\*([^*]|\\*+[^*/])*\\*+/" "/*x*/" ,ere-opts "/*x*/")
    (253 0 "/\\*([^*]|\\*+[^*/])*\\*+/" "/*x*/y/*z*/" ,ere-opts "/*x*/")
    (254 0 "/\\*([^*]|\\*+[^*/])*\\*+/" "/*x**/y/*z*/" ,ere-opts "/*x**/")
    (255 0 "/\\*([^*]|\\*+[^*/])*\\*+/" "/*x****/y/*z*/" ,ere-opts "/*x****/")
    (256 0 "/\\*([^*]|\\*+[^*/])*\\*+/" "/*x**x*/y/*z*/" ,ere-opts "/*x**x*/")
    (257 0 "/\\*([^*]|\\*+[^*/])*\\*+/" "/*x***x/y/*z*/" ,ere-opts "/*x***x/y/*z*/")
    (258 0 "a(b)(c)d" "abcd" ,ere-opts "abcd" "b,c")
    (259 0 "a(((b)))c" "abc" ,ere-opts "abc" "b,b,b")
    (260 0 "a(b|(c))d" "abd" ,ere-opts "abd" "b,-")
    (261 0 "a(b*|c|e)d" "abbd" ,ere-opts "abbd" "bb")
    (262 0 "a(b*|c|e)d" "acd" ,ere-opts "acd" "c")
    (263 0 "a(b*|c|e)d" "ad" ,ere-opts "ad" "")
    (264 0 "a(b?)c" "abc" ,ere-opts "abc" "b")
    (265 0 "a(b?)c" "ac" ,ere-opts "ac" "")
    (266 0 "a(b+)c" "abc" ,ere-opts "abc" "b")
    (267 0 "a(b+)c" "abbbc" ,ere-opts "abbbc" "bbb")
    (268 0 "a(b*)c" "ac" ,ere-opts "ac" "")
    (269 0 "(a|ab)(bc([de]+)f|cde)" "abcdef" ,ere-opts "abcdef" "a,bcdef,de")
    (270 0 "a(b)(c)(d)(e)(f)(g)(h)(i)(j)k" "abcdefghijk" ,ere-opts "abcdefghijk" "b,c,d,e,f,g,h,i,j")
    (271 0 "a(b)(c)(d)(e)(f)(g)(h)(i)(j)(k)l" "abcdefghijkl" ,ere-opts "abcdefghijkl" "b,c,d,e,f,g,h,i,j,k")
    (272 0 "a([bc]?)c" "abc" ,ere-opts "abc" "b")
    (273 0 "a([bc]?)c" "ac" ,ere-opts "ac" "")
    (274 0 "a([bc]+)c" "abc" ,ere-opts "abc" "b")
    (275 0 "a([bc]+)c" "abcc" ,ere-opts "abcc" "bc")
    (276 0 "a([bc]+)bc" "abcbc" ,ere-opts "abcbc" "bc")
    (277 0 "a(bb+|b)b" "abb" ,ere-opts "abb" "b")
    (278 0 "a(bbb+|bb+|b)b" "abb" ,ere-opts "abb" "b")
    (279 0 "a(bbb+|bb+|b)b" "abbb" ,ere-opts "abbb" "bb")
    (280 0 "a(bbb+|bb+|b)bb" "abbb" ,ere-opts "abbb" "b")
    (281 0 "(.*).*" "abcdef" ,ere-opts "abcdef" "abcdef")
    (282 0 "a(b|c)*d" "ad" ,ere-opts "ad" "")
    (283 0 "a(b|c)*d" "abcd" ,ere-opts "abcd" "c")
    (284 0 "a(b|c)+d" "abd" ,ere-opts "abd" "b")
    (285 0 "a(b|c)+d" "abcd" ,ere-opts "abcd" "c")
    (286 0 "a(b|c?)+d" "ad" ,ere-opts "ad" "")
    (287 0 "a(b|c?)+d" "abcd" ,ere-opts "abcd" "")
    (288 0 "a(b|c){0,0}d" "ad" ,ere-opts "ad" "")
    (289 0 "a(b|c){0,1}d" "ad" ,ere-opts "ad" "")
    (290 0 "a(b|c){0,1}d" "abd" ,ere-opts "abd" "b")
    (291 0 "a(b|c){0,2}d" "ad" ,ere-opts "ad" "")
    (292 0 "a(b|c){0,2}d" "abcd" ,ere-opts "abcd" "c")
    (293 0 "a(b|c){0,}d" "ad" ,ere-opts "ad" "")
    (294 0 "a(b|c){0,}d" "abcd" ,ere-opts "abcd" "c")
    (295 0 "a(b|c){1,1}d" "abd" ,ere-opts "abd" "b")
    (296 0 "a(b|c){1,1}d" "acd" ,ere-opts "acd" "c")
    (297 0 "a(b|c){1,2}d" "abd" ,ere-opts "abd" "b")
    (298 0 "a(b|c){1,2}d" "abcd" ,ere-opts "abcd" "c")
    (299 0 "a(b|c){1,}d" "abd" ,ere-opts "abd" "b")
    (300 0 "a(b|c){1,}d" "abcd" ,ere-opts "abcd" "c")
    (301 0 "a(b|c){2,2}d" "acbd" ,ere-opts "acbd" "b")
    (302 0 "a(b|c){2,2}d" "abcd" ,ere-opts "abcd" "c")
    (303 0 "a(b|c){2,4}d" "abcd" ,ere-opts "abcd" "c")
    (304 0 "a(b|c){2,4}d" "abcbd" ,ere-opts "abcbd" "b")
    (305 0 "a(b|c){2,4}d" "abcbcd" ,ere-opts "abcbcd" "c")
    (306 0 "a(b|c){2,}d" "abcd" ,ere-opts "abcd" "c")
    (307 0 "a(b|c){2,}d" "abcbd" ,ere-opts "abcbd" "b")
    (308 0 "a b" "a" ,bre-opts "a")
    (308 0 "a b" "a" ,ere-opts "a")
    (309 0 "[[:<:]]a" "a" ,bre-opts "a")
    (309 0 "[[:<:]]a" "a" ,ere-opts "a")
    (310 1 "[[:<:]]a" "ba" ,bre-opts)
    (310 1 "[[:<:]]a" "ba" ,ere-opts)
    (311 0 "[[:<:]]a" "-a" ,bre-opts "a")
    (311 0 "[[:<:]]a" "-a" ,ere-opts "a")
    (312 0 "a[[:>:]]" "a" ,bre-opts "a")
    (312 0 "a[[:>:]]" "a" ,ere-opts "a")
    (313 1 "a[[:>:]]" "ab" ,bre-opts)
    (313 1 "a[[:>:]]" "ab" ,ere-opts)
    (314 0 "a[[:>:]]" "a-" ,bre-opts "a")
    (314 0 "a[[:>:]]" "a-" ,ere-opts "a")
    (315 0 "[[:<:]]a.c[[:>:]]" "axcd-dayc-dazce-abc" ,bre-opts "abc")
    (315 0 "[[:<:]]a.c[[:>:]]" "axcd-dayc-dazce-abc" ,ere-opts "abc")
    (316 0 "[[:<:]]a.c[[:>:]]" "axcd-dayc-dazce-abc-q" ,bre-opts "abc")
    (316 0 "[[:<:]]a.c[[:>:]]" "axcd-dayc-dazce-abc-q" ,ere-opts "abc")
    (317 0 "[[:<:]]a.c[[:>:]]" "axc-dayc-dazce-abc" ,bre-opts "axc")
    (317 0 "[[:<:]]a.c[[:>:]]" "axc-dayc-dazce-abc" ,ere-opts "axc")
    (318 0 "[[:<:]]b.c[[:>:]]" "a_bxc-byc_d-bzc-q" ,bre-opts "bzc")
    (318 0 "[[:<:]]b.c[[:>:]]" "a_bxc-byc_d-bzc-q" ,ere-opts "bzc")
    (319 0 "[[:<:]].x..[[:>:]]" "y_xa_-_xb_y-_xc_-axdc" ,bre-opts "_xc_")
    (319 0 "[[:<:]].x..[[:>:]]" "y_xa_-_xb_y-_xc_-axdc" ,ere-opts "_xc_")
    (320 1 "[[:<:]]a_b[[:>:]]" "x_a_b" ,bre-opts)
    (320 1 "[[:<:]]a_b[[:>:]]" "x_a_b" ,ere-opts)
    (321 0 "(A[1])|(A[2])|(A[3])|(A[4])|(A[5])|(A[6])|(A[7])|(A[8])|(A[9])|(A[A])" "A1" ,ere-opts "A1")

    (323 0 "CC[13]1|a{21}[23][EO][123][Es][12]a{15}aa[34][EW]aaaaaaa[X]a" "CC11" ,ere-opts "CC11")

    (325 0 "a?b" "ab" ,ere-opts "ab")
    (326 0 "-\\{0,1\\}[0-9]*$" "-5" ,bre-opts "-5")
    )))

(defun do-tests-bre (opts)
  (declare (xargs :guard (parse-opts-p opts)))
  (prog2$ (cw "BRE tests failed:~%")
          (do-test-list opts *bre-tests*)))

;; Various sets of extended regexp tests that come with the
;; grep source distribution
(defun do-tests-ere (opts)
  (declare (xargs :guard (parse-opts-p opts)))
  (prog2$ (cw "ERE tests failed:~%")
          (do-test-list opts *ere-tests*)))

(defun do-tests-khadafy (opts)
  (declare (xargs :guard (parse-opts-p opts)))
  (prog2$ 
   (cw "Khadafy tests failed: ~%")
   (do-test-list opts *khadafy-tests*)))


(defun do-tests-spencer1 (opts)
  (declare (xargs :guard (parse-opts-p opts)))
  (prog2$ (cw "Spencer1 tests failed:~%")
          (do-test-list opts *spencer1-tests*)))

(defun do-tests-backref (opts)
  (declare (xargs :guard (parse-opts-p opts)))
  (prog2$ (cw "Backref tests failed:~%")
          (do-test-list opts
                        '((1 0 "radar" "(.)(.).\\2\\1")
                          (2 0 "civic"
                             "(.?)(.?)(.?)(.?)(.?)(.?)(.?)(.?)(.?).?\\9\\8\\7\\6\\5\\4\\3\\2\\1$")
                          ))))


(defun do-tests-spencer2 (opts)
  (declare (xargs :guard (parse-opts-p opts)))
  (prog2$ (cw "Spencer2 tests failed:~%")
          (do-test-list opts *spencer2-tests*)))


(defun do-tests-palindrome (opts)
  (declare (xargs :guard (parse-opts-p opts)))
    (prog2$ (cw "Palindrome tests failed:~%")
            (do-test-list opts *palindrome-tests*)))


(defun do-tests-misc ()
  (declare (xargs :guard t))
  (prog2$ (cw "Misc tests missed:~%")
          (do-other-test-list *misc-tests*)))
             



;; Run all the test cases
(defun do-all-tests ()
  (declare (xargs :guard t
                  :guard-hints
                  (("Goal"
                    :in-theory (disable 
                                (:EXECUTABLE-COUNTERPART DO-TEST-LIST)
                                (:EXECUTABLE-COUNTERPART DO-TESTS-BRE)
                                (:EXECUTABLE-COUNTERPART DO-TESTS-ERE)
                                (:EXECUTABLE-COUNTERPART DO-TESTS-KHADAFY)
                                (:EXECUTABLE-COUNTERPART DO-TESTS-PALINDROME)
                                (:EXECUTABLE-COUNTERPART DO-TESTS-SPENCER1)
                                (:EXECUTABLE-COUNTERPART DO-TESTS-SPENCER2)
                                (:EXECUTABLE-COUNTERPART DO-TESTS-BACKREF))))))
  (let* ((ere-opts (parse-options 'ere t nil nil nil))
         (bre-opts (parse-options 'bre t t nil nil))
         (nwrong
          (+ (do-tests-bre bre-opts)
             (do-tests-ere ere-opts)
             (do-tests-khadafy ere-opts)
             (do-tests-backref ere-opts)
             (do-tests-spencer1 ere-opts)
             (do-tests-spencer2 ere-opts)
             (do-tests-palindrome ere-opts))))
    (prog2$ (cw "Number wrong: ~s0~%" nwrong)
            nwrong)))

;; Run all the test cases n times
(defun do-tests-n-times (n)
  (declare (xargs :guard (natp n)))
  (if (zp n) 
      0
    (+ (do-all-tests)
       (do-tests-n-times (1- n)))))


(defthm no-regress
  (<= (do-all-tests) 
      ;; Number of failures out of 646 tests from the GNU grep test suite
      67))


(defun regex-main (regex string)
  (declare (xargs :guard (and (stringp regex) (stringp string))))
  (run-regex-test string regex (parse-options 'ere t nil nil nil)))