# C runtime object model.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/util"
require "adlint/c/syntax"
require "adlint/c/value"
require "adlint/c/scope"
require "adlint/c/seqp"

module AdLint #:nodoc:
module C #:nodoc:

  module Bindable
    attr_accessor :binding

    def bind_to(target)
      case target
      when Memory
        Binding.bind(self, target)
      when Object
        Binding.bind(target, self)
      end
    end

    def be_alias_to(memory)
      case memory
      when Memory
        Binding.create_alias(self, memory)
      else
        raise TypeError, "an object cannot be an alias to other objects."
      end
    end
  end

  class Binding
    def self.bind(object, memory)
      binding = new(object, memory)
      object.binding = binding
      memory.binding = binding
      binding
    end

    def self.create_alias(object, memory)
      binding = new(object, memory)
      object.binding = binding
      binding
    end

    def initialize(object, memory)
      @object = object
      @memory = memory
    end
    private_class_method :new

    attr_reader :object
    attr_reader :memory
  end

  class Object
    include Bindable

    def initialize(declaration_or_definition = nil)
      @declarations_and_definitions = [declaration_or_definition].compact
    end

    attr_reader :declarations_and_definitions

    def storage_class_specifiers
      @declarations_and_definitions.map { |decl_or_def|
        decl_or_def.storage_class_specifier
      }.compact
    end

    def declared_as_extern?
      first_sc_spec = first_storage_class_specifier
      first_sc_spec.nil? || first_sc_spec.type == :EXTERN
    end

    def declared_as_static?
      first_sc_spec = first_storage_class_specifier and
        first_sc_spec.type == :STATIC
    end

    def declared_as_auto?
      first_sc_spec = first_storage_class_specifier and
        first_sc_spec.type == :AUTO
    end

    def declared_as_register?
      first_sc_spec = first_storage_class_specifier and
        first_sc_spec.type == :REGISTER
    end

    def designated_by_lvalue?
      subclass_responsibility
    end

    def named?
      subclass_responsibility
    end

    def temporary?
      subclass_responsibility
    end

    def function?
      subclass_responsibility
    end

    def variable?
      subclass_responsibility
    end

    private
    def first_storage_class_specifier
      if @declarations_and_definitions.empty?
        nil
      else
        @declarations_and_definitions.first.storage_class_specifier
      end
    end
  end

  class TypedObject < Object
    def initialize(type, declaration_or_definition = nil)
      super(declaration_or_definition)
      @type = type
    end

    attr_reader :type
  end

  # == DESCRIPTION
  # === Variable class hierarchy
  #  Variable
  #    <--- ScopedVariable
  #           <--- OuterVariable
  #                  <--- NamedVariable --------- Nameable <<module>>
  #                         <--- AliasVariable       |
  #                  <--- TemporaryVariable          |
  #                  <--- InnerVariable -------------+
  #                         <--- ArrayElementVariable
  #                         <--- CompositeMemberVariable
  class Variable < TypedObject
    def initialize(memory, declaration_or_definition, type)
      super(type, declaration_or_definition)

      relate_to_memory(memory)
    end

    def function?
      false
    end

    def variable?
      true
    end

    def name
      subclass_responsibility
    end

    def named?
      false
    end

    def value
      binding.memory.read
    end

    def assign!(value)
      # NOTE: Length of the incomplete array type should be deducted while
      #       initializer evaluation.  So, adjustment of the assigning value
      #       can be done at this point by Value#coerce_to(type).
      # NOTE: Domain of the assigning value must be narrowed before writing to
      #       the memory by Value#coerce_to(type).
      binding.memory.write(value.coerce_to(type))
    end

    def uninitialize!
      assign!(self.type.undefined_value)
    end

    def narrow_value_domain!(operator_symbol, value)
      unless self_value = modifiable_value
        assign!(type.arbitrary_value)
        self_value = modifiable_value
      end

      self_value.narrow_domain!(operator_symbol, value.coerce_to(type))
      self_value.exist?
    end

    def widen_value_domain!(operator_symbol, value)
      unless self_value = modifiable_value
        assign!(type.nil_value)
        self_value = modifiable_value
      end

      self_value.widen_domain!(operator_symbol, value.coerce_to(type))
      self_value.exist?
    end

    def enter_value_versioning_group
      value.enter_versioning_group
    end

    def leave_value_versioning_group(raise_complement)
      value.leave_versioning_group(raise_complement)
    end

    def begin_value_versioning
      value.begin_versioning
    end

    def end_value_versioning
      value.end_versioning
    end

    def rollback_latest_value_version!
      value.rollback_latest_version!
    end

    def rollback_all_value_versions!
      value.rollback_all_versions!
    end

    private
    def relate_to_memory(memory)
      bind_to(memory)
    end

    def modifiable_value
      binding.memory.read_modifiable
    end
  end

  class ScopedVariable < Variable
    def initialize(memory, declaration_or_definition, type, scope)
      super(memory, declaration_or_definition, type)
      @scope = scope
    end

    attr_accessor :scope

    def declared_as_extern?
      if @scope.global?
        super
      else
      first_sc_spec = first_storage_class_specifier and
        first_sc_spec.type == :EXTERN
      end
    end

    def declared_as_auto?
      if @scope.global?
        super
      else
        first_sc_spec = first_storage_class_specifier
        first_sc_spec.nil? || first_sc_spec.type == :AUTO
      end
    end
  end

  class OuterVariable < ScopedVariable
    def initialize(memory, declaration_or_definition, type, scope)
      super(memory, declaration_or_definition, type, scope)

      # TODO: If too slow, make an index of inner variables.
      @inner_variables = create_inner_variables(type, scope)
    end

    def enter_value_versioning_group
      super
      if @inner_variables
        @inner_variables.each do |inner|
          inner.enter_value_versioning_group
        end
      end
    end

    def leave_value_versioning_group(raise_complement)
      super
      if @inner_variables
        @inner_variables.each do |inner|
          inner.leave_value_versioning_group(raise_complement)
        end
      end
    end

    def begin_value_versioning
      super
      if @inner_variables
        @inner_variables.each do |inner|
          inner.begin_value_versioning
        end
      end
    end

    def end_value_versioning
      super
      if @inner_variables
        @inner_variables.each do |inner|
          inner.end_value_versioning
        end
      end
    end

    def rollback_latest_value_version!
      super
      if @inner_variables
        @inner_variables.each do |inner|
          inner.rollback_latest_value_version!
        end
      end
    end

    def rollback_all_value_versions!
      super
      if @inner_variables
        @inner_variables.each do |inner|
          inner.rollback_all_value_versions!
        end
      end
    end

    def inner_variable_at(index)
      if @type.array?
        # TODO: If linear searching is too slow, use an index of inner
        #       variables.
        target_name = ArrayElementVariable.component_name_of(index)
        @inner_variables.find { |iv| iv.component_name == target_name }
      else
        nil
      end
    end

    def inner_variable_named(name)
      if @type.composite?
        # TODO: If linear searching is too slow, use an index of inner
        #       variables.
        target_name = CompositeMemberVariable.component_name_of(name)
        @inner_variables.find { |iv| iv.component_name == target_name }
      else
        nil
      end
    end

    private
    def create_inner_variables(type, scope)
      case
      when type.array?
        create_array_elements(type, scope, binding.memory)
      when type.composite?
        create_composite_members(type, scope, binding.memory)
      else
        nil
      end
    end

    def create_array_elements(type, scope, memory)
      offset = 0
      type.impl_length.times.map do |index|
        window = memory.create_window(offset, type.base_type.aligned_byte_size)
        offset += window.byte_size
        ArrayElementVariable.new(window, self, type.base_type, index)
      end
    end

    def create_composite_members(type, scope, memory)
      offset = 0
      type.members.map do |member|
        window = memory.create_window(offset, member.type.aligned_byte_size)
        offset += window.byte_size
        CompositeMemberVariable.new(window, self, member.type, member.name)
      end
    end
  end

  module Nameable
    attr_reader :name

    def named?
      true
    end

    private
    def name=(name)
      # NOTE: Private attr_writer to suppress `private attribute?' warning.
      @name = name
    end
  end

  class NamedVariable < OuterVariable
    include Nameable

    def initialize(memory, declaration_or_definition, scope)
      self.name = declaration_or_definition.identifier.value

      super(memory, declaration_or_definition, declaration_or_definition.type,
            scope)
    end

    def temporary?
      false
    end

    def designated_by_lvalue?
      true
    end

    # NOTE: This method should be overridden by PhantomVariable.
    def to_named_variable
      self
    end

    def pretty_print(pp)
      Summary.new(object_id, name, type, binding.memory).pretty_print(pp)
    end

    Summary = Struct.new(:object_id, :name, :type, :memory)
  end

  class TemporaryVariable < OuterVariable
    def initialize(memory, type, scope)
      super(memory, nil, type, scope)
    end

    def temporary?
      true
    end

    def designated_by_lvalue?
      false
    end

    def named?
      false
    end

    def pretty_print(pp)
      Summary.new(object_id, type, binding.memory).pretty_print(pp)
    end

    Summary = Struct.new(:object_id, :type, :memory)
  end

  class InnerVariable < OuterVariable
    include Nameable

    def initialize(memory, outer_variable, type, component_name)
      @owner = outer_variable
      @component_name = component_name
      self.name = create_qualified_name(outer_variable, component_name)

      super(memory, nil, type, outer_variable.scope)
    end

    attr_reader :owner
    attr_reader :component_name

    def storage_class_specifiers
      @owner.storage_class_specifiers
    end

    def declared_as_extern?
      @owner.declared_as_extern?
    end

    def declared_as_static?
      @owner.declared_as_static?
    end

    def declared_as_auto?
      @owner.declared_as_auto?
    end

    def declared_as_register?
      @owner.declared_as_register?
    end

    def named?
      @owner.named?
    end

    def temporary?
      @owner.temporary?
    end

    def designated_by_lvalue?
      true
    end

    def to_named_variable
      self
    end

    private
    def create_qualified_name(outer_variable, component_name)
      if outer_variable.named?
        "#{outer_variable.name}#{component_name}"
      else
        "__adlint__tempvar#{component_name}"
      end
    end
  end

  class ArrayElementVariable < InnerVariable
    def self.component_name_of(index)
      "[#{index}]"
    end

    def initialize(memory, outer_variable, type, index)
      super(memory, outer_variable, type, self.class.component_name_of(index))
    end
  end

  class CompositeMemberVariable < InnerVariable
    def self.component_name_of(name)
      ".#{name}"
    end

    def initialize(memory, outer_variable, type, name)
      super(memory, outer_variable, type, self.class.component_name_of(name))
    end
  end

  class AliasVariable < NamedVariable
    def initialize(variable)
      super(variable.binding.memory,
            variable.declarations_and_definitions.first,
            variable.scope)
    end

    private
    def relate_to_memory(memory)
      be_alias_to(memory)
    end
  end

  class VariableTable
    def initialize(memory_pool)
      @memory_pool = memory_pool
      @named_variables = [{}]
      @temporary_variables = [[]]
      @scope_stack = [GlobalScope.new]
    end

    def all_named_variables
      @named_variables.map { |hash| hash.values }.flatten
    end

    def enter_scope
      @named_variables.push({})
      @temporary_variables.push([])
      @scope_stack.push(Scope.new(@scope_stack.size))
    end

    def leave_scope
      @named_variables.pop.each_value do |variable|
        @memory_pool.free(variable.binding.memory)
      end
      @temporary_variables.pop.each do |variable|
        @memory_pool.free(variable.binding.memory)
      end
      @scope_stack.pop
      rollback_all_global_variables_value! if current_scope.global?
    end

    def declare(declaration)
      if variable = lookup(declaration.identifier.value)
        variable.declarations_and_definitions.push(declaration)
        return variable
      end

      # NOTE: External variable may have undefined values.
      define_variable(declaration, declaration.type,
                      allocate_memory(declaration),
                      declaration.type.undefined_value)
    end

    def define(definition, init_value = nil)
      if storage_duration_of(definition) == :static
        # NOTE: Value of the static duration object should be arbitrary because
        #       execution of its accessors are out of order.
        #       So, a value of the initializer should be ignored.
        init_value = definition.type.arbitrary_value
      else
        init_value ||= definition.type.undefined_value
      end

      if variable = lookup(definition.identifier.value)
        if variable.scope == current_scope
          variable.declarations_and_definitions.push(definition)
          variable.assign!(init_value)
          return variable
        end
      end

      # NOTE: Domain of the init-value will be restricted by type's min-max in
      #       define_variable.
      define_variable(definition, definition.type,
                      allocate_memory(definition), init_value)
    end

    def define_temporary(type, init_value)
      memory = @memory_pool.allocate_dynamic(type.aligned_byte_size)

      # NOTE: Domain of the init-value will be restricted by type's min-max in
      #       define_variable.
      define_variable(nil, type, memory, init_value)
    end

    def lookup(name_str)
      @named_variables.reverse_each do |hash|
        if variable = hash[name_str]
          return variable
        end
      end
      nil
    end

    def enter_variables_value_versioning_group
      @named_variables.each do |hash|
        hash.each_value { |variable| variable.enter_value_versioning_group }
      end
    end

    def leave_variables_value_versioning_group(raise_complement)
      @named_variables.each do |hash|
        hash.each_value do |variable|
          variable.leave_value_versioning_group(raise_complement)
        end
      end
    end

    def begin_variables_value_versioning
      @named_variables.each do |hash|
        hash.each_value { |variable| variable.begin_value_versioning }
      end
    end

    def end_variables_value_versioning
      @named_variables.each do |hash|
        hash.each_value { |variable| variable.end_value_versioning }
      end
    end

    def rollback_latest_variables_value_version!
      @named_variables.each do |hash|
        hash.each_value { |variable| variable.rollback_latest_value_version! }
      end
    end

    def storage_duration_of(declaration_or_definition)
      # NOTE: The ISO C99 standard saids;
      #
      # 6.2.2 Linkages of identifiers
      #
      # 1 An identifier declared in different scopes or in the same scope more
      #   than once can be made to refer to the same object or function by a
      #   process called linkage.  There are three kinds of linkage: external,
      #   internal, and none.
      #
      # 3 If the declaration of a file scope identifier for an object or a
      #   function contains the storage-class specifier static, the identifier
      #   has internal linkage.
      #
      # 4 For an identifier declared with the storage-class specifier extern in
      #   a scope in which a prior declaration of that identifier is visible,
      #   if the prior declaration specifies internal or external linkage, the
      #   linkage of the identifier at the later declaration is the same as the
      #   linkage specified at the prior declaration. If no prior declaration
      #   is visible, or if the prior declaration specifies no linkage, then
      #   the identifier has external linkage.
      #
      # 5 If the declaration of an identifier for a function has no
      #   storage-class specifier, its linkage is determined exactly as if it
      #   were declared with the storage-class specifier extern. If the
      #   declaration of an identifier for an object has file scope and no
      #   storage-class specifier, its linkage is external.
      #
      # 6 The following identifiers have no linkage: an identifier declared to
      #   be anything other than an object or a function; an identifier
      #   declared to be a function parameter; a block scope identifier for an
      #   object declared without the storage-class specifier extern.
      #
      # 6.2.4 Storage durations of objects
      #
      # 1 An object has a storage duration that determines its lifetime. There
      #   are three storage durations: static, automatic, and allocated.
      #   Allocated storage is described in 7.20.3.
      #
      # 3 An object whose identifier is declared with external or internal
      #   linkage, or with the storage-class specifier static has static
      #   storage duration. Its lifetime is the entire execution of the program
      #   and its stored value is initialized only once, prior to program
      #   startup.
      #
      # 4 An object whose identifier is declared with no linkage and without
      #   the storage-class specifier static has automatic storage duration.

      sc_specifier = declaration_or_definition.storage_class_specifier
      if sc_specifier &&
          (sc_specifier.type == :EXTERN || sc_specifier.type == :STATIC)
        :static
      else
        current_scope.global? ? :static : :automatic
      end
    end

    private
    def define_variable(declaration_or_definition, type, memory, init_value)
      variable = create_variable(declaration_or_definition, type, memory)
      variable.assign!(init_value)

      if variable.named?
        @named_variables.last[variable.name] = variable
      else
        @temporary_variables.last.push(variable)
      end

      variable
    end

    def allocate_memory(declaration_or_definition)
      byte_size = declaration_or_definition.type.aligned_byte_size
      if storage_duration_of(declaration_or_definition) == :static
        @memory_pool.allocate_static(byte_size)
      else
        @memory_pool.allocate_dynamic(byte_size)
      end
    end

    def create_variable(declaration_or_definition, type, memory)
      if declaration_or_definition
        NamedVariable.new(memory, declaration_or_definition, current_scope)
      else
        TemporaryVariable.new(memory, type, current_scope)
      end
    end

    def current_scope
      @scope_stack.last
    end

    def rollback_all_global_variables_value!
      @named_variables.first.each_value do |variable|
        variable.rollback_all_value_versions!
      end
    end
  end

  # == DESCRIPTION
  # === Function class hierarchy
  #  Function
  #    <--- NamedFunction --- Nameable <<module>>
  #           <--- ExplicitFunction
  #           <--- ImplicitFunction
  #           <--- BuiltinFunction
  #    <--- AnonymousFunction
  class Function < TypedObject
    def initialize(declaration_or_definition, type)
      super(type, declaration_or_definition)
    end

    def name
      subclass_responsibility
    end

    def named?
      false
    end

    def temporary?
      false
    end

    def variable?
      false
    end

    def function?
      true
    end

    def explicit?
      subclass_responsibility
    end

    def implicit?
      !explicit?
    end

    def builtin?
      subclass_responsibility
    end

    def call(interpreter, funcall_expr, args)
      assign_arguments_to_parameters(interpreter, args)
      return_values_via_pointer_arguments(interpreter, funcall_expr, args)

      if type.return_type.function?
        interpreter.temporary_variable
      else
        interpreter.temporary_variable(type.return_type,
                                       type.return_type.return_value)
      end
    end

    def signature
      subclass_responsibility
    end

    private
    def assign_arguments_to_parameters(interpreter, args)
      args.zip(type.parameter_types).each do |(arg, expr), param_type|
        next if param_type && param_type.void?

        if param_type
          case
          when arg.type.array? && param_type.pointer?
            arg_value = interpreter.pointer_value_of(arg)
            converted = interpreter.temporary_variable(param_type, arg_value)
          when arg.type.pointer? && param_type.array?
            converted = interpreter.pointee_of(arg)
          when arg.type.function? && param_type.pointer?
            arg_value = interpreter.pointer_value_of(arg)
            converted = interpreter.temporary_variable(param_type, arg_value)
          when arg.variable? && !arg.type.same_as?(param_type)
            converted =
              interpreter.do_conversion(arg, param_type) ||
              interpreter.temporary_variable(param_type)
          else
            converted = arg
          end
        else
          converted = interpreter.do_default_argument_promotion(arg)
        end

        # NOTE: Value of the argument is referred when the assignment to the
        #       parameter is performed.
        if arg.variable? && !arg.type.array?
          interpreter.notify_variable_value_referred(expr, arg)
        end

        if converted && arg != converted
          interpreter.notify_implicit_conv_performed(expr, arg, converted)
        end
      end
    end

    def return_values_via_pointer_arguments(interpreter, funcall_expr, args)
      args.zip(type.parameter_types).each do |(arg, expr), param_type|
        next if param_type && param_type.void?
        next unless arg.variable? && (arg.type.pointer? || arg.type.array?)

        param_type = param_type.unqualify if param_type

        case
        when param_type.nil? && (arg.type.pointer? || arg.type.array?),
             param_type && param_type.pointer? && !param_type.base_type.const?,
             param_type && param_type.array? && !param_type.base_type.const?
        else
          next
        end

        case
        when arg.type.pointer?
          pointee = interpreter.pointee_of(arg)
          if pointee && pointee.designated_by_lvalue? && pointee.variable?
            sink = pointee
          else
            next
          end
        when arg.type.array?
          sink = arg
        end

        sink.assign!(sink.type.return_value)
        interpreter.notify_variable_value_updated(expr, sink)

        # NOTE: Returning a value via a pointer parameter can be considered as
        #       an evaluation of a statement-expression with a
        #       simple-assignment-expression.
        #       Control will reach to a sequence-point at the end of a full
        #       expression.
        interpreter.notify_sequence_point_reached(
          SequencePoint.new(funcall_expr, false))
      end
    end
  end

  class NamedFunction < Function
    include Nameable

    def initialize(declaration_or_definition, type, name)
      super(declaration_or_definition, type)

      self.name = name
    end

    def designated_by_lvalue?
      true
    end

    def call(interpreter, funcall_expr, args)
      case name
      when "exit", "_exit", "abort"
        BreakEvent.of_return.throw
      when "longjmp", "siglongjmp"
        BreakEvent.of_return.throw
      else
        super
      end
    end

    def signature
      FunctionSignature.new(name, type)
    end
  end

  class ExplicitFunction < NamedFunction
    def initialize(declaration_or_definition)
      super(declaration_or_definition, declaration_or_definition.type,
            declaration_or_definition.identifier.value)
    end

    def explicit?
      true
    end

    def builtin?
      false
    end
  end

  class ImplicitFunction < NamedFunction
    def initialize(type, name)
      super(nil, type, name)
    end

    def explicit?
      false
    end

    def builtin?
      false
    end
  end

  class AnonymousFunction < Function
    def initialize(type)
      super(nil, type)
    end

    def designated_by_lvalue?
      false
    end

    def explicit?
      false
    end

    def builtin?
      false
    end

    def signature
      FunctionSignature.new("__adlint__anonymous_func", type)
    end
  end

  class FunctionTable
    def initialize(memory_pool)
      @memory_pool = memory_pool
      @functions = [{}]
      @scope_stack = [GlobalScope.new]
    end

    def enter_scope
      @functions.push({})
      @scope_stack.push(Scope.new(@scope_stack.size))
    end

    def leave_scope
      @functions.pop.each_value do |function|
        @memory_pool.free(function.binding.memory)
      end
      @scope_stack.pop
    end

    def declare(declaration)
      if function = lookup(declaration.identifier.value)
        function.declarations_and_definitions.push(declaration)
        return function
      end

      define(ExplicitFunction.new(declaration))
    end

    def define(function)
      # NOTE: A function has a starting address in the TEXT segment.
      #       This is ad-hoc implementation, but it's enough for analysis.
      function.bind_to(@memory_pool.allocate_static(0))

      @functions.last[function.name] = function if function.named?
      function
    end

    def lookup(name_str)
      @functions.reverse_each do |hash|
        if function = hash[name_str]
          return function
        end
      end
      nil
    end
  end

  class Memory
    include Bindable

    def initialize(address, byte_size)
      @address = address
      @byte_size = byte_size
      @value = nil
    end

    attr_reader :address
    attr_reader :byte_size

    def static?
      subclass_responsibility
    end

    def dynamic?
      subclass_responsibility
    end

    def read
      @value ? @frozen_value ||= FrozenValue.new(@value) : nil
    end

    def read_modifiable
      @value
    end

    def write(value)
      if @value
        @value.overwrite!(value)
      else
        @value = VersionedValue.new(value)
      end
    end
  end

  class MemoryBlock < Memory
    def initialize(address, byte_size)
      super

      @windows = []
    end

    attr_reader :windows

    def create_window(offset, byte_size)
      window = MemoryWindow.new(self, @address + offset, byte_size)
      window.on_written += method(:handle_written_through_window)
      @windows.push(window)
      window
    end

    def write(value)
      super
      if !@windows.empty? &&
          (@value.array? && value.array?) ||
          (@value.composite? && value.composite?)
        single_value = value.to_single_value
        @windows.zip(single_value.values).each do |window, inner_value|
          window.write(inner_value, false)
        end
      end
    end

    protected
    def create_value_from_windows
      case
      when @value.scalar?
        @value
      when @value.array?
        ArrayValue.new(@windows.map { |w| w.create_value_from_windows })
      when @value.composite?
        CompositeValue.new(@windows.map { |w| w.create_value_from_windows })
      end
    end

    private
    def handle_written_through_window(window)
      value = create_value_from_windows

      if @value
        @value.overwrite!(value)
      else
        @value = VersionedValue.new(value)
      end
    end
  end

  class MemoryWindow < MemoryBlock
    def initialize(owner, address, byte_size)
      super(address, byte_size)
      @owner = owner
    end

    extend Pluggable

    def_plugin :on_written

    def static?
      @owner.static?
    end

    def dynamic?
      @owner.dynamic?
    end

    def write(value, cascade = true)
      super(value)
      on_written.invoke(self) if cascade
    end

    private
    def handle_written_through_window(window)
      super
      on_written.invoke(self)
    end
  end

  class StaticMemoryBlock < MemoryBlock
    def static?
      true
    end

    def dynamic?
      false
    end
  end

  class DynamicMemoryBlock < MemoryBlock
    def static?
      false
    end

    def dynamic?
      true
    end
  end

  class MemoryPool
    def initialize
      @memory_blocks = {}
      @address_ranges = []
      # NOTE: To make room for NULL and controlling expressions.
      @free_address = 10
    end

    def allocate_static(byte_size)
      memory_block = StaticMemoryBlock.new(@free_address, byte_size)
      @free_address += allocating_byte_size(byte_size)
      @memory_blocks[memory_block.address] = memory_block
      @address_ranges.push(memory_block.address...@free_address)
      memory_block
    end

    def allocate_dynamic(byte_size)
      memory_block = DynamicMemoryBlock.new(@free_address, byte_size)
      @free_address += allocating_byte_size(byte_size)
      @memory_blocks[memory_block.address] = memory_block
      @address_ranges.push(memory_block.address...@free_address)
      memory_block
    end

    def free(memory_block)
      @memory_blocks.delete(memory_block.address)
      @address_ranges.reject! { |range| range.include?(memory_block.address) }
    end

    def lookup(address)
      if memory_block = @memory_blocks[address]
        return memory_block
      else
        if addr_range = @address_ranges.find { |r| r.include?(address) }
          memory_block = @memory_blocks[addr_range.first]
          return memory_block.windows.find { |w| w.address == address }
        end
      end
      nil
    end

    private
    def allocating_byte_size(byte_size)
      byte_size == 0 ? 1 : byte_size
    end
  end

end
end
