# Miscellaneous utilities.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

module AdLint #:nodoc:

  module Validation
    def ensure_validity_of(*args)
      @validators ||= []
      attr_names = _attr_names_in(args)
      attr_names.each do |attr_name|
        @validators.push(ObjectValidator.new(attr_name))
      end
      self
    end

    def ensure_presence_of(*args)
      @validators ||= []
      attr_names = _attr_names_in(args)
      attr_names.each do |attr_name|
        @validators.push(PresenceValidator.new(attr_name))
      end
      self
    end

    def ensure_numericality_of(*args)
      @validators ||= []
      attr_names, options = _attr_names_in(args), _options_in(args)
      only_integer = options[:only_integer] ? true : false
      min_value = options[:min]
      max_value = options[:max]
      attr_names.each do |attr_name|
        @validators.push(NumericalityValidator.new(attr_name, only_integer,
                                                   min_value, max_value))
      end
      self
    end

    def ensure_file_presence_of(*args)
      @validators ||= []
      attr_names, options = _attr_names_in(args), _options_in(args)
      allow_nil = options[:allow_nil]
      attr_names.each do |attr_name|
        @validators.push(FilePresenceValidator.new(attr_name, allow_nil))
      end
      self
    end

    def ensure_dirs_presence_of(*args)
      @validators ||= []
      attr_names = _attr_names_in(args)
      attr_names.each do |attr_name|
        @validators.push(DirsPresenceValidator.new(attr_name))
      end
      self
    end

    def ensure_inclusion_of(*args)
      @validators ||= []
      attr_names, options = _attr_names_in(args), _options_in(args)
      values = options[:values] || []
      attr_names.each do |attr_name|
        @validators.push(InclusionValidator.new(attr_name, values))
      end
      self
    end

    def ensure_true_or_false_of(*args)
      @validators ||= []
      attr_names = _attr_names_in(args)
      attr_names.each do |attr_name|
        @validators.push(TrueOrFalseValidator.new(attr_name))
      end
      self
    end

    def ensure_with(*args)
      @validators ||= []
      attr_names, options = _attr_names_in(args), _options_in(args)
      message = options[:message] || "is not valid."
      validator = options[:validator] || lambda { |value| value }
      attr_names.each do |attr_name|
        @validators.push(CustomValidator.new(attr_name, message, validator))
      end
      self
    end

    attr_reader :validators

    def name
      subclass_responsibility
    end

    def valid?
      if self.class.validators
        self.class.validators.map { |validator| validator.execute(self) }.all?
      end
    end

    def errors
      if self.class.validators
        self.class.validators.map { |validator| validator.errors }.flatten
      else
        []
      end
    end

    def self.included(class_or_module)
      class_or_module.extend(self)
    end

    private
    def _attr_names_in(args)
      args.select { |obj| obj.kind_of?(Symbol) }
    end

    def _options_in(args)
      args.find { |obj| obj.kind_of?(Hash) } || {}
    end

    class Validator
      def initialize(attr_name)
        @attr_name = attr_name
        @errors = []
      end

      attr_reader :errors

      def execute(attr_owner)
        subclass_responsibility
      end

      private
      def target_value(attr_owner)
        attr_owner.instance_variable_get("@#{@attr_name}")
      end
    end
    private_constant :Validator

    class ObjectValidator < Validator
      def execute(attr_owner)
        if object = target_value(attr_owner)
          return true if object.valid?
          @errors.concat(object.errors)
        end
        false
      end
    end
    private_constant :ObjectValidator

    class PresenceValidator < Validator
      def execute(attr_owner)
        if target_value(attr_owner).nil?
          @errors.push("`#{attr_owner.name}:#{@attr_name}' is not specified.")
          return false
        end
        true
      end
    end
    private_constant :PresenceValidator

    class NumericalityValidator < PresenceValidator
      def initialize(attr_name, only_integer, min_value, max_value)
        super(attr_name)
        @only_integer = only_integer
        @min_value = min_value
        @max_value = max_value
      end

      def execute(attr_owner)
        return false unless super
        value = target_value(attr_owner)

        case value
        when Numeric
          if @only_integer && !value.integer?
            @errors.push("`#{attr_owner.name}:#{@attr_name}' " +
                         "is not an integer.")
            return false
          end
          if @min_value && value < @min_value
            @errors.push("`#{attr_owner.name}:#{@attr_name}' " +
                         "is not greater than or equal to #{@min_value}.")
            return false
          end
          if @max_value && @max_value < value
            @errors.push("`#{attr_owner.name}:#{@attr_name}' " +
                         "is not less than or equal to #{@max_value}.")
            return false
          end
        else
          @errors.push("`#{attr_owner.name}:#{@attr_name}' " +
                       "is not a numerical value.")
          return false
        end

        true
      end
    end
    private_constant :NumericalityValidator

    class FilePresenceValidator < Validator
      def initialize(attr_name, allow_nil)
        super(attr_name)
        @allow_nil = allow_nil
      end

      def execute(attr_owner)
        value = target_value(attr_owner)

        unless value
          return true if @allow_nil
          @errors.push("`#{attr_owner.name}:#{@attr_name}' is not specified.")
          return false
        end

        unless File.exist?(value) && File.file?(value)
          @errors.push("`#{attr_owner.name}:#{@attr_name}' " +
                       "is non-existent pathname (#{value.to_s}).")
          return false
        end
        true
      end
    end
    private_constant :FilePresenceValidator

    class DirsPresenceValidator < Validator
      def execute(attr_owner)
        value = target_value(attr_owner)

        bad_paths = value.reject { |pathname|
          File.exist?(pathname) && File.directory?(pathname)
        }

        unless bad_paths.empty?
          bad_paths.each do |pathname|
            @errors.push("`#{attr_owner.name}:#{@attr_name}' " +
                         "contains non-existent pathname (#{pathname.to_s}).")
          end
          return false
        end
        true
      end
    end
    private_constant :DirsPresenceValidator

    class InclusionValidator < PresenceValidator
      def initialize(attr_name, values)
        super(attr_name)
        @values = values
      end

      def execute(attr_owner)
        return false unless super
        value = target_value(attr_owner)

        unless @values.include?(value)
          @errors.push("`#{attr_owner.name}:#{@attr_name}' " +
                       "is not one of #{@values.join(", ")}.")
          return false
        end
        true
      end
    end
    private_constant :InclusionValidator

    class TrueOrFalseValidator < PresenceValidator
      def execute(attr_owner)
        return false unless super
        case target_value(attr_owner)
        when TrueClass, FalseClass
          true
        else
          @errors.push("`#{attr_owner.name}:#{@attr_name}' " +
                       "is not a boolean value.")
          false
        end
      end
    end
    private_constant :TrueOrFalseValidator

    class CustomValidator < Validator
      def initialize(attr_name, message, validator)
        super(attr_name)
        @message = message
        @validator = validator
      end

      def execute(attr_owner)
        unless @validator[target_value(attr_owner)]
          @errors.push("`#{attr_owner.name}:#{@attr_name}' " + @message)
          return false
        end
        true
      end
    end
    private_constant :CustomValidator
  end

  module Visitable
    def accept(visitor)
      visitor.__send__(visitor_method_name, self)
    end

    private
    def visitor_method_name
      unless defined?(@visitor_method_name)
        node_name = self.class.name.sub(/\A.*::/, "")
        node_name = node_name.gsub(/([A-Z][a-z])/, "_\\1")
        node_name = node_name.sub(/\A_/, "").tr("A-Z", "a-z")
        @visitor_method_name = "visit_#{node_name}".intern
      end
      @visitor_method_name
    end
  end

  class CsvRecord
    def initialize(array)
      @array = array
    end

    def field_at(index)
      @array.fetch(index)
    end
  end

  module Memoizable
    def memoize(name, *key_indices)
      orig_name = "_orig_#{name}"
      class_eval <<-EOS
        alias_method("#{orig_name}", "#{name}")
        private("#{orig_name}")
      EOS
      if key_indices.empty?
        if instance_method("#{name}").arity == 0
          define_cache_manipulator(name)
          class_eval(<<-EOS, "util.rb", 71)
            define_method("#{name}") do |*args|
              @_#{name}_cache ||= nil
              @_#{name}_cache_forbidden ||= false
              if result = @_#{name}_cache
                @_#{name}_cache_forbidden = false
              else
                result = #{orig_name}(*args)
                if @_#{name}_cache_forbidden
                  @_#{name}_cache_forbidden = false
                else
                  @_#{name}_cache = result
                end
              end
              result
            end
          EOS
        else
          define_cache_manipulator(name, key_indices)
          class_eval(<<-EOS, "util.rb", 90)
            define_method("#{name}") do |*args|
              @_#{name}_cache ||= {}
              @_#{name}_cache_forbidden ||= false
              if result = @_#{name}_cache[args]
                @_#{name}_cache_forbidden = false
              else
                result = #{orig_name}(*args)
                if @_#{name}_cache_forbidden
                  @_#{name}_cache_forbidden = false
                else
                  @_#{name}_cache[args] = result
                end
              end
              result
            end
          EOS
        end
      else
        define_cache_manipulator(name, key_indices)
        class_eval(<<-EOS, "util.rb", 110)
          define_method("#{name}") do |*args|
            @_#{name}_cache ||= {}
            @_#{name}_cache_forbidden ||= false
            key = args.values_at(#{key_indices.join(',')})
            if result = @_#{name}_cache[key]
              @_#{name}_cache_forbidden = false
            else
              result = #{orig_name}(*args)
              if @_#{name}_cache_forbidden
                @_#{name}_cache_forbidden = false
              else
                @_#{name}_cache[key] = result
              end
            end
            result
          end
        EOS
      end
    end

    private
    def define_cache_manipulator(name, key_indices = nil)
      class_eval(<<-EOS, "util.rb", 133)
        define_method("forbid_#{name}_memoizing_once") do |*args|
          @_#{name}_cache_forbidden = true
        end
      EOS
      case
      when key_indices && key_indices.empty?
        class_eval(<<-EOS, "util.rb", 140)
          define_method("forget_#{name}_memo") do |*args|
            @_#{name}_cache.delete(args) if defined?(@_#{name}_cache)
          end
          define_method("clear_#{name}_memo") do |*args|
            @_#{name}_cache.clear if defined?(@_#{name}_cache)
          end
        EOS
      when key_indices
        class_eval(<<-EOS, "util.rb", 149)
          define_method("forget_#{name}_memo") do |*args|
            if defined?(@_#{name}_cache)
              @_#{name}_cache.delete(args.values_at(#{key_indices.join(',')}))
            end
          end
          define_method("clear_#{name}_memo") do |*args|
            @_#{name}_cache.clear if defined?(@_#{name}_cache)
          end
        EOS
      else
        class_eval(<<-EOS, "util.rb", 160)
          define_method("forget_#{name}_memo") do |*args|
            @_#{name}_cache = nil
          end
          define_method("clear_#{name}_memo") do |*args|
            @_#{name}_cache = nil
          end
        EOS
      end
    end
  end

  class Plugin
    def initialize(methods = [])
      @methods = methods
    end

    def +(method)
      Plugin.new(@methods + [method])
    end

    def invoke(*args)
      @methods.each { |method| method.call(*args) }
    end
  end

  module Pluggable
    def def_plugin(event)
      class_eval(<<-EOS, "util.rb", 188)
        define_method("#{event}") do |*args|
          @#{event}_plugin ||= Plugin.new
        end
        define_method("#{event}=") do |*args|
          @#{event}_plugin = args.first
        end
      EOS
    end
  end

end

if $0 == __FILE__
  require_relative "prelude.rb"

  class Foo
    def initialize
      p "foo"
    end

    extend AdLint::Pluggable

    def_plugin :on_initialization

    def run
      on_initialization.invoke(1, 2)
    end
  end

  def bar(a1, a2)
    p a1, a2
  end

  def baz(a1, a2)
    p "baz"
  end

  foo = Foo.lazy_new
  p "foo?"
  foo.on_initialization += method(:bar)
  foo.on_initialization += method(:baz)
  foo.run
end
