/*
-----------------------------------------------------------------
File LUsolve.cpp of ADOL-C version 1.10.0        as of July/01/05
-----------------------------------------------------------------
 
   Example:  * Solution of a linear equation system
               by LU-decomposition of the system matrix without pivoting 
             * application of tapedoc to observe taping of
               the new op_codes for the elementary operations
                  
                     y += x1 * x2;
                     y -= x1 * x2;           
 
   Last changes:
     991210 olvo    debugging 
     991122 olvo    first version 
 
-----------------------------------------------------------------
*/

/****************************************************************************/
/*                                                                 INCLUDES */
#include "LU.h"              // LU-decomposition

/****************************************************************************/
/*                                                             MAIN PROGRAM */
int main() { /*------------------------------------------------------------------------*/
    /* variables */
    const int tag   = 1;                       // tape tag
    const int size  = 5;                       // system size
    const int indep = size*size+size;          // # of indeps
    const int depen = size;                    // # of deps

    double  A[size][size], a1[size], a2[size], // passive variables
    b[size], x[size];
    adouble **AA, *AAp, *Abx;                  // active variables
    double *args = myalloc1(indep);            // arguments
    double **jac = myalloc2(depen,indep);      // the Jacobian
    double *laghessvec = myalloc1(indep);      // Hessian-vector product

    int i,j;


    /*------------------------------------------------------------------------*/
    /* Info */
    fprintf(stdout,"LINEAR SYSTEM SOLVING by "
            "LU-DECOMPOSITION (ADOL-C Example)\n\n");


    /*------------------------------------------------------------------------*/
    /* Allocation und initialization of the system matrix */
    AA  = new adouble*[size];
    AAp = new adouble[size*size];
    for (i=0; i<size; i++) {
        AA[i] = AAp;
        AAp += size;
    }
    Abx = new adouble[size];
    for(i=0; i<size; i++) {
        a1[i] = i*0.25;
        a2[i] = i*0.33;
    }
    for(i=0; i<size; i++) {
        for(j=0; j<size; j++)
            A[i][j] = a1[i]*a2[j];
        A[i][i] += i+1;
        b[i] = -i-1;
    }


    /*------------------------------------------------------------------------*/
    /* Taping the computation of the determinant */
    trace_on(tag);
    /* marking indeps */
    for(i=0; i<size; i++)
        for(j=0; j<size; j++)
            AA[i][j] <<= (args[i*size+j] = A[i][j]);
    for(i=0; i<size; i++)
        Abx[i] <<= (args[size*size+i] = b[i]);
    /* LU-factorization and computation of solution */
    LUfact(size,AA);
    LUsolve(size,AA,Abx);
    /* marking deps */
    for (i=0; i<size; i++)
        Abx[i] >>= x[i];
    trace_off();
    fprintf(stdout," x[0] (original):  %16.4le\n",x[0]);


    /*------------------------------------------------------------------------*/
    /* Recomputation  */
    function(tag,depen,indep,args,x);
    fprintf(stdout," x[0] (from tape): %16.4le\n",x[0]);


    /*------------------------------------------------------------------------*/
    /* Computation of Jacobian */
    jacobian(tag,depen,indep,args,jac);
    fprintf(stdout," Jacobian:\n");
    for (i=0; i<depen; i++) {
        for (j=0; j<indep; j++)
            fprintf(stdout," %14.6le",jac[i][j]);
        fprintf(stdout,"\n");
    }

    /*------------------------------------------------------------------------*/
    /* Computation of Lagrange-Hessian-vector product */
    lagra_hess_vec(tag,depen,indep,args,args,x,laghessvec);
    fprintf(stdout," Part of Lagrange-Hessian-vector product:\n");
    for (i=0; i<size; i++) {
        for (j=0; j<size; j++)
            fprintf(stdout," %14.6le",laghessvec[i*size+j]);
        fprintf(stdout,"\n");
    }


    /*------------------------------------------------------------------------*/
    /* Tape-documentation */
    tape_doc(tag,depen,indep,args,x);


    /*------------------------------------------------------------------------*/
    /* Tape statistics */
    int tape_stats[11];
    tapestats(tag,tape_stats);
    fprintf(stdout,"\n    independents   %d\n",tape_stats[0]);
    fprintf(stdout,"    dependents     %d\n",tape_stats[1]);
    fprintf(stdout,"    operations     %d\n",tape_stats[5]);
    fprintf(stdout,"    buffer size    %d\n",tape_stats[4]);
    fprintf(stdout,"    maxlive        %d\n",tape_stats[2]);
    fprintf(stdout,"    valstack size  %d\n\n",tape_stats[3]);

    /*------------------------------------------------------------------------*/
    /* That's it */
    return 1;
}








