/*
   Project: Adun

   Copyright (C) 2005 Michael Johnston & Jordi Villa-Freixa

   Author: Michael Johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/
//! \file
//! \brief Header file containing definitions. These constants define the untis used in the simulation 

/**
The basic simulation units are
  - Time : femtoseconds
  - Length : angstroms
  - Mass : Atomic Mass Unit
  - Charge : Electron Charge Units
  - Energy : amu.fs²/A²
  - Temperature : Kelvin
**/		 

#ifndef _ADUN_DEFINITIONS
#define _ADUN_DEFINITIONS

#include <fenv.h>
#include <Base/AdMatrix.h>
#include <Foundation/Foundation.h>

//! \brief Conversion factor for simulation units to kcal/mol

#define STCAL 2390.05735688

//! \breif Simulation Units to joules/mol (accurate to nine significant digits)

#define STJMOL 1E7

//! \brief Boltzmanns constant in simulation units

#define KB 8.3144726887E-7

//! \brief Reciprocal of Boltzmanns constant in simulation units

#define KB_1 1202722.09368

//! \brief Conversion factor for kcal/mol to simulation units

#define BOND_FACTOR 4.184E-4 

//! \brief Conversion factor for degrees to radians

#define DEG_2_RAD M_PI/180

//! \brief Conversion factor for m/s to A/fs

#define MS_2_AFS 0.00001

//! \brief Value of sigma for finite difference functions

#define FD_SIGMA 0.000001

//! \brief 4Pi times the permitivity of free space in eu^2.fd^2/amu.A^3 (a lovely unit!)

#define EPSILON 7.19758673876

//! \brief The reciprocal of epsilon

#define PI4EP_R 0.1389354566

//! \brief Conversion factor for kg/m3 to amu/A3

#define DENSITY_FACTOR 6.02214151134E-4

//! \brief Conversion factor for Debye to eA

#define FROMDEBYE 0.208194346224

//! \brief Conversion factor for Hartress to Sim Energy

#define FROMHARTREE 0.262549962955 

typedef double AdMatrixSize;

//typedefs for AdMatrix - we make AdMatrix refer to the DoubleMatrix type

typedef DoubleMatrix AdMatrix;

//constants - First Interaction Types

//! \brief The standard bonded interaction

extern NSString* const HarmonicBond;
 
//! \brief The standard angle interaction
extern NSString* const HarmonicAngle;

//! \brief The standard torsion interaction

extern NSString* const FourierTorsion;

//! \brief The standard electrostatic interaction

extern NSString* const CoulombElectrostatic;

//! \brief pure arithmetic vdw interaction (Enzymix, gromos)

extern NSString* const TypeOneVDWInteraction;

//! \brief geometric plus arithmetic vdw interactions (CHARMM, AMBER)

extern NSString* const TypeTwoVDWInteraction;

//nonbonded parameter types

//! \brief Parameters for TypeOneVDWInteraction

extern NSString* const TypeOneVDWParameters;

//! \brief Parameters for TypeTwoVDWInteraction

extern NSString* const TypeTwoVDWParameters;

//! \brief Partial charges per atom

extern NSString* const PartialCharges;

//! \brief Bitwise OR of all supported floating point exceptions

int floatingPointExceptionMask;

#endif
