/*
   Project: Adun

   Copyright (C) 2005 Michael Johnston & Jordi Villa-Freixa

   Author: Michael Johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/
#ifndef ADUN_NONBONDED_CALCULATOR
#define ADUN_NONBONDED_CALCULATOR

#include "AdunKernel/AdunForceFieldTerm.h"
#include "AdunKernel/AdunDefinitions.h"

/*!
\ingroup Inter
AdNonBondedCalculator is the abstract base class for a strategy hierarchy. An AdForceField
object needs to be able to evaluate the Nonbonded portion of the calculation (defined to be the electrostatic and van der waals terms )
in a number of different ways (nocuttoff , shift, ewald etc).
Each of these ways requires a different algorithm. Concrete instances of AdNonbondedCalculater encapsulate one of these algorithms.
At run time the correct object is instantiated using the Environment settings and AdForceField delgates the non bonded calculation to it. \n\n
AdNonbondedCalculators can either calculate the force simultaneously using evaluateForces or seperately using evaluateVDWForce and evaluateElectrostaticForce.
The results of the force calculation will be written to an internal matrix if useInternalForceMatrix is called. This matrix can then be retrieved using
the Forces method. The matrix is zeroed before each force calculation. useExternalForceMatrix: causes the object to write the force to the provided matrix. 
This matrix will not be zeroe'd before the calculation. The default behaviour is to use an externalMatrix.

*/


@interface AdNonbondedCalculator: AdForceFieldTerm <AdAbstractFactory>
{
	AdMatrix* coordinates;
	AdMatrix* forces;
}
- (void) evaluateElectrostaticForces;
- (void) evaluateVDWForces;
- (void) setInteractions: (NSValue*) interactions;
- (void) setCoordinates: (NSValue*) value;
- (void) useInternalForceMatrix;
- (void) useExternalForceMatrix: (AdMatrix*) forceMatrix;
- (double) VDWPotential;
- (double) ElectrostaticPotential;
@end

#endif
