/*
 *	aegis - project change supervisor
 *	Copyright (C) 2001 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to manipulate sends
 */

#include <ac/ctype.h>
#include <ac/stdio.h>
#include <ac/stdlib.h>

#include <arglex3.h>
#include <change.h>
#include <change/file.h>
#include <error.h>
#include <help.h>
#include <input/file.h>
#include <option.h>
#include <os.h>
#include <output/conten_encod.h>
#include <output/file.h>
#include <output/gzip.h>
#include <output/prefix.h>
#include <output/wrap.h>
#include <progname.h>
#include <project/file.h>
#include <project/file/roll_forward.h>
#include <project.h>
#include <send.h>
#include <str.h>
#include <sub.h>
#include <trace.h>
#include <user.h>
#include <undo.h>


static int len_printable _((string_ty *, int));

static int
len_printable(s, max)
	string_ty	*s;
	int		max;
{
	const char 	*cp;
	int		result;

	/* Intentionally the C locale, not the user's locale */
	for (cp = s->str_text; *cp && isprint((unsigned char)*cp); ++cp)
		;
	result = (cp - s->str_text);
	if (result > max)
		result = max;
	return result;
}


static time_t change_finish_time _((change_ty *));

static time_t
change_finish_time(cp)
	change_ty	*cp;
{
	cstate		cstate_data;
	cstate_history	hp;
	cstate_history_list hlp;

	cstate_data = change_cstate_get(cp);
	hlp = cstate_data->history;
	assert(hlp);
	assert(hlp->length);
	hp = hlp->list[hlp->length - 1];
	assert(hp);
	return hp->when;
}


static void usage _((void));

static void
usage()
{
	char	*progname;

	progname = progname_get();
	fprintf(stderr, "Usage: %s --send [ <option>... ]\n", progname);
	fprintf(stderr, "       %s --help\n", progname);
	exit(1);
}


static string_ty *change_description_get _((change_ty *));

static string_ty *
change_description_get(cp)
	change_ty	*cp;
{
	cstate		cstate_data;

	cstate_data = change_cstate_get(cp);
	return cstate_data->description;
}


static string_ty *project_file_version_path _((project_ty *, fstate_src,
	int *));

static string_ty *
project_file_version_path(pp, src, unlink_p)
	project_ty	*pp;
	fstate_src	src;
	int		*unlink_p;
{
	fstate_src	old_src;
	project_ty	*ppp;
	change_ty	*cp;
	string_ty	*filename;
	history_version	ed;
	fstate_src	reconstruct;

	trace(("project_file_version_path(pp = %08lX, src = %08lX, \
unlink_p = %08lX)\n{\n", (long)pp, (long)src, (long)unlink_p));
	assert(src);
	assert(src->file_name);
	trace(("fn \"%s\"\n", src->file_name->str_text));
	assert(src->edit || src->edit_origin);
	ed = src->edit ? src->edit : src->edit_origin;
	assert(ed->revision);
	trace(("rev \"%s\"\n", ed->revision->str_text));
	if (unlink_p)
		*unlink_p = 0;
	for (ppp = pp; ppp; ppp = ppp->parent)
	{
		cp = project_change_get(ppp);
		old_src = change_file_find(cp, src->file_name);
		if (!old_src)
			continue;
		if (old_src->about_to_be_created_by)
			continue;
		if (old_src->about_to_be_copied_by)
			continue;
		if (old_src->action == file_action_remove)
			continue;
		assert(old_src->edit);
		assert(old_src->edit->revision);
		if (str_equal(old_src->edit->revision, ed->revision))
		{
			filename = change_file_path(cp, src->file_name);
			trace(("return \"%s\";\n", filename->str_text));
			trace(("}\n"));
			return filename;
		}
	}

	filename = os_edit_filename(0);
	os_become_orig();
	undo_unlink_errok(filename);
	os_become_undo();
	if (unlink_p)
		*unlink_p = 1;

	reconstruct = fstate_src_type.alloc();
	reconstruct->file_name = str_copy(src->file_name);
	reconstruct->edit = history_version_copy(ed);

	cp = project_change_get(pp);
	change_run_history_get_command
	(
		cp,
		reconstruct,
		filename,
		user_executing(pp)
	);
	history_version_type.free(reconstruct);
	trace(("return \"%s\";\n", filename->str_text));
	trace(("}\n"));
	return filename;
}


void
send()
{
	string_ty	*original_filename;
	int		original_filename_unlink;
	string_ty	*input_filename;
	int		input_filename_unlink;
	string_ty	*output_file_name;
	output_ty	*t1;
	output_ty	*t2;
	string_ty	*project_name;
	long		change_number;
	char		*branch;
	int		grandparent;
	int		trunk;
	output_ty	*ofp;
	input_ty	*ifp;
	project_ty	*pp;
	change_ty	*cp;
	user_ty		*up;
	cstate		cstate_data;
	string_ty	*output;
	string_ty	*s;
	string_ty	*s2;
	size_t		j;
	content_encoding_t ascii_armor;
	int		compress;
	string_ty	*dev_null;

	branch = 0;
	change_number = 0;
	grandparent = 0;
	project_name = 0;
	trunk = 0;
	output = 0;
	ascii_armor = content_encoding_unset;
	compress = -1;
	arglex();
	while (arglex_token != arglex_token_eoln)
	{
		switch (arglex_token)
		{
		default:
			generic_argument(usage);
			continue;

		case arglex_token_change:
			if (arglex() != arglex_token_number)
				option_needs_number(arglex_token_change, usage);
			/* fall throught... */

		case arglex_token_number:
			if (change_number)
				duplicate_option_by_name(arglex_token_change, usage);
			change_number = arglex_value.alv_number;
			if (!change_number)
				change_number = MAGIC_ZERO;
			else if (change_number < 0)
			{
				sub_context_ty	*scp;

				scp = sub_context_new();
				sub_var_set_long(scp, "Number", change_number);
				fatal_intl(scp, i18n("change $number out of range"));
				/* NOTREACHED */
			}
			break;

		 case arglex_token_project:
			if (arglex() != arglex_token_string)
				option_needs_name(arglex_token_project, usage);
			if (project_name)
				duplicate_option_by_name(arglex_token_project, usage);
			project_name = str_from_c(arglex_value.alv_string);
			break;

		case arglex_token_branch:
			if (branch)
				duplicate_option(usage);
			switch (arglex())
			{
			default:
				option_needs_number(arglex_token_branch, usage);

			case arglex_token_number:
			case arglex_token_string:
				branch = arglex_value.alv_string;
				break;
			}
			break;

		case arglex_token_trunk:
			if (trunk)
				duplicate_option(usage);
			++trunk;
			break;

		case arglex_token_grandparent:
			if (grandparent)
				duplicate_option(usage);
			++grandparent;
			break;

		case arglex_token_output:
			if (output)
				duplicate_option(usage);
			switch (arglex())
			{
			default:
				option_needs_file(arglex_token_output, usage);
				/* NOTREACHED */

			case arglex_token_stdio:
				output = str_from_c("");
				break;

			case arglex_token_string:
				output = str_from_c(arglex_value.alv_string);
				break;
			}
			break;

		case arglex_token_ascii_armor:
			if (ascii_armor != content_encoding_unset)
			{
				duplicate_option_by_name
				(
					arglex_token_content_transfer_encoding,
					usage
				);
			}
			ascii_armor = content_encoding_base64;
			break;

		case arglex_token_ascii_armor_not:
			if (ascii_armor != content_encoding_unset)
			{
				duplicate_option_by_name
				(
					arglex_token_content_transfer_encoding,
					usage
				);
			}
			ascii_armor = content_encoding_none;
			break;

		case arglex_token_content_transfer_encoding:
			if (ascii_armor != content_encoding_unset)
				duplicate_option(usage);
			if (arglex() != arglex_token_string)
			{
				option_needs_string
				(
					arglex_token_content_transfer_encoding,
					usage
				);
			}
			ascii_armor =
				content_encoding_grok(arglex_value.alv_string);
			break;

		case arglex_token_compress:
			if (compress > 0)
				duplicate_option(usage);
			else if (compress >= 0)
			{
				compress_yuck:
				mutually_exclusive_options
				(
					arglex_token_compress,
					arglex_token_compress_not,
					usage
				);
			}
			compress = 1;
			break;

		case arglex_token_compress_not:
			if (compress == 0)
				duplicate_option(usage);
			else if (compress >= 0)
				goto compress_yuck;
			compress = 0;
			break;
		}
		arglex();
	}

	/*
	 * reject illegal combinations of options
	 */
	if (grandparent)
	{
		if (branch)
		{
			mutually_exclusive_options
			(
				arglex_token_branch,
				arglex_token_grandparent,
				usage
			);
		}
		if (trunk)
		{
			mutually_exclusive_options
			(
				arglex_token_trunk,
				arglex_token_grandparent,
				usage
			);
		}
		branch = "..";
	}
	if (trunk)
	{
		if (branch)
		{
			mutually_exclusive_options
			(
				arglex_token_branch,
				arglex_token_trunk,
				usage
			);
		}
		branch = "";
	}

	/*
	 * locate project data
	 */
	if (!project_name)
		project_name = user_default_project();
	pp = project_alloc(project_name);
	str_free(project_name);
	project_bind_existing(pp);

	/*
	 * locate the other branch
	 */
	if (branch)
		pp = project_find_branch(pp, branch);

	/*
	 * locate user data
	 */
	up = user_executing(pp);

	/*
	 * locate change data
	 */
	if (!change_number)
		change_number = user_default_change(up);
	cp = change_alloc(pp, change_number);
	change_bind_existing(cp);

	cstate_data = change_cstate_get(cp);
	switch (cstate_data->state)
	{
	default:
		change_fatal(cp, 0, i18n("bad patch send state"));

	case cstate_state_completed:
		/*
		 * Need to reconstruct the appropriate file histories.
		 */
		project_file_roll_forward(pp, change_finish_time(cp), 0);
		break;

	case cstate_state_being_integrated:
	case cstate_state_awaiting_integration:
	case cstate_state_being_reviewed:
	case cstate_state_being_developed:
		break;
	}

	/* open the output */
	os_become_orig();
	if (ascii_armor == content_encoding_unset)
		ascii_armor = content_encoding_base64;
	if (ascii_armor != content_encoding_none || !compress)
		ofp = output_file_text_open(output);
	else
		ofp = output_file_binary_open(output);
	output_fputs(ofp, "MIME-Version: 1.0\n");
	output_fputs(ofp, "Content-Type: application/aegis-patch\n");
	content_encoding_header(ofp, ascii_armor);
	s = project_name_get(pp);
	s2 = cstate_data->brief_description;
	output_fprintf
	(
		ofp,
		"Subject: %.*s - %.*s\n",
		len_printable(s, 40),
		s->str_text,
		len_printable(s2, 80),
		s2->str_text
	);
	output_fprintf
	(
		ofp,
		"Content-Name: %s.C%3.3ld.patch\n",
		project_name_get(pp)->str_text,
		change_number
	);
	output_fprintf
	(
		ofp,
		"Content-Disposition: attachment; filename=%s.C%3.3ld.patch\n",
		project_name_get(pp)->str_text,
		change_number
	);
	output_fprintf
	(
		ofp,
		"X-Aegis-Project-Name: %s\n",
		project_name_get(pp)->str_text
	);
	output_fprintf
	(
		ofp,
		"X-Aegis-Change-Number: %ld\n",
		cp->number
	);
	output_fputc(ofp, '\n');
	ofp = output_content_encoding(ofp, ascii_armor);
	if (compress)
		ofp = output_gzip(ofp);

	/*
	 * Add the change details to the archive.
	 * This is done as a simple comment.
	 */
	t1 = output_prefix(ofp, 0, "#\t");
	t2 = output_wrap_open(t1, 1, 70);
	os_become_undo();
	s = change_description_get(cp);
	os_become_orig();
	output_fputc(t2, '\n');
	output_put_str(t2, s);
	output_end_of_line(t2);
	output_fputc(t2, '\n');
	output_delete(t2);
	os_become_undo();

	/*
	 * We need a whole bunch of temporary files.
	 */
	output_file_name = os_edit_filename(0);
	os_become_orig();
	undo_unlink_errok(output_file_name);
	os_become_undo();
	dev_null = str_from_c("/dev/null");

	/*
	 * Add each of the relevant source files to the patch.
	 */
	for (j = 0; ; ++j)
	{
		fstate_src	csrc;

		original_filename_unlink = 0;
		input_filename_unlink = 0;

		csrc = change_file_nth(cp, j);
		if (!csrc)
			break;
		trace(("fn = \"%s\"\n", csrc->file_name->str_text));
		if (csrc->usage == file_usage_build)
			continue;
		output_fputs(ofp, "Index: ");
		output_put_str(ofp, csrc->file_name);
		output_fputc(ofp, '\n');

		/*
		 * Find a source file.	Depending on the change state,
		 * it could be in the development directory, or in the
		 * baseline or in history.
		 *
		 * original_filename
		 *	The oldest version of the file.
		 * input_filename
		 *	The youngest version of the file.
		 * input_filename
		 *	Where to write the output.
		 *
		 * These names are taken from the substitutions for
		 * the diff_command.  It's historical.
		 */
		ifp = 0;
		switch (cstate_data->state)
		{
		default:
			assert(0);
			continue;

		case cstate_state_being_developed:
		case cstate_state_being_reviewed:
		case cstate_state_awaiting_integration:
		case cstate_state_being_integrated:
			if (csrc->action == file_action_create)
				original_filename = str_copy(dev_null);
			else
			{
				original_filename =
					project_file_version_path
					(
						pp,
						csrc,
						&original_filename_unlink
					);
			}
			if (csrc->action != file_action_remove)
			{
				input_filename =
					change_file_path(cp, csrc->file_name);
			}
			else
				input_filename = str_copy(dev_null);
			break;

		case cstate_state_completed:
			/*
			 * Both the versions to be diffed come out
			 * of history.
			 */
			switch (csrc->action)
			{
				file_event_list_ty *felp;
				file_event_ty	*fep;
				fstate_src	old_src;

			case file_action_create:
				original_filename = dev_null;
				input_filename = 
					project_file_version_path
					(
						pp,
						csrc,
						&input_filename_unlink
					);
				break;

			case file_action_remove:
				felp =
					project_file_roll_forward_get
					(
						csrc->file_name
					);
				assert(felp);
				assert(felp->length >= 2);

				fep = &felp->item[felp->length - 2]; 
				old_src =
					change_file_find
					(
						fep->cp,
						csrc->file_name
					);
				assert(old_src);
				original_filename =
					project_file_version_path
					(
						pp,
						old_src,
						&original_filename_unlink
					);

				input_filename = str_copy(dev_null);
				break;
			
			default:
				felp =
					project_file_roll_forward_get
					(
						csrc->file_name
					);
				assert(felp);
				assert(felp->length >= 2);

				fep = &felp->item[felp->length - 2]; 
				old_src =
					change_file_find
					(
						fep->cp,
						csrc->file_name
					);
				assert(old_src);
				original_filename =
					project_file_version_path
					(
						pp,
						old_src,
						&original_filename_unlink
					);

				fep = &felp->item[felp->length - 1]; 
				old_src =
					change_file_find
					(
						fep->cp,
						csrc->file_name
					);
				assert(old_src);
				input_filename =
					project_file_version_path
					(
						pp,
						old_src,
						&input_filename_unlink
					);
			}
			break;
		}

		/*
		 * Generate the difference file.
		 */
		change_run_patch_diff_command
		(
			cp,
			up,
			original_filename,
			input_filename,
			output_file_name,
			csrc->file_name
		);

		os_become_orig();
		if (original_filename_unlink)
		{
			os_unlink_errok(original_filename);
			str_free(original_filename);
		}
		if (input_filename_unlink)
		{
			os_unlink_errok(input_filename);
			str_free(input_filename);
		}
		os_become_undo();

		/*
		 * Read the diff into the patch output.
		 */
		trace(("open \"%s\"\n", output_file_name->str_text));
		os_become_orig();
		ifp = input_file_open(output_file_name);
		input_file_unlink_on_close(ifp);
		input_to_output(ifp, ofp);
		os_become_undo();
		str_free(output_file_name);
	}

	/*
	 * Get rid of all the temporary files.
	 */
	os_become_orig();
	os_unlink_errok(output_file_name);
	os_become_undo();
	str_free(output_file_name);
	str_free(dev_null);

	/*
	 * Mark the end of the patch.
	 */
	os_become_orig();
	output_delete(ofp);
	os_become_undo();

	/* clean up and go home */
	change_free(cp);
	project_free(pp);
}
