/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <collectionmanager.h>
#include <kconfig.h>
#include <filefolder.h>
#include <blockdev.h>
#include <qdir.h>
#include <orb.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <transactionserver.h>
#include <remotemailfolder.h>
#include <imap4handler.h>

extern KConfig *GlobalConfig;

CollectionManager::CollectionManager()
{
	// system folders
  addSysContacts();
  addSysGroups();
  addSysNotes();
	addSysMailFolders();
	
  // init the collections
  addMailFolders();
	addBlockFolders();
	addFileFolders();
	
  //addJournals();
  //addScheduler();
  //addTasks();
  //addToDos();
  //addScripts();
	
	mailDict.setAutoDelete(true);
	collectionDict.setAutoDelete(true);
}

CollectionManager::~CollectionManager()
{
  // nothing here
}

void CollectionManager::addMailFolders()
{
  addLocalMailFolders();
  addRemoteMailFolders();
}

void CollectionManager::addLocalMailFolders()
{
	printf("Scanning for the local mail folders...\n");
	
	// scan for additional mail directories
	GlobalConfig->setGroup("Directory");
	QDir mdir(GlobalConfig->readEntry("AetheraDir"));
	mdir.setFilter(QDir::Dirs | QDir::Readable | QDir::Writable);
	mdir.setNameFilter("mailfolder*");
	QStringList mfolders=mdir.entryList();
	for(unsigned i=0;i<mfolders.count();i++)
	{
		// debug
		printf("Adding the local mail folder %s\n", (const char *)(mdir.path()+"/"+mfolders[i]));
		
		mailDict.insert(mfolders[i], new LocalMailFolder(mdir.path()+"/"+mfolders[i]));
	}
	
	printf("Scan complete\n");
}

void CollectionManager::addRemoteMailFolders()
{
	printf("Scanning for the remote mail folders...\n");
	
	// scan for additional mail directories
	GlobalConfig->setGroup("Directory");
	QDir mdir(GlobalConfig->readEntry("AetheraDir"));
	mdir.setFilter(QDir::Dirs | QDir::Readable | QDir::Writable);
	mdir.setNameFilter("imap_*");
	QStringList mfolders=mdir.entryList();
	for(unsigned i=0;i<mfolders.count();i++)
	{
		// debug
		printf("Adding the remote mail folder %s\n", (const char *)(mdir.path()+"/"+mfolders[i]));
		fflush(stdout);
		RemoteMailFolder *mailFolder=new RemoteMailFolder(mdir.path()+"/"+mfolders[i]);
		mailDict.insert(mfolders[i], mailFolder);
		QString key=mailFolder->account()+"/"+mailFolder->mailbox();
		remoteMailDict.insert(key, mailFolder);
	}
	
	printf("Scan complete\n");
}

void CollectionManager::addSysMailFolders()
{
	GlobalConfig->setGroup("Directory");
	QString mdir=GlobalConfig->readEntry("AetheraDir");
	
	// sanity checks
	if(!QFile::exists(mdir+"/sysmail_inbox"))
	{
		printf("Warning: sysmail_inbox does not exists.\n");
		createFolderStructure("sysmail_inbox");
	}
	if(!QFile::exists(mdir+"/sysmail_outbox"))
	{
		printf("Fatal: sysmail_outbox does not exists.\n");
		createFolderStructure("sysmail_outbox");
	}
	if(!QFile::exists(mdir+"/sysmail_drafts"))
	{
		printf("Fatal: sysmail_drafts does not exists.\n");
		createFolderStructure("sysmail_drafts");
	}
	if(!QFile::exists(mdir+"/sysmail_sent"))
	{
		printf("Fatal: sysmail_sent does not exists.\n");
		createFolderStructure("sysmail_sent");
	}
	if(!QFile::exists(mdir+"/sysmail_saved"))
	{
		printf("Fatal: sysmail_saved does not exists.\n");
		createFolderStructure("sysmail_saved");
	}
	if(!QFile::exists(mdir+"/sysmail_deleted"))
	{
		printf("Fatal: sysmail_deleted does not exists.\n");
		createFolderStructure("sysmail_deleted");
	}
	
	// init system inbox
	mailDict.insert("sysmail_inbox", new LocalMailFolder(mdir+"/sysmail_inbox"));
	// debug
	printf("sysmail_inbox added\n");
	
	// init system outbox
	mailDict.insert("sysmail_outbox", new LocalMailFolder(mdir+"/sysmail_outbox"));
	// debug
	printf("sysmail_outbox added\n");
	
	// init system drafts
	mailDict.insert("sysmail_drafts", new LocalMailFolder(mdir+"/sysmail_drafts"));
	// debug
	printf("sysmail_drafts added\n");
	
	// init system sent mail
	mailDict.insert("sysmail_sent", new LocalMailFolder(mdir+"/sysmail_sent"));
	// debug
	printf("sysmail_sent added\n");
	
	// init system saved messages
	mailDict.insert("sysmail_saved", new LocalMailFolder(mdir+"/sysmail_saved"));
	// debug
	printf("sysmail_saved added\n");
	
	// init system deleted messages
	mailDict.insert("sysmail_deleted", new LocalMailFolder(mdir+"/sysmail_deleted"));
	// debug
	printf("sysmail_deleted added\n");
}

void CollectionManager::addSysContacts()
{
	// insert the system contacts folder
	GlobalConfig->setGroup("Directory");
	QString syscontacts=GlobalConfig->readEntry("AetheraDir")+"/syscontacts";
	if(!QFile::exists(syscontacts))
	{
		printf("Fatal: system folder [%s] not found, improper installation.\n", (const char *)syscontacts);
		createBlockStructure("syscontacts");
	}
	collectionDict.insert("syscontacts", new BlockDevice(syscontacts, "i"));
	
	// debug
	printf("Adding system contacts folder %s...\n", (const char *)syscontacts);
}

void CollectionManager::addBlockFolders()
{
	// scan the data dir
	GlobalConfig->setGroup("Directory");
	QString mdir=GlobalConfig->readEntry("AetheraDir");
	
	QDir d(mdir);
	d.setNameFilter("blockdev*");
	d.setFilter(QDir::Files | QDir::Readable | QDir::Writable);
	QStringList files=d.entryList();
	
	for(unsigned i=0;i<files.count();i++)
	{
		// add a new folder
		collectionDict.insert(files[i], new BlockDevice(d.path()+"/"+files[i], "b"));
		
		// debug
		printf("Adding block device %s...\n", (const char *)(d.path()+"/"+files[i]));
	}
}

void CollectionManager::addSysGroups()
{
	// insert the system groups folder
  GlobalConfig->setGroup("Directory");
  QString sysgroups=GlobalConfig->readEntry("AetheraDir")+"/sysgroups";
	if(!QFile::exists(sysgroups))
	{
		printf("Fatal: system folder [%s] not found, improper installation.\n", (const char *)sysgroups);
		createBlockStructure("sysgroups");
	}
  collectionDict.insert("sysgroups", new BlockDevice(sysgroups, "b"));
	
	// debug
	printf("Adding system groups folder %s...\n", (const char *)sysgroups);
}

void CollectionManager::addFileFolders()
{
	// scan the data dir
	GlobalConfig->setGroup("Directory");
	QString mdir=GlobalConfig->readEntry("AetheraDir");
	
	QDir d(mdir);
	d.setNameFilter("filefolder*");
	d.setFilter(QDir::Dirs | QDir::Readable | QDir::Writable);
	QStringList files=d.entryList();
	
	for(unsigned i=0;i<files.count();i++)
	{
		// add a new folder
		collectionDict.insert(files[i], new FileFolder(d.path()+"/"+files[i], "f"));
		
		// debug
		printf("Adding file folder %s...\n", (const char *)(d.path()+"/"+files[i]));
	}
}

void CollectionManager::addSysNotes()
{
	// insert the system notes folder
  GlobalConfig->setGroup("Directory");
  QString sysnotes=GlobalConfig->readEntry("AetheraDir")+"/sysnotes";
	if(!QFile::exists(sysnotes))
	{
		printf("Fatal: system folder [%s] not found, improper installation.\n", (const char *)sysnotes);
		createFolderStructure("sysnotes");
	}
  collectionDict.insert("sysnotes", new FileFolder(sysnotes, "f"));

  printf("Adding system notes folder %s...\n", (const char *)sysnotes);
}

// void CollectionManager::addJournals()
// {
//   GlobalConfig->setGroup("Directory");
//   QString path=GlobalConfig->readEntry("AetheraDir");
//   path+="/Journals/";
//   QDir d(path);
//   d.setFilter(QDir::Files | QDir::Readable | QDir::Writable | QDir::NoSymLinks);
//   d.setNameFilter("*.jf");
//   for(int i=0;i<d.count();i++)
//   {
//     QString jname=d[i];
//     jname.truncate(jname.length()-3);
//     collectionDict.insert("journals://Journals/"+jname, new iFolder(path+d[i], "JEV"));
//
//     printf("Adding journals://Journals/%s ...\n", (const char *)jname);
//   }
// }

// void CollectionManager::addScheduler()
// {
//   GlobalConfig->setGroup("Directory");
//   QString path=GlobalConfig->readEntry("AetheraDir");
//   path+="/Calendar/schedule";
//   collectionDict.insert("scheduler://Scheduler", new iFolder(path, "SEV"));
//
//   printf("Adding scheduler://Scheduler ...\n");
// }
//
// void CollectionManager::addTasks()
// {
//   GlobalConfig->setGroup("Directory");
//   QString path=GlobalConfig->readEntry("AetheraDir");
//   path+="/Calendar/tasks";
//   collectionDict.insert("tasks://Tasks", new iFolder(path, "TEV"));
//
//   printf("Adding tasks://Tasks ...\n");
// }
//
// void CollectionManager::addToDos()
// {
//   GlobalConfig->setGroup("Directory");
//   QString path=GlobalConfig->readEntry("AetheraDir");
//   path+="/Calendar/ToDos";
//   QDir d(path);
//   d.setFilter(QDir::Files | QDir::Readable | QDir::Writable | QDir::NoSymLinks);
//   d.setNameFilter("*.tf");
//   for(int i=0;i<d.count();i++)
//   {
//     QString tname=d[i];
//     tname.truncate(tname.length()-3);
//     collectionDict.insert("todos://ToDos/"+tname, new iFolder(path+d[i], "MTD"));
//
//     printf("Adding todos://ToDos/%s ...\n", (const char *)tname);
//   }
// }
//
// void CollectionManager::addScripts()
// {
// #warning TODO: CollectionManager::addScripts()
// }

QString CollectionManager::reparentObject(QString &url, QString &dest_folder)
{
  DataCollection *t=collectionDict[dest_folder];
  if(!t)
  {
    printf("Warning: the collection dictionary returned null when a valid pointer was expected, bailing out\n");
    return QString::null;
  }
  QString tobj=ObjectRequestBroker::thisInstance()->object(url);
  if(tobj.isEmpty())
  {
    printf("Warning: ORB::object() returned an empty string when a valid object id was expected, bailing out\n");
    return QString::null;
  }
  QString sfolder=ObjectRequestBroker::thisInstance()->folderpath(url);
  DataCollection *s=collectionDict[sfolder];
  if(!s)
  {
    printf("Warning: the collection dictionary returned null when a valid pointer was expected, bailing out\n");
    return QString::null;
  }
  if(s->type!=t->type)
  {
    printf("Warning: reparenting can be done only between storage devices of the same type, bailing out\n");
    return QString::null;
  }
  QString tid=t->addEntry();
  ObjectReference *tref=t->ref(tid), *sref=s->ref(tobj);
  if(!tref || !sref)
  {
    printf("Warning: DataCollection::ref() returned null when a valid pointer was expected, bailing out\n");
    return QString::null;
  }
  tref->rawData=sref->rawData.copy();
//  tref->filename=sref->filename; // hmm... here we should check if the filename does not already exist in our destination
  s->removeEntry(tobj);

  if(s->type==DataCollection::FileStorage)
    TransactionServer::thisInstance()->save(t);
  else
    TransactionServer::thisInstance()->save(tref);

  return dest_folder+"/"+tid;
}

QString CollectionManager::reparentObjectCollection(QString &src_folder, QString &dest_folder)
{
  DataCollection *t=collectionDict[dest_folder], *s=collectionDict[src_folder];
  if(!t || !s)
  {
    printf("Warning: the collection dictionary returned null when a valid pointer was expected, bailing out\n");
    return QString::null;
  }
  if(t->type!=s->type)
  {
    printf("Warning: reparenting can be done only between devices of the same type, bailing out\n");
    return QString::null;
  }
  QStringList ids=s->getEntryList();
  for(unsigned i=0;i<ids.count();i++)
  {
    ObjectReference *sref=s->ref(ids[i]), *tref=t->ref(t->addEntry());
    if(!sref||!tref)
    {
      printf("Warning: at least one object reference was null while reparenting, aborting\n");
      if(s->type==DataCollection::FileStorage) // save what we can, then abort
      {
        TransactionServer::thisInstance()->save(s);
        TransactionServer::thisInstance()->save(t);
      }
      return dest_folder;
    }
    tref->filename=sref->filename;
    tref->rawData=sref->rawData.copy();
    s->removeEntry(ids[i], false);
    if(s->type==DataCollection::FolderStorage)
      TransactionServer::thisInstance()->save(tref);
  }

  if(s->type==DataCollection::FileStorage)
  {
    TransactionServer::thisInstance()->save(s);
    TransactionServer::thisInstance()->save(t);
  }

  return dest_folder;
}

QStringList CollectionManager::reparentObjectSet(QStringList &list, QString &dest_folder)
{
  QStringList ids;
  QString lastfolder;
  unsigned i=0;
  DataCollection *t=collectionDict[dest_folder], *s=0;
  if(!t)
  {
    printf("Warning: the collection dictionary returned null when a valid pointer was expected, bailing out\n");
    return ids;
  }
  list.sort();
  while(i<list.count())
  {
    QString cfolder=ObjectRequestBroker::thisInstance()->folderpath(list[i]);
    if(cfolder!=lastfolder)
    {
      if(t->type==DataCollection::FileStorage && s)
        TransactionServer::thisInstance()->save(s);
      lastfolder=cfolder;
      s=collectionDict[cfolder];
      if(!s)
      {
        printf("Warning: the collection dictionary returned null when a valid pointer was expected, aborting\n");
        return ids;
      }
      if(s->type!=t->type)
      {
        printf("Warning: storage type mismatch occured during reparenting, aborting\n");
        return ids;
      }
    }
    QString tid=t->addEntry();
    ObjectReference *sref=s->ref(ObjectRequestBroker::thisInstance()->object(list[i])), *tref=t->ref(tid);
    if(!sref || !tref)
    {
      printf("Warning: at least one object reference was null while reparenting, aborting\n");
      if(t->type==DataCollection::FileStorage)
      {
        TransactionServer::thisInstance()->save(s);
        TransactionServer::thisInstance()->save(t);
      }
      return ids;
    }
    tref->filename=sref->filename; // we should check if the filename already exists in the destination
    tref->rawData=sref->rawData.copy();
    if(t->type==DataCollection::FolderStorage)
      TransactionServer::thisInstance()->save(tref);
    s->removeEntry(list[i], false);
    ids.append(dest_folder+"/"+tid);
    i++;
  }
  if(t->type==DataCollection::FileStorage)
  {
    TransactionServer::thisInstance()->save(s);
    TransactionServer::thisInstance()->save(t);
  }

  return ids;
}

void CollectionManager::createFolder(QString parent, QString folderPath, QString type, QString viewType)
{
  QString folderName;
	GlobalConfig->setGroup("Directory");
	QString aetheraDir(GlobalConfig->readEntry("AetheraDir"));
	
	if(type=="mail")
	{
		folderName=createFolderStructure("mailfolder");
		if(!folderName.isEmpty())
			mailDict.insert(folderName, new LocalMailFolder(aetheraDir+"/"+folderName));
	}
	else if(type=="imap")
	{
    if(viewType=="mail")
    {
   		QString key=parent+"/"+folderPath;
   		MailFolder *folder=remoteMailDict[key];
      if( !folder )
      {
    		folderName=createFolderStructure("imap_");
    		if(!folderName.isEmpty())
    		{
    	    // create the folder in Aethera server
  		    RemoteMailFolder *mailFolder=new RemoteMailFolder(aetheraDir+"/"+folderName, parent, folderPath);
    			mailDict.insert(folderName, mailFolder);
      		remoteMailDict.insert(key, mailFolder);
    		}
    		else
    		  return;
      }
      else
        folderName=folder->name();
		}
	}
	else if(type=="contacts" || type=="groups")
	{
		folderName=createBlockStructure("blockdev");
		if(!folderName.isEmpty())
			collectionDict.insert(folderName, new BlockDevice(aetheraDir+"/"+folderName, "b"));
	}
	else if(type=="notes")
	{
		folderName=createFolderStructure("filefolder");
		if(!folderName.isEmpty())
			collectionDict.insert(folderName, new FileFolder(aetheraDir+"/"+folderName, "f"));
	}
	
	// Send the create file notification
	QStringList parameters;
	parameters.append(parent);
	parameters.append(folderPath);
	parameters.append(type);
	parameters.append(viewType);
	parameters.append(folderName);
	ServerNotifier::thisInstance()->folderCreated(parameters);
}

QString CollectionManager::createFolder(QStringList paramList)
{
	QString account, mailbox, type, viewType;
	unsigned i=0;
	
	for(QStringList::Iterator it=paramList.begin(); it!=paramList.end(); ++it)
	{
	  switch(i)
	  {
	    case 0:
	      type=(*it);
	      break;
	    case 1:
	      viewType=(*it);
	      break;
	    case 2:
	      account=(*it);
	      break;
	    case 3:
	      mailbox=(*it);
	      break;
	  }
	  i++;
	}
	
	printf("collectionmanager: trying to create folder, dev type %s\n", (const char *)type);
	
	GlobalConfig->setGroup("Directory");
	QString aetheraDir(GlobalConfig->readEntry("AetheraDir"));
	
	if(type=="mail")
	{
		QString folderName=createFolderStructure("mailfolder");
		if(!folderName.isEmpty())
			mailDict.insert(folderName, new LocalMailFolder(aetheraDir+"/"+folderName));
		
		return folderName;
	}
	else if(type=="imap")
	{
	
    if(viewType=="mail")
	    IMAP4Handler::ref()->createFolder(account, mailbox, viewType);
    else
 	    IMAP4Handler::ref()->createFolder(account, mailbox+"/", viewType);
 	
    return QString::null;
	}
	else if(type=="contacts" || type=="groups")
	{
		QString folderName=createBlockStructure("blockdev");
		if(!folderName.isEmpty())
			collectionDict.insert(folderName, new BlockDevice(aetheraDir+"/"+folderName, "b"));
		
		return folderName;
	}
	else if(type=="notes")
	{
		QString folderName=createFolderStructure("filefolder");
		if(!folderName.isEmpty())
			collectionDict.insert(folderName, new FileFolder(aetheraDir+"/"+folderName, "f"));
		
		return folderName;
	}
	else
	{
		return QString::null;
	}
}

QString CollectionManager::createFolderStructure(QString baseName)
{
	GlobalConfig->setGroup("Directory");
	QDir aetheraDir(GlobalConfig->readEntry("AetheraDir"));
	int baseIndex=0;
	
	while(1)
	{
		QString folderName;
		
		if(baseName.left(3)!="sys")
			folderName.sprintf(baseName+"%05d", ++baseIndex);
		else
			folderName=baseName;

		// debug
		printf("collectionmanager: creating folder %s...\n", (const char *)folderName);
				
		if(!aetheraDir.exists(folderName))
		{
			if(aetheraDir.mkdir(folderName))
			{
				return folderName;
			}
			else
			{
   		  printf("\ncollectionmanager: couldn't create a new folder!\n");
				return QString::null;
			}
		}
	}
}

QString CollectionManager::createBlockStructure(QString baseName)
{
	GlobalConfig->setGroup("Directory");
	QDir aetheraDir(GlobalConfig->readEntry("AetheraDir"));
	int baseIndex=0;
	
	while(1)
	{
		QString blockName;
		
		if(baseName.left(3)!="sys")
			blockName.sprintf(baseName+"%05d", ++baseIndex);
		else
			blockName=baseName;

		// debug
		printf("collectionmanager: creating block device %s...\n", (const char *)blockName);
				
		if(!aetheraDir.exists(blockName))
		{
			QFile blockFile(aetheraDir.path()+"/"+blockName);
			if(blockFile.open(IO_WriteOnly))
			{
				blockFile.close();
				return blockName;
			}
			else
			{
				return QString::null;
			}
		}
	}
}

bool CollectionManager::removeFolder(const QString &name)
{

  mailDict.remove(name) || collectionDict.remove(name);

	if(name.find("mailfolder")==0) return removeLocalMailFolder(name);
	else if(name.find("imap_")==0) return removeRemoteMailFolder(name);
	else if(name.find("filefolder")==0) return removeFileFolder(name);
	else if(name.find("blockdev")==0) return removeBlockDevice(name);
	
	return false;
}

void CollectionManager::deleteFolder(QString parent, QString folderPath, QString type, QString serverPath)
{
  if( type=="imap" )
  {
    if( serverPath!="default" )
      removeFolder(serverPath);
  	// Send the create file notification
  	QStringList parameters;
  	parameters.append(parent);
  	parameters.append(folderPath);
  	parameters.append(type);
  	ServerNotifier::thisInstance()->folderDeleted(parameters);
  }
}

bool CollectionManager::removeLocalMailFolder(const QString &name)
{
	GlobalConfig->setGroup("Directory");
	QDir aetheraDir(GlobalConfig->readEntry("AetheraDir"));
	QDir mailFolder(GlobalConfig->readEntry("AetheraDir")+"/"+name);
	
	mailFolder.remove("descriptor");
	mailFolder.remove("index");
	mailFolder.remove("messages");
	
	return aetheraDir.rmdir(mailFolder.dirName());
}

bool CollectionManager::removeRemoteMailFolder(const QString &name)
{
  QString account, mailbox;
	GlobalConfig->setGroup("Directory");
	QDir aetheraDir(GlobalConfig->readEntry("AetheraDir"));
	QDir mailFolder(GlobalConfig->readEntry("AetheraDir")+"/"+name);
	
	// Read the account name and the mailbox name from the config file.
  QFile f(GlobalConfig->readEntry("AetheraDir")+"/"+name+"/"+"config");
  f.open( IO_ReadOnly );
  QDataStream s( &f );
  s >> account;
  s >> mailbox;
  f.close();
  // Remove it from the IMAP server
  IMAP4Handler::ref()->deleteFolder(account, mailbox, name);

	mailFolder.remove("deluids");
	mailFolder.remove("descriptor");
	mailFolder.remove("index");
	mailFolder.remove("headers");
	mailFolder.remove("rcvuids");
	mailFolder.remove("config");
	
	return aetheraDir.rmdir(mailFolder.dirName());
}

bool CollectionManager::removeBlockDevice(const QString &name)
{
	GlobalConfig->setGroup("Directory");
	QDir aetheraDir(GlobalConfig->readEntry("AetheraDir"));
	
	return aetheraDir.remove(name);
}

bool CollectionManager::removeFileFolder(const QString &name)
{
	GlobalConfig->setGroup("Directory");
	QDir aetheraDir(GlobalConfig->readEntry("AetheraDir"));
	QDir fileFolder(GlobalConfig->readEntry("AetheraDir")+"/"+name);

	QStringList fileList=fileFolder.entryList();
	for(unsigned i=0;i<fileList.count();i++)
		fileFolder.remove(fileList[i]);
	
	return aetheraDir.rmdir(fileFolder.dirName());
}

















