/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <eventprocessor.h>
#include <statuscodes.h>
#include <commframe.h>
#include <commands.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qcstring.h>
#include <qdatastream.h>
#include <conversions.h>
#include <serverrmi.h>

EventProcessor *inst;

EventProcessor::EventProcessor()
{
}

EventProcessor::~EventProcessor()
{
}

EventProcessor *EventProcessor::thisInstance()
{
	return inst?inst:inst=new EventProcessor();
}

void EventProcessor::processEvent(CommFrame &inFrame, CommFrame &outFrame)
{
	// check the frame version
	if(inFrame.majorRevNo!=1 || inFrame.minorRevNo!=1)
	{
		// compose the error frame
		outFrame.statusCode=StatusInfo::WrongSoftwareVersion;
		outFrame.completionStatus=StatusInfo::Completed_No;
		outFrame.hasStatus=true;
		
		// debug
		printf("eventprocessor: frame received with wrong software version, discarding\n");
		
		return;
	}
	
	outFrame.statusCode=StatusInfo::Ok;
	outFrame.hasStatus=true;
	
	switch(inFrame.commandCode)
	{
		case MagellanComm::RMICall:
			{
				ServerRMI::ref()->__process(inFrame, outFrame);
				break;
			}
		case MagellanComm::RcptOk:
			{
				outFrame.completionStatus=StatusInfo::Completed_Yes;
				break;
			}
		case MagellanComm::UrlIsFolder:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				outFrame.extraData=_stream(ObjectRequestBroker::thisInstance()->isFolder(_narrowQString(inFrame.extraData)));
				outFrame.completionStatus=StatusInfo::Completed_Yes;
				break;
			}
		case MagellanComm::UrlExists:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				outFrame.extraData=_stream(ObjectRequestBroker::thisInstance()->exists(_narrowQString(inFrame.extraData)));
				outFrame.completionStatus=StatusInfo::Completed_Yes;
				break;
			}
		case MagellanComm::GetParent:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				outFrame.extraData=_stream(ObjectRequestBroker::thisInstance()->parent(_narrowQString(inFrame.extraData)));
				if(ObjectRequestBroker::thisInstance()->lastError().isNull())
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::GetFolder:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				outFrame.extraData=_stream(ObjectRequestBroker::thisInstance()->folder(_narrowQString(inFrame.extraData)));
				if(ObjectRequestBroker::thisInstance()->lastError().isNull())
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::GetChildren:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				outFrame.extraData=_stream(ObjectRequestBroker::thisInstance()->children(_narrowQString(inFrame.extraData)));
				if(ObjectRequestBroker::thisInstance()->lastError().isNull())
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::GetRelated:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				outFrame.extraData=_stream(ObjectRequestBroker::thisInstance()->related(_narrowQString(inFrame.extraData)));
				if(ObjectRequestBroker::thisInstance()->lastError().isNull())
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::GetType:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				outFrame.extraData=_stream(ObjectRequestBroker::thisInstance()->type(_narrowQString(inFrame.extraData)));
				if(ObjectRequestBroker::thisInstance()->lastError().isNull())
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::GetSubtype:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				outFrame.extraData=_stream(ObjectRequestBroker::thisInstance()->subtype(_narrowQString(inFrame.extraData)));
				outFrame.completionStatus=StatusInfo::Completed_Yes;
				if(ObjectRequestBroker::thisInstance()->lastError().isNull())
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::SetRelated:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				QString url;
				QStringList rList;
				QDataStream s(inFrame.extraData, IO_ReadOnly);
				s>>url>>rList;
				if(ObjectRequestBroker::thisInstance()->setRelated(url, rList))
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::SetParent:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				QString url, parentUrl;
				QDataStream s(inFrame.extraData, IO_ReadOnly);
				s>>url>>parentUrl;
				if(ObjectRequestBroker::thisInstance()->setParent(url, parentUrl))
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::GetData:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				if(ObjectRequestBroker::thisInstance()->getObjectData(_narrowQString(inFrame.extraData), outFrame.extraData))
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::GetChildrenData:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				if(ObjectRequestBroker::thisInstance()->getChildrenData(_narrowQString(inFrame.extraData), outFrame.extraData))
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::GetObjectsData:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				if(ObjectRequestBroker::thisInstance()->getObjectData(_narrowQStringList(inFrame.extraData), outFrame.extraData))
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::SetData:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				QString url;
				QByteArray data;
				QDataStream s(inFrame.extraData, IO_ReadOnly);
				s>>url>>data;
				// debug
				if(ObjectRequestBroker::thisInstance()->setObjectData(url, data))
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::CreateObject:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				QString url=ObjectRequestBroker::thisInstance()->createObject(_narrowQString(inFrame.extraData));
				if(url.isNull())
					outFrame.completionStatus=StatusInfo::Completed_No;
				else
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				outFrame.extraData=_stream(url);
				break;
			}
		case MagellanComm::CreateFolder:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				QStringList paramList=_narrowQStringList(inFrame.extraData);
				QString device=ObjectRequestBroker::thisInstance()->createFolder(paramList);
				if(device.isNull())
					outFrame.completionStatus=StatusInfo::Completed_No;
				else
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				outFrame.extraData=_stream(device);
				break;
			}
		case MagellanComm::CopyUrl:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				QString url, dest;
				QDataStream s(inFrame.extraData, IO_ReadOnly);
				s>>url>>dest;
				QString newUrl=ObjectRequestBroker::thisInstance()->copy(url, dest);
				if(newUrl.isNull())
					outFrame.completionStatus=StatusInfo::Completed_No;
				else
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				outFrame.extraData=_stream(newUrl);
				break;
			}
		case MagellanComm::MoveUrl:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				QString url, dest;
				QDataStream s(inFrame.extraData, IO_ReadOnly);
				s>>url>>dest;
				QString newUrl=ObjectRequestBroker::thisInstance()->move(url, dest);
				if(newUrl.isNull())
					outFrame.completionStatus=StatusInfo::Completed_No;
				else
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				outFrame.extraData=_stream(newUrl);
				break;
			}
		case MagellanComm::RemoveUrl:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				if(ObjectRequestBroker::thisInstance()->remove(_narrowQString(inFrame.extraData)))
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::RemoveFolder:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				if(ObjectRequestBroker::thisInstance()->removeFolder(_narrowQStringList(inFrame.extraData)))
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::SendUrl:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				if(ObjectRequestBroker::thisInstance()->send(_narrowQString(inFrame.extraData)))
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		case MagellanComm::RegetLastError:
			{
				outFrame.extraData=_stream(ObjectRequestBroker::thisInstance()->lastError());
				outFrame.completionStatus=StatusInfo::Completed_Yes;
				break;
			}
		case MagellanComm::RegisterNotify:
			{
				ObjectRequestBroker::thisInstance()->clearStatus();
				if(ObjectRequestBroker::thisInstance()->registerNotify(_narrowQString(inFrame.extraData)))
					outFrame.completionStatus=StatusInfo::Completed_Yes;
				else
					outFrame.completionStatus=StatusInfo::Completed_No;
				break;
			}
		default:
			{
				outFrame.completionStatus=StatusInfo::Completed_No;
				outFrame.statusCode=StatusInfo::UnknownExtension;
			}
	}
}		

void EventProcessor::processEvent(CommFrame &inFrame)
{
	// check the frame version
	if(inFrame.majorRevNo!=1 || inFrame.minorRevNo!=1)
	{
		// nothing we can do, since the client would not accept a return frame
		return;
	}
	
	switch(inFrame.commandCode)
	{
		case MagellanComm::RMICall:
			{
				ServerRMI::ref()->__process(inFrame);
				break;
			}
		case MagellanComm::SetRelated:
			{
				QString url;
				QStringList rList;
				QDataStream s(inFrame.extraData, IO_ReadOnly);
				s>>url>>rList;
				ObjectRequestBroker::thisInstance()->setRelated(url, rList);
				break;
			}
		case MagellanComm::SetParent:
			{
				QString url, parentUrl;
				QDataStream s(inFrame.extraData, IO_ReadOnly);
				s>>url>>parentUrl;
				ObjectRequestBroker::thisInstance()->setParent(url, parentUrl);
				break;
			}
		case MagellanComm::SetData:
			{
				QString url;
				QByteArray data;
				QDataStream s(inFrame.extraData, IO_ReadOnly);
				s>>url>>data;
				ObjectRequestBroker::thisInstance()->setObjectData(url, data);
				break;
			}
		case MagellanComm::CreateObject:
			{
				ObjectRequestBroker::thisInstance()->createObject(_narrowQString(inFrame.extraData));
				break;
			}
		case MagellanComm::CreateFolder:
			{
				ObjectRequestBroker::thisInstance()->createFolder(_narrowQStringList(inFrame.extraData));
				break;
			}
		case MagellanComm::CopyUrl:
			{
				QString url, dest;
				QDataStream s(inFrame.extraData, IO_ReadOnly);
				s>>url>>dest;
				ObjectRequestBroker::thisInstance()->copy(url, dest);
				break;
			}
		case MagellanComm::MoveUrl:
			{
				QString url, dest;
				QDataStream s(inFrame.extraData, IO_ReadOnly);
				s>>url>>dest;
				ObjectRequestBroker::thisInstance()->move(url, dest);
				break;
			}
		case MagellanComm::RemoveUrl:
			{
				ObjectRequestBroker::thisInstance()->remove(_narrowQString(inFrame.extraData));
				break;
			}
		case MagellanComm::RemoveFolder:
			{
				ObjectRequestBroker::thisInstance()->removeFolder(_narrowQStringList(inFrame.extraData));
				break;
			}
		case MagellanComm::SendUrl:
			{
				ObjectRequestBroker::thisInstance()->send(_narrowQString(inFrame.extraData));
				break;
			}
		case MagellanComm::RegisterNotify:
			{
				ObjectRequestBroker::thisInstance()->registerNotify(_narrowQString(inFrame.extraData));
				break;
			}
	}
}


