/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <stdlib.h>
#include <lsvariable.h>
#include <lscondition.h>
#include <layoutscript.h>
#include <kprocwrap.h>

#ifdef IDSTRING
#undef IDSTRING
#endif

#define IDSTRING "LayoutScript: "

LayoutScript::LayoutScript(QString text=QString::null)
{
//	variables.setAutoDelete(true);
	setErrorCode(NO_ERROR);
	setText(text);
	splitContent();
	generateVariables();
}

bool LayoutScript::letFun(int lineno, QString line)
{
	QString value;
	QString variable;
	int eqpos=line.find('=');
	value=line.mid(eqpos+1,line.length()-eqpos).stripWhiteSpace();
	variable=line.mid(3,eqpos-3).stripWhiteSpace();
	if(variable[0]!='$' || variable[variable.length()-1]!='$')
	{
		setErrorLine(lineno);
		setErrorCode(YES_ERROR);
		setErrorString("Error: $ symbol missing from variable name at line " +
				itoa(lineno)+".\n["+line+"]");
		return false;
	}
	// getting rid of the $$
	if(variable[variable.length()-1]=='$')
		variable.remove(variable.length()-1,1);
	if(variable[0]=='$') variable.remove(0,1);
	// and the quotes from the text
	if(value[value.length()-1]=='\"') 
	{
		value.remove(value.length()-1,1);
		if(value[0]=='\"') value.remove(0,1);
		else
		{
			setErrorLine(lineno);
			setErrorCode(YES_ERROR);
			setErrorString("Error: (\") missing from string at line " +
					itoa(lineno)+".\n["+line+"]");
			return false;
		}
	}
	setVariable(variable,value);
#ifdef DEBUG_MAIL_GENERATOR
	printf("%s=%s\n",(const char *)variable, (const char *)value);
#endif
}

bool LayoutScript::declareFun(int lineno, QString line)
{
	QString value;
	QString variable;
	int eqpos=line.find(':');
	if(eqpos==-1)
	{
		setErrorCode(YES_WARNING);
		setWarningString("Warning: type specification missing at line " +
				itoa(lineno)+". Assigning to \"String\".\n\t["+line+"]");
	}
	value=line.mid(eqpos+1,line.length()-eqpos).stripWhiteSpace();
	variable=line.mid(7,eqpos-7).stripWhiteSpace();
	if(variable[0]!='$' || variable[variable.length()-1]!='$')
	{
		setErrorLine(lineno);
		setErrorCode(YES_ERROR);
		setErrorString("Error: $ symbol missing from variable name at line " +
				itoa(lineno)+".\n["+line+"]");
		return false;
	}
	// getting rid of the $$
	if(variable[variable.length()-1]=='$')
		variable.remove(variable.length()-1,1);
	if(variable[0]=='$') variable.remove(0,1);
	setVariable(variable,"",value);
	setInputVariable(variable,false);
#ifdef DEBUG_MAIL_GENERATOR
	printf("\"%s\":[%s]\n",(const char *)variable, (const char *)value);
#endif
}

bool LayoutScript::inputFun(int lineno, QString line)
{
	QString value=QString::null;
	QString variable=QString::null;
	int eqpos=line.find('=');
	bool hasDefaultValue=true;
	if(eqpos==-1)
	{
		eqpos=line.length();
		hasDefaultValue=false;
	}
	variable=line.mid(5,eqpos-5).stripWhiteSpace();
	if(hasDefaultValue)
	{
		value=line.mid(eqpos+1,line.length()-eqpos).stripWhiteSpace();
#ifdef DEBUG_MAIL_GENERATOR
		printf(IDSTRING"inputFun: [%s] has default value: <%s>\n",
			(const char *)variable, (const char *)value);
#endif
	}
#ifdef DEBUG_MAIL_GENERATOR
	else
	{
		printf(IDSTRING"inputFun: [%s] has no default value.\n",
			(const char *)variable);
	}
#endif
	if(variable[0]!='$' || variable[variable.length()-1]!='$')
	{
		setErrorLine(lineno);
		setErrorCode(YES_ERROR);
		setErrorString("Error: $ symbol missing from variable name at line " +
				itoa(lineno)+".\n["+line+"]");
		return false;
	}
	// getting rid of the $$
	if(variable[variable.length()-1]=='$')
		variable.remove(variable.length()-1,1);
	if(variable[0]=='$') variable.remove(0,1);
	// and the quotes from the text
	if(value[value.length()-1]=='\"') 
	{
		value.remove(value.length()-1,1);
		if(value[0]=='\"') value.remove(0,1);
		else
		{
			setErrorLine(lineno);
			setErrorCode(YES_ERROR);
			setErrorString("Error: (\") missing from string at line " +
					itoa(lineno)+".\n["+line+"]");
			return false;
		}
	}
	setVariable(variable,value);
	setInputVariable(variable,true);
#ifdef DEBUG_MAIL_GENERATOR
	printf(IDSTRING"inputFun: \"%s\":[%s]\n",(const char *)variable, 
			(const char *)value);
#endif
	return true;
}

bool LayoutScript::printFun(int lineno, QString line)
{
#ifdef DEBUG_MAIL_GENERATOR
	printf(" LINE: %s\n",(const char *)line);
	printf(" RSLT: %s\n",(const char *)result);
#endif
	QString value;
	value=line.mid(5,line.length()-5).stripWhiteSpace();
	// and the quotes from the text
	if(value[value.length()-1]=='\"') 
	{
		value.remove(value.length()-1,1);
		if(value[0]=='\"') value.remove(0,1);
		else
		{
			setErrorLine(lineno);
			setErrorCode(YES_ERROR);
			setErrorString("Error: (\") missing from string at line " +
					itoa(lineno)+".\n["+line+"]");
			return false;
		}
	}
	result+=value;
#ifdef DEBUG_MAIL_GENERATOR
	printf(" RSLT: %s\n",(const char *)result);
#endif
}

bool LayoutScript::includeFun(int lineno, QString line)
{
	QString value;
	value=line.mid(8,line.length()-5).stripWhiteSpace();
	// and the quotes from the text
	value.remove(value.length()-1,1);
	value.remove(0,1);
	FILE *f;
	bool flag=(value.find(".mScript")!=-1);
	f=fopen((const char *)value, "rt");
	if(!f)
	{
		setErrorLine(lineno);
		setErrorCode(YES_ERROR);
		setErrorString("Error: Could not open file \"" + value +"\" " +
				itoa(lineno)+".\n["+line+"]");
		return false;
	}
	char *buffer;
	buffer=new char[201];
	unsigned readsz;
	value="";
	while(!feof(f))
	{
		readsz=fread(buffer,1,200,f);
		buffer[readsz]=0;
		value+=buffer;
	}
	delete buffer;
	fclose(f);
	if(flag)
		internalParse(value);
	else
		result+=value;
}


// ifFun: function for parsing the "if condition" line.
int LayoutScript::ifFun(int lineno, QString line)
{
	QString condition;
	condition=line.mid(2,line.length()-2).stripWhiteSpace();
#ifdef DEBUG_MAIL_GENERATOR
		printf("If: %s\n",(const char *)condition);
#endif
	MailCondition mc;
	return mc.testCondition(condition.simplifyWhiteSpace());
}


bool LayoutScript::generateExec()
{
	static ProcessRun process;
	unsigned position=0;
	int startExec=0, endExec=0;
	while (position<script.length())
	{
		startExec=script.find("stdout(",position,false);
		if(startExec!=-1 && (startExec==0 || script[startExec-1]!='\\'))
		{
			endExec=script.find(')',startExec);
			if(endExec!=-1)
			{
				endExec++;
				QString execLine=script.mid(startExec, endExec-startExec).stripWhiteSpace();
				execLine.remove(0,7);
				execLine.remove(execLine.length()-1,1);
				execLine=execLine.stripWhiteSpace();
#ifdef DEBUG_MAIL_GENERATOR
				printf("Executing: \"%s\"...\n",(const char *)execLine);
#endif
				process.reset();
				process.run(execLine);
				execLine=process.getOutput();
				for(int i=0;i>-1;)
				{
					i=execLine.find('\n',i+1);
					if(i!=-1)
						execLine.replace(i,1,"\\n");
				}
				script.remove(startExec, endExec-startExec);
				script.insert(startExec,execLine);
#ifdef DEBUG_MAIL_GENERATOR
				printf("[%s]\n",(const char *)script);
				printf(" %d %d\n",startExec,endExec);
#endif
				position=endExec+1;
			}
				else	position=script.length();
		}
			else	position++;
	} //while

	/* The same thing for stderr */

	position=0;
	while (position<script.length())
	{
		startExec=script.find("stderr(",position,false);
		if(startExec!=-1 && (startExec==0 || script[startExec-1]!='\\'))
		{
			endExec=script.find(')',startExec);
			if(endExec!=-1)
			{
				endExec++;
				QString execLine=script.mid(startExec, endExec-startExec).stripWhiteSpace();
				execLine.remove(0,7);
				execLine.remove(execLine.length()-1,1);
				execLine=execLine.stripWhiteSpace();
#ifdef DEBUG_MAIL_GENERATOR
				printf("Executing: \"%s\"...\n",(const char *)execLine);
#endif
				process.reset();
				process.run(execLine);
				execLine=process.getError();
				for(int i=0;i>-1;)
				{
					i=execLine.find('\n',i+1);
					if(i!=-1)
						execLine.replace(i,1,"\\n");
				}
				script.replace(startExec, endExec-startExec,execLine);
#ifdef DEBUG_MAIL_GENERATOR
				printf("[%s]\n",(const char *)script);
				printf(" %d %d\n",startExec,endExec);
#endif
				position=endExec+1;
			}
				else	position=script.length();
		}
			else	position++;
	} //while

//#ifdef DEBUG_MAIL_GENERATOR
	printf("<[%s]>\n",(const char *) script);
//#endif
	return true;
}




QString LayoutScript::internalParse(QString inputText)
{
#ifdef DEBUG_MAIL_GENERATOR
		printf("internal: %s\n",(const char *)inputText);
#endif
	if(getErrorCode() & YES_ERROR)
		return QString::null;
	QString line;
	unsigned pos=0;
	unsigned lineno=1;
	int id;
	bool flag;
	line=getLine(inputText,0);
	while((unsigned)pos<=inputText.length())
	{
		id=0;
		if(!id && line.find("print",0,false)==0)
			id=PRINT;
		if(!id && line.find("if",0,false)==0)
			id=IF;
		if(!id && line.find("include",0,false)==0)
			id=INCLUDE;
		flag=true;
		switch(id)
		{
			case INCLUDE:
				flag=includeFun(lineno, line);
				break;
			case PRINT:
				flag=printFun(lineno, line);
				break;
			case IF:
			{
				bool result=ifFun(lineno, line);
				if(result==true)
				{
					QString newBlock="";
					QString myLine=line;
					bool flag=true;
					while(flag)
					{
						pos+=myLine.length()+1;
						myLine=getLine(inputText,pos);
						if((myLine.find("else",0,false)==0) ||
								(myLine.find("endif",0,false)==0) || 
								(pos>=inputText.length()))
						{
							flag=false;
						}
						else
						{
							newBlock+=myLine+"\n";
						}
					} //while
					// and now skip the rest
					if(myLine.find("else",0,false)==0)
					{
						flag=true;
						while(flag)
						{
							pos+=myLine.length()+1;
							myLine=getLine(inputText,pos);
							if(myLine.find("else",0,false)==0 || 
								(myLine.find("endif",0,false)==0) || 
								(pos>=inputText.length()))
								flag=false;
						}
					}
					internalParse(newBlock);
					line=myLine;
				}
				else
				{
				// i have to skip to else/endif
					QString newBlock="";
					QString myLine;
					bool flag=true;
					while(flag)
					{
						pos+=myLine.length()+1;
						myLine=getLine(inputText,pos);
						if(myLine.find("else",0,false)==0 || 
							(myLine.find("endif",0,false)==0) || 
							(pos>=inputText.length()))
							flag=false;
					}
					if(myLine.find("else",0,false)==0)
					{
						while(flag)
						{
							pos+=myLine.length()+1;
							myLine=getLine(inputText,pos);
							if((myLine.find("else",0,false)==0) ||
									(myLine.find("endif",0,false)==0) || 
									(pos>=inputText.length()))
								flag=false;
							else
							newBlock+=myLine+"\n";
						} //while
						internalParse(newBlock);
					}
					line=myLine;
				}
			}
			break;
			default:
				break;
		}
		if(getErrorCode() & YES_ERROR)
			return getErrorString();
		if(!flag) return line;
		pos+=line.length()+1;
		line=getLine(inputText,pos);
		lineno++;
#ifdef DEBUG_MAIL_GENERATOR
			printf("------------------------------<><><>\n");
			printf(" %s\n",(const char *)result);
			printf("------------------------------<><><>\n");
#endif
	}
	
	return QString::null;
}


///////////////////////// From old mlparser.cpp

void LayoutScript::splitContent()
{
#ifdef DEBUG_MAIL_GENERATOR
	printf("content:\n %s\n",(const char *)content);
#endif
	script="";
	QString line;
	int pos=0;
	unsigned lineno=1;
	line=getLine(content,0);
	while((unsigned)pos<=content.length())
	{
		if(!line.isEmpty())
		{
			bool myflag=true;
			if(line.find("let",0,false)==0)
			{
				letFun(lineno, line);
				myflag=false;
			}
			if(myflag && line.find("declare",0,false)==0)
			{
				declareFun(lineno, line);
				myflag=false;
			}
			if(myflag && line.find("input",0,false)==0)
			{
				inputFun(lineno, line);
				myflag=false;
			}
			if(myflag && !line.isEmpty())
				script+=line+"\n";
		}
		lineno ++;
		pos+=line.length()+1;
		line=getLine(content,pos);
	}
#ifdef DEBUG_MAIL_GENERATOR
	printf("LETS:\n%s\nInput:\n%s\nRest:\n%s\n",
		(const char *)attributions,
		(const char *)input,
		(const char *)content);
#endif
}




/**************************************************************************
 * THESE ARE VERIFIED FUNCTIONS
 **************************************************************************/
 QString LayoutScript::getLine(QString text, int position)
{
	QString line;
	int pos;
	pos=text.find('\n',position);
	if(pos==-1) pos=text.length();
	line=text.mid(position,pos-position);
	return line.stripWhiteSpace();
}
QString LayoutScript::unfold(QString foldedText)
{
	QString unfolded;
	QString unfoldedLine;
	QString line=getLine(foldedText);
	QString strippedLine;
	int pos=0;
	bool flag;
	while((unsigned)pos<=foldedText.length())
	{
		pos+=line.length()+1;
		strippedLine=line.stripWhiteSpace();
		line=getLine(foldedText,pos);
		flag=(strippedLine[strippedLine.length()-1]=='\\');
		if(flag)
		{
			unfoldedLine+=strippedLine;
			unfoldedLine.remove(unfoldedLine.length()-1,1);
#ifdef DEBUG_MAIL_GENERATOR
				printf("[%s]\n", (const char *)unfoldedLine);
#endif
		}
		else
		{
			unfolded+="\n"+unfoldedLine;
			unfoldedLine="";
			unfolded+=strippedLine;
		}
	}
	return unfolded;
}

void LayoutScript::generateVariables()
{
	unsigned pos=(unsigned)-1;
	int dollars=0;
	unsigned oldpos=0;
	while((pos+1)<(content.length()+1))
	{
		oldpos=pos;
		bool flag=true;
		while(flag && (pos+1)<(content.length()+1))
		{
			pos=content.find('$',pos+1);
			if(pos==0)
			{
				flag=false;
				dollars++;
			}
			else
				if(content[pos-1]!='\\')
				{
					flag=false;
					dollars++;
				}
			if(pos==(unsigned)-1)
				return;
		}
		if(!(dollars%2) && (dollars!=0))
		{
			QString varName=content.mid(oldpos+1,pos-oldpos-1);
			bool exists=false;
			for(unsigned i=0;i<variables.count();i++)
				if(variables[i]->match(varName)) exists=true;
			if(!exists)
				variables.append(new Variable(varName));
#ifdef DEBUG_MAIL_GENERATOR
			else
				printf("variable \'%s\' already in the list.\n",(const char *)varName);
			printf("<%s>\n",(const char *)varName);
#endif
		}
	}
}


QValueList<QString> LayoutScript::getVariablesNames()
{
	QValueList<QString> names;
	for(unsigned i=0;i<variables.count();i++)
	{
		names+=variables[i]->getName();
	}
	return names;
}

void LayoutScript::setInputVariable(QString name, bool in)
{
	if(name.isEmpty()) return;
	for(unsigned i=0;i<variables.count();i++)
	{
		if(variables[i]->getName().find(name,0,false)==0)
		{
#ifdef DEBUG_MAIL_GENERATOR
			printf(IDSTRING"setInputVariable: idx=%d %d\n",i, in);
#endif
			variables[i]->setInput(in);
			return;
		}
	}
#ifdef DEBUG_MAIL_GENERATOR
	printf(IDSTRING"setInputVariable: Warning: variable \"%s\" "
	"type set as default \"string\"\n",(const char *)name);
#endif
	Variable *v;
	v=new Variable(name, QString::null,"String");
	v->setInput(in);
	variables.append(v);
}

void LayoutScript::setVariable(QString name, QString value, QString type)
{
	if(name.isEmpty()) return;
	for(unsigned i=0;i<variables.count();i++)
	{
		if(variables[i]->getName().find(name,0,false)==0)
		{
			if(!value.isEmpty()) variables[i]->setValue(value);
			if(!type.isEmpty()) variables[i]->setType(type);
			return;
		}
	}
#ifdef DEBUG_MAIL_GENERATOR
	printf(IDSTRING"setVariable: Warning: variable \"%s\" not used by "
	"the script\n",(const char *)name);
#endif
	variables.append(new Variable(name,value,type));
}

Variable *LayoutScript::getVariablePtr(QString name)
{
	for(unsigned i=0;i<variables.count();i++)
	{
		if(variables[i]->getName().find(name,0,false)==0)
		{
			return variables[i];
		}
	}
	return &defaultVar;
}

Variable &LayoutScript::operator[](QString name)
{
	return *getVariablePtr(name);
}

void LayoutScript::multReplace(QString &str, QString oldS, QString newS)
{
	int pos=0;
	while (pos!=-1)
	{
		pos=str.find(oldS,pos);
		if(pos!=-1)
		{
			str.replace(pos,oldS.length(),newS);
			pos+=newS.length()-oldS.length()+1;
		}
	}
}

void LayoutScript::replaceWhiteSpaces()
{
	multReplace(result,"\\n","\n");
	multReplace(result,"\\t","\t");
	multReplace(result,"\\r","\r");
	multReplace(result,"\\\'","\'");
	multReplace(result,"\\\"","\"");
	multReplace(result,"\\\%","\%");
	multReplace(result,"\\$","$");
}

void LayoutScript::replaceVariables(bool all)
{
	content=script;
	unsigned pos=(unsigned)-1;
	int dollars=0;
	unsigned oldpos=0;
	unsigned lineno=1;
	while((pos+1)<(content.length()+1))
	{
		oldpos=pos;
		bool flag=true;
		while(flag && (pos+1)<(content.length()+1))
		{
			pos=content.find('$',pos+1);
			if(pos==0)
			{
				flag=false;
				dollars++;
			}
			else
				if(content[pos-1]!='\\')
				{
					flag=false;
					dollars++;
				}
			if(pos==(unsigned)-1)
				return;
		}
		if(!(dollars%2) && (dollars!=0))
		{
			QString varName=content.mid(oldpos+1,pos-oldpos-1);
			if(all || (!all && !getVariablePtr(varName)->getValue().isEmpty()))
			{
				if(getVariablePtr(varName)->getValue().isEmpty())
				{
					setErrorLine(lineno);
					setErrorCode(YES_WARNING);
					setWarningString("Warning: Empty value " +
							getVariablePtr(varName)->getName());
				}
				content.replace(oldpos,
						pos-oldpos+1,
						getVariablePtr(varName)->getValue());
			}
		}
	}
}

QValueList<Variable *> LayoutScript::getInputVariables()
{
	QValueList<Variable *> result;
	for(unsigned i=0;i<variables.count();i++)
	{
		if(variables[i]->getInput()==true) 
			result.append(variables[i]);
	}
#ifdef DEBUG_MAIL_GENERATOR
	printf(IDSTRING"getInputVariables: %d\n",result.count());
#endif
	return result;
}

void LayoutScript::showValues()
{
	for(unsigned i=0;i<variables.count();i++)
		printf("%s:%s=[%s](%s)\n",(const char *)variables[i]->getName(),
		(const char *)variables[i]->getType(),
		(const char *)variables[i]->getValue(),
		(variables[i]->getInput())?"input":"script");
}





